/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details about a composite alarm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CompositeAlarm implements SdkPojo, Serializable, ToCopyableBuilder<CompositeAlarm.Builder, CompositeAlarm> {
    private static final SdkField<Boolean> ACTIONS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ActionsEnabled").getter(getter(CompositeAlarm::actionsEnabled)).setter(setter(Builder::actionsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsEnabled").build()).build();

    private static final SdkField<List<String>> ALARM_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AlarmActions")
            .getter(getter(CompositeAlarm::alarmActions))
            .setter(setter(Builder::alarmActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ALARM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmArn").getter(getter(CompositeAlarm::alarmArn)).setter(setter(Builder::alarmArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmArn").build()).build();

    private static final SdkField<Instant> ALARM_CONFIGURATION_UPDATED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("AlarmConfigurationUpdatedTimestamp")
            .getter(getter(CompositeAlarm::alarmConfigurationUpdatedTimestamp))
            .setter(setter(Builder::alarmConfigurationUpdatedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmConfigurationUpdatedTimestamp")
                    .build()).build();

    private static final SdkField<String> ALARM_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmDescription").getter(getter(CompositeAlarm::alarmDescription))
            .setter(setter(Builder::alarmDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmDescription").build()).build();

    private static final SdkField<String> ALARM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmName").getter(getter(CompositeAlarm::alarmName)).setter(setter(Builder::alarmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmName").build()).build();

    private static final SdkField<String> ALARM_RULE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmRule").getter(getter(CompositeAlarm::alarmRule)).setter(setter(Builder::alarmRule))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmRule").build()).build();

    private static final SdkField<List<String>> INSUFFICIENT_DATA_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InsufficientDataActions")
            .getter(getter(CompositeAlarm::insufficientDataActions))
            .setter(setter(Builder::insufficientDataActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InsufficientDataActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> OK_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OKActions")
            .getter(getter(CompositeAlarm::okActions))
            .setter(setter(Builder::okActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OKActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateReason").getter(getter(CompositeAlarm::stateReason)).setter(setter(Builder::stateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateReason").build()).build();

    private static final SdkField<String> STATE_REASON_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateReasonData").getter(getter(CompositeAlarm::stateReasonData))
            .setter(setter(Builder::stateReasonData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateReasonData").build()).build();

    private static final SdkField<Instant> STATE_UPDATED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StateUpdatedTimestamp").getter(getter(CompositeAlarm::stateUpdatedTimestamp))
            .setter(setter(Builder::stateUpdatedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateUpdatedTimestamp").build())
            .build();

    private static final SdkField<String> STATE_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateValue").getter(getter(CompositeAlarm::stateValueAsString)).setter(setter(Builder::stateValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateValue").build()).build();

    private static final SdkField<Instant> STATE_TRANSITIONED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StateTransitionedTimestamp")
            .getter(getter(CompositeAlarm::stateTransitionedTimestamp))
            .setter(setter(Builder::stateTransitionedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateTransitionedTimestamp").build())
            .build();

    private static final SdkField<String> ACTIONS_SUPPRESSED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActionsSuppressedBy").getter(getter(CompositeAlarm::actionsSuppressedByAsString))
            .setter(setter(Builder::actionsSuppressedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsSuppressedBy").build())
            .build();

    private static final SdkField<String> ACTIONS_SUPPRESSED_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActionsSuppressedReason").getter(getter(CompositeAlarm::actionsSuppressedReason))
            .setter(setter(Builder::actionsSuppressedReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsSuppressedReason").build())
            .build();

    private static final SdkField<String> ACTIONS_SUPPRESSOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActionsSuppressor").getter(getter(CompositeAlarm::actionsSuppressor))
            .setter(setter(Builder::actionsSuppressor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsSuppressor").build()).build();

    private static final SdkField<Integer> ACTIONS_SUPPRESSOR_WAIT_PERIOD_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ActionsSuppressorWaitPeriod")
            .getter(getter(CompositeAlarm::actionsSuppressorWaitPeriod))
            .setter(setter(Builder::actionsSuppressorWaitPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsSuppressorWaitPeriod")
                    .build()).build();

    private static final SdkField<Integer> ACTIONS_SUPPRESSOR_EXTENSION_PERIOD_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("ActionsSuppressorExtensionPeriod")
            .getter(getter(CompositeAlarm::actionsSuppressorExtensionPeriod))
            .setter(setter(Builder::actionsSuppressorExtensionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActionsSuppressorExtensionPeriod")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIONS_ENABLED_FIELD,
            ALARM_ACTIONS_FIELD, ALARM_ARN_FIELD, ALARM_CONFIGURATION_UPDATED_TIMESTAMP_FIELD, ALARM_DESCRIPTION_FIELD,
            ALARM_NAME_FIELD, ALARM_RULE_FIELD, INSUFFICIENT_DATA_ACTIONS_FIELD, OK_ACTIONS_FIELD, STATE_REASON_FIELD,
            STATE_REASON_DATA_FIELD, STATE_UPDATED_TIMESTAMP_FIELD, STATE_VALUE_FIELD, STATE_TRANSITIONED_TIMESTAMP_FIELD,
            ACTIONS_SUPPRESSED_BY_FIELD, ACTIONS_SUPPRESSED_REASON_FIELD, ACTIONS_SUPPRESSOR_FIELD,
            ACTIONS_SUPPRESSOR_WAIT_PERIOD_FIELD, ACTIONS_SUPPRESSOR_EXTENSION_PERIOD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("ActionsEnabled", ACTIONS_ENABLED_FIELD);
                    put("AlarmActions", ALARM_ACTIONS_FIELD);
                    put("AlarmArn", ALARM_ARN_FIELD);
                    put("AlarmConfigurationUpdatedTimestamp", ALARM_CONFIGURATION_UPDATED_TIMESTAMP_FIELD);
                    put("AlarmDescription", ALARM_DESCRIPTION_FIELD);
                    put("AlarmName", ALARM_NAME_FIELD);
                    put("AlarmRule", ALARM_RULE_FIELD);
                    put("InsufficientDataActions", INSUFFICIENT_DATA_ACTIONS_FIELD);
                    put("OKActions", OK_ACTIONS_FIELD);
                    put("StateReason", STATE_REASON_FIELD);
                    put("StateReasonData", STATE_REASON_DATA_FIELD);
                    put("StateUpdatedTimestamp", STATE_UPDATED_TIMESTAMP_FIELD);
                    put("StateValue", STATE_VALUE_FIELD);
                    put("StateTransitionedTimestamp", STATE_TRANSITIONED_TIMESTAMP_FIELD);
                    put("ActionsSuppressedBy", ACTIONS_SUPPRESSED_BY_FIELD);
                    put("ActionsSuppressedReason", ACTIONS_SUPPRESSED_REASON_FIELD);
                    put("ActionsSuppressor", ACTIONS_SUPPRESSOR_FIELD);
                    put("ActionsSuppressorWaitPeriod", ACTIONS_SUPPRESSOR_WAIT_PERIOD_FIELD);
                    put("ActionsSuppressorExtensionPeriod", ACTIONS_SUPPRESSOR_EXTENSION_PERIOD_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Boolean actionsEnabled;

    private final List<String> alarmActions;

    private final String alarmArn;

    private final Instant alarmConfigurationUpdatedTimestamp;

    private final String alarmDescription;

    private final String alarmName;

    private final String alarmRule;

    private final List<String> insufficientDataActions;

    private final List<String> okActions;

    private final String stateReason;

    private final String stateReasonData;

    private final Instant stateUpdatedTimestamp;

    private final String stateValue;

    private final Instant stateTransitionedTimestamp;

    private final String actionsSuppressedBy;

    private final String actionsSuppressedReason;

    private final String actionsSuppressor;

    private final Integer actionsSuppressorWaitPeriod;

    private final Integer actionsSuppressorExtensionPeriod;

    private CompositeAlarm(BuilderImpl builder) {
        this.actionsEnabled = builder.actionsEnabled;
        this.alarmActions = builder.alarmActions;
        this.alarmArn = builder.alarmArn;
        this.alarmConfigurationUpdatedTimestamp = builder.alarmConfigurationUpdatedTimestamp;
        this.alarmDescription = builder.alarmDescription;
        this.alarmName = builder.alarmName;
        this.alarmRule = builder.alarmRule;
        this.insufficientDataActions = builder.insufficientDataActions;
        this.okActions = builder.okActions;
        this.stateReason = builder.stateReason;
        this.stateReasonData = builder.stateReasonData;
        this.stateUpdatedTimestamp = builder.stateUpdatedTimestamp;
        this.stateValue = builder.stateValue;
        this.stateTransitionedTimestamp = builder.stateTransitionedTimestamp;
        this.actionsSuppressedBy = builder.actionsSuppressedBy;
        this.actionsSuppressedReason = builder.actionsSuppressedReason;
        this.actionsSuppressor = builder.actionsSuppressor;
        this.actionsSuppressorWaitPeriod = builder.actionsSuppressorWaitPeriod;
        this.actionsSuppressorExtensionPeriod = builder.actionsSuppressorExtensionPeriod;
    }

    /**
     * <p>
     * Indicates whether actions should be executed during any changes to the alarm state.
     * </p>
     * 
     * @return Indicates whether actions should be executed during any changes to the alarm state.
     */
    public final Boolean actionsEnabled() {
        return actionsEnabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the AlarmActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAlarmActions() {
        return alarmActions != null && !(alarmActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The actions to execute when this alarm transitions to the ALARM state from any other state. Each action is
     * specified as an Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlarmActions} method.
     * </p>
     * 
     * @return The actions to execute when this alarm transitions to the ALARM state from any other state. Each action
     *         is specified as an Amazon Resource Name (ARN).
     */
    public final List<String> alarmActions() {
        return alarmActions;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the alarm.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the alarm.
     */
    public final String alarmArn() {
        return alarmArn;
    }

    /**
     * <p>
     * The time stamp of the last update to the alarm configuration.
     * </p>
     * 
     * @return The time stamp of the last update to the alarm configuration.
     */
    public final Instant alarmConfigurationUpdatedTimestamp() {
        return alarmConfigurationUpdatedTimestamp;
    }

    /**
     * <p>
     * The description of the alarm.
     * </p>
     * 
     * @return The description of the alarm.
     */
    public final String alarmDescription() {
        return alarmDescription;
    }

    /**
     * <p>
     * The name of the alarm.
     * </p>
     * 
     * @return The name of the alarm.
     */
    public final String alarmName() {
        return alarmName;
    }

    /**
     * <p>
     * The rule that this alarm uses to evaluate its alarm state.
     * </p>
     * 
     * @return The rule that this alarm uses to evaluate its alarm state.
     */
    public final String alarmRule() {
        return alarmRule;
    }

    /**
     * For responses, this returns true if the service returned a value for the InsufficientDataActions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInsufficientDataActions() {
        return insufficientDataActions != null && !(insufficientDataActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The actions to execute when this alarm transitions to the INSUFFICIENT_DATA state from any other state. Each
     * action is specified as an Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInsufficientDataActions} method.
     * </p>
     * 
     * @return The actions to execute when this alarm transitions to the INSUFFICIENT_DATA state from any other state.
     *         Each action is specified as an Amazon Resource Name (ARN).
     */
    public final List<String> insufficientDataActions() {
        return insufficientDataActions;
    }

    /**
     * For responses, this returns true if the service returned a value for the OKActions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOkActions() {
        return okActions != null && !(okActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The actions to execute when this alarm transitions to the OK state from any other state. Each action is specified
     * as an Amazon Resource Name (ARN).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOkActions} method.
     * </p>
     * 
     * @return The actions to execute when this alarm transitions to the OK state from any other state. Each action is
     *         specified as an Amazon Resource Name (ARN).
     */
    public final List<String> okActions() {
        return okActions;
    }

    /**
     * <p>
     * An explanation for the alarm state, in text format.
     * </p>
     * 
     * @return An explanation for the alarm state, in text format.
     */
    public final String stateReason() {
        return stateReason;
    }

    /**
     * <p>
     * An explanation for the alarm state, in JSON format.
     * </p>
     * 
     * @return An explanation for the alarm state, in JSON format.
     */
    public final String stateReasonData() {
        return stateReasonData;
    }

    /**
     * <p>
     * Tracks the timestamp of any state update, even if <code>StateValue</code> doesn't change.
     * </p>
     * 
     * @return Tracks the timestamp of any state update, even if <code>StateValue</code> doesn't change.
     */
    public final Instant stateUpdatedTimestamp() {
        return stateUpdatedTimestamp;
    }

    /**
     * <p>
     * The state value for the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateValue} will
     * return {@link StateValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateValueAsString}.
     * </p>
     * 
     * @return The state value for the alarm.
     * @see StateValue
     */
    public final StateValue stateValue() {
        return StateValue.fromValue(stateValue);
    }

    /**
     * <p>
     * The state value for the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateValue} will
     * return {@link StateValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateValueAsString}.
     * </p>
     * 
     * @return The state value for the alarm.
     * @see StateValue
     */
    public final String stateValueAsString() {
        return stateValue;
    }

    /**
     * <p>
     * The timestamp of the last change to the alarm's <code>StateValue</code>.
     * </p>
     * 
     * @return The timestamp of the last change to the alarm's <code>StateValue</code>.
     */
    public final Instant stateTransitionedTimestamp() {
        return stateTransitionedTimestamp;
    }

    /**
     * <p>
     * When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor alarm is in
     * <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions are suppressed because
     * the composite alarm is waiting for the suppressor alarm to go into into the <code>ALARM</code> state. The maximum
     * waiting time is as specified in <code>ActionsSuppressorWaitPeriod</code>. After this time, the composite alarm
     * performs its actions. When the value is <code>ExtensionPeriod</code>, it means that the actions are suppressed
     * because the composite alarm is waiting after the suppressor alarm went out of the <code>ALARM</code> state. The
     * maximum waiting time is as specified in <code>ActionsSuppressorExtensionPeriod</code>. After this time, the
     * composite alarm performs its actions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #actionsSuppressedBy} will return {@link ActionsSuppressedBy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #actionsSuppressedByAsString}.
     * </p>
     * 
     * @return When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor
     *         alarm is in <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions are
     *         suppressed because the composite alarm is waiting for the suppressor alarm to go into into the
     *         <code>ALARM</code> state. The maximum waiting time is as specified in
     *         <code>ActionsSuppressorWaitPeriod</code>. After this time, the composite alarm performs its actions. When
     *         the value is <code>ExtensionPeriod</code>, it means that the actions are suppressed because the composite
     *         alarm is waiting after the suppressor alarm went out of the <code>ALARM</code> state. The maximum waiting
     *         time is as specified in <code>ActionsSuppressorExtensionPeriod</code>. After this time, the composite
     *         alarm performs its actions.
     * @see ActionsSuppressedBy
     */
    public final ActionsSuppressedBy actionsSuppressedBy() {
        return ActionsSuppressedBy.fromValue(actionsSuppressedBy);
    }

    /**
     * <p>
     * When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor alarm is in
     * <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions are suppressed because
     * the composite alarm is waiting for the suppressor alarm to go into into the <code>ALARM</code> state. The maximum
     * waiting time is as specified in <code>ActionsSuppressorWaitPeriod</code>. After this time, the composite alarm
     * performs its actions. When the value is <code>ExtensionPeriod</code>, it means that the actions are suppressed
     * because the composite alarm is waiting after the suppressor alarm went out of the <code>ALARM</code> state. The
     * maximum waiting time is as specified in <code>ActionsSuppressorExtensionPeriod</code>. After this time, the
     * composite alarm performs its actions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #actionsSuppressedBy} will return {@link ActionsSuppressedBy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #actionsSuppressedByAsString}.
     * </p>
     * 
     * @return When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor
     *         alarm is in <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions are
     *         suppressed because the composite alarm is waiting for the suppressor alarm to go into into the
     *         <code>ALARM</code> state. The maximum waiting time is as specified in
     *         <code>ActionsSuppressorWaitPeriod</code>. After this time, the composite alarm performs its actions. When
     *         the value is <code>ExtensionPeriod</code>, it means that the actions are suppressed because the composite
     *         alarm is waiting after the suppressor alarm went out of the <code>ALARM</code> state. The maximum waiting
     *         time is as specified in <code>ActionsSuppressorExtensionPeriod</code>. After this time, the composite
     *         alarm performs its actions.
     * @see ActionsSuppressedBy
     */
    public final String actionsSuppressedByAsString() {
        return actionsSuppressedBy;
    }

    /**
     * <p>
     * Captures the reason for action suppression.
     * </p>
     * 
     * @return Captures the reason for action suppression.
     */
    public final String actionsSuppressedReason() {
        return actionsSuppressedReason;
    }

    /**
     * <p>
     * Actions will be suppressed if the suppressor alarm is in the <code>ALARM</code> state.
     * <code>ActionsSuppressor</code> can be an AlarmName or an Amazon Resource Name (ARN) from an existing alarm.
     * </p>
     * 
     * @return Actions will be suppressed if the suppressor alarm is in the <code>ALARM</code> state.
     *         <code>ActionsSuppressor</code> can be an AlarmName or an Amazon Resource Name (ARN) from an existing
     *         alarm.
     */
    public final String actionsSuppressor() {
        return actionsSuppressor;
    }

    /**
     * <p>
     * The maximum time in seconds that the composite alarm waits for the suppressor alarm to go into the
     * <code>ALARM</code> state. After this time, the composite alarm performs its actions.
     * </p>
     * <important>
     * <p>
     * <code>WaitPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
     * </p>
     * </important>
     * 
     * @return The maximum time in seconds that the composite alarm waits for the suppressor alarm to go into the
     *         <code>ALARM</code> state. After this time, the composite alarm performs its actions. </p> <important>
     *         <p>
     *         <code>WaitPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
     *         </p>
     */
    public final Integer actionsSuppressorWaitPeriod() {
        return actionsSuppressorWaitPeriod;
    }

    /**
     * <p>
     * The maximum time in seconds that the composite alarm waits after suppressor alarm goes out of the
     * <code>ALARM</code> state. After this time, the composite alarm performs its actions.
     * </p>
     * <important>
     * <p>
     * <code>ExtensionPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
     * </p>
     * </important>
     * 
     * @return The maximum time in seconds that the composite alarm waits after suppressor alarm goes out of the
     *         <code>ALARM</code> state. After this time, the composite alarm performs its actions. </p> <important>
     *         <p>
     *         <code>ExtensionPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
     *         </p>
     */
    public final Integer actionsSuppressorExtensionPeriod() {
        return actionsSuppressorExtensionPeriod;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlarmActions() ? alarmActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(alarmArn());
        hashCode = 31 * hashCode + Objects.hashCode(alarmConfigurationUpdatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(alarmDescription());
        hashCode = 31 * hashCode + Objects.hashCode(alarmName());
        hashCode = 31 * hashCode + Objects.hashCode(alarmRule());
        hashCode = 31 * hashCode + Objects.hashCode(hasInsufficientDataActions() ? insufficientDataActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOkActions() ? okActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateReason());
        hashCode = 31 * hashCode + Objects.hashCode(stateReasonData());
        hashCode = 31 * hashCode + Objects.hashCode(stateUpdatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(stateValueAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateTransitionedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(actionsSuppressedByAsString());
        hashCode = 31 * hashCode + Objects.hashCode(actionsSuppressedReason());
        hashCode = 31 * hashCode + Objects.hashCode(actionsSuppressor());
        hashCode = 31 * hashCode + Objects.hashCode(actionsSuppressorWaitPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(actionsSuppressorExtensionPeriod());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CompositeAlarm)) {
            return false;
        }
        CompositeAlarm other = (CompositeAlarm) obj;
        return Objects.equals(actionsEnabled(), other.actionsEnabled()) && hasAlarmActions() == other.hasAlarmActions()
                && Objects.equals(alarmActions(), other.alarmActions()) && Objects.equals(alarmArn(), other.alarmArn())
                && Objects.equals(alarmConfigurationUpdatedTimestamp(), other.alarmConfigurationUpdatedTimestamp())
                && Objects.equals(alarmDescription(), other.alarmDescription()) && Objects.equals(alarmName(), other.alarmName())
                && Objects.equals(alarmRule(), other.alarmRule())
                && hasInsufficientDataActions() == other.hasInsufficientDataActions()
                && Objects.equals(insufficientDataActions(), other.insufficientDataActions())
                && hasOkActions() == other.hasOkActions() && Objects.equals(okActions(), other.okActions())
                && Objects.equals(stateReason(), other.stateReason())
                && Objects.equals(stateReasonData(), other.stateReasonData())
                && Objects.equals(stateUpdatedTimestamp(), other.stateUpdatedTimestamp())
                && Objects.equals(stateValueAsString(), other.stateValueAsString())
                && Objects.equals(stateTransitionedTimestamp(), other.stateTransitionedTimestamp())
                && Objects.equals(actionsSuppressedByAsString(), other.actionsSuppressedByAsString())
                && Objects.equals(actionsSuppressedReason(), other.actionsSuppressedReason())
                && Objects.equals(actionsSuppressor(), other.actionsSuppressor())
                && Objects.equals(actionsSuppressorWaitPeriod(), other.actionsSuppressorWaitPeriod())
                && Objects.equals(actionsSuppressorExtensionPeriod(), other.actionsSuppressorExtensionPeriod());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CompositeAlarm").add("ActionsEnabled", actionsEnabled())
                .add("AlarmActions", hasAlarmActions() ? alarmActions() : null).add("AlarmArn", alarmArn())
                .add("AlarmConfigurationUpdatedTimestamp", alarmConfigurationUpdatedTimestamp())
                .add("AlarmDescription", alarmDescription()).add("AlarmName", alarmName()).add("AlarmRule", alarmRule())
                .add("InsufficientDataActions", hasInsufficientDataActions() ? insufficientDataActions() : null)
                .add("OKActions", hasOkActions() ? okActions() : null).add("StateReason", stateReason())
                .add("StateReasonData", stateReasonData()).add("StateUpdatedTimestamp", stateUpdatedTimestamp())
                .add("StateValue", stateValueAsString()).add("StateTransitionedTimestamp", stateTransitionedTimestamp())
                .add("ActionsSuppressedBy", actionsSuppressedByAsString())
                .add("ActionsSuppressedReason", actionsSuppressedReason()).add("ActionsSuppressor", actionsSuppressor())
                .add("ActionsSuppressorWaitPeriod", actionsSuppressorWaitPeriod())
                .add("ActionsSuppressorExtensionPeriod", actionsSuppressorExtensionPeriod()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActionsEnabled":
            return Optional.ofNullable(clazz.cast(actionsEnabled()));
        case "AlarmActions":
            return Optional.ofNullable(clazz.cast(alarmActions()));
        case "AlarmArn":
            return Optional.ofNullable(clazz.cast(alarmArn()));
        case "AlarmConfigurationUpdatedTimestamp":
            return Optional.ofNullable(clazz.cast(alarmConfigurationUpdatedTimestamp()));
        case "AlarmDescription":
            return Optional.ofNullable(clazz.cast(alarmDescription()));
        case "AlarmName":
            return Optional.ofNullable(clazz.cast(alarmName()));
        case "AlarmRule":
            return Optional.ofNullable(clazz.cast(alarmRule()));
        case "InsufficientDataActions":
            return Optional.ofNullable(clazz.cast(insufficientDataActions()));
        case "OKActions":
            return Optional.ofNullable(clazz.cast(okActions()));
        case "StateReason":
            return Optional.ofNullable(clazz.cast(stateReason()));
        case "StateReasonData":
            return Optional.ofNullable(clazz.cast(stateReasonData()));
        case "StateUpdatedTimestamp":
            return Optional.ofNullable(clazz.cast(stateUpdatedTimestamp()));
        case "StateValue":
            return Optional.ofNullable(clazz.cast(stateValueAsString()));
        case "StateTransitionedTimestamp":
            return Optional.ofNullable(clazz.cast(stateTransitionedTimestamp()));
        case "ActionsSuppressedBy":
            return Optional.ofNullable(clazz.cast(actionsSuppressedByAsString()));
        case "ActionsSuppressedReason":
            return Optional.ofNullable(clazz.cast(actionsSuppressedReason()));
        case "ActionsSuppressor":
            return Optional.ofNullable(clazz.cast(actionsSuppressor()));
        case "ActionsSuppressorWaitPeriod":
            return Optional.ofNullable(clazz.cast(actionsSuppressorWaitPeriod()));
        case "ActionsSuppressorExtensionPeriod":
            return Optional.ofNullable(clazz.cast(actionsSuppressorExtensionPeriod()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CompositeAlarm, T> g) {
        return obj -> g.apply((CompositeAlarm) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CompositeAlarm> {
        /**
         * <p>
         * Indicates whether actions should be executed during any changes to the alarm state.
         * </p>
         * 
         * @param actionsEnabled
         *        Indicates whether actions should be executed during any changes to the alarm state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsEnabled(Boolean actionsEnabled);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the ALARM state from any other state. Each action is
         * specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param alarmActions
         *        The actions to execute when this alarm transitions to the ALARM state from any other state. Each
         *        action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmActions(Collection<String> alarmActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the ALARM state from any other state. Each action is
         * specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param alarmActions
         *        The actions to execute when this alarm transitions to the ALARM state from any other state. Each
         *        action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmActions(String... alarmActions);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the alarm.
         * </p>
         * 
         * @param alarmArn
         *        The Amazon Resource Name (ARN) of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmArn(String alarmArn);

        /**
         * <p>
         * The time stamp of the last update to the alarm configuration.
         * </p>
         * 
         * @param alarmConfigurationUpdatedTimestamp
         *        The time stamp of the last update to the alarm configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmConfigurationUpdatedTimestamp(Instant alarmConfigurationUpdatedTimestamp);

        /**
         * <p>
         * The description of the alarm.
         * </p>
         * 
         * @param alarmDescription
         *        The description of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmDescription(String alarmDescription);

        /**
         * <p>
         * The name of the alarm.
         * </p>
         * 
         * @param alarmName
         *        The name of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmName(String alarmName);

        /**
         * <p>
         * The rule that this alarm uses to evaluate its alarm state.
         * </p>
         * 
         * @param alarmRule
         *        The rule that this alarm uses to evaluate its alarm state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmRule(String alarmRule);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the INSUFFICIENT_DATA state from any other state. Each
         * action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param insufficientDataActions
         *        The actions to execute when this alarm transitions to the INSUFFICIENT_DATA state from any other
         *        state. Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insufficientDataActions(Collection<String> insufficientDataActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the INSUFFICIENT_DATA state from any other state. Each
         * action is specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param insufficientDataActions
         *        The actions to execute when this alarm transitions to the INSUFFICIENT_DATA state from any other
         *        state. Each action is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insufficientDataActions(String... insufficientDataActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the OK state from any other state. Each action is
         * specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param okActions
         *        The actions to execute when this alarm transitions to the OK state from any other state. Each action
         *        is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder okActions(Collection<String> okActions);

        /**
         * <p>
         * The actions to execute when this alarm transitions to the OK state from any other state. Each action is
         * specified as an Amazon Resource Name (ARN).
         * </p>
         * 
         * @param okActions
         *        The actions to execute when this alarm transitions to the OK state from any other state. Each action
         *        is specified as an Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder okActions(String... okActions);

        /**
         * <p>
         * An explanation for the alarm state, in text format.
         * </p>
         * 
         * @param stateReason
         *        An explanation for the alarm state, in text format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReason(String stateReason);

        /**
         * <p>
         * An explanation for the alarm state, in JSON format.
         * </p>
         * 
         * @param stateReasonData
         *        An explanation for the alarm state, in JSON format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReasonData(String stateReasonData);

        /**
         * <p>
         * Tracks the timestamp of any state update, even if <code>StateValue</code> doesn't change.
         * </p>
         * 
         * @param stateUpdatedTimestamp
         *        Tracks the timestamp of any state update, even if <code>StateValue</code> doesn't change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateUpdatedTimestamp(Instant stateUpdatedTimestamp);

        /**
         * <p>
         * The state value for the alarm.
         * </p>
         * 
         * @param stateValue
         *        The state value for the alarm.
         * @see StateValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateValue
         */
        Builder stateValue(String stateValue);

        /**
         * <p>
         * The state value for the alarm.
         * </p>
         * 
         * @param stateValue
         *        The state value for the alarm.
         * @see StateValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StateValue
         */
        Builder stateValue(StateValue stateValue);

        /**
         * <p>
         * The timestamp of the last change to the alarm's <code>StateValue</code>.
         * </p>
         * 
         * @param stateTransitionedTimestamp
         *        The timestamp of the last change to the alarm's <code>StateValue</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateTransitionedTimestamp(Instant stateTransitionedTimestamp);

        /**
         * <p>
         * When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor alarm
         * is in <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions are suppressed
         * because the composite alarm is waiting for the suppressor alarm to go into into the <code>ALARM</code> state.
         * The maximum waiting time is as specified in <code>ActionsSuppressorWaitPeriod</code>. After this time, the
         * composite alarm performs its actions. When the value is <code>ExtensionPeriod</code>, it means that the
         * actions are suppressed because the composite alarm is waiting after the suppressor alarm went out of the
         * <code>ALARM</code> state. The maximum waiting time is as specified in
         * <code>ActionsSuppressorExtensionPeriod</code>. After this time, the composite alarm performs its actions.
         * </p>
         * 
         * @param actionsSuppressedBy
         *        When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor
         *        alarm is in <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions
         *        are suppressed because the composite alarm is waiting for the suppressor alarm to go into into the
         *        <code>ALARM</code> state. The maximum waiting time is as specified in
         *        <code>ActionsSuppressorWaitPeriod</code>. After this time, the composite alarm performs its actions.
         *        When the value is <code>ExtensionPeriod</code>, it means that the actions are suppressed because the
         *        composite alarm is waiting after the suppressor alarm went out of the <code>ALARM</code> state. The
         *        maximum waiting time is as specified in <code>ActionsSuppressorExtensionPeriod</code>. After this
         *        time, the composite alarm performs its actions.
         * @see ActionsSuppressedBy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionsSuppressedBy
         */
        Builder actionsSuppressedBy(String actionsSuppressedBy);

        /**
         * <p>
         * When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor alarm
         * is in <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions are suppressed
         * because the composite alarm is waiting for the suppressor alarm to go into into the <code>ALARM</code> state.
         * The maximum waiting time is as specified in <code>ActionsSuppressorWaitPeriod</code>. After this time, the
         * composite alarm performs its actions. When the value is <code>ExtensionPeriod</code>, it means that the
         * actions are suppressed because the composite alarm is waiting after the suppressor alarm went out of the
         * <code>ALARM</code> state. The maximum waiting time is as specified in
         * <code>ActionsSuppressorExtensionPeriod</code>. After this time, the composite alarm performs its actions.
         * </p>
         * 
         * @param actionsSuppressedBy
         *        When the value is <code>ALARM</code>, it means that the actions are suppressed because the suppressor
         *        alarm is in <code>ALARM</code> When the value is <code>WaitPeriod</code>, it means that the actions
         *        are suppressed because the composite alarm is waiting for the suppressor alarm to go into into the
         *        <code>ALARM</code> state. The maximum waiting time is as specified in
         *        <code>ActionsSuppressorWaitPeriod</code>. After this time, the composite alarm performs its actions.
         *        When the value is <code>ExtensionPeriod</code>, it means that the actions are suppressed because the
         *        composite alarm is waiting after the suppressor alarm went out of the <code>ALARM</code> state. The
         *        maximum waiting time is as specified in <code>ActionsSuppressorExtensionPeriod</code>. After this
         *        time, the composite alarm performs its actions.
         * @see ActionsSuppressedBy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionsSuppressedBy
         */
        Builder actionsSuppressedBy(ActionsSuppressedBy actionsSuppressedBy);

        /**
         * <p>
         * Captures the reason for action suppression.
         * </p>
         * 
         * @param actionsSuppressedReason
         *        Captures the reason for action suppression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsSuppressedReason(String actionsSuppressedReason);

        /**
         * <p>
         * Actions will be suppressed if the suppressor alarm is in the <code>ALARM</code> state.
         * <code>ActionsSuppressor</code> can be an AlarmName or an Amazon Resource Name (ARN) from an existing alarm.
         * </p>
         * 
         * @param actionsSuppressor
         *        Actions will be suppressed if the suppressor alarm is in the <code>ALARM</code> state.
         *        <code>ActionsSuppressor</code> can be an AlarmName or an Amazon Resource Name (ARN) from an existing
         *        alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsSuppressor(String actionsSuppressor);

        /**
         * <p>
         * The maximum time in seconds that the composite alarm waits for the suppressor alarm to go into the
         * <code>ALARM</code> state. After this time, the composite alarm performs its actions.
         * </p>
         * <important>
         * <p>
         * <code>WaitPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
         * </p>
         * </important>
         * 
         * @param actionsSuppressorWaitPeriod
         *        The maximum time in seconds that the composite alarm waits for the suppressor alarm to go into the
         *        <code>ALARM</code> state. After this time, the composite alarm performs its actions. </p> <important>
         *        <p>
         *        <code>WaitPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsSuppressorWaitPeriod(Integer actionsSuppressorWaitPeriod);

        /**
         * <p>
         * The maximum time in seconds that the composite alarm waits after suppressor alarm goes out of the
         * <code>ALARM</code> state. After this time, the composite alarm performs its actions.
         * </p>
         * <important>
         * <p>
         * <code>ExtensionPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
         * </p>
         * </important>
         * 
         * @param actionsSuppressorExtensionPeriod
         *        The maximum time in seconds that the composite alarm waits after suppressor alarm goes out of the
         *        <code>ALARM</code> state. After this time, the composite alarm performs its actions. </p> <important>
         *        <p>
         *        <code>ExtensionPeriod</code> is required only when <code>ActionsSuppressor</code> is specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsSuppressorExtensionPeriod(Integer actionsSuppressorExtensionPeriod);
    }

    static final class BuilderImpl implements Builder {
        private Boolean actionsEnabled;

        private List<String> alarmActions = DefaultSdkAutoConstructList.getInstance();

        private String alarmArn;

        private Instant alarmConfigurationUpdatedTimestamp;

        private String alarmDescription;

        private String alarmName;

        private String alarmRule;

        private List<String> insufficientDataActions = DefaultSdkAutoConstructList.getInstance();

        private List<String> okActions = DefaultSdkAutoConstructList.getInstance();

        private String stateReason;

        private String stateReasonData;

        private Instant stateUpdatedTimestamp;

        private String stateValue;

        private Instant stateTransitionedTimestamp;

        private String actionsSuppressedBy;

        private String actionsSuppressedReason;

        private String actionsSuppressor;

        private Integer actionsSuppressorWaitPeriod;

        private Integer actionsSuppressorExtensionPeriod;

        private BuilderImpl() {
        }

        private BuilderImpl(CompositeAlarm model) {
            actionsEnabled(model.actionsEnabled);
            alarmActions(model.alarmActions);
            alarmArn(model.alarmArn);
            alarmConfigurationUpdatedTimestamp(model.alarmConfigurationUpdatedTimestamp);
            alarmDescription(model.alarmDescription);
            alarmName(model.alarmName);
            alarmRule(model.alarmRule);
            insufficientDataActions(model.insufficientDataActions);
            okActions(model.okActions);
            stateReason(model.stateReason);
            stateReasonData(model.stateReasonData);
            stateUpdatedTimestamp(model.stateUpdatedTimestamp);
            stateValue(model.stateValue);
            stateTransitionedTimestamp(model.stateTransitionedTimestamp);
            actionsSuppressedBy(model.actionsSuppressedBy);
            actionsSuppressedReason(model.actionsSuppressedReason);
            actionsSuppressor(model.actionsSuppressor);
            actionsSuppressorWaitPeriod(model.actionsSuppressorWaitPeriod);
            actionsSuppressorExtensionPeriod(model.actionsSuppressorExtensionPeriod);
        }

        public final Boolean getActionsEnabled() {
            return actionsEnabled;
        }

        public final void setActionsEnabled(Boolean actionsEnabled) {
            this.actionsEnabled = actionsEnabled;
        }

        @Override
        public final Builder actionsEnabled(Boolean actionsEnabled) {
            this.actionsEnabled = actionsEnabled;
            return this;
        }

        public final Collection<String> getAlarmActions() {
            if (alarmActions instanceof SdkAutoConstructList) {
                return null;
            }
            return alarmActions;
        }

        public final void setAlarmActions(Collection<String> alarmActions) {
            this.alarmActions = ResourceListCopier.copy(alarmActions);
        }

        @Override
        public final Builder alarmActions(Collection<String> alarmActions) {
            this.alarmActions = ResourceListCopier.copy(alarmActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alarmActions(String... alarmActions) {
            alarmActions(Arrays.asList(alarmActions));
            return this;
        }

        public final String getAlarmArn() {
            return alarmArn;
        }

        public final void setAlarmArn(String alarmArn) {
            this.alarmArn = alarmArn;
        }

        @Override
        public final Builder alarmArn(String alarmArn) {
            this.alarmArn = alarmArn;
            return this;
        }

        public final Instant getAlarmConfigurationUpdatedTimestamp() {
            return alarmConfigurationUpdatedTimestamp;
        }

        public final void setAlarmConfigurationUpdatedTimestamp(Instant alarmConfigurationUpdatedTimestamp) {
            this.alarmConfigurationUpdatedTimestamp = alarmConfigurationUpdatedTimestamp;
        }

        @Override
        public final Builder alarmConfigurationUpdatedTimestamp(Instant alarmConfigurationUpdatedTimestamp) {
            this.alarmConfigurationUpdatedTimestamp = alarmConfigurationUpdatedTimestamp;
            return this;
        }

        public final String getAlarmDescription() {
            return alarmDescription;
        }

        public final void setAlarmDescription(String alarmDescription) {
            this.alarmDescription = alarmDescription;
        }

        @Override
        public final Builder alarmDescription(String alarmDescription) {
            this.alarmDescription = alarmDescription;
            return this;
        }

        public final String getAlarmName() {
            return alarmName;
        }

        public final void setAlarmName(String alarmName) {
            this.alarmName = alarmName;
        }

        @Override
        public final Builder alarmName(String alarmName) {
            this.alarmName = alarmName;
            return this;
        }

        public final String getAlarmRule() {
            return alarmRule;
        }

        public final void setAlarmRule(String alarmRule) {
            this.alarmRule = alarmRule;
        }

        @Override
        public final Builder alarmRule(String alarmRule) {
            this.alarmRule = alarmRule;
            return this;
        }

        public final Collection<String> getInsufficientDataActions() {
            if (insufficientDataActions instanceof SdkAutoConstructList) {
                return null;
            }
            return insufficientDataActions;
        }

        public final void setInsufficientDataActions(Collection<String> insufficientDataActions) {
            this.insufficientDataActions = ResourceListCopier.copy(insufficientDataActions);
        }

        @Override
        public final Builder insufficientDataActions(Collection<String> insufficientDataActions) {
            this.insufficientDataActions = ResourceListCopier.copy(insufficientDataActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder insufficientDataActions(String... insufficientDataActions) {
            insufficientDataActions(Arrays.asList(insufficientDataActions));
            return this;
        }

        public final Collection<String> getOkActions() {
            if (okActions instanceof SdkAutoConstructList) {
                return null;
            }
            return okActions;
        }

        public final void setOkActions(Collection<String> okActions) {
            this.okActions = ResourceListCopier.copy(okActions);
        }

        @Override
        public final Builder okActions(Collection<String> okActions) {
            this.okActions = ResourceListCopier.copy(okActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder okActions(String... okActions) {
            okActions(Arrays.asList(okActions));
            return this;
        }

        public final String getStateReason() {
            return stateReason;
        }

        public final void setStateReason(String stateReason) {
            this.stateReason = stateReason;
        }

        @Override
        public final Builder stateReason(String stateReason) {
            this.stateReason = stateReason;
            return this;
        }

        public final String getStateReasonData() {
            return stateReasonData;
        }

        public final void setStateReasonData(String stateReasonData) {
            this.stateReasonData = stateReasonData;
        }

        @Override
        public final Builder stateReasonData(String stateReasonData) {
            this.stateReasonData = stateReasonData;
            return this;
        }

        public final Instant getStateUpdatedTimestamp() {
            return stateUpdatedTimestamp;
        }

        public final void setStateUpdatedTimestamp(Instant stateUpdatedTimestamp) {
            this.stateUpdatedTimestamp = stateUpdatedTimestamp;
        }

        @Override
        public final Builder stateUpdatedTimestamp(Instant stateUpdatedTimestamp) {
            this.stateUpdatedTimestamp = stateUpdatedTimestamp;
            return this;
        }

        public final String getStateValue() {
            return stateValue;
        }

        public final void setStateValue(String stateValue) {
            this.stateValue = stateValue;
        }

        @Override
        public final Builder stateValue(String stateValue) {
            this.stateValue = stateValue;
            return this;
        }

        @Override
        public final Builder stateValue(StateValue stateValue) {
            this.stateValue(stateValue == null ? null : stateValue.toString());
            return this;
        }

        public final Instant getStateTransitionedTimestamp() {
            return stateTransitionedTimestamp;
        }

        public final void setStateTransitionedTimestamp(Instant stateTransitionedTimestamp) {
            this.stateTransitionedTimestamp = stateTransitionedTimestamp;
        }

        @Override
        public final Builder stateTransitionedTimestamp(Instant stateTransitionedTimestamp) {
            this.stateTransitionedTimestamp = stateTransitionedTimestamp;
            return this;
        }

        public final String getActionsSuppressedBy() {
            return actionsSuppressedBy;
        }

        public final void setActionsSuppressedBy(String actionsSuppressedBy) {
            this.actionsSuppressedBy = actionsSuppressedBy;
        }

        @Override
        public final Builder actionsSuppressedBy(String actionsSuppressedBy) {
            this.actionsSuppressedBy = actionsSuppressedBy;
            return this;
        }

        @Override
        public final Builder actionsSuppressedBy(ActionsSuppressedBy actionsSuppressedBy) {
            this.actionsSuppressedBy(actionsSuppressedBy == null ? null : actionsSuppressedBy.toString());
            return this;
        }

        public final String getActionsSuppressedReason() {
            return actionsSuppressedReason;
        }

        public final void setActionsSuppressedReason(String actionsSuppressedReason) {
            this.actionsSuppressedReason = actionsSuppressedReason;
        }

        @Override
        public final Builder actionsSuppressedReason(String actionsSuppressedReason) {
            this.actionsSuppressedReason = actionsSuppressedReason;
            return this;
        }

        public final String getActionsSuppressor() {
            return actionsSuppressor;
        }

        public final void setActionsSuppressor(String actionsSuppressor) {
            this.actionsSuppressor = actionsSuppressor;
        }

        @Override
        public final Builder actionsSuppressor(String actionsSuppressor) {
            this.actionsSuppressor = actionsSuppressor;
            return this;
        }

        public final Integer getActionsSuppressorWaitPeriod() {
            return actionsSuppressorWaitPeriod;
        }

        public final void setActionsSuppressorWaitPeriod(Integer actionsSuppressorWaitPeriod) {
            this.actionsSuppressorWaitPeriod = actionsSuppressorWaitPeriod;
        }

        @Override
        public final Builder actionsSuppressorWaitPeriod(Integer actionsSuppressorWaitPeriod) {
            this.actionsSuppressorWaitPeriod = actionsSuppressorWaitPeriod;
            return this;
        }

        public final Integer getActionsSuppressorExtensionPeriod() {
            return actionsSuppressorExtensionPeriod;
        }

        public final void setActionsSuppressorExtensionPeriod(Integer actionsSuppressorExtensionPeriod) {
            this.actionsSuppressorExtensionPeriod = actionsSuppressorExtensionPeriod;
        }

        @Override
        public final Builder actionsSuppressorExtensionPeriod(Integer actionsSuppressorExtensionPeriod) {
            this.actionsSuppressorExtensionPeriod = actionsSuppressorExtensionPeriod;
            return this;
        }

        @Override
        public CompositeAlarm build() {
            return new CompositeAlarm(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
