/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.internal.interceptor.trait.RequestCompression;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.cloudwatch.internal.CloudWatchServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.cloudwatch.model.CloudWatchException;
import software.amazon.awssdk.services.cloudwatch.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cloudwatch.model.ConflictException;
import software.amazon.awssdk.services.cloudwatch.model.DashboardInvalidInputErrorException;
import software.amazon.awssdk.services.cloudwatch.model.DashboardNotFoundErrorException;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAnomalyDetectorRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAnomalyDetectorResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteDashboardsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteDashboardsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteMetricStreamRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteMetricStreamResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsForMetricRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsForMetricResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAnomalyDetectorsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAnomalyDetectorsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.DisableAlarmActionsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DisableAlarmActionsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DisableInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.DisableInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.EnableAlarmActionsRequest;
import software.amazon.awssdk.services.cloudwatch.model.EnableAlarmActionsResponse;
import software.amazon.awssdk.services.cloudwatch.model.EnableInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.EnableInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetDashboardRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetDashboardResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetInsightRuleReportRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetInsightRuleReportResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricDataRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricDataResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStatisticsRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStatisticsResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStreamRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStreamResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricWidgetImageRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricWidgetImageResponse;
import software.amazon.awssdk.services.cloudwatch.model.InternalServiceException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidFormatException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidNextTokenException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidParameterValueException;
import software.amazon.awssdk.services.cloudwatch.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatch.model.ListDashboardsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListDashboardsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListManagedInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListManagedInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloudwatch.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.cloudwatch.model.PutAnomalyDetectorRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutAnomalyDetectorResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutCompositeAlarmRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutCompositeAlarmResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutDashboardRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutDashboardResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutInsightRuleRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutInsightRuleResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutManagedInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutManagedInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricAlarmRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricAlarmResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricDataRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricDataResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricStreamRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricStreamResponse;
import software.amazon.awssdk.services.cloudwatch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatch.model.SetAlarmStateRequest;
import software.amazon.awssdk.services.cloudwatch.model.SetAlarmStateResponse;
import software.amazon.awssdk.services.cloudwatch.model.StartMetricStreamsRequest;
import software.amazon.awssdk.services.cloudwatch.model.StartMetricStreamsResponse;
import software.amazon.awssdk.services.cloudwatch.model.StopMetricStreamsRequest;
import software.amazon.awssdk.services.cloudwatch.model.StopMetricStreamsResponse;
import software.amazon.awssdk.services.cloudwatch.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudwatch.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudwatch.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudwatch.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteAlarmsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteDashboardsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteMetricStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmHistoryRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmsForMetricRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAnomalyDetectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DisableAlarmActionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DisableInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.EnableAlarmActionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.EnableInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetDashboardRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetInsightRuleReportRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricDataRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricStatisticsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricWidgetImageRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListDashboardsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListManagedInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListMetricStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListMetricsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutCompositeAlarmRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutDashboardRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutInsightRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutManagedInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricAlarmRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricDataRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.SetAlarmStateRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.StartMetricStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.StopMetricStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.waiters.CloudWatchWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudWatchClient}.
 *
 * @see CloudWatchClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchClient implements CloudWatchClient {
    private static final Logger log = Logger.loggerFor(DefaultCloudWatchClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.QUERY).build();

    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init();
    }

    /**
     * <p>
     * Deletes the specified alarms. You can delete up to 100 alarms in one operation. However, this total can include
     * no more than one composite alarm. For example, you could delete 99 metric alarms and one composite alarms with
     * one operation, but you can't delete two composite alarms with one operation.
     * </p>
     * <p>
     * If you specify any incorrect alarm names, the alarms you specify with correct names are still deleted. Other
     * syntax errors might result in no alarms being deleted. To confirm that alarms were deleted successfully, you can
     * use the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html">DescribeAlarms
     * </a> operation after using <code>DeleteAlarms</code>.
     * </p>
     * <note>
     * <p>
     * It is possible to create a loop or cycle of composite alarms, where composite alarm A depends on composite alarm
     * B, and composite alarm B also depends on composite alarm A. In this scenario, you can't delete any composite
     * alarm that is part of the cycle because there is always still a composite alarm that depends on that alarm that
     * you want to delete.
     * </p>
     * <p>
     * To get out of such a situation, you must break the cycle by changing the rule of one of the composite alarms in
     * the cycle to remove a dependency that creates the cycle. The simplest change to make to break a cycle is to
     * change the <code>AlarmRule</code> of one of the alarms to <code>false</code>.
     * </p>
     * <p>
     * Additionally, the evaluation of composite alarms stops if CloudWatch detects a cycle in the evaluation path.
     * </p>
     * </note>
     *
     * @param deleteAlarmsRequest
     * @return Result of the DeleteAlarms operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteAlarms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAlarmsResponse deleteAlarms(DeleteAlarmsRequest deleteAlarmsRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteAlarmsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAlarmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAlarmsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlarmsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlarms");

            return clientHandler.execute(new ClientExecutionParams<DeleteAlarmsRequest, DeleteAlarmsResponse>()
                    .withOperationName("DeleteAlarms").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAlarmsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAlarmsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified anomaly detection model from your account. For more information about how to delete an
     * anomaly detection model, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Create_Anomaly_Detection_Alarm.html#Delete_Anomaly_Detection_Model"
     * >Deleting an anomaly detection model</a> in the <i>CloudWatch User Guide</i>.
     * </p>
     *
     * @param deleteAnomalyDetectorRequest
     * @return Result of the DeleteAnomalyDetector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAnomalyDetectorResponse deleteAnomalyDetector(DeleteAnomalyDetectorRequest deleteAnomalyDetectorRequest)
            throws ResourceNotFoundException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<DeleteAnomalyDetectorResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAnomalyDetectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAnomalyDetectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAnomalyDetector");

            return clientHandler.execute(new ClientExecutionParams<DeleteAnomalyDetectorRequest, DeleteAnomalyDetectorResponse>()
                    .withOperationName("DeleteAnomalyDetector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAnomalyDetectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAnomalyDetectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all dashboards that you specify. You can specify up to 100 dashboards to delete. If there is an error
     * during this call, no dashboards are deleted.
     * </p>
     *
     * @param deleteDashboardsRequest
     * @return Result of the DeleteDashboards operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws DashboardNotFoundErrorException
     *         The specified dashboard does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDashboardsResponse deleteDashboards(DeleteDashboardsRequest deleteDashboardsRequest)
            throws InvalidParameterValueException, DashboardNotFoundErrorException, InternalServiceException, ConflictException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteDashboardsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteDashboardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDashboardsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDashboardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDashboards");

            return clientHandler.execute(new ClientExecutionParams<DeleteDashboardsRequest, DeleteDashboardsResponse>()
                    .withOperationName("DeleteDashboards").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDashboardsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDashboardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes the specified Contributor Insights rules.
     * </p>
     * <p>
     * If you create a rule, delete it, and then re-create it with the same name, historical data from the first time
     * the rule was created might not be available.
     * </p>
     *
     * @param deleteInsightRulesRequest
     * @return Result of the DeleteInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteInsightRules" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteInsightRulesResponse deleteInsightRules(DeleteInsightRulesRequest deleteInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<DeleteInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteInsightRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInsightRules");

            return clientHandler.execute(new ClientExecutionParams<DeleteInsightRulesRequest, DeleteInsightRulesResponse>()
                    .withOperationName("DeleteInsightRules").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes the metric stream that you specify.
     * </p>
     *
     * @param deleteMetricStreamRequest
     * @return Result of the DeleteMetricStream operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteMetricStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteMetricStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteMetricStreamResponse deleteMetricStream(DeleteMetricStreamRequest deleteMetricStreamRequest)
            throws InternalServiceException, InvalidParameterValueException, MissingRequiredParameterException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteMetricStreamResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteMetricStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMetricStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMetricStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMetricStream");

            return clientHandler.execute(new ClientExecutionParams<DeleteMetricStreamRequest, DeleteMetricStreamResponse>()
                    .withOperationName("DeleteMetricStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteMetricStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMetricStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the history for the specified alarm. You can filter the results by date range or item type. If an alarm
     * name is not specified, the histories for either all metric alarms or all composite alarms are returned.
     * </p>
     * <p>
     * CloudWatch retains the history of an alarm even if you delete the alarm.
     * </p>
     * <p>
     * To use this operation and return information about a composite alarm, you must be signed on with the
     * <code>cloudwatch:DescribeAlarmHistory</code> permission that is scoped to <code>*</code>. You can't return
     * information about composite alarms if your <code>cloudwatch:DescribeAlarmHistory</code> permission has a narrower
     * scope.
     * </p>
     *
     * @param describeAlarmHistoryRequest
     * @return Result of the DescribeAlarmHistory operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmHistoryResponse describeAlarmHistory(DescribeAlarmHistoryRequest describeAlarmHistoryRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmHistoryResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmHistoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAlarmHistoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarmHistory");

            return clientHandler.execute(new ClientExecutionParams<DescribeAlarmHistoryRequest, DescribeAlarmHistoryResponse>()
                    .withOperationName("DescribeAlarmHistory").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAlarmHistoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAlarmHistoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified alarms. You can filter the results by specifying a prefix for the alarm name, the alarm
     * state, or a prefix for any action.
     * </p>
     * <p>
     * To use this operation and return information about composite alarms, you must be signed on with the
     * <code>cloudwatch:DescribeAlarms</code> permission that is scoped to <code>*</code>. You can't return information
     * about composite alarms if your <code>cloudwatch:DescribeAlarms</code> permission has a narrower scope.
     * </p>
     *
     * @param describeAlarmsRequest
     * @return Result of the DescribeAlarms operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAlarmsResponse describeAlarms(DescribeAlarmsRequest describeAlarmsRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAlarmsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarms");

            return clientHandler.execute(new ClientExecutionParams<DescribeAlarmsRequest, DescribeAlarmsResponse>()
                    .withOperationName("DescribeAlarms").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeAlarmsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAlarmsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the alarms for the specified metric. To filter the results, specify a statistic, period, or unit.
     * </p>
     * <p>
     * This operation retrieves only standard alarms that are based on the specified metric. It does not return alarms
     * based on math expressions that use the specified metric, or composite alarms that use the specified metric.
     * </p>
     *
     * @param describeAlarmsForMetricRequest
     * @return Result of the DescribeAlarmsForMetric operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmsForMetric
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmsForMetric"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmsForMetricResponse describeAlarmsForMetric(DescribeAlarmsForMetricRequest describeAlarmsForMetricRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmsForMetricResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmsForMetricResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAlarmsForMetricRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmsForMetricRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarmsForMetric");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAlarmsForMetricRequest, DescribeAlarmsForMetricResponse>()
                            .withOperationName("DescribeAlarmsForMetric").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeAlarmsForMetricRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAlarmsForMetricRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the anomaly detection models that you have created in your account. For single metric anomaly detectors,
     * you can list all of the models in your account or filter the results to only the models that are related to a
     * certain namespace, metric name, or metric dimension. For metric math anomaly detectors, you can list them by
     * adding <code>METRIC_MATH</code> to the <code>AnomalyDetectorTypes</code> array. This will return all metric math
     * anomaly detectors in your account.
     * </p>
     *
     * @param describeAnomalyDetectorsRequest
     * @return Result of the DescribeAnomalyDetectors operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAnomalyDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAnomalyDetectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAnomalyDetectorsResponse describeAnomalyDetectors(
            DescribeAnomalyDetectorsRequest describeAnomalyDetectorsRequest) throws InvalidNextTokenException,
            InternalServiceException, InvalidParameterValueException, InvalidParameterCombinationException, AwsServiceException,
            SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAnomalyDetectorsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAnomalyDetectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAnomalyDetectorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomalyDetectors");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAnomalyDetectorsRequest, DescribeAnomalyDetectorsResponse>()
                            .withOperationName("DescribeAnomalyDetectors").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeAnomalyDetectorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAnomalyDetectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the Contributor Insights rules in your account.
     * </p>
     * <p>
     * For more information about Contributor Insights, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/ContributorInsights.html">Using Contributor
     * Insights to Analyze High-Cardinality Data</a>.
     * </p>
     *
     * @param describeInsightRulesRequest
     * @return Result of the DescribeInsightRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeInsightRulesResponse describeInsightRules(DescribeInsightRulesRequest describeInsightRulesRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeInsightRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsightRules");

            return clientHandler.execute(new ClientExecutionParams<DescribeInsightRulesRequest, DescribeInsightRulesResponse>()
                    .withOperationName("DescribeInsightRules").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables the actions for the specified alarms. When an alarm's actions are disabled, the alarm actions do not
     * execute when the alarm state changes.
     * </p>
     *
     * @param disableAlarmActionsRequest
     * @return Result of the DisableAlarmActions operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DisableAlarmActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DisableAlarmActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableAlarmActionsResponse disableAlarmActions(DisableAlarmActionsRequest disableAlarmActionsRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DisableAlarmActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DisableAlarmActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableAlarmActionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableAlarmActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableAlarmActions");

            return clientHandler.execute(new ClientExecutionParams<DisableAlarmActionsRequest, DisableAlarmActionsResponse>()
                    .withOperationName("DisableAlarmActions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableAlarmActionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableAlarmActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables the specified Contributor Insights rules. When rules are disabled, they do not analyze log groups and do
     * not incur costs.
     * </p>
     *
     * @param disableInsightRulesRequest
     * @return Result of the DisableInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DisableInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DisableInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableInsightRulesResponse disableInsightRules(DisableInsightRulesRequest disableInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<DisableInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(DisableInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableInsightRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableInsightRules");

            return clientHandler.execute(new ClientExecutionParams<DisableInsightRulesRequest, DisableInsightRulesResponse>()
                    .withOperationName("DisableInsightRules").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables the actions for the specified alarms.
     * </p>
     *
     * @param enableAlarmActionsRequest
     * @return Result of the EnableAlarmActions operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.EnableAlarmActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/EnableAlarmActions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableAlarmActionsResponse enableAlarmActions(EnableAlarmActionsRequest enableAlarmActionsRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<EnableAlarmActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(EnableAlarmActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableAlarmActionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableAlarmActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableAlarmActions");

            return clientHandler.execute(new ClientExecutionParams<EnableAlarmActionsRequest, EnableAlarmActionsResponse>()
                    .withOperationName("EnableAlarmActions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableAlarmActionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableAlarmActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables the specified Contributor Insights rules. When rules are enabled, they immediately begin analyzing log
     * data.
     * </p>
     *
     * @param enableInsightRulesRequest
     * @return Result of the EnableInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws LimitExceededException
     *         The operation exceeded one or more limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.EnableInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/EnableInsightRules" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableInsightRulesResponse enableInsightRules(EnableInsightRulesRequest enableInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<EnableInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(EnableInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableInsightRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableInsightRules");

            return clientHandler.execute(new ClientExecutionParams<EnableInsightRulesRequest, EnableInsightRulesResponse>()
                    .withOperationName("EnableInsightRules").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the details of the dashboard that you specify.
     * </p>
     * <p>
     * To copy an existing dashboard, use <code>GetDashboard</code>, and then use the data returned within
     * <code>DashboardBody</code> as the template for the new dashboard when you call <code>PutDashboard</code> to
     * create the copy.
     * </p>
     *
     * @param getDashboardRequest
     * @return Result of the GetDashboard operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws DashboardNotFoundErrorException
     *         The specified dashboard does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetDashboard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDashboardResponse getDashboard(GetDashboardRequest getDashboardRequest) throws InvalidParameterValueException,
            DashboardNotFoundErrorException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<GetDashboardResponse> responseHandler = protocolFactory
                .createResponseHandler(GetDashboardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDashboardRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDashboard");

            return clientHandler.execute(new ClientExecutionParams<GetDashboardRequest, GetDashboardResponse>()
                    .withOperationName("GetDashboard").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns the time series data collected by a Contributor Insights rule. The data includes the
     * identity and number of contributors to the log group.
     * </p>
     * <p>
     * You can also optionally return one or more statistics about each data point in the time series. These statistics
     * can include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>UniqueContributors</code> -- the number of unique contributors for each data point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaxContributorValue</code> -- the value of the top contributor for each data point. The identity of the
     * contributor might change for each data point in the graph.
     * </p>
     * <p>
     * If this rule aggregates by COUNT, the top contributor for each data point is the contributor with the most
     * occurrences in that period. If the rule aggregates by SUM, the top contributor is the contributor with the
     * highest sum in the log field specified by the rule's <code>Value</code>, during that period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> -- the number of data points matched by the rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> -- the sum of the values from all contributors during the time period represented by that data
     * point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Minimum</code> -- the minimum value from a single observation during the time period represented by that
     * data point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> -- the maximum value from a single observation during the time period represented by that
     * data point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> -- the average value from all contributors during the time period represented by that data
     * point.
     * </p>
     * </li>
     * </ul>
     *
     * @param getInsightRuleReportRequest
     * @return Result of the GetInsightRuleReport operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetInsightRuleReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetInsightRuleReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInsightRuleReportResponse getInsightRuleReport(GetInsightRuleReportRequest getInsightRuleReportRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetInsightRuleReportResponse> responseHandler = protocolFactory
                .createResponseHandler(GetInsightRuleReportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInsightRuleReportRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightRuleReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightRuleReport");

            return clientHandler.execute(new ClientExecutionParams<GetInsightRuleReportRequest, GetInsightRuleReportResponse>()
                    .withOperationName("GetInsightRuleReport").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getInsightRuleReportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInsightRuleReportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use the <code>GetMetricData</code> API to retrieve CloudWatch metric values. The operation can also
     * include a CloudWatch Metrics Insights query, and one or more metric math functions.
     * </p>
     * <p>
     * A <code>GetMetricData</code> operation that does not include a query can retrieve as many as 500 different
     * metrics in a single request, with a total of as many as 100,800 data points. You can also optionally perform
     * metric math expressions on the values of the returned statistics, to create new time series that represent new
     * insights into your data. For example, using Lambda metrics, you could divide the Errors metric by the Invocations
     * metric to get an error rate time series. For more information about metric math expressions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax"
     * >Metric Math Syntax and Functions</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * If you include a Metrics Insights query, each <code>GetMetricData</code> operation can include only one query.
     * But the same <code>GetMetricData</code> operation can also retrieve other metrics. Metrics Insights queries can
     * query only the most recent three hours of metric data. For more information about Metrics Insights, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/query_with_cloudwatch-metrics-insights.html"
     * >Query your metrics with CloudWatch Metrics Insights</a>.
     * </p>
     * <p>
     * Calls to the <code>GetMetricData</code> API have a different pricing structure than calls to
     * <code>GetMetricStatistics</code>. For more information about pricing, see <a
     * href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     * <p>
     * If you omit <code>Unit</code> in your request, all data that was collected with any unit is returned, along with
     * the corresponding units that were specified when the data was reported to CloudWatch. If you specify a unit, the
     * operation returns only data that was collected with that unit specified. If you specify a unit that does not
     * match the data collected, the results of the operation are null. CloudWatch does not perform unit conversions.
     * </p>
     * <p>
     * <b>Using Metrics Insights queries with metric math</b>
     * </p>
     * <p>
     * You can't mix a Metric Insights query and metric math syntax in the same expression, but you can reference
     * results from a Metrics Insights query within other Metric math expressions. A Metrics Insights query without a
     * <b>GROUP BY</b> clause returns a single time-series (TS), and can be used as input for a metric math expression
     * that expects a single time series. A Metrics Insights query with a <b>GROUP BY</b> clause returns an array of
     * time-series (TS[]), and can be used as input for a metric math expression that expects an array of time series.
     * </p>
     *
     * @param getMetricDataRequest
     * @return Result of the GetMetricData operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataResponse getMetricData(GetMetricDataRequest getMetricDataRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricDataResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMetricDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricData");

            return clientHandler.execute(new ClientExecutionParams<GetMetricDataRequest, GetMetricDataResponse>()
                    .withOperationName("GetMetricData").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMetricDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets statistics for the specified metric.
     * </p>
     * <p>
     * The maximum number of data points returned from a single call is 1,440. If you request more than 1,440 data
     * points, CloudWatch returns an error. To reduce the number of data points, you can narrow the specified time range
     * and make multiple requests across adjacent time ranges, or you can increase the specified period. Data points are
     * not returned in chronological order.
     * </p>
     * <p>
     * CloudWatch aggregates data points based on the length of the period that you specify. For example, if you request
     * statistics with a one-hour period, CloudWatch aggregates all data points with time stamps that fall within each
     * one-hour period. Therefore, the number of values aggregated by CloudWatch is larger than the number of data
     * points returned.
     * </p>
     * <p>
     * CloudWatch needs raw data points to calculate percentile statistics. If you publish data using a statistic set
     * instead, you can only retrieve percentile statistics for this data if one of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The SampleCount value of the statistic set is 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Min and the Max values of the statistic set are equal.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Percentile statistics are not available for metrics when any of the metric values are negative numbers.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     * <p>
     * CloudWatch started retaining 5-minute and 1-hour metric data as of July 9, 2016.
     * </p>
     * <p>
     * For information about metrics and dimensions supported by Amazon Web Services services, see the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CW_Support_For_AWS.html">Amazon CloudWatch
     * Metrics and Dimensions Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param getMetricStatisticsRequest
     * @return Result of the GetMetricStatistics operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMetricStatisticsResponse getMetricStatistics(GetMetricStatisticsRequest getMetricStatisticsRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, InvalidParameterCombinationException,
            InternalServiceException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricStatisticsResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMetricStatisticsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricStatistics");

            return clientHandler.execute(new ClientExecutionParams<GetMetricStatisticsRequest, GetMetricStatisticsResponse>()
                    .withOperationName("GetMetricStatistics").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMetricStatisticsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricStatisticsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the metric stream that you specify.
     * </p>
     *
     * @param getMetricStreamRequest
     * @return Result of the GetMetricStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMetricStreamResponse getMetricStream(GetMetricStreamRequest getMetricStreamRequest)
            throws ResourceNotFoundException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<GetMetricStreamResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMetricStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricStream");

            return clientHandler.execute(new ClientExecutionParams<GetMetricStreamRequest, GetMetricStreamResponse>()
                    .withOperationName("GetMetricStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMetricStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use the <code>GetMetricWidgetImage</code> API to retrieve a snapshot graph of one or more Amazon
     * CloudWatch metrics as a bitmap image. You can then embed this image into your services and products, such as wiki
     * pages, reports, and documents. You could also retrieve images regularly, such as every minute, and create your
     * own custom live dashboard.
     * </p>
     * <p>
     * The graph you retrieve can include all CloudWatch metric graph features, including metric math and horizontal and
     * vertical annotations.
     * </p>
     * <p>
     * There is a limit of 20 transactions per second for this API. Each <code>GetMetricWidgetImage</code> action has
     * the following limits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * As many as 100 metrics in the graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * Up to 100 KB uncompressed payload.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMetricWidgetImageRequest
     * @return Result of the GetMetricWidgetImage operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricWidgetImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricWidgetImage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMetricWidgetImageResponse getMetricWidgetImage(GetMetricWidgetImageRequest getMetricWidgetImageRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricWidgetImageResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricWidgetImageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMetricWidgetImageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricWidgetImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricWidgetImage");

            return clientHandler.execute(new ClientExecutionParams<GetMetricWidgetImageRequest, GetMetricWidgetImageResponse>()
                    .withOperationName("GetMetricWidgetImage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMetricWidgetImageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricWidgetImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the dashboards for your account. If you include <code>DashboardNamePrefix</code>, only those
     * dashboards with names starting with the prefix are listed. Otherwise, all dashboards in your account are listed.
     * </p>
     * <p>
     * <code>ListDashboards</code> returns up to 1000 results on one page. If there are more than 1000 dashboards, you
     * can call <code>ListDashboards</code> again and include the value you received for <code>NextToken</code> in the
     * first call, to receive the next 1000 results.
     * </p>
     *
     * @param listDashboardsRequest
     * @return Result of the ListDashboards operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListDashboards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDashboardsResponse listDashboards(ListDashboardsRequest listDashboardsRequest)
            throws InvalidParameterValueException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<ListDashboardsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListDashboardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDashboardsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDashboardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDashboards");

            return clientHandler.execute(new ClientExecutionParams<ListDashboardsRequest, ListDashboardsResponse>()
                    .withOperationName("ListDashboards").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDashboardsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDashboardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list that contains the number of managed Contributor Insights rules in your account.
     * </p>
     *
     * @param listManagedInsightRulesRequest
     * @return Result of the ListManagedInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListManagedInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListManagedInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedInsightRulesResponse listManagedInsightRules(ListManagedInsightRulesRequest listManagedInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<ListManagedInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(ListManagedInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listManagedInsightRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedInsightRules");

            return clientHandler
                    .execute(new ClientExecutionParams<ListManagedInsightRulesRequest, ListManagedInsightRulesResponse>()
                            .withOperationName("ListManagedInsightRules").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listManagedInsightRulesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListManagedInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of metric streams in this account.
     * </p>
     *
     * @param listMetricStreamsRequest
     * @return Result of the ListMetricStreams operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMetricStreamsResponse listMetricStreams(ListMetricStreamsRequest listMetricStreamsRequest)
            throws InvalidNextTokenException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<ListMetricStreamsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListMetricStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMetricStreamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMetricStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMetricStreams");

            return clientHandler.execute(new ClientExecutionParams<ListMetricStreamsRequest, ListMetricStreamsResponse>()
                    .withOperationName("ListMetricStreams").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMetricStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMetricStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the specified metrics. You can use the returned metrics with <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a> to get statistical data.
     * </p>
     * <p>
     * Up to 500 results are returned for any one call. To retrieve additional results, use the returned token with
     * subsequent calls.
     * </p>
     * <p>
     * After you create a metric, allow up to 15 minutes for the metric to appear. To see metric statistics sooner, use
     * <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     * <p>
     * If you are using CloudWatch cross-account observability, you can use this operation in a monitoring account and
     * view metrics from the linked source accounts. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account.html"
     * >CloudWatch cross-account observability</a>.
     * </p>
     * <p>
     * <code>ListMetrics</code> doesn't return information about metrics if those metrics haven't reported data in the
     * past two weeks. To retrieve those metrics, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     *
     * @param listMetricsRequest
     * @return Result of the ListMetrics operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMetricsResponse listMetrics(ListMetricsRequest listMetricsRequest) throws InternalServiceException,
            InvalidParameterValueException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<ListMetricsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMetricsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMetrics");

            return clientHandler.execute(new ClientExecutionParams<ListMetricsRequest, ListMetricsResponse>()
                    .withOperationName("ListMetrics").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listMetricsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch resource. Currently, alarms and Contributor Insights rules support
     * tagging.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterValueException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, CloudWatchException {

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an anomaly detection model for a CloudWatch metric. You can use the model to display a band of expected
     * normal values when the metric is graphed.
     * </p>
     * <p>
     * If you have enabled unified cross-account observability, and this account is a monitoring account, the metric can
     * be in the same account or a source account. You can specify the account ID in the object you specify in the
     * <code>SingleMetricAnomalyDetector</code> parameter.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Anomaly_Detection.html"
     * >CloudWatch Anomaly Detection</a>.
     * </p>
     *
     * @param putAnomalyDetectorRequest
     * @return Result of the PutAnomalyDetector operation returned by the service.
     * @throws LimitExceededException
     *         The operation exceeded one or more limits.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutAnomalyDetector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutAnomalyDetectorResponse putAnomalyDetector(PutAnomalyDetectorRequest putAnomalyDetectorRequest)
            throws LimitExceededException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<PutAnomalyDetectorResponse> responseHandler = protocolFactory
                .createResponseHandler(PutAnomalyDetectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putAnomalyDetectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAnomalyDetector");

            return clientHandler.execute(new ClientExecutionParams<PutAnomalyDetectorRequest, PutAnomalyDetectorResponse>()
                    .withOperationName("PutAnomalyDetector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putAnomalyDetectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAnomalyDetectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a <i>composite alarm</i>. When you create a composite alarm, you specify a rule expression for
     * the alarm that takes into account the alarm states of other alarms that you have created. The composite alarm
     * goes into ALARM state only if all conditions of the rule are met.
     * </p>
     * <p>
     * The alarms specified in a composite alarm's rule expression can include metric alarms and other composite alarms.
     * The rule expression of a composite alarm can include as many as 100 underlying alarms. Any single alarm can be
     * included in the rule expressions of as many as 150 composite alarms.
     * </p>
     * <p>
     * Using composite alarms can reduce alarm noise. You can create multiple metric alarms, and also create a composite
     * alarm and set up alerts only for the composite alarm. For example, you could create a composite alarm that goes
     * into ALARM state only when more than one of the underlying metric alarms are in ALARM state.
     * </p>
     * <p>
     * Composite alarms can take the following actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Notify Amazon SNS topics.
     * </p>
     * </li>
     * <li>
     * <p>
     * Invoke Lambda functions.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create OpsItems in Systems Manager Ops Center.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create incidents in Systems Manager Incident Manager.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * It is possible to create a loop or cycle of composite alarms, where composite alarm A depends on composite alarm
     * B, and composite alarm B also depends on composite alarm A. In this scenario, you can't delete any composite
     * alarm that is part of the cycle because there is always still a composite alarm that depends on that alarm that
     * you want to delete.
     * </p>
     * <p>
     * To get out of such a situation, you must break the cycle by changing the rule of one of the composite alarms in
     * the cycle to remove a dependency that creates the cycle. The simplest change to make to break a cycle is to
     * change the <code>AlarmRule</code> of one of the alarms to <code>false</code>.
     * </p>
     * <p>
     * Additionally, the evaluation of composite alarms stops if CloudWatch detects a cycle in the evaluation path.
     * </p>
     * </note>
     * <p>
     * When this operation creates an alarm, the alarm state is immediately set to <code>INSUFFICIENT_DATA</code>. The
     * alarm is then evaluated and its state is set appropriately. Any actions associated with the new state are then
     * executed. For a composite alarm, this initial time after creation is the only time that the alarm can be in
     * <code>INSUFFICIENT_DATA</code> state.
     * </p>
     * <p>
     * When you update an existing alarm, its state is left unchanged, but the update completely overwrites the previous
     * configuration of the alarm.
     * </p>
     * <p>
     * To use this operation, you must be signed on with the <code>cloudwatch:PutCompositeAlarm</code> permission that
     * is scoped to <code>*</code>. You can't create a composite alarms if your
     * <code>cloudwatch:PutCompositeAlarm</code> permission has a narrower scope.
     * </p>
     * <p>
     * If you are an IAM user, you must have <code>iam:CreateServiceLinkedRole</code> to create a composite alarm that
     * has Systems Manager OpsItem actions.
     * </p>
     *
     * @param putCompositeAlarmRequest
     * @return Result of the PutCompositeAlarm operation returned by the service.
     * @throws LimitExceededException
     *         The quota for alarms for this customer has already been reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutCompositeAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutCompositeAlarm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutCompositeAlarmResponse putCompositeAlarm(PutCompositeAlarmRequest putCompositeAlarmRequest)
            throws LimitExceededException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutCompositeAlarmResponse> responseHandler = protocolFactory
                .createResponseHandler(PutCompositeAlarmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putCompositeAlarmRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCompositeAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCompositeAlarm");

            return clientHandler.execute(new ClientExecutionParams<PutCompositeAlarmRequest, PutCompositeAlarmResponse>()
                    .withOperationName("PutCompositeAlarm").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putCompositeAlarmRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutCompositeAlarmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a dashboard if it does not already exist, or updates an existing dashboard. If you update a dashboard,
     * the entire contents are replaced with what you specify here.
     * </p>
     * <p>
     * All dashboards in your account are global, not region-specific.
     * </p>
     * <p>
     * A simple way to create a dashboard using <code>PutDashboard</code> is to copy an existing dashboard. To copy an
     * existing dashboard using the console, you can load the dashboard and then use the View/edit source command in the
     * Actions menu to display the JSON block for that dashboard. Another way to copy a dashboard is to use
     * <code>GetDashboard</code>, and then use the data returned within <code>DashboardBody</code> as the template for
     * the new dashboard when you call <code>PutDashboard</code>.
     * </p>
     * <p>
     * When you create a dashboard with <code>PutDashboard</code>, a good practice is to add a text widget at the top of
     * the dashboard with a message that the dashboard was created by script and should not be changed in the console.
     * This message could also point console users to the location of the <code>DashboardBody</code> script or the
     * CloudFormation template used to create the dashboard.
     * </p>
     *
     * @param putDashboardRequest
     * @return Result of the PutDashboard operation returned by the service.
     * @throws DashboardInvalidInputErrorException
     *         Some part of the dashboard data is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutDashboard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutDashboardResponse putDashboard(PutDashboardRequest putDashboardRequest) throws DashboardInvalidInputErrorException,
            InternalServiceException, ConflictException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutDashboardResponse> responseHandler = protocolFactory
                .createResponseHandler(PutDashboardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putDashboardRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDashboard");

            return clientHandler.execute(new ClientExecutionParams<PutDashboardRequest, PutDashboardResponse>()
                    .withOperationName("PutDashboard").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Contributor Insights rule. Rules evaluate log events in a CloudWatch Logs log group, enabling you to
     * find contributor data for the log events in that log group. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/ContributorInsights.html">Using Contributor
     * Insights to Analyze High-Cardinality Data</a>.
     * </p>
     * <p>
     * If you create a rule, delete it, and then re-create it with the same name, historical data from the first time
     * the rule was created might not be available.
     * </p>
     *
     * @param putInsightRuleRequest
     * @return Result of the PutInsightRule operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws LimitExceededException
     *         The operation exceeded one or more limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutInsightRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutInsightRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutInsightRuleResponse putInsightRule(PutInsightRuleRequest putInsightRuleRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutInsightRuleResponse> responseHandler = protocolFactory
                .createResponseHandler(PutInsightRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putInsightRuleRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInsightRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInsightRule");

            return clientHandler.execute(new ClientExecutionParams<PutInsightRuleRequest, PutInsightRuleResponse>()
                    .withOperationName("PutInsightRule").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putInsightRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutInsightRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a managed Contributor Insights rule for a specified Amazon Web Services resource. When you enable a
     * managed rule, you create a Contributor Insights rule that collects data from Amazon Web Services services. You
     * cannot edit these rules with <code>PutInsightRule</code>. The rules can be enabled, disabled, and deleted using
     * <code>EnableInsightRules</code>, <code>DisableInsightRules</code>, and <code>DeleteInsightRules</code>. If a
     * previously created managed rule is currently disabled, a subsequent call to this API will re-enable it. Use
     * <code>ListManagedInsightRules</code> to describe all available rules.
     * </p>
     *
     * @param putManagedInsightRulesRequest
     * @return Result of the PutManagedInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutManagedInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutManagedInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutManagedInsightRulesResponse putManagedInsightRules(PutManagedInsightRulesRequest putManagedInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<PutManagedInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(PutManagedInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putManagedInsightRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putManagedInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutManagedInsightRules");

            return clientHandler
                    .execute(new ClientExecutionParams<PutManagedInsightRulesRequest, PutManagedInsightRulesResponse>()
                            .withOperationName("PutManagedInsightRules").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putManagedInsightRulesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutManagedInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates an alarm and associates it with the specified metric, metric math expression, anomaly
     * detection model, or Metrics Insights query. For more information about using a Metrics Insights query for an
     * alarm, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Create_Metrics_Insights_Alarm.html">Create
     * alarms on Metrics Insights queries</a>.
     * </p>
     * <p>
     * Alarms based on anomaly detection models cannot have Auto Scaling actions.
     * </p>
     * <p>
     * When this operation creates an alarm, the alarm state is immediately set to <code>INSUFFICIENT_DATA</code>. The
     * alarm is then evaluated and its state is set appropriately. Any actions associated with the new state are then
     * executed.
     * </p>
     * <p>
     * When you update an existing alarm, its state is left unchanged, but the update completely overwrites the previous
     * configuration of the alarm.
     * </p>
     * <p>
     * If you are an IAM user, you must have Amazon EC2 permissions for some alarm operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>iam:CreateServiceLinkedRole</code> permission for all alarms with EC2 actions
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>iam:CreateServiceLinkedRole</code> permissions to create an alarm with Systems Manager OpsItem or
     * response plan actions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The first time you create an alarm in the Amazon Web Services Management Console, the CLI, or by using the
     * PutMetricAlarm API, CloudWatch creates the necessary service-linked role for you. The service-linked roles are
     * called <code>AWSServiceRoleForCloudWatchEvents</code> and
     * <code>AWSServiceRoleForCloudWatchAlarms_ActionSSM</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role"
     * >Amazon Web Services service-linked role</a>.
     * </p>
     * <p>
     * Each <code>PutMetricAlarm</code> action has a maximum uncompressed payload of 120 KB.
     * </p>
     * <p>
     * <b>Cross-account alarms</b>
     * </p>
     * <p>
     * You can set an alarm on metrics in the current account, or in another account. To create a cross-account alarm
     * that watches a metric in a different account, you must have completed the following pre-requisites:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The account where the metrics are located (the <i>sharing account</i>) must already have a sharing role named
     * <b>CloudWatch-CrossAccountSharingRole</b>. If it does not already have this role, you must create it using the
     * instructions in <b>Set up a sharing account</b> in <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Cross-Account-Cross-Region.html#enable-cross-account-cross-Region"
     * > Cross-account cross-Region CloudWatch console</a>. The policy for that role must grant access to the ID of the
     * account where you are creating the alarm.
     * </p>
     * </li>
     * <li>
     * <p>
     * The account where you are creating the alarm (the <i>monitoring account</i>) must already have a service-linked
     * role named <b>AWSServiceRoleForCloudWatchCrossAccount</b> to allow CloudWatch to assume the sharing role in the
     * sharing account. If it does not, you must create it following the directions in <b>Set up a monitoring
     * account</b> in <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Cross-Account-Cross-Region.html#enable-cross-account-cross-Region"
     * > Cross-account cross-Region CloudWatch console</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param putMetricAlarmRequest
     * @return Result of the PutMetricAlarm operation returned by the service.
     * @throws LimitExceededException
     *         The quota for alarms for this customer has already been reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricAlarm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricAlarmResponse putMetricAlarm(PutMetricAlarmRequest putMetricAlarmRequest) throws LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutMetricAlarmResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricAlarmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMetricAlarmRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricAlarm");

            return clientHandler.execute(new ClientExecutionParams<PutMetricAlarmRequest, PutMetricAlarmResponse>()
                    .withOperationName("PutMetricAlarm").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putMetricAlarmRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutMetricAlarmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Publishes metric data to Amazon CloudWatch. CloudWatch associates the data with the specified metric. If the
     * specified metric does not exist, CloudWatch creates the metric. When CloudWatch creates a metric, it can take up
     * to fifteen minutes for the metric to appear in calls to <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html">ListMetrics</a>.
     * </p>
     * <p>
     * You can publish metrics with associated entity data (so that related telemetry can be found and viewed together),
     * or publish metric data by itself. To send entity data with your metrics, use the <code>EntityMetricData</code>
     * parameter. To send metrics without entity data, use the <code>MetricData</code> parameter. The
     * <code>EntityMetricData</code> structure includes <code>MetricData</code> structures for the metric data.
     * </p>
     * <p>
     * You can publish either individual values in the <code>Value</code> field, or arrays of values and the number of
     * times each value occurred during the period by using the <code>Values</code> and <code>Counts</code> fields in
     * the <code>MetricData</code> structure. Using the <code>Values</code> and <code>Counts</code> method enables you
     * to publish up to 150 values per metric with one <code>PutMetricData</code> request, and supports retrieving
     * percentile statistics on this data.
     * </p>
     * <p>
     * Each <code>PutMetricData</code> request is limited to 1 MB in size for HTTP POST requests. You can send a payload
     * compressed by gzip. Each request is also limited to no more than 1000 different metrics (across both the
     * <code>MetricData</code> and <code>EntityMetricData</code> properties).
     * </p>
     * <p>
     * Although the <code>Value</code> parameter accepts numbers of type <code>Double</code>, CloudWatch rejects values
     * that are either too small or too large. Values must be in the range of -2^360 to 2^360. In addition, special
     * values (for example, NaN, +Infinity, -Infinity) are not supported.
     * </p>
     * <p>
     * You can use up to 30 dimensions per metric to further clarify what data the metric collects. Each dimension
     * consists of a Name and Value pair. For more information about specifying dimensions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
     * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * You specify the time stamp to be associated with each data point. You can specify time stamps that are as much as
     * two weeks before the current date, and as much as 2 hours after the current day and time.
     * </p>
     * <p>
     * Data points with time stamps from 24 hours ago or longer can take at least 48 hours to become available for <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a> from the time they are submitted. Data points with time stamps between 3 and 24 hours ago
     * can take as much as 2 hours to become available for <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     * <p>
     * CloudWatch needs raw data points to calculate percentile statistics. If you publish data using a statistic set
     * instead, you can only retrieve percentile statistics for this data if one of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>SampleCount</code> value of the statistic set is 1 and <code>Min</code>, <code>Max</code>, and
     * <code>Sum</code> are all equal.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>Min</code> and <code>Max</code> are equal, and <code>Sum</code> is equal to <code>Min</code> multiplied
     * by <code>SampleCount</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param putMetricDataRequest
     * @return Result of the PutMetricData operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricDataResponse putMetricData(PutMetricDataRequest putMetricDataRequest) throws InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, InternalServiceException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutMetricDataResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMetricDataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricData");

            return clientHandler.execute(new ClientExecutionParams<PutMetricDataRequest, PutMetricDataResponse>()
                    .withOperationName("PutMetricData")
                    .withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration)
                    .withInput(putMetricDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .putExecutionAttribute(SdkInternalExecutionAttribute.REQUEST_COMPRESSION,
                            RequestCompression.builder().encodings("gzip").isStreaming(false).build())
                    .withMarshaller(new PutMetricDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a metric stream. Metric streams can automatically stream CloudWatch metrics to Amazon Web
     * Services destinations, including Amazon S3, and to many third-party solutions.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Metric-Streams.html"> Using
     * Metric Streams</a>.
     * </p>
     * <p>
     * To create a metric stream, you must be signed in to an account that has the <code>iam:PassRole</code> permission
     * and either the <code>CloudWatchFullAccess</code> policy or the <code>cloudwatch:PutMetricStream</code>
     * permission.
     * </p>
     * <p>
     * When you create or update a metric stream, you choose one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Stream metrics from all metric namespaces in the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream metrics from all metric namespaces in the account, except for the namespaces that you list in
     * <code>ExcludeFilters</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream metrics from only the metric namespaces that you list in <code>IncludeFilters</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By default, a metric stream always sends the <code>MAX</code>, <code>MIN</code>, <code>SUM</code>, and
     * <code>SAMPLECOUNT</code> statistics for each metric that is streamed. You can use the
     * <code>StatisticsConfigurations</code> parameter to have the metric stream send additional statistics in the
     * stream. Streaming additional statistics incurs additional costs. For more information, see <a
     * href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * When you use <code>PutMetricStream</code> to create a new metric stream, the stream is created in the
     * <code>running</code> state. If you use it to update an existing stream, the state of the stream is not changed.
     * </p>
     * <p>
     * If you are using CloudWatch cross-account observability and you create a metric stream in a monitoring account,
     * you can choose whether to include metrics from source accounts in the stream. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account.html"
     * >CloudWatch cross-account observability</a>.
     * </p>
     *
     * @param putMetricStreamRequest
     * @return Result of the PutMetricStream operation returned by the service.
     * @throws ConcurrentModificationException
     *         More than one process tried to modify a resource at the same time.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutMetricStreamResponse putMetricStream(PutMetricStreamRequest putMetricStreamRequest)
            throws ConcurrentModificationException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<PutMetricStreamResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMetricStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricStream");

            return clientHandler.execute(new ClientExecutionParams<PutMetricStreamRequest, PutMetricStreamResponse>()
                    .withOperationName("PutMetricStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putMetricStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutMetricStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Temporarily sets the state of an alarm for testing purposes. When the updated state differs from the previous
     * value, the action configured for the appropriate state is invoked. For example, if your alarm is configured to
     * send an Amazon SNS message when an alarm is triggered, temporarily changing the alarm state to <code>ALARM</code>
     * sends an SNS message.
     * </p>
     * <p>
     * Metric alarms returns to their actual state quickly, often within seconds. Because the metric alarm state change
     * happens quickly, it is typically only visible in the alarm's <b>History</b> tab in the Amazon CloudWatch console
     * or through <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmHistory.html"
     * >DescribeAlarmHistory</a>.
     * </p>
     * <p>
     * If you use <code>SetAlarmState</code> on a composite alarm, the composite alarm is not guaranteed to return to
     * its actual state. It returns to its actual state only once any of its children alarms change state. It is also
     * reevaluated if you update its configuration.
     * </p>
     * <p>
     * If an alarm triggers EC2 Auto Scaling policies or application Auto Scaling policies, you must include information
     * in the <code>StateReasonData</code> parameter to enable the policy to take the correct action.
     * </p>
     *
     * @param setAlarmStateRequest
     * @return Result of the SetAlarmState operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InvalidFormatException
     *         Data was not syntactically valid JSON.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.SetAlarmState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/SetAlarmState" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetAlarmStateResponse setAlarmState(SetAlarmStateRequest setAlarmStateRequest) throws ResourceNotFoundException,
            InvalidFormatException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<SetAlarmStateResponse> responseHandler = protocolFactory
                .createResponseHandler(SetAlarmStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setAlarmStateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setAlarmStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetAlarmState");

            return clientHandler.execute(new ClientExecutionParams<SetAlarmStateRequest, SetAlarmStateResponse>()
                    .withOperationName("SetAlarmState").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(setAlarmStateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetAlarmStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the streaming of metrics for one or more of your metric streams.
     * </p>
     *
     * @param startMetricStreamsRequest
     * @return Result of the StartMetricStreams operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.StartMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/StartMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartMetricStreamsResponse startMetricStreams(StartMetricStreamsRequest startMetricStreamsRequest)
            throws InternalServiceException, InvalidParameterValueException, MissingRequiredParameterException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<StartMetricStreamsResponse> responseHandler = protocolFactory
                .createResponseHandler(StartMetricStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMetricStreamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMetricStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMetricStreams");

            return clientHandler.execute(new ClientExecutionParams<StartMetricStreamsRequest, StartMetricStreamsResponse>()
                    .withOperationName("StartMetricStreams").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startMetricStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartMetricStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the streaming of metrics for one or more of your metric streams.
     * </p>
     *
     * @param stopMetricStreamsRequest
     * @return Result of the StopMetricStreams operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.StopMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/StopMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopMetricStreamsResponse stopMetricStreams(StopMetricStreamsRequest stopMetricStreamsRequest)
            throws InternalServiceException, InvalidParameterValueException, MissingRequiredParameterException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<StopMetricStreamsResponse> responseHandler = protocolFactory
                .createResponseHandler(StopMetricStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopMetricStreamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMetricStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMetricStreams");

            return clientHandler.execute(new ClientExecutionParams<StopMetricStreamsRequest, StopMetricStreamsResponse>()
                    .withOperationName("StopMetricStreams").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopMetricStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopMetricStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch resource. Currently, the only CloudWatch
     * resources that can be tagged are alarms and Contributor Insights rules.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with an alarm that already has tags. If you specify a new tag key
     * for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key that
     * is already associated with the alarm, the new tag value that you specify replaces the previous value for that
     * tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a CloudWatch resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws ConcurrentModificationException
     *         More than one process tried to modify a resource at the same time.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterValueException,
            ResourceNotFoundException, ConcurrentModificationException, InternalServiceException, ConflictException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws ConcurrentModificationException
     *         More than one process tried to modify a resource at the same time.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterValueException,
            ResourceNotFoundException, ConcurrentModificationException, InternalServiceException, ConflictException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link CloudWatchWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link CloudWatchWaiter}
     */
    @Override
    public CloudWatchWaiter waiter() {
        return CloudWatchWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        CloudWatchServiceClientConfigurationBuilder serviceConfigBuilder = new CloudWatchServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValue")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFormat")
                                .exceptionBuilderSupplier(InvalidFormatException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingParameter")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextToken")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombination")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(DashboardNotFoundErrorException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterInput")
                                .exceptionBuilderSupplier(DashboardInvalidInputErrorException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(CloudWatchException::builder).build();
    }

    @Override
    public final CloudWatchServiceClientConfiguration serviceClientConfiguration() {
        return new CloudWatchServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
