/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the history of a specific alarm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AlarmHistoryItem implements SdkPojo, Serializable,
        ToCopyableBuilder<AlarmHistoryItem.Builder, AlarmHistoryItem> {
    private static final SdkField<String> ALARM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmName").getter(getter(AlarmHistoryItem::alarmName)).setter(setter(Builder::alarmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmName").build()).build();

    private static final SdkField<String> ALARM_CONTRIBUTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmContributorId").getter(getter(AlarmHistoryItem::alarmContributorId))
            .setter(setter(Builder::alarmContributorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmContributorId").build())
            .build();

    private static final SdkField<String> ALARM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AlarmType").getter(getter(AlarmHistoryItem::alarmTypeAsString)).setter(setter(Builder::alarmType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmType").build()).build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(AlarmHistoryItem::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<String> HISTORY_ITEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HistoryItemType").getter(getter(AlarmHistoryItem::historyItemTypeAsString))
            .setter(setter(Builder::historyItemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HistoryItemType").build()).build();

    private static final SdkField<String> HISTORY_SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HistorySummary").getter(getter(AlarmHistoryItem::historySummary))
            .setter(setter(Builder::historySummary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HistorySummary").build()).build();

    private static final SdkField<String> HISTORY_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("HistoryData").getter(getter(AlarmHistoryItem::historyData)).setter(setter(Builder::historyData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HistoryData").build()).build();

    private static final SdkField<Map<String, String>> ALARM_CONTRIBUTOR_ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AlarmContributorAttributes")
            .getter(getter(AlarmHistoryItem::alarmContributorAttributes))
            .setter(setter(Builder::alarmContributorAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmContributorAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALARM_NAME_FIELD,
            ALARM_CONTRIBUTOR_ID_FIELD, ALARM_TYPE_FIELD, TIMESTAMP_FIELD, HISTORY_ITEM_TYPE_FIELD, HISTORY_SUMMARY_FIELD,
            HISTORY_DATA_FIELD, ALARM_CONTRIBUTOR_ATTRIBUTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String alarmName;

    private final String alarmContributorId;

    private final String alarmType;

    private final Instant timestamp;

    private final String historyItemType;

    private final String historySummary;

    private final String historyData;

    private final Map<String, String> alarmContributorAttributes;

    private AlarmHistoryItem(BuilderImpl builder) {
        this.alarmName = builder.alarmName;
        this.alarmContributorId = builder.alarmContributorId;
        this.alarmType = builder.alarmType;
        this.timestamp = builder.timestamp;
        this.historyItemType = builder.historyItemType;
        this.historySummary = builder.historySummary;
        this.historyData = builder.historyData;
        this.alarmContributorAttributes = builder.alarmContributorAttributes;
    }

    /**
     * <p>
     * The descriptive name for the alarm.
     * </p>
     * 
     * @return The descriptive name for the alarm.
     */
    public final String alarmName() {
        return alarmName;
    }

    /**
     * <p>
     * The unique identifier of the alarm contributor associated with this history item, if applicable.
     * </p>
     * 
     * @return The unique identifier of the alarm contributor associated with this history item, if applicable.
     */
    public final String alarmContributorId() {
        return alarmContributorId;
    }

    /**
     * <p>
     * The type of alarm, either metric alarm or composite alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #alarmType} will
     * return {@link AlarmType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #alarmTypeAsString}.
     * </p>
     * 
     * @return The type of alarm, either metric alarm or composite alarm.
     * @see AlarmType
     */
    public final AlarmType alarmType() {
        return AlarmType.fromValue(alarmType);
    }

    /**
     * <p>
     * The type of alarm, either metric alarm or composite alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #alarmType} will
     * return {@link AlarmType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #alarmTypeAsString}.
     * </p>
     * 
     * @return The type of alarm, either metric alarm or composite alarm.
     * @see AlarmType
     */
    public final String alarmTypeAsString() {
        return alarmType;
    }

    /**
     * <p>
     * The time stamp for the alarm history item.
     * </p>
     * 
     * @return The time stamp for the alarm history item.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The type of alarm history item.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #historyItemType}
     * will return {@link HistoryItemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #historyItemTypeAsString}.
     * </p>
     * 
     * @return The type of alarm history item.
     * @see HistoryItemType
     */
    public final HistoryItemType historyItemType() {
        return HistoryItemType.fromValue(historyItemType);
    }

    /**
     * <p>
     * The type of alarm history item.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #historyItemType}
     * will return {@link HistoryItemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #historyItemTypeAsString}.
     * </p>
     * 
     * @return The type of alarm history item.
     * @see HistoryItemType
     */
    public final String historyItemTypeAsString() {
        return historyItemType;
    }

    /**
     * <p>
     * A summary of the alarm history, in text format.
     * </p>
     * 
     * @return A summary of the alarm history, in text format.
     */
    public final String historySummary() {
        return historySummary;
    }

    /**
     * <p>
     * Data about the alarm, in JSON format.
     * </p>
     * 
     * @return Data about the alarm, in JSON format.
     */
    public final String historyData() {
        return historyData;
    }

    /**
     * For responses, this returns true if the service returned a value for the AlarmContributorAttributes property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAlarmContributorAttributes() {
        return alarmContributorAttributes != null && !(alarmContributorAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of attributes that describe the alarm contributor associated with this history item, providing context
     * about the contributor's characteristics at the time of the event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlarmContributorAttributes} method.
     * </p>
     * 
     * @return A map of attributes that describe the alarm contributor associated with this history item, providing
     *         context about the contributor's characteristics at the time of the event.
     */
    public final Map<String, String> alarmContributorAttributes() {
        return alarmContributorAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(alarmName());
        hashCode = 31 * hashCode + Objects.hashCode(alarmContributorId());
        hashCode = 31 * hashCode + Objects.hashCode(alarmTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(historyItemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(historySummary());
        hashCode = 31 * hashCode + Objects.hashCode(historyData());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlarmContributorAttributes() ? alarmContributorAttributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AlarmHistoryItem)) {
            return false;
        }
        AlarmHistoryItem other = (AlarmHistoryItem) obj;
        return Objects.equals(alarmName(), other.alarmName()) && Objects.equals(alarmContributorId(), other.alarmContributorId())
                && Objects.equals(alarmTypeAsString(), other.alarmTypeAsString())
                && Objects.equals(timestamp(), other.timestamp())
                && Objects.equals(historyItemTypeAsString(), other.historyItemTypeAsString())
                && Objects.equals(historySummary(), other.historySummary()) && Objects.equals(historyData(), other.historyData())
                && hasAlarmContributorAttributes() == other.hasAlarmContributorAttributes()
                && Objects.equals(alarmContributorAttributes(), other.alarmContributorAttributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AlarmHistoryItem").add("AlarmName", alarmName()).add("AlarmContributorId", alarmContributorId())
                .add("AlarmType", alarmTypeAsString()).add("Timestamp", timestamp())
                .add("HistoryItemType", historyItemTypeAsString()).add("HistorySummary", historySummary())
                .add("HistoryData", historyData())
                .add("AlarmContributorAttributes", hasAlarmContributorAttributes() ? alarmContributorAttributes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AlarmName":
            return Optional.ofNullable(clazz.cast(alarmName()));
        case "AlarmContributorId":
            return Optional.ofNullable(clazz.cast(alarmContributorId()));
        case "AlarmType":
            return Optional.ofNullable(clazz.cast(alarmTypeAsString()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "HistoryItemType":
            return Optional.ofNullable(clazz.cast(historyItemTypeAsString()));
        case "HistorySummary":
            return Optional.ofNullable(clazz.cast(historySummary()));
        case "HistoryData":
            return Optional.ofNullable(clazz.cast(historyData()));
        case "AlarmContributorAttributes":
            return Optional.ofNullable(clazz.cast(alarmContributorAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AlarmName", ALARM_NAME_FIELD);
        map.put("AlarmContributorId", ALARM_CONTRIBUTOR_ID_FIELD);
        map.put("AlarmType", ALARM_TYPE_FIELD);
        map.put("Timestamp", TIMESTAMP_FIELD);
        map.put("HistoryItemType", HISTORY_ITEM_TYPE_FIELD);
        map.put("HistorySummary", HISTORY_SUMMARY_FIELD);
        map.put("HistoryData", HISTORY_DATA_FIELD);
        map.put("AlarmContributorAttributes", ALARM_CONTRIBUTOR_ATTRIBUTES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AlarmHistoryItem, T> g) {
        return obj -> g.apply((AlarmHistoryItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AlarmHistoryItem> {
        /**
         * <p>
         * The descriptive name for the alarm.
         * </p>
         * 
         * @param alarmName
         *        The descriptive name for the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmName(String alarmName);

        /**
         * <p>
         * The unique identifier of the alarm contributor associated with this history item, if applicable.
         * </p>
         * 
         * @param alarmContributorId
         *        The unique identifier of the alarm contributor associated with this history item, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmContributorId(String alarmContributorId);

        /**
         * <p>
         * The type of alarm, either metric alarm or composite alarm.
         * </p>
         * 
         * @param alarmType
         *        The type of alarm, either metric alarm or composite alarm.
         * @see AlarmType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AlarmType
         */
        Builder alarmType(String alarmType);

        /**
         * <p>
         * The type of alarm, either metric alarm or composite alarm.
         * </p>
         * 
         * @param alarmType
         *        The type of alarm, either metric alarm or composite alarm.
         * @see AlarmType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AlarmType
         */
        Builder alarmType(AlarmType alarmType);

        /**
         * <p>
         * The time stamp for the alarm history item.
         * </p>
         * 
         * @param timestamp
         *        The time stamp for the alarm history item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The type of alarm history item.
         * </p>
         * 
         * @param historyItemType
         *        The type of alarm history item.
         * @see HistoryItemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HistoryItemType
         */
        Builder historyItemType(String historyItemType);

        /**
         * <p>
         * The type of alarm history item.
         * </p>
         * 
         * @param historyItemType
         *        The type of alarm history item.
         * @see HistoryItemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HistoryItemType
         */
        Builder historyItemType(HistoryItemType historyItemType);

        /**
         * <p>
         * A summary of the alarm history, in text format.
         * </p>
         * 
         * @param historySummary
         *        A summary of the alarm history, in text format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder historySummary(String historySummary);

        /**
         * <p>
         * Data about the alarm, in JSON format.
         * </p>
         * 
         * @param historyData
         *        Data about the alarm, in JSON format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder historyData(String historyData);

        /**
         * <p>
         * A map of attributes that describe the alarm contributor associated with this history item, providing context
         * about the contributor's characteristics at the time of the event.
         * </p>
         * 
         * @param alarmContributorAttributes
         *        A map of attributes that describe the alarm contributor associated with this history item, providing
         *        context about the contributor's characteristics at the time of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmContributorAttributes(Map<String, String> alarmContributorAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String alarmName;

        private String alarmContributorId;

        private String alarmType;

        private Instant timestamp;

        private String historyItemType;

        private String historySummary;

        private String historyData;

        private Map<String, String> alarmContributorAttributes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AlarmHistoryItem model) {
            alarmName(model.alarmName);
            alarmContributorId(model.alarmContributorId);
            alarmType(model.alarmType);
            timestamp(model.timestamp);
            historyItemType(model.historyItemType);
            historySummary(model.historySummary);
            historyData(model.historyData);
            alarmContributorAttributes(model.alarmContributorAttributes);
        }

        public final String getAlarmName() {
            return alarmName;
        }

        public final void setAlarmName(String alarmName) {
            this.alarmName = alarmName;
        }

        @Override
        public final Builder alarmName(String alarmName) {
            this.alarmName = alarmName;
            return this;
        }

        public final String getAlarmContributorId() {
            return alarmContributorId;
        }

        public final void setAlarmContributorId(String alarmContributorId) {
            this.alarmContributorId = alarmContributorId;
        }

        @Override
        public final Builder alarmContributorId(String alarmContributorId) {
            this.alarmContributorId = alarmContributorId;
            return this;
        }

        public final String getAlarmType() {
            return alarmType;
        }

        public final void setAlarmType(String alarmType) {
            this.alarmType = alarmType;
        }

        @Override
        public final Builder alarmType(String alarmType) {
            this.alarmType = alarmType;
            return this;
        }

        @Override
        public final Builder alarmType(AlarmType alarmType) {
            this.alarmType(alarmType == null ? null : alarmType.toString());
            return this;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final String getHistoryItemType() {
            return historyItemType;
        }

        public final void setHistoryItemType(String historyItemType) {
            this.historyItemType = historyItemType;
        }

        @Override
        public final Builder historyItemType(String historyItemType) {
            this.historyItemType = historyItemType;
            return this;
        }

        @Override
        public final Builder historyItemType(HistoryItemType historyItemType) {
            this.historyItemType(historyItemType == null ? null : historyItemType.toString());
            return this;
        }

        public final String getHistorySummary() {
            return historySummary;
        }

        public final void setHistorySummary(String historySummary) {
            this.historySummary = historySummary;
        }

        @Override
        public final Builder historySummary(String historySummary) {
            this.historySummary = historySummary;
            return this;
        }

        public final String getHistoryData() {
            return historyData;
        }

        public final void setHistoryData(String historyData) {
            this.historyData = historyData;
        }

        @Override
        public final Builder historyData(String historyData) {
            this.historyData = historyData;
            return this;
        }

        public final Map<String, String> getAlarmContributorAttributes() {
            if (alarmContributorAttributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return alarmContributorAttributes;
        }

        public final void setAlarmContributorAttributes(Map<String, String> alarmContributorAttributes) {
            this.alarmContributorAttributes = ContributorAttributesCopier.copy(alarmContributorAttributes);
        }

        @Override
        public final Builder alarmContributorAttributes(Map<String, String> alarmContributorAttributes) {
            this.alarmContributorAttributes = ContributorAttributesCopier.copy(alarmContributorAttributes);
            return this;
        }

        @Override
        public AlarmHistoryItem build() {
            return new AlarmHistoryItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
