/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchevents.model;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.cloudwatchevents.transform.BatchParametersMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The custom parameters to be used when the target is an AWS Batch job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchParameters implements StructuredPojo, ToCopyableBuilder<BatchParameters.Builder, BatchParameters> {
    private final String jobDefinition;

    private final String jobName;

    private final BatchArrayProperties arrayProperties;

    private final BatchRetryStrategy retryStrategy;

    private BatchParameters(BuilderImpl builder) {
        this.jobDefinition = builder.jobDefinition;
        this.jobName = builder.jobName;
        this.arrayProperties = builder.arrayProperties;
        this.retryStrategy = builder.retryStrategy;
    }

    /**
     * <p>
     * The ARN or name of the job definition to use if the event target is an AWS Batch job. This job definition must
     * already exist.
     * </p>
     * 
     * @return The ARN or name of the job definition to use if the event target is an AWS Batch job. This job definition
     *         must already exist.
     */
    public String jobDefinition() {
        return jobDefinition;
    }

    /**
     * <p>
     * The name to use for this execution of the job, if the target is an AWS Batch job.
     * </p>
     * 
     * @return The name to use for this execution of the job, if the target is an AWS Batch job.
     */
    public String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The array properties for the submitted job, such as the size of the array. The array size can be between 2 and
     * 10,000. If you specify array properties for a job, it becomes an array job. This parameter is used only if the
     * target is an AWS Batch job.
     * </p>
     * 
     * @return The array properties for the submitted job, such as the size of the array. The array size can be between
     *         2 and 10,000. If you specify array properties for a job, it becomes an array job. This parameter is used
     *         only if the target is an AWS Batch job.
     */
    public BatchArrayProperties arrayProperties() {
        return arrayProperties;
    }

    /**
     * <p>
     * The retry strategy to use for failed jobs, if the target is an AWS Batch job. The retry strategy is the number of
     * times to retry the failed job execution. Valid values are 1–10. When you specify a retry strategy here, it
     * overrides the retry strategy defined in the job definition.
     * </p>
     * 
     * @return The retry strategy to use for failed jobs, if the target is an AWS Batch job. The retry strategy is the
     *         number of times to retry the failed job execution. Valid values are 1–10. When you specify a retry
     *         strategy here, it overrides the retry strategy defined in the job definition.
     */
    public BatchRetryStrategy retryStrategy() {
        return retryStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(arrayProperties());
        hashCode = 31 * hashCode + Objects.hashCode(retryStrategy());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchParameters)) {
            return false;
        }
        BatchParameters other = (BatchParameters) obj;
        return Objects.equals(jobDefinition(), other.jobDefinition()) && Objects.equals(jobName(), other.jobName())
                && Objects.equals(arrayProperties(), other.arrayProperties())
                && Objects.equals(retryStrategy(), other.retryStrategy());
    }

    @Override
    public String toString() {
        return ToString.builder("BatchParameters").add("JobDefinition", jobDefinition()).add("JobName", jobName())
                .add("ArrayProperties", arrayProperties()).add("RetryStrategy", retryStrategy()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JobDefinition":
            return Optional.ofNullable(clazz.cast(jobDefinition()));
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "ArrayProperties":
            return Optional.ofNullable(clazz.cast(arrayProperties()));
        case "RetryStrategy":
            return Optional.ofNullable(clazz.cast(retryStrategy()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        BatchParametersMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, BatchParameters> {
        /**
         * <p>
         * The ARN or name of the job definition to use if the event target is an AWS Batch job. This job definition
         * must already exist.
         * </p>
         * 
         * @param jobDefinition
         *        The ARN or name of the job definition to use if the event target is an AWS Batch job. This job
         *        definition must already exist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDefinition(String jobDefinition);

        /**
         * <p>
         * The name to use for this execution of the job, if the target is an AWS Batch job.
         * </p>
         * 
         * @param jobName
         *        The name to use for this execution of the job, if the target is an AWS Batch job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The array properties for the submitted job, such as the size of the array. The array size can be between 2
         * and 10,000. If you specify array properties for a job, it becomes an array job. This parameter is used only
         * if the target is an AWS Batch job.
         * </p>
         * 
         * @param arrayProperties
         *        The array properties for the submitted job, such as the size of the array. The array size can be
         *        between 2 and 10,000. If you specify array properties for a job, it becomes an array job. This
         *        parameter is used only if the target is an AWS Batch job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arrayProperties(BatchArrayProperties arrayProperties);

        /**
         * <p>
         * The array properties for the submitted job, such as the size of the array. The array size can be between 2
         * and 10,000. If you specify array properties for a job, it becomes an array job. This parameter is used only
         * if the target is an AWS Batch job.
         * </p>
         * This is a convenience that creates an instance of the {@link BatchArrayProperties.Builder} avoiding the need
         * to create one manually via {@link BatchArrayProperties#builder()}.
         *
         * When the {@link Consumer} completes, {@link BatchArrayProperties.Builder#build()} is called immediately and
         * its result is passed to {@link #arrayProperties(BatchArrayProperties)}.
         * 
         * @param arrayProperties
         *        a consumer that will call methods on {@link BatchArrayProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #arrayProperties(BatchArrayProperties)
         */
        default Builder arrayProperties(Consumer<BatchArrayProperties.Builder> arrayProperties) {
            return arrayProperties(BatchArrayProperties.builder().applyMutation(arrayProperties).build());
        }

        /**
         * <p>
         * The retry strategy to use for failed jobs, if the target is an AWS Batch job. The retry strategy is the
         * number of times to retry the failed job execution. Valid values are 1–10. When you specify a retry strategy
         * here, it overrides the retry strategy defined in the job definition.
         * </p>
         * 
         * @param retryStrategy
         *        The retry strategy to use for failed jobs, if the target is an AWS Batch job. The retry strategy is
         *        the number of times to retry the failed job execution. Valid values are 1–10. When you specify a retry
         *        strategy here, it overrides the retry strategy defined in the job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryStrategy(BatchRetryStrategy retryStrategy);

        /**
         * <p>
         * The retry strategy to use for failed jobs, if the target is an AWS Batch job. The retry strategy is the
         * number of times to retry the failed job execution. Valid values are 1–10. When you specify a retry strategy
         * here, it overrides the retry strategy defined in the job definition.
         * </p>
         * This is a convenience that creates an instance of the {@link BatchRetryStrategy.Builder} avoiding the need to
         * create one manually via {@link BatchRetryStrategy#builder()}.
         *
         * When the {@link Consumer} completes, {@link BatchRetryStrategy.Builder#build()} is called immediately and its
         * result is passed to {@link #retryStrategy(BatchRetryStrategy)}.
         * 
         * @param retryStrategy
         *        a consumer that will call methods on {@link BatchRetryStrategy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retryStrategy(BatchRetryStrategy)
         */
        default Builder retryStrategy(Consumer<BatchRetryStrategy.Builder> retryStrategy) {
            return retryStrategy(BatchRetryStrategy.builder().applyMutation(retryStrategy).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String jobDefinition;

        private String jobName;

        private BatchArrayProperties arrayProperties;

        private BatchRetryStrategy retryStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(BatchParameters model) {
            jobDefinition(model.jobDefinition);
            jobName(model.jobName);
            arrayProperties(model.arrayProperties);
            retryStrategy(model.retryStrategy);
        }

        public final String getJobDefinition() {
            return jobDefinition;
        }

        @Override
        public final Builder jobDefinition(String jobDefinition) {
            this.jobDefinition = jobDefinition;
            return this;
        }

        public final void setJobDefinition(String jobDefinition) {
            this.jobDefinition = jobDefinition;
        }

        public final String getJobName() {
            return jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        public final BatchArrayProperties.Builder getArrayProperties() {
            return arrayProperties != null ? arrayProperties.toBuilder() : null;
        }

        @Override
        public final Builder arrayProperties(BatchArrayProperties arrayProperties) {
            this.arrayProperties = arrayProperties;
            return this;
        }

        public final void setArrayProperties(BatchArrayProperties.BuilderImpl arrayProperties) {
            this.arrayProperties = arrayProperties != null ? arrayProperties.build() : null;
        }

        public final BatchRetryStrategy.Builder getRetryStrategy() {
            return retryStrategy != null ? retryStrategy.toBuilder() : null;
        }

        @Override
        public final Builder retryStrategy(BatchRetryStrategy retryStrategy) {
            this.retryStrategy = retryStrategy;
            return this;
        }

        public final void setRetryStrategy(BatchRetryStrategy.BuilderImpl retryStrategy) {
            this.retryStrategy = retryStrategy != null ? retryStrategy.build() : null;
        }

        @Override
        public BatchParameters build() {
            return new BatchParameters(this);
        }
    }
}
