/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchevents;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudwatchevents.model.ActivateEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ActivateEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CancelReplayRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CancelReplayResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CloudWatchEventsException;
import software.amazon.awssdk.services.cloudwatchevents.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateApiDestinationRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateApiDestinationResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateArchiveRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateArchiveResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateConnectionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateConnectionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreateEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.CreatePartnerEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.CreatePartnerEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeactivateEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeactivateEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeauthorizeConnectionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeauthorizeConnectionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteApiDestinationRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteApiDestinationResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteArchiveRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteArchiveResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeletePartnerEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeletePartnerEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DeleteRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeApiDestinationRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeApiDestinationResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeArchiveRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeArchiveResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeConnectionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeConnectionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventBusRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventBusResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribePartnerEventSourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribePartnerEventSourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeReplayRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeReplayResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DescribeRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.DisableRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.DisableRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.EnableRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.EnableRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.IllegalStatusException;
import software.amazon.awssdk.services.cloudwatchevents.model.InternalException;
import software.amazon.awssdk.services.cloudwatchevents.model.InvalidEventPatternException;
import software.amazon.awssdk.services.cloudwatchevents.model.InvalidStateException;
import software.amazon.awssdk.services.cloudwatchevents.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatchevents.model.ListApiDestinationsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListApiDestinationsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListArchivesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListArchivesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListConnectionsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListConnectionsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventBusesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventBusesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventSourcesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListEventSourcesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourceAccountsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourceAccountsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourcesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListPartnerEventSourcesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListReplaysRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListReplaysResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRuleNamesByTargetRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRuleNamesByTargetResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRulesRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListRulesResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTargetsByRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.ListTargetsByRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ManagedRuleException;
import software.amazon.awssdk.services.cloudwatchevents.model.OperationDisabledException;
import software.amazon.awssdk.services.cloudwatchevents.model.PolicyLengthExceededException;
import software.amazon.awssdk.services.cloudwatchevents.model.PutEventsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutEventsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPartnerEventsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPartnerEventsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPermissionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutPermissionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutRuleRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutRuleResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.PutTargetsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.PutTargetsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.RemovePermissionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.RemovePermissionResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.RemoveTargetsRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.RemoveTargetsResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.cloudwatchevents.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatchevents.model.StartReplayRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.StartReplayResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.TestEventPatternRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.TestEventPatternResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.UpdateApiDestinationRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.UpdateApiDestinationResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.UpdateArchiveRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.UpdateArchiveResponse;
import software.amazon.awssdk.services.cloudwatchevents.model.UpdateConnectionRequest;
import software.amazon.awssdk.services.cloudwatchevents.model.UpdateConnectionResponse;
import software.amazon.awssdk.services.cloudwatchevents.transform.ActivateEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CancelReplayRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreateApiDestinationRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreateArchiveRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreateConnectionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreateEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.CreatePartnerEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeactivateEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeauthorizeConnectionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteApiDestinationRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteArchiveRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteConnectionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeletePartnerEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DeleteRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeApiDestinationRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeArchiveRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeConnectionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeEventBusRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribePartnerEventSourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeReplayRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DescribeRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.DisableRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.EnableRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListApiDestinationsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListArchivesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListConnectionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListEventBusesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListEventSourcesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListPartnerEventSourceAccountsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListPartnerEventSourcesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListReplaysRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRuleNamesByTargetRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.ListTargetsByRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutPartnerEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutPermissionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.PutTargetsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemovePermissionRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.RemoveTargetsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.StartReplayRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.TestEventPatternRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.UpdateApiDestinationRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.UpdateArchiveRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchevents.transform.UpdateConnectionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudWatchEventsAsyncClient}.
 *
 * @see CloudWatchEventsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchEventsAsyncClient implements CloudWatchEventsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudWatchEventsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchEventsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Activates a partner event source that has been deactivated. Once activated, your matching event bus will start
     * receiving events from the event source.
     * </p>
     *
     * @param activateEventSourceRequest
     * @return A Java Future containing the result of the ActivateEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>InvalidStateException The specified state is not a valid state for an event source.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ActivateEventSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ActivateEventSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateEventSourceResponse> activateEventSource(
            ActivateEventSourceRequest activateEventSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateEventSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateEventSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateEventSourceRequest, ActivateEventSourceResponse>()
                            .withOperationName("ActivateEventSource")
                            .withMarshaller(new ActivateEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activateEventSourceRequest));
            CompletableFuture<ActivateEventSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the specified replay.
     * </p>
     *
     * @param cancelReplayRequest
     * @return A Java Future containing the result of the CancelReplay operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>IllegalStatusException An error occurred because a replay can be canceled only when the state is
     *         Running or Starting.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CancelReplay
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CancelReplay" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelReplayResponse> cancelReplay(CancelReplayRequest cancelReplayRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelReplayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelReplay");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelReplayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelReplayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelReplayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelReplayRequest, CancelReplayResponse>()
                            .withOperationName("CancelReplay").withMarshaller(new CancelReplayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelReplayRequest));
            CompletableFuture<CancelReplayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an API destination, which is an HTTP invocation endpoint configured as a target for events.
     * </p>
     *
     * @param createApiDestinationRequest
     * @return A Java Future containing the result of the CreateApiDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreateApiDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreateApiDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApiDestinationResponse> createApiDestination(
            CreateApiDestinationRequest createApiDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApiDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApiDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApiDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApiDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApiDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApiDestinationRequest, CreateApiDestinationResponse>()
                            .withOperationName("CreateApiDestination")
                            .withMarshaller(new CreateApiDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApiDestinationRequest));
            CompletableFuture<CreateApiDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an archive of events with the specified settings. When you create an archive, incoming events might not
     * immediately start being sent to the archive. Allow a short period of time for changes to take effect. If you do
     * not specify a pattern to filter events sent to the archive, all events are sent to the archive except replayed
     * events. Replayed events are not sent to an archive.
     * </p>
     *
     * @param createArchiveRequest
     * @return A Java Future containing the result of the CreateArchive operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>InvalidEventPatternException The event pattern is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreateArchive
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreateArchive" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateArchiveResponse> createArchive(CreateArchiveRequest createArchiveRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createArchiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateArchive");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateArchiveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateArchiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateArchiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateArchiveRequest, CreateArchiveResponse>()
                            .withOperationName("CreateArchive")
                            .withMarshaller(new CreateArchiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createArchiveRequest));
            CompletableFuture<CreateArchiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a connection. A connection defines the authorization type and credentials to use for authorization with
     * an API destination HTTP endpoint.
     * </p>
     *
     * @param createConnectionRequest
     * @return A Java Future containing the result of the CreateConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreateConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectionResponse> createConnection(CreateConnectionRequest createConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectionRequest, CreateConnectionResponse>()
                            .withOperationName("CreateConnection")
                            .withMarshaller(new CreateConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConnectionRequest));
            CompletableFuture<CreateConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new event bus within your account. This can be a custom event bus which you can use to receive events
     * from your custom applications and services, or it can be a partner event bus which can be matched to a partner
     * event source.
     * </p>
     *
     * @param createEventBusRequest
     * @return A Java Future containing the result of the CreateEventBus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InvalidStateException The specified state is not a valid state for an event source.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreateEventBus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreateEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEventBusResponse> createEventBus(CreateEventBusRequest createEventBusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventBusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventBus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEventBusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEventBusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEventBusRequest, CreateEventBusResponse>()
                            .withOperationName("CreateEventBus")
                            .withMarshaller(new CreateEventBusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createEventBusRequest));
            CompletableFuture<CreateEventBusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Called by an SaaS partner to create a partner event source. This operation is not used by Amazon Web Services
     * customers.
     * </p>
     * <p>
     * Each partner event source can be used by one Amazon Web Services account to create a matching partner event bus
     * in that Amazon Web Services account. A SaaS partner must create one partner event source for each Amazon Web
     * Services account that wants to receive those event types.
     * </p>
     * <p>
     * A partner event source creates events based on resources within the SaaS partner's service or application.
     * </p>
     * <p>
     * An Amazon Web Services account that creates a partner event bus that matches the partner event source can use
     * that event bus to receive events from the partner, and then process them using Amazon Web Services Events rules
     * and targets.
     * </p>
     * <p>
     * Partner event source names follow this format:
     * </p>
     * <p>
     * <code> <i>partner_name</i>/<i>event_namespace</i>/<i>event_name</i> </code>
     * </p>
     * <p>
     * <i>partner_name</i> is determined during partner registration and identifies the partner to Amazon Web Services
     * customers. <i>event_namespace</i> is determined by the partner and is a way for the partner to categorize their
     * events. <i>event_name</i> is determined by the partner, and should uniquely identify an event-generating resource
     * within the partner system. The combination of <i>event_namespace</i> and <i>event_name</i> should help Amazon Web
     * Services customers decide whether to create an event bus to receive these events.
     * </p>
     *
     * @param createPartnerEventSourceRequest
     * @return A Java Future containing the result of the CreatePartnerEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.CreatePartnerEventSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/CreatePartnerEventSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePartnerEventSourceResponse> createPartnerEventSource(
            CreatePartnerEventSourceRequest createPartnerEventSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPartnerEventSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePartnerEventSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePartnerEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePartnerEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePartnerEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePartnerEventSourceRequest, CreatePartnerEventSourceResponse>()
                            .withOperationName("CreatePartnerEventSource")
                            .withMarshaller(new CreatePartnerEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPartnerEventSourceRequest));
            CompletableFuture<CreatePartnerEventSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use this operation to temporarily stop receiving events from the specified partner event source. The
     * matching event bus is not deleted.
     * </p>
     * <p>
     * When you deactivate a partner event source, the source goes into PENDING state. If it remains in PENDING state
     * for more than two weeks, it is deleted.
     * </p>
     * <p>
     * To activate a deactivated partner event source, use <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html"
     * >ActivateEventSource</a>.
     * </p>
     *
     * @param deactivateEventSourceRequest
     * @return A Java Future containing the result of the DeactivateEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>InvalidStateException The specified state is not a valid state for an event source.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeactivateEventSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeactivateEventSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateEventSourceResponse> deactivateEventSource(
            DeactivateEventSourceRequest deactivateEventSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateEventSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateEventSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateEventSourceRequest, DeactivateEventSourceResponse>()
                            .withOperationName("DeactivateEventSource")
                            .withMarshaller(new DeactivateEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deactivateEventSourceRequest));
            CompletableFuture<DeactivateEventSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes all authorization parameters from the connection. This lets you remove the secret from the connection so
     * you can reuse it without having to create a new connection.
     * </p>
     *
     * @param deauthorizeConnectionRequest
     * @return A Java Future containing the result of the DeauthorizeConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeauthorizeConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeauthorizeConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeauthorizeConnectionResponse> deauthorizeConnection(
            DeauthorizeConnectionRequest deauthorizeConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deauthorizeConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeauthorizeConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeauthorizeConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeauthorizeConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeauthorizeConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeauthorizeConnectionRequest, DeauthorizeConnectionResponse>()
                            .withOperationName("DeauthorizeConnection")
                            .withMarshaller(new DeauthorizeConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deauthorizeConnectionRequest));
            CompletableFuture<DeauthorizeConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified API destination.
     * </p>
     *
     * @param deleteApiDestinationRequest
     * @return A Java Future containing the result of the DeleteApiDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteApiDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteApiDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApiDestinationResponse> deleteApiDestination(
            DeleteApiDestinationRequest deleteApiDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApiDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApiDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApiDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApiDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApiDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApiDestinationRequest, DeleteApiDestinationResponse>()
                            .withOperationName("DeleteApiDestination")
                            .withMarshaller(new DeleteApiDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApiDestinationRequest));
            CompletableFuture<DeleteApiDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified archive.
     * </p>
     *
     * @param deleteArchiveRequest
     * @return A Java Future containing the result of the DeleteArchive operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteArchive
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteArchive" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteArchiveResponse> deleteArchive(DeleteArchiveRequest deleteArchiveRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteArchiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteArchive");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteArchiveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteArchiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteArchiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteArchiveRequest, DeleteArchiveResponse>()
                            .withOperationName("DeleteArchive")
                            .withMarshaller(new DeleteArchiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteArchiveRequest));
            CompletableFuture<DeleteArchiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a connection.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectionResponse> deleteConnection(DeleteConnectionRequest deleteConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectionRequest, DeleteConnectionResponse>()
                            .withOperationName("DeleteConnection")
                            .withMarshaller(new DeleteConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConnectionRequest));
            CompletableFuture<DeleteConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified custom event bus or partner event bus. All rules associated with this event bus need to be
     * deleted. You can't delete your account's default event bus.
     * </p>
     *
     * @param deleteEventBusRequest
     * @return A Java Future containing the result of the DeleteEventBus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteEventBus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventBusResponse> deleteEventBus(DeleteEventBusRequest deleteEventBusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventBusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventBus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEventBusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEventBusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventBusRequest, DeleteEventBusResponse>()
                            .withOperationName("DeleteEventBus")
                            .withMarshaller(new DeleteEventBusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteEventBusRequest));
            CompletableFuture<DeleteEventBusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation is used by SaaS partners to delete a partner event source. This operation is not used by Amazon
     * Web Services customers.
     * </p>
     * <p>
     * When you delete an event source, the status of the corresponding partner event bus in the Amazon Web Services
     * customer account becomes DELETED.
     * </p>
     * <p/>
     *
     * @param deletePartnerEventSourceRequest
     * @return A Java Future containing the result of the DeletePartnerEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li> <li>
     *         ConcurrentModificationException There is concurrent modification on a rule, target, archive, or replay.
     *         </li> <li>OperationDisabledException The operation you are attempting is not available in this region.
     *         </li> <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and
     *         client). Can be used for catch all scenarios.</li> <li>SdkClientException If any client side error occurs
     *         such as an IO related failure, failure to get credentials, etc.</li> <li>CloudWatchEventsException Base
     *         class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeletePartnerEventSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeletePartnerEventSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePartnerEventSourceResponse> deletePartnerEventSource(
            DeletePartnerEventSourceRequest deletePartnerEventSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePartnerEventSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePartnerEventSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePartnerEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePartnerEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePartnerEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePartnerEventSourceRequest, DeletePartnerEventSourceResponse>()
                            .withOperationName("DeletePartnerEventSource")
                            .withMarshaller(new DeletePartnerEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePartnerEventSourceRequest));
            CompletableFuture<DeletePartnerEventSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified rule.
     * </p>
     * <p>
     * Before you can delete the rule, you must remove all targets, using <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html">RemoveTargets</a>.
     * </p>
     * <p>
     * When you delete a rule, incoming events might continue to match to the deleted rule. Allow a short period of time
     * for changes to take effect.
     * </p>
     * <p>
     * If you call delete rule multiple times for the same rule, all calls will succeed. When you call delete rule for a
     * non-existent custom eventbus, <code>ResourceNotFoundException</code> is returned.
     * </p>
     * <p>
     * Managed rules are rules created and managed by another Amazon Web Services service on your behalf. These rules
     * are created by those other Amazon Web Services services to support functionality in those services. You can
     * delete these rules using the <code>Force</code> option, but you should do so only if you are sure the other
     * service is not still using that rule.
     * </p>
     *
     * @param deleteRuleRequest
     * @return A Java Future containing the result of the DeleteRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DeleteRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DeleteRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRuleResponse> deleteRule(DeleteRuleRequest deleteRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRuleRequest, DeleteRuleResponse>().withOperationName("DeleteRule")
                            .withMarshaller(new DeleteRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRuleRequest));
            CompletableFuture<DeleteRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about an API destination.
     * </p>
     *
     * @param describeApiDestinationRequest
     * @return A Java Future containing the result of the DescribeApiDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeApiDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeApiDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeApiDestinationResponse> describeApiDestination(
            DescribeApiDestinationRequest describeApiDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApiDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApiDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeApiDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeApiDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeApiDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeApiDestinationRequest, DescribeApiDestinationResponse>()
                            .withOperationName("DescribeApiDestination")
                            .withMarshaller(new DescribeApiDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeApiDestinationRequest));
            CompletableFuture<DescribeApiDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about an archive.
     * </p>
     *
     * @param describeArchiveRequest
     * @return A Java Future containing the result of the DescribeArchive operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeArchive
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeArchive" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeArchiveResponse> describeArchive(DescribeArchiveRequest describeArchiveRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeArchiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeArchive");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeArchiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeArchiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeArchiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeArchiveRequest, DescribeArchiveResponse>()
                            .withOperationName("DescribeArchive")
                            .withMarshaller(new DescribeArchiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeArchiveRequest));
            CompletableFuture<DescribeArchiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a connection.
     * </p>
     *
     * @param describeConnectionRequest
     * @return A Java Future containing the result of the DescribeConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConnectionResponse> describeConnection(DescribeConnectionRequest describeConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectionRequest, DescribeConnectionResponse>()
                            .withOperationName("DescribeConnection")
                            .withMarshaller(new DescribeConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeConnectionRequest));
            CompletableFuture<DescribeConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays details about an event bus in your account. This can include the external Amazon Web Services accounts
     * that are permitted to write events to your default event bus, and the associated policy. For custom event buses
     * and partner event buses, it displays the name, ARN, policy, state, and creation time.
     * </p>
     * <p>
     * To enable your account to receive events from other accounts on its default event bus, use <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html">PutPermission</a>.
     * </p>
     * <p>
     * For more information about partner event buses, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html">CreateEventBus</a>.
     * </p>
     *
     * @param describeEventBusRequest
     * @return A Java Future containing the result of the DescribeEventBus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeEventBus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeEventBus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventBusResponse> describeEventBus(DescribeEventBusRequest describeEventBusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventBusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventBus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventBusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventBusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventBusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventBusRequest, DescribeEventBusResponse>()
                            .withOperationName("DescribeEventBus")
                            .withMarshaller(new DescribeEventBusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEventBusRequest));
            CompletableFuture<DescribeEventBusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation lists details about a partner event source that is shared with your account.
     * </p>
     *
     * @param describeEventSourceRequest
     * @return A Java Future containing the result of the DescribeEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeEventSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeEventSource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventSourceResponse> describeEventSource(
            DescribeEventSourceRequest describeEventSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventSourceRequest, DescribeEventSourceResponse>()
                            .withOperationName("DescribeEventSource")
                            .withMarshaller(new DescribeEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEventSourceRequest));
            CompletableFuture<DescribeEventSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An SaaS partner can use this operation to list details about a partner event source that they have created.
     * Amazon Web Services customers do not use this operation. Instead, Amazon Web Services customers can use <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html"
     * >DescribeEventSource</a> to see details about a partner event source that is shared with them.
     * </p>
     *
     * @param describePartnerEventSourceRequest
     * @return A Java Future containing the result of the DescribePartnerEventSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribePartnerEventSource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribePartnerEventSource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePartnerEventSourceResponse> describePartnerEventSource(
            DescribePartnerEventSourceRequest describePartnerEventSourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePartnerEventSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePartnerEventSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePartnerEventSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePartnerEventSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePartnerEventSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePartnerEventSourceRequest, DescribePartnerEventSourceResponse>()
                            .withOperationName("DescribePartnerEventSource")
                            .withMarshaller(new DescribePartnerEventSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePartnerEventSourceRequest));
            CompletableFuture<DescribePartnerEventSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a replay. Use <code>DescribeReplay</code> to determine the progress of a running replay.
     * A replay processes events to replay based on the time in the event, and replays them using 1 minute intervals. If
     * you use <code>StartReplay</code> and specify an <code>EventStartTime</code> and an <code>EventEndTime</code> that
     * covers a 20 minute time range, the events are replayed from the first minute of that 20 minute range first. Then
     * the events from the second minute are replayed. You can use <code>DescribeReplay</code> to determine the progress
     * of a replay. The value returned for <code>EventLastReplayedTime</code> indicates the time within the specified
     * time range associated with the last event replayed.
     * </p>
     *
     * @param describeReplayRequest
     * @return A Java Future containing the result of the DescribeReplay operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeReplay
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeReplay" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeReplayResponse> describeReplay(DescribeReplayRequest describeReplayRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReplayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReplay");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReplayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeReplayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReplayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReplayRequest, DescribeReplayResponse>()
                            .withOperationName("DescribeReplay")
                            .withMarshaller(new DescribeReplayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReplayRequest));
            CompletableFuture<DescribeReplayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the specified rule.
     * </p>
     * <p>
     * DescribeRule does not list the targets of a rule. To see the targets associated with a rule, use <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html"
     * >ListTargetsByRule</a>.
     * </p>
     *
     * @param describeRuleRequest
     * @return A Java Future containing the result of the DescribeRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DescribeRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DescribeRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRuleResponse> describeRule(DescribeRuleRequest describeRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRuleRequest, DescribeRuleResponse>()
                            .withOperationName("DescribeRule").withMarshaller(new DescribeRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRuleRequest));
            CompletableFuture<DescribeRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the specified rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule
     * expression.
     * </p>
     * <p>
     * When you disable a rule, incoming events might continue to match to the disabled rule. Allow a short period of
     * time for changes to take effect.
     * </p>
     *
     * @param disableRuleRequest
     * @return A Java Future containing the result of the DisableRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.DisableRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/DisableRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableRuleResponse> disableRule(DisableRuleRequest disableRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableRuleRequest, DisableRuleResponse>()
                            .withOperationName("DisableRule").withMarshaller(new DisableRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableRuleRequest));
            CompletableFuture<DisableRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables the specified rule. If the rule does not exist, the operation fails.
     * </p>
     * <p>
     * When you enable a rule, incoming events might not immediately start matching to a newly enabled rule. Allow a
     * short period of time for changes to take effect.
     * </p>
     *
     * @param enableRuleRequest
     * @return A Java Future containing the result of the EnableRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.EnableRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/EnableRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableRuleResponse> enableRule(EnableRuleRequest enableRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableRuleRequest, EnableRuleResponse>().withOperationName("EnableRule")
                            .withMarshaller(new EnableRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableRuleRequest));
            CompletableFuture<EnableRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of API destination in the account in the current Region.
     * </p>
     *
     * @param listApiDestinationsRequest
     * @return A Java Future containing the result of the ListApiDestinations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListApiDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListApiDestinations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApiDestinationsResponse> listApiDestinations(
            ListApiDestinationsRequest listApiDestinationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApiDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApiDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApiDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApiDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApiDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApiDestinationsRequest, ListApiDestinationsResponse>()
                            .withOperationName("ListApiDestinations")
                            .withMarshaller(new ListApiDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApiDestinationsRequest));
            CompletableFuture<ListApiDestinationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your archives. You can either list all the archives or you can provide a prefix to match to the archive
     * names. Filter parameters are exclusive.
     * </p>
     *
     * @param listArchivesRequest
     * @return A Java Future containing the result of the ListArchives operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListArchives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListArchives" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListArchivesResponse> listArchives(ListArchivesRequest listArchivesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listArchivesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListArchives");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListArchivesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListArchivesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListArchivesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListArchivesRequest, ListArchivesResponse>()
                            .withOperationName("ListArchives").withMarshaller(new ListArchivesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listArchivesRequest));
            CompletableFuture<ListArchivesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of connections from the account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return A Java Future containing the result of the ListConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListConnections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectionsResponse> listConnections(ListConnectionsRequest listConnectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectionsRequest, ListConnectionsResponse>()
                            .withOperationName("ListConnections")
                            .withMarshaller(new ListConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listConnectionsRequest));
            CompletableFuture<ListConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the event buses in your account, including the default event bus, custom event buses, and partner event
     * buses.
     * </p>
     *
     * @param listEventBusesRequest
     * @return A Java Future containing the result of the ListEventBuses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListEventBuses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListEventBuses" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventBusesResponse> listEventBuses(ListEventBusesRequest listEventBusesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventBusesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventBuses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventBusesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventBusesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventBusesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventBusesRequest, ListEventBusesResponse>()
                            .withOperationName("ListEventBuses")
                            .withMarshaller(new ListEventBusesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEventBusesRequest));
            CompletableFuture<ListEventBusesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use this to see all the partner event sources that have been shared with your Amazon Web Services
     * account. For more information about partner event sources, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html">CreateEventBus</a>.
     * </p>
     *
     * @param listEventSourcesRequest
     * @return A Java Future containing the result of the ListEventSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListEventSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListEventSources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventSourcesResponse> listEventSources(ListEventSourcesRequest listEventSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventSourcesRequest, ListEventSourcesResponse>()
                            .withOperationName("ListEventSources")
                            .withMarshaller(new ListEventSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEventSourcesRequest));
            CompletableFuture<ListEventSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An SaaS partner can use this operation to display the Amazon Web Services account ID that a particular partner
     * event source name is associated with. This operation is not used by Amazon Web Services customers.
     * </p>
     *
     * @param listPartnerEventSourceAccountsRequest
     * @return A Java Future containing the result of the ListPartnerEventSourceAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListPartnerEventSourceAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListPartnerEventSourceAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPartnerEventSourceAccountsResponse> listPartnerEventSourceAccounts(
            ListPartnerEventSourceAccountsRequest listPartnerEventSourceAccountsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listPartnerEventSourceAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPartnerEventSourceAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPartnerEventSourceAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPartnerEventSourceAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPartnerEventSourceAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPartnerEventSourceAccountsRequest, ListPartnerEventSourceAccountsResponse>()
                            .withOperationName("ListPartnerEventSourceAccounts")
                            .withMarshaller(new ListPartnerEventSourceAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPartnerEventSourceAccountsRequest));
            CompletableFuture<ListPartnerEventSourceAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * An SaaS partner can use this operation to list all the partner event source names that they have created. This
     * operation is not used by Amazon Web Services customers.
     * </p>
     *
     * @param listPartnerEventSourcesRequest
     * @return A Java Future containing the result of the ListPartnerEventSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListPartnerEventSources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListPartnerEventSources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPartnerEventSourcesResponse> listPartnerEventSources(
            ListPartnerEventSourcesRequest listPartnerEventSourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPartnerEventSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPartnerEventSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPartnerEventSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPartnerEventSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPartnerEventSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPartnerEventSourcesRequest, ListPartnerEventSourcesResponse>()
                            .withOperationName("ListPartnerEventSources")
                            .withMarshaller(new ListPartnerEventSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPartnerEventSourcesRequest));
            CompletableFuture<ListPartnerEventSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your replays. You can either list all the replays or you can provide a prefix to match to the replay names.
     * Filter parameters are exclusive.
     * </p>
     *
     * @param listReplaysRequest
     * @return A Java Future containing the result of the ListReplays operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListReplays
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListReplays" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListReplaysResponse> listReplays(ListReplaysRequest listReplaysRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReplaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReplays");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReplaysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListReplaysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReplaysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReplaysRequest, ListReplaysResponse>()
                            .withOperationName("ListReplays").withMarshaller(new ListReplaysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listReplaysRequest));
            CompletableFuture<ListReplaysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the rules for the specified target. You can see which of the rules in Amazon EventBridge can invoke a
     * specific target in your account.
     * </p>
     *
     * @param listRuleNamesByTargetRequest
     * @return A Java Future containing the result of the ListRuleNamesByTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListRuleNamesByTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListRuleNamesByTarget" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRuleNamesByTargetResponse> listRuleNamesByTarget(
            ListRuleNamesByTargetRequest listRuleNamesByTargetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRuleNamesByTargetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRuleNamesByTarget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRuleNamesByTargetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRuleNamesByTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRuleNamesByTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRuleNamesByTargetRequest, ListRuleNamesByTargetResponse>()
                            .withOperationName("ListRuleNamesByTarget")
                            .withMarshaller(new ListRuleNamesByTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRuleNamesByTargetRequest));
            CompletableFuture<ListRuleNamesByTargetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your Amazon EventBridge rules. You can either list all the rules or you can provide a prefix to match to
     * the rule names.
     * </p>
     * <p>
     * ListRules does not list the targets of a rule. To see the targets associated with a rule, use <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html"
     * >ListTargetsByRule</a>.
     * </p>
     *
     * @param listRulesRequest
     * @return A Java Future containing the result of the ListRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListRulesResponse> listRules(ListRulesRequest listRulesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>().withOperationName("ListRules")
                            .withMarshaller(new ListRulesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRulesRequest));
            CompletableFuture<ListRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with an EventBridge resource. In EventBridge, rules and event buses can be tagged.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the targets assigned to the specified rule.
     * </p>
     *
     * @param listTargetsByRuleRequest
     * @return A Java Future containing the result of the ListTargetsByRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.ListTargetsByRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/ListTargetsByRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTargetsByRuleResponse> listTargetsByRule(ListTargetsByRuleRequest listTargetsByRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTargetsByRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTargetsByRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTargetsByRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTargetsByRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTargetsByRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTargetsByRuleRequest, ListTargetsByRuleResponse>()
                            .withOperationName("ListTargetsByRule")
                            .withMarshaller(new ListTargetsByRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTargetsByRuleRequest));
            CompletableFuture<ListTargetsByRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     * </p>
     *
     * @param putEventsRequest
     * @return A Java Future containing the result of the PutEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutEventsResponse> putEvents(PutEventsRequest putEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEventsRequest, PutEventsResponse>().withOperationName("PutEvents")
                            .withMarshaller(new PutEventsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(putEventsRequest));
            CompletableFuture<PutEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This is used by SaaS partners to write events to a customer's partner event bus. Amazon Web Services customers do
     * not use this operation.
     * </p>
     *
     * @param putPartnerEventsRequest
     * @return A Java Future containing the result of the PutPartnerEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutPartnerEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutPartnerEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPartnerEventsResponse> putPartnerEvents(PutPartnerEventsRequest putPartnerEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPartnerEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPartnerEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPartnerEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutPartnerEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPartnerEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPartnerEventsRequest, PutPartnerEventsResponse>()
                            .withOperationName("PutPartnerEvents")
                            .withMarshaller(new PutPartnerEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPartnerEventsRequest));
            CompletableFuture<PutPartnerEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Running <code>PutPermission</code> permits the specified Amazon Web Services account or Amazon Web Services
     * organization to put events to the specified <i>event bus</i>. Amazon EventBridge (CloudWatch Events) rules in
     * your account are triggered by these events arriving to an event bus in your account.
     * </p>
     * <p>
     * For another account to send events to your account, that external account must have an EventBridge rule with your
     * account's event bus as a target.
     * </p>
     * <p>
     * To enable multiple Amazon Web Services accounts to put events to your event bus, run <code>PutPermission</code>
     * once for each of these accounts. Or, if all the accounts are members of the same Amazon Web Services
     * organization, you can run <code>PutPermission</code> once specifying <code>Principal</code> as "*" and specifying
     * the Amazon Web Services organization ID in <code>Condition</code>, to grant permissions to all accounts in that
     * organization.
     * </p>
     * <p>
     * If you grant permissions using an organization, then accounts in that organization must specify a
     * <code>RoleArn</code> with proper permissions when they use <code>PutTarget</code> to add your account's event bus
     * as a target. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html"
     * >Sending and Receiving Events Between Amazon Web Services Accounts</a> in the <i>Amazon EventBridge User
     * Guide</i>.
     * </p>
     * <p>
     * The permission policy on the event bus cannot exceed 10 KB in size.
     * </p>
     *
     * @param putPermissionRequest
     * @return A Java Future containing the result of the PutPermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>PolicyLengthExceededException The event bus policy is too long. For more information, see the limits.
     *         </li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutPermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutPermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPermissionResponse> putPermission(PutPermissionRequest putPermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPermissionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutPermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPermissionRequest, PutPermissionResponse>()
                            .withOperationName("PutPermission")
                            .withMarshaller(new PutPermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPermissionRequest));
            CompletableFuture<PutPermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates the specified rule. Rules are enabled by default, or based on value of the state. You can
     * disable a rule using <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html">DisableRule</a>.
     * </p>
     * <p>
     * A single rule watches for events from a single event bus. Events generated by Amazon Web Services services go to
     * your account's default event bus. Events generated by SaaS partner services or applications go to the matching
     * partner event bus. If you have custom applications or services, you can specify whether their events go to your
     * default event bus or a custom event bus that you have created. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html">CreateEventBus</a>.
     * </p>
     * <p>
     * If you are updating an existing rule, the rule is replaced with what you specify in this <code>PutRule</code>
     * command. If you omit arguments in <code>PutRule</code>, the old values for those arguments are not kept. Instead,
     * they are replaced with null values.
     * </p>
     * <p>
     * When you create or update a rule, incoming events might not immediately start matching to new or updated rules.
     * Allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * A rule must contain at least an EventPattern or ScheduleExpression. Rules with EventPatterns are triggered when a
     * matching event is observed. Rules with ScheduleExpressions self-trigger based on the given schedule. A rule can
     * have both an EventPattern and a ScheduleExpression, in which case the rule triggers on matching events as well as
     * on a schedule.
     * </p>
     * <p>
     * When you initially create a rule, you can optionally assign one or more tags to the rule. Tags can help you
     * organize and categorize your resources. You can also use them to scope user permissions, by granting a user
     * permission to access or change only rules with certain tag values. To use the <code>PutRule</code> operation and
     * assign tags, you must have both the <code>events:PutRule</code> and <code>events:TagResource</code> permissions.
     * </p>
     * <p>
     * If you are updating an existing rule, any tags you specify in the <code>PutRule</code> operation are ignored. To
     * update the tags of an existing rule, use <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html">TagResource</a> and <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html">UntagResource</a>.
     * </p>
     * <p>
     * Most services in Amazon Web Services treat : or / as the same character in Amazon Resource Names (ARNs). However,
     * EventBridge uses an exact match in event patterns and rules. Be sure to use the correct ARN characters when
     * creating event patterns so that they match the ARN syntax in the event you want to match.
     * </p>
     * <p>
     * In EventBridge, it is possible to create rules that lead to infinite loops, where a rule is fired repeatedly. For
     * example, a rule might detect that ACLs have changed on an S3 bucket, and trigger software to change them to the
     * desired state. If the rule is not written carefully, the subsequent change to the ACLs fires the rule again,
     * creating an infinite loop.
     * </p>
     * <p>
     * To prevent this, write the rules so that the triggered actions do not re-fire the same rule. For example, your
     * rule could fire only if ACLs are found to be in a bad state, instead of after any change.
     * </p>
     * <p>
     * An infinite loop can quickly cause higher than expected charges. We recommend that you use budgeting, which
     * alerts you when charges exceed your specified limit. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-managing-costs.html">Managing Your
     * Costs with Budgets</a>.
     * </p>
     *
     * @param putRuleRequest
     * @return A Java Future containing the result of the PutRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidEventPatternException The event pattern is not valid.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRuleResponse> putRule(PutRuleRequest putRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRuleRequest, PutRuleResponse>().withOperationName("PutRule")
                            .withMarshaller(new PutRuleRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRuleRequest));
            CompletableFuture<PutRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified targets to the specified rule, or updates the targets if they are already associated with the
     * rule.
     * </p>
     * <p>
     * Targets are the resources that are invoked when a rule is triggered.
     * </p>
     * <p>
     * You can configure the following as targets for Events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-api-destinations.html">API destination</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon API Gateway REST API endpoints
     * </p>
     * </li>
     * <li>
     * <p>
     * API Gateway
     * </p>
     * </li>
     * <li>
     * <p>
     * Batch job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * CloudWatch Logs group
     * </p>
     * </li>
     * <li>
     * <p>
     * CodeBuild project
     * </p>
     * </li>
     * <li>
     * <p>
     * CodePipeline
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon EC2 <code>CreateSnapshot</code> API call
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon EC2 <code>RebootInstances</code> API call
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon EC2 <code>StopInstances</code> API call
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon EC2 <code>TerminateInstances</code> API call
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ECS tasks
     * </p>
     * </li>
     * <li>
     * <p>
     * Event bus in a different Amazon Web Services account or Region.
     * </p>
     * <p>
     * You can use an event bus in the US East (N. Virginia) us-east-1, US West (Oregon) us-west-2, or Europe (Ireland)
     * eu-west-1 Regions as a target for a rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * Firehose delivery stream (Kinesis Data Firehose)
     * </p>
     * </li>
     * <li>
     * <p>
     * Inspector assessment template (Amazon Inspector)
     * </p>
     * </li>
     * <li>
     * <p>
     * Kinesis stream (Kinesis Data Stream)
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda function
     * </p>
     * </li>
     * <li>
     * <p>
     * Redshift clusters (Data API statement execution)
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SNS topic
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SQS queues (includes FIFO queues
     * </p>
     * </li>
     * <li>
     * <p>
     * SSM Automation
     * </p>
     * </li>
     * <li>
     * <p>
     * SSM OpsItem
     * </p>
     * </li>
     * <li>
     * <p>
     * SSM Run Command
     * </p>
     * </li>
     * <li>
     * <p>
     * Step Functions state machines
     * </p>
     * </li>
     * </ul>
     * <p>
     * Creating rules with built-in targets is supported only in the Management Console. The built-in targets are
     * <code>EC2 CreateSnapshot API call</code>, <code>EC2 RebootInstances API call</code>,
     * <code>EC2 StopInstances API call</code>, and <code>EC2 TerminateInstances API call</code>.
     * </p>
     * <p>
     * For some target types, <code>PutTargets</code> provides target-specific parameters. If the target is a Kinesis
     * data stream, you can optionally specify which shard the event goes to by using the <code>KinesisParameters</code>
     * argument. To invoke a command on multiple EC2 instances with one rule, you can use the
     * <code>RunCommandParameters</code> field.
     * </p>
     * <p>
     * To be able to make API calls against the resources that you own, Amazon EventBridge needs the appropriate
     * permissions. For Lambda and Amazon SNS resources, EventBridge relies on resource-based policies. For EC2
     * instances, Kinesis Data Streams, Step Functions state machines and API Gateway REST APIs, EventBridge relies on
     * IAM roles that you specify in the <code>RoleARN</code> argument in <code>PutTargets</code>. For more information,
     * see <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/auth-and-access-control-eventbridge.html">
     * Authentication and Access Control</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * <p>
     * If another Amazon Web Services account is in the same region and has granted you permission (using
     * <code>PutPermission</code>), you can send events to that account. Set that account's event bus as a target of the
     * rules in your account. To send the matched events to the other account, specify that account's event bus as the
     * <code>Arn</code> value when you run <code>PutTargets</code>. If your account sends events to another account,
     * your account is charged for each sent event. Each event sent to another account is charged as a custom event. The
     * account receiving the event is not charged. For more information, see <a
     * href="http://aws.amazon.com/eventbridge/pricing/">Amazon EventBridge Pricing</a>.
     * </p>
     * <note>
     * <p>
     * <code>Input</code>, <code>InputPath</code>, and <code>InputTransformer</code> are not available with
     * <code>PutTarget</code> if the target is an event bus of a different Amazon Web Services account.
     * </p>
     * </note>
     * <p>
     * If you are setting the event bus of another account as the target, and that account granted permission to your
     * account through an organization instead of directly by the account ID, then you must specify a
     * <code>RoleArn</code> with proper permissions in the <code>Target</code> structure. For more information, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html">
     * Sending and Receiving Events Between Amazon Web Services Accounts</a> in the <i>Amazon EventBridge User
     * Guide</i>.
     * </p>
     * <p>
     * For more information about enabling cross-account events, see <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html">PutPermission</a>.
     * </p>
     * <p>
     * <b>Input</b>, <b>InputPath</b>, and <b>InputTransformer</b> are mutually exclusive and optional parameters of a
     * target. When a rule is triggered due to a matched event:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If none of the following arguments are specified for a target, then the entire event is passed to the target in
     * JSON format (unless the target is Amazon EC2 Run Command or Amazon ECS task, in which case nothing from the event
     * is passed to the target).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <b>Input</b> is specified in the form of valid JSON, then the matched event is overridden with this constant.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <b>InputPath</b> is specified in the form of JSONPath (for example, <code>&#36.detail</code>), then only the
     * part of the event specified in the path is passed to the target (for example, only the detail part of the event
     * is passed).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <b>InputTransformer</b> is specified, then one or more specified JSONPaths are extracted from the event and
     * used as values in a template that you specify as the input to the target.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you specify <code>InputPath</code> or <code>InputTransformer</code>, you must use JSON dot notation, not
     * bracket notation.
     * </p>
     * <p>
     * When you add targets to a rule and the associated rule triggers soon after, new or updated targets might not be
     * immediately invoked. Allow a short period of time for changes to take effect.
     * </p>
     * <p>
     * This action can partially fail if too many requests are made at the same time. If that happens,
     * <code>FailedEntryCount</code> is non-zero in the response and each entry in <code>FailedEntries</code> provides
     * the ID of the failed target and the error code.
     * </p>
     *
     * @param putTargetsRequest
     * @return A Java Future containing the result of the PutTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.PutTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/PutTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutTargetsResponse> putTargets(PutTargetsRequest putTargetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutTargets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutTargetsRequest, PutTargetsResponse>().withOperationName("PutTargets")
                            .withMarshaller(new PutTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putTargetsRequest));
            CompletableFuture<PutTargetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes the permission of another Amazon Web Services account to be able to put events to the specified event
     * bus. Specify the account to revoke by the <code>StatementId</code> value that you associated with the account
     * when you granted it permission with <code>PutPermission</code>. You can find the <code>StatementId</code> by
     * using <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html">DescribeEventBus
     * </a>.
     * </p>
     *
     * @param removePermissionRequest
     * @return A Java Future containing the result of the RemovePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>OperationDisabledException The operation you are attempting is not available in this region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.RemovePermission
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/RemovePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemovePermissionResponse> removePermission(RemovePermissionRequest removePermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemovePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemovePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemovePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemovePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemovePermissionRequest, RemovePermissionResponse>()
                            .withOperationName("RemovePermission")
                            .withMarshaller(new RemovePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removePermissionRequest));
            CompletableFuture<RemovePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified targets from the specified rule. When the rule is triggered, those targets are no longer be
     * invoked.
     * </p>
     * <p>
     * When you remove a target, when the associated rule triggers, removed targets might continue to be invoked. Allow
     * a short period of time for changes to take effect.
     * </p>
     * <p>
     * This action can partially fail if too many requests are made at the same time. If that happens,
     * <code>FailedEntryCount</code> is non-zero in the response and each entry in <code>FailedEntries</code> provides
     * the ID of the failed target and the error code.
     * </p>
     *
     * @param removeTargetsRequest
     * @return A Java Future containing the result of the RemoveTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.RemoveTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/RemoveTargets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveTargetsResponse> removeTargets(RemoveTargetsRequest removeTargetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTargetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTargets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RemoveTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveTargetsRequest, RemoveTargetsResponse>()
                            .withOperationName("RemoveTargets")
                            .withMarshaller(new RemoveTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeTargetsRequest));
            CompletableFuture<RemoveTargetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the specified replay. Events are not necessarily replayed in the exact same order that they were added to
     * the archive. A replay processes events to replay based on the time in the event, and replays them using 1 minute
     * intervals. If you specify an <code>EventStartTime</code> and an <code>EventEndTime</code> that covers a 20 minute
     * time range, the events are replayed from the first minute of that 20 minute range first. Then the events from the
     * second minute are replayed. You can use <code>DescribeReplay</code> to determine the progress of a replay. The
     * value returned for <code>EventLastReplayedTime</code> indicates the time within the specified time range
     * associated with the last event replayed.
     * </p>
     *
     * @param startReplayRequest
     * @return A Java Future containing the result of the StartReplay operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ResourceAlreadyExistsException The resource you are trying to create already exists.</li>
     *         <li>InvalidEventPatternException The event pattern is not valid.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.StartReplay
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/StartReplay" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartReplayResponse> startReplay(StartReplayRequest startReplayRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startReplayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartReplay");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartReplayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartReplayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartReplayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartReplayRequest, StartReplayResponse>()
                            .withOperationName("StartReplay").withMarshaller(new StartReplayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startReplayRequest));
            CompletableFuture<StartReplayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified EventBridge resource. Tags can help you organize and
     * categorize your resources. You can also use them to scope user permissions by granting a user permission to
     * access or change only resources with certain tag values. In EventBridge, rules and event buses can be tagged.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with a resource that already has tags. If you specify a new tag
     * key, this tag is appended to the list of tags associated with the resource. If you specify a tag key that is
     * already associated with the resource, the new tag value that you specify replaces the previous value for that
     * tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tests whether the specified event pattern matches the provided event.
     * </p>
     * <p>
     * Most services in Amazon Web Services treat : or / as the same character in Amazon Resource Names (ARNs). However,
     * EventBridge uses an exact match in event patterns and rules. Be sure to use the correct ARN characters when
     * creating event patterns so that they match the ARN syntax in the event you want to match.
     * </p>
     *
     * @param testEventPatternRequest
     * @return A Java Future containing the result of the TestEventPattern operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidEventPatternException The event pattern is not valid.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.TestEventPattern
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/TestEventPattern" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TestEventPatternResponse> testEventPattern(TestEventPatternRequest testEventPatternRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testEventPatternRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestEventPattern");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestEventPatternResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestEventPatternResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestEventPatternResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestEventPatternRequest, TestEventPatternResponse>()
                            .withOperationName("TestEventPattern")
                            .withMarshaller(new TestEventPatternRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(testEventPatternRequest));
            CompletableFuture<TestEventPatternResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified EventBridge resource. In Amazon EventBridge (CloudWatch Events),
     * rules and event buses can be tagged.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ManagedRuleException This rule was created by an Amazon Web Services service on behalf of your
     *         account. It is managed by that service. If you see this error in response to <code>DeleteRule</code> or
     *         <code>RemoveTargets</code>, you can use the <code>Force</code> parameter in those calls to delete the
     *         rule or remove targets from the rule. You cannot modify these managed rules by using
     *         <code>DisableRule</code>, <code>EnableRule</code>, <code>PutTargets</code>, <code>PutRule</code>,
     *         <code>TagResource</code>, or <code>UntagResource</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an API destination.
     * </p>
     *
     * @param updateApiDestinationRequest
     * @return A Java Future containing the result of the UpdateApiDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.UpdateApiDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/UpdateApiDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApiDestinationResponse> updateApiDestination(
            UpdateApiDestinationRequest updateApiDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApiDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApiDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApiDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApiDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApiDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApiDestinationRequest, UpdateApiDestinationResponse>()
                            .withOperationName("UpdateApiDestination")
                            .withMarshaller(new UpdateApiDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApiDestinationRequest));
            CompletableFuture<UpdateApiDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified archive.
     * </p>
     *
     * @param updateArchiveRequest
     * @return A Java Future containing the result of the UpdateArchive operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>InvalidEventPatternException The event pattern is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.UpdateArchive
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/UpdateArchive" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateArchiveResponse> updateArchive(UpdateArchiveRequest updateArchiveRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateArchiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateArchive");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateArchiveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateArchiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateArchiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateArchiveRequest, UpdateArchiveResponse>()
                            .withOperationName("UpdateArchive")
                            .withMarshaller(new UpdateArchiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateArchiveRequest));
            CompletableFuture<UpdateArchiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings for a connection.
     * </p>
     *
     * @param updateConnectionRequest
     * @return A Java Future containing the result of the UpdateConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException There is concurrent modification on a rule, target, archive, or
     *         replay.</li>
     *         <li>ResourceNotFoundException An entity that you specified does not exist.</li>
     *         <li>InternalException This exception occurs due to unexpected causes.</li>
     *         <li>LimitExceededException The request failed because it attempted to create resource beyond the allowed
     *         service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchEventsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchEventsAsyncClient.UpdateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/events-2015-10-07/UpdateConnection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConnectionResponse> updateConnection(UpdateConnectionRequest updateConnectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch Events");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectionRequest, UpdateConnectionResponse>()
                            .withOperationName("UpdateConnection")
                            .withMarshaller(new UpdateConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConnectionRequest));
            CompletableFuture<UpdateConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudWatchEventsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyLengthExceededException")
                                .exceptionBuilderSupplier(PolicyLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventPatternException")
                                .exceptionBuilderSupplier(InvalidEventPatternException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationDisabledException")
                                .exceptionBuilderSupplier(OperationDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalStatusException")
                                .exceptionBuilderSupplier(IllegalStatusException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ManagedRuleException")
                                .exceptionBuilderSupplier(ManagedRuleException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
