/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudwatchlogs.model.AssociateKmsKeyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.AssociateKmsKeyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.CancelExportTaskRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.CancelExportTaskResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.CloudWatchLogsException;
import software.amazon.awssdk.services.cloudwatchlogs.model.CloudWatchLogsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.CreateExportTaskRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.CreateExportTaskResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.CreateLogGroupRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.CreateLogGroupResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.CreateLogStreamRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.CreateLogStreamResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DataAlreadyAcceptedException;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteDestinationRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteDestinationResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteLogGroupRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteLogGroupResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteLogStreamRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteLogStreamResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteMetricFilterRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteMetricFilterResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteRetentionPolicyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteRetentionPolicyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteSubscriptionFilterRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DeleteSubscriptionFilterResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeDestinationsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeDestinationsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeExportTasksRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeExportTasksResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogStreamsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogStreamsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeMetricFiltersRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeMetricFiltersResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeQueriesRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeQueriesResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeResourcePoliciesRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeResourcePoliciesResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeSubscriptionFiltersRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeSubscriptionFiltersResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.DisassociateKmsKeyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DisassociateKmsKeyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.FilterLogEventsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.FilterLogEventsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogGroupFieldsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogGroupFieldsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogRecordRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogRecordResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetQueryResultsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.InvalidOperationException;
import software.amazon.awssdk.services.cloudwatchlogs.model.InvalidParameterException;
import software.amazon.awssdk.services.cloudwatchlogs.model.InvalidSequenceTokenException;
import software.amazon.awssdk.services.cloudwatchlogs.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatchlogs.model.ListTagsLogGroupRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.ListTagsLogGroupResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.MalformedQueryException;
import software.amazon.awssdk.services.cloudwatchlogs.model.OperationAbortedException;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutDestinationPolicyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutDestinationPolicyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutDestinationRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutDestinationResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutLogEventsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutLogEventsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutMetricFilterRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutMetricFilterResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutRetentionPolicyRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutRetentionPolicyResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutSubscriptionFilterRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutSubscriptionFilterResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.cloudwatchlogs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatchlogs.model.ServiceUnavailableException;
import software.amazon.awssdk.services.cloudwatchlogs.model.StartQueryRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.StartQueryResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.StopQueryRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.StopQueryResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.TagLogGroupRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.TagLogGroupResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.TestMetricFilterRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.TestMetricFilterResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.UnrecognizedClientException;
import software.amazon.awssdk.services.cloudwatchlogs.model.UntagLogGroupRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.UntagLogGroupResponse;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeDestinationsPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogGroupsPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogStreamsPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeMetricFiltersPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeSubscriptionFiltersPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.FilterLogEventsPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.GetLogEventsPublisher;
import software.amazon.awssdk.services.cloudwatchlogs.transform.AssociateKmsKeyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.CancelExportTaskRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.CreateExportTaskRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.CreateLogGroupRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.CreateLogStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteDestinationRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteLogGroupRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteLogStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteMetricFilterRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteRetentionPolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DeleteSubscriptionFilterRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeDestinationsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeExportTasksRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeLogGroupsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeLogStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeMetricFiltersRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeQueriesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeResourcePoliciesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DescribeSubscriptionFiltersRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.DisassociateKmsKeyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.FilterLogEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.GetLogEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.GetLogGroupFieldsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.GetLogRecordRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.GetQueryResultsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.ListTagsLogGroupRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutDestinationPolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutDestinationRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutLogEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutMetricFilterRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutRetentionPolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.PutSubscriptionFilterRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.StartQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.StopQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.TagLogGroupRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.TestMetricFilterRequestMarshaller;
import software.amazon.awssdk.services.cloudwatchlogs.transform.UntagLogGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudWatchLogsAsyncClient}.
 *
 * @see CloudWatchLogsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchLogsAsyncClient implements CloudWatchLogsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudWatchLogsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchLogsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log
     * group.
     * </p>
     * <p>
     * Associating an AWS KMS CMK with a log group overrides any existing associations between the log group and a CMK.
     * After a CMK is associated with a log group, all newly ingested data for the log group is encrypted using the CMK.
     * This association is stored as long as the data encrypted with the CMK is still within Amazon CloudWatch Logs.
     * This enables Amazon CloudWatch Logs to decrypt this data whenever it is requested.
     * </p>
     * <p>
     * Note that it can take up to 5 minutes for this operation to take effect.
     * </p>
     * <p>
     * If you attempt to associate a CMK with a log group but the CMK does not exist or the CMK is disabled, you will
     * receive an <code>InvalidParameterException</code> error.
     * </p>
     *
     * @param associateKmsKeyRequest
     * @return A Java Future containing the result of the AssociateKmsKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.AssociateKmsKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/AssociateKmsKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateKmsKeyResponse> associateKmsKey(AssociateKmsKeyRequest associateKmsKeyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateKmsKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateKmsKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateKmsKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateKmsKeyRequest, AssociateKmsKeyResponse>()
                            .withOperationName("AssociateKmsKey")
                            .withMarshaller(new AssociateKmsKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateKmsKeyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the specified export task.
     * </p>
     * <p>
     * The task must be in the <code>PENDING</code> or <code>RUNNING</code> state.
     * </p>
     *
     * @param cancelExportTaskRequest
     * @return A Java Future containing the result of the CancelExportTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidOperationException The operation is not valid on the specified resource.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.CancelExportTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/CancelExportTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelExportTaskResponse> cancelExportTask(CancelExportTaskRequest cancelExportTaskRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelExportTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelExportTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelExportTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelExportTaskRequest, CancelExportTaskResponse>()
                            .withOperationName("CancelExportTask")
                            .withMarshaller(new CancelExportTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(cancelExportTaskRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an export task, which allows you to efficiently export data from a log group to an Amazon S3 bucket.
     * </p>
     * <p>
     * This is an asynchronous call. If all the required information is provided, this operation initiates an export
     * task and responds with the ID of the task. After the task has started, you can use <a>DescribeExportTasks</a> to
     * get the status of the export task. Each account can only have one active (<code>RUNNING</code> or
     * <code>PENDING</code>) export task at a time. To cancel an export task, use <a>CancelExportTask</a>.
     * </p>
     * <p>
     * You can export logs from multiple log groups or multiple time ranges to the same S3 bucket. To separate out log
     * data for each export task, you can specify a prefix to be used as the Amazon S3 key prefix for all exported
     * objects.
     * </p>
     * <p>
     * Exporting to S3 buckets that are encrypted with AES-256 is supported. Exporting to S3 buckets encrypted with
     * SSE-KMS is not supported.
     * </p>
     *
     * @param createExportTaskRequest
     * @return A Java Future containing the result of the CreateExportTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ResourceAlreadyExistsException The specified resource already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.CreateExportTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/CreateExportTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExportTaskResponse> createExportTask(CreateExportTaskRequest createExportTaskRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExportTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExportTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExportTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExportTaskRequest, CreateExportTaskResponse>()
                            .withOperationName("CreateExportTask")
                            .withMarshaller(new CreateExportTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createExportTaskRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a log group with the specified name.
     * </p>
     * <p>
     * You can create up to 20,000 log groups per account.
     * </p>
     * <p>
     * You must use the following guidelines when naming a log group:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Log group names must be unique within a region for an AWS account.
     * </p>
     * </li>
     * <li>
     * <p>
     * Log group names can be between 1 and 512 characters long.
     * </p>
     * </li>
     * <li>
     * <p>
     * Log group names consist of the following characters: a-z, A-Z, 0-9, '_' (underscore), '-' (hyphen), '/' (forward
     * slash), '.' (period), and '#' (number sign)
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you associate a AWS Key Management Service (AWS KMS) customer master key (CMK) with the log group, ingested
     * data is encrypted using the CMK. This association is stored as long as the data encrypted with the CMK is still
     * within Amazon CloudWatch Logs. This enables Amazon CloudWatch Logs to decrypt this data whenever it is requested.
     * </p>
     * <p>
     * If you attempt to associate a CMK with the log group but the CMK does not exist or the CMK is disabled, you will
     * receive an <code>InvalidParameterException</code> error.
     * </p>
     *
     * @param createLogGroupRequest
     * @return A Java Future containing the result of the CreateLogGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceAlreadyExistsException The specified resource already exists.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.CreateLogGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/CreateLogGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLogGroupResponse> createLogGroup(CreateLogGroupRequest createLogGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLogGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLogGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLogGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLogGroupRequest, CreateLogGroupResponse>()
                            .withOperationName("CreateLogGroup")
                            .withMarshaller(new CreateLogGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createLogGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a log stream for the specified log group.
     * </p>
     * <p>
     * There is no limit on the number of log streams that you can create for a log group.
     * </p>
     * <p>
     * You must use the following guidelines when naming a log stream:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Log stream names must be unique within the log group.
     * </p>
     * </li>
     * <li>
     * <p>
     * Log stream names can be between 1 and 512 characters long.
     * </p>
     * </li>
     * <li>
     * <p>
     * The ':' (colon) and '*' (asterisk) characters are not allowed.
     * </p>
     * </li>
     * </ul>
     *
     * @param createLogStreamRequest
     * @return A Java Future containing the result of the CreateLogStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceAlreadyExistsException The specified resource already exists.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.CreateLogStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/CreateLogStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLogStreamResponse> createLogStream(CreateLogStreamRequest createLogStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLogStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLogStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLogStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLogStreamRequest, CreateLogStreamResponse>()
                            .withOperationName("CreateLogStream")
                            .withMarshaller(new CreateLogStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createLogStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified destination, and eventually disables all the subscription filters that publish to it. This
     * operation does not delete the physical resource encapsulated by the destination.
     * </p>
     *
     * @param deleteDestinationRequest
     * @return A Java Future containing the result of the DeleteDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteDestination
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteDestination" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDestinationResponse> deleteDestination(DeleteDestinationRequest deleteDestinationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDestinationRequest, DeleteDestinationResponse>()
                            .withOperationName("DeleteDestination")
                            .withMarshaller(new DeleteDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDestinationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified log group and permanently deletes all the archived log events associated with the log
     * group.
     * </p>
     *
     * @param deleteLogGroupRequest
     * @return A Java Future containing the result of the DeleteLogGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteLogGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteLogGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLogGroupResponse> deleteLogGroup(DeleteLogGroupRequest deleteLogGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLogGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLogGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLogGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLogGroupRequest, DeleteLogGroupResponse>()
                            .withOperationName("DeleteLogGroup")
                            .withMarshaller(new DeleteLogGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteLogGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified log stream and permanently deletes all the archived log events associated with the log
     * stream.
     * </p>
     *
     * @param deleteLogStreamRequest
     * @return A Java Future containing the result of the DeleteLogStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteLogStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteLogStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLogStreamResponse> deleteLogStream(DeleteLogStreamRequest deleteLogStreamRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLogStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLogStreamResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLogStreamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLogStreamRequest, DeleteLogStreamResponse>()
                            .withOperationName("DeleteLogStream")
                            .withMarshaller(new DeleteLogStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteLogStreamRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified metric filter.
     * </p>
     *
     * @param deleteMetricFilterRequest
     * @return A Java Future containing the result of the DeleteMetricFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteMetricFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteMetricFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMetricFilterResponse> deleteMetricFilter(DeleteMetricFilterRequest deleteMetricFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMetricFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMetricFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMetricFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMetricFilterRequest, DeleteMetricFilterResponse>()
                            .withOperationName("DeleteMetricFilter")
                            .withMarshaller(new DeleteMetricFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteMetricFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a resource policy from this account. This revokes the access of the identities in that policy to put log
     * events to this account.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy")
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteResourcePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified retention policy.
     * </p>
     * <p>
     * Log events do not expire if they belong to log groups without a retention policy.
     * </p>
     *
     * @param deleteRetentionPolicyRequest
     * @return A Java Future containing the result of the DeleteRetentionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteRetentionPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteRetentionPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRetentionPolicyResponse> deleteRetentionPolicy(
            DeleteRetentionPolicyRequest deleteRetentionPolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRetentionPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRetentionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRetentionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRetentionPolicyRequest, DeleteRetentionPolicyResponse>()
                            .withOperationName("DeleteRetentionPolicy")
                            .withMarshaller(new DeleteRetentionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteRetentionPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified subscription filter.
     * </p>
     *
     * @param deleteSubscriptionFilterRequest
     * @return A Java Future containing the result of the DeleteSubscriptionFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DeleteSubscriptionFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DeleteSubscriptionFilter" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubscriptionFilterResponse> deleteSubscriptionFilter(
            DeleteSubscriptionFilterRequest deleteSubscriptionFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubscriptionFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubscriptionFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubscriptionFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriptionFilterRequest, DeleteSubscriptionFilterResponse>()
                            .withOperationName("DeleteSubscriptionFilter")
                            .withMarshaller(new DeleteSubscriptionFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteSubscriptionFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all your destinations. The results are ASCII-sorted by destination name.
     * </p>
     *
     * @param describeDestinationsRequest
     * @return A Java Future containing the result of the DescribeDestinations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeDestinations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeDestinations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDestinationsResponse> describeDestinations(
            DescribeDestinationsRequest describeDestinationsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDestinationsRequest, DescribeDestinationsResponse>()
                            .withOperationName("DescribeDestinations")
                            .withMarshaller(new DescribeDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDestinationsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all your destinations. The results are ASCII-sorted by destination name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDestinations(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeDestinationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeDestinationsPublisher publisher = client.describeDestinationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeDestinationsPublisher publisher = client.describeDestinationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.DescribeDestinationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeDestinationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDestinations(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeDestinationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDestinationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeDestinations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeDestinations" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeDestinationsPublisher describeDestinationsPaginator(DescribeDestinationsRequest describeDestinationsRequest) {
        return new DescribeDestinationsPublisher(this, applyPaginatorUserAgent(describeDestinationsRequest));
    }

    /**
     * <p>
     * Lists the specified export tasks. You can list all your export tasks or filter the results based on task ID or
     * task status.
     * </p>
     *
     * @param describeExportTasksRequest
     * @return A Java Future containing the result of the DescribeExportTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeExportTasks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeExportTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExportTasksResponse> describeExportTasks(
            DescribeExportTasksRequest describeExportTasksRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExportTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExportTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExportTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExportTasksRequest, DescribeExportTasksResponse>()
                            .withOperationName("DescribeExportTasks")
                            .withMarshaller(new DescribeExportTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeExportTasksRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the specified log groups. You can list all your log groups or filter the results by prefix. The results are
     * ASCII-sorted by log group name.
     * </p>
     *
     * @param describeLogGroupsRequest
     * @return A Java Future containing the result of the DescribeLogGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeLogGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeLogGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLogGroupsResponse> describeLogGroups(DescribeLogGroupsRequest describeLogGroupsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLogGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLogGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLogGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLogGroupsRequest, DescribeLogGroupsResponse>()
                            .withOperationName("DescribeLogGroups")
                            .withMarshaller(new DescribeLogGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeLogGroupsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the specified log groups. You can list all your log groups or filter the results by prefix. The results are
     * ASCII-sorted by log group name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeLogGroups(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogGroupsPublisher publisher = client.describeLogGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogGroupsPublisher publisher = client.describeLogGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeLogGroups(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeLogGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeLogGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeLogGroups" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeLogGroupsPublisher describeLogGroupsPaginator(DescribeLogGroupsRequest describeLogGroupsRequest) {
        return new DescribeLogGroupsPublisher(this, applyPaginatorUserAgent(describeLogGroupsRequest));
    }

    /**
     * <p>
     * Lists the log streams for the specified log group. You can list all the log streams or filter the results by
     * prefix. You can also control how the results are ordered.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second, after which transactions are throttled.
     * </p>
     *
     * @param describeLogStreamsRequest
     * @return A Java Future containing the result of the DescribeLogStreams operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeLogStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeLogStreams" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLogStreamsResponse> describeLogStreams(DescribeLogStreamsRequest describeLogStreamsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLogStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLogStreamsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLogStreamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLogStreamsRequest, DescribeLogStreamsResponse>()
                            .withOperationName("DescribeLogStreams")
                            .withMarshaller(new DescribeLogStreamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeLogStreamsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the log streams for the specified log group. You can list all the log streams or filter the results by
     * prefix. You can also control how the results are ordered.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second, after which transactions are throttled.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeLogStreams(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogStreamsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogStreamsPublisher publisher = client.describeLogStreamsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogStreamsPublisher publisher = client.describeLogStreamsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogStreamsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogStreamsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeLogStreams(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogStreamsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeLogStreamsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeLogStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeLogStreams" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeLogStreamsPublisher describeLogStreamsPaginator(DescribeLogStreamsRequest describeLogStreamsRequest) {
        return new DescribeLogStreamsPublisher(this, applyPaginatorUserAgent(describeLogStreamsRequest));
    }

    /**
     * <p>
     * Lists the specified metric filters. You can list all the metric filters or filter the results by log name,
     * prefix, metric name, or metric namespace. The results are ASCII-sorted by filter name.
     * </p>
     *
     * @param describeMetricFiltersRequest
     * @return A Java Future containing the result of the DescribeMetricFilters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeMetricFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeMetricFilters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMetricFiltersResponse> describeMetricFilters(
            DescribeMetricFiltersRequest describeMetricFiltersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMetricFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMetricFiltersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMetricFiltersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMetricFiltersRequest, DescribeMetricFiltersResponse>()
                            .withOperationName("DescribeMetricFilters")
                            .withMarshaller(new DescribeMetricFiltersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeMetricFiltersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the specified metric filters. You can list all the metric filters or filter the results by log name,
     * prefix, metric name, or metric namespace. The results are ASCII-sorted by filter name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeMetricFilters(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeMetricFiltersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeMetricFiltersPublisher publisher = client.describeMetricFiltersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeMetricFiltersPublisher publisher = client.describeMetricFiltersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.DescribeMetricFiltersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeMetricFiltersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeMetricFilters(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeMetricFiltersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeMetricFiltersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeMetricFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeMetricFilters" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeMetricFiltersPublisher describeMetricFiltersPaginator(DescribeMetricFiltersRequest describeMetricFiltersRequest) {
        return new DescribeMetricFiltersPublisher(this, applyPaginatorUserAgent(describeMetricFiltersRequest));
    }

    /**
     * <p>
     * Returns a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently
     * in this account. You can request all queries, or limit it to queries of a specific log group or queries with a
     * certain status.
     * </p>
     *
     * @param describeQueriesRequest
     * @return A Java Future containing the result of the DescribeQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeQueries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeQueries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeQueriesResponse> describeQueries(DescribeQueriesRequest describeQueriesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeQueriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeQueriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeQueriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeQueriesRequest, DescribeQueriesResponse>()
                            .withOperationName("DescribeQueries")
                            .withMarshaller(new DescribeQueriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeQueriesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource policies in this account.
     * </p>
     *
     * @param describeResourcePoliciesRequest
     * @return A Java Future containing the result of the DescribeResourcePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeResourcePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeResourcePolicies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeResourcePoliciesResponse> describeResourcePolicies(
            DescribeResourcePoliciesRequest describeResourcePoliciesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeResourcePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeResourcePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeResourcePoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeResourcePoliciesRequest, DescribeResourcePoliciesResponse>()
                            .withOperationName("DescribeResourcePolicies")
                            .withMarshaller(new DescribeResourcePoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeResourcePoliciesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the subscription filters for the specified log group. You can list all the subscription filters or filter
     * the results by prefix. The results are ASCII-sorted by filter name.
     * </p>
     *
     * @param describeSubscriptionFiltersRequest
     * @return A Java Future containing the result of the DescribeSubscriptionFilters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeSubscriptionFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeSubscriptionFilters"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSubscriptionFiltersResponse> describeSubscriptionFilters(
            DescribeSubscriptionFiltersRequest describeSubscriptionFiltersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSubscriptionFiltersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSubscriptionFiltersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSubscriptionFiltersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSubscriptionFiltersRequest, DescribeSubscriptionFiltersResponse>()
                            .withOperationName("DescribeSubscriptionFilters")
                            .withMarshaller(new DescribeSubscriptionFiltersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeSubscriptionFiltersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the subscription filters for the specified log group. You can list all the subscription filters or filter
     * the results by prefix. The results are ASCII-sorted by filter name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeSubscriptionFilters(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeSubscriptionFiltersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeSubscriptionFiltersPublisher publisher = client.describeSubscriptionFiltersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeSubscriptionFiltersPublisher publisher = client.describeSubscriptionFiltersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.DescribeSubscriptionFiltersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeSubscriptionFiltersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSubscriptionFilters(software.amazon.awssdk.services.cloudwatchlogs.model.DescribeSubscriptionFiltersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeSubscriptionFiltersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DescribeSubscriptionFilters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DescribeSubscriptionFilters"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeSubscriptionFiltersPublisher describeSubscriptionFiltersPaginator(
            DescribeSubscriptionFiltersRequest describeSubscriptionFiltersRequest) {
        return new DescribeSubscriptionFiltersPublisher(this, applyPaginatorUserAgent(describeSubscriptionFiltersRequest));
    }

    /**
     * <p>
     * Disassociates the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified
     * log group.
     * </p>
     * <p>
     * After the AWS KMS CMK is disassociated from the log group, AWS CloudWatch Logs stops encrypting newly ingested
     * data for the log group. All previously ingested data remains encrypted, and AWS CloudWatch Logs requires
     * permissions for the CMK whenever the encrypted data is requested.
     * </p>
     * <p>
     * Note that it can take up to 5 minutes for this operation to take effect.
     * </p>
     *
     * @param disassociateKmsKeyRequest
     * @return A Java Future containing the result of the DisassociateKmsKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.DisassociateKmsKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/DisassociateKmsKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateKmsKeyResponse> disassociateKmsKey(DisassociateKmsKeyRequest disassociateKmsKeyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateKmsKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateKmsKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateKmsKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateKmsKeyRequest, DisassociateKmsKeyResponse>()
                            .withOperationName("DisassociateKmsKey")
                            .withMarshaller(new DisassociateKmsKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateKmsKeyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists log events from the specified log group. You can list all the log events or filter the results using a
     * filter pattern, a time range, and the name of the log stream.
     * </p>
     * <p>
     * By default, this operation returns as many log events as can fit in 1 MB (up to 10,000 log events), or all the
     * events found within the time range that you specify. If the results include a token, then there are more log
     * events available, and you can get additional results by specifying the token in a subsequent call.
     * </p>
     *
     * @param filterLogEventsRequest
     * @return A Java Future containing the result of the FilterLogEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.FilterLogEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/FilterLogEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<FilterLogEventsResponse> filterLogEvents(FilterLogEventsRequest filterLogEventsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<FilterLogEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, FilterLogEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<FilterLogEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<FilterLogEventsRequest, FilterLogEventsResponse>()
                            .withOperationName("FilterLogEvents")
                            .withMarshaller(new FilterLogEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(filterLogEventsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists log events from the specified log group. You can list all the log events or filter the results using a
     * filter pattern, a time range, and the name of the log stream.
     * </p>
     * <p>
     * By default, this operation returns as many log events as can fit in 1 MB (up to 10,000 log events), or all the
     * events found within the time range that you specify. If the results include a token, then there are more log
     * events available, and you can get additional results by specifying the token in a subsequent call.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #filterLogEvents(software.amazon.awssdk.services.cloudwatchlogs.model.FilterLogEventsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.FilterLogEventsPublisher publisher = client.filterLogEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.FilterLogEventsPublisher publisher = client.filterLogEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.FilterLogEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.FilterLogEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #filterLogEvents(software.amazon.awssdk.services.cloudwatchlogs.model.FilterLogEventsRequest)}
     * operation.</b>
     * </p>
     *
     * @param filterLogEventsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.FilterLogEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/FilterLogEvents" target="_top">AWS API
     *      Documentation</a>
     */
    public FilterLogEventsPublisher filterLogEventsPaginator(FilterLogEventsRequest filterLogEventsRequest) {
        return new FilterLogEventsPublisher(this, applyPaginatorUserAgent(filterLogEventsRequest));
    }

    /**
     * <p>
     * Lists log events from the specified log stream. You can list all the log events or filter using a time range.
     * </p>
     * <p>
     * By default, this operation returns as many log events as can fit in a response size of 1MB (up to 10,000 log
     * events). You can get additional log events by specifying one of the tokens in a subsequent call.
     * </p>
     *
     * @param getLogEventsRequest
     * @return A Java Future containing the result of the GetLogEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.GetLogEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/GetLogEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLogEventsResponse> getLogEvents(GetLogEventsRequest getLogEventsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLogEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLogEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLogEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLogEventsRequest, GetLogEventsResponse>()
                            .withOperationName("GetLogEvents").withMarshaller(new GetLogEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getLogEventsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists log events from the specified log stream. You can list all the log events or filter using a time range.
     * </p>
     * <p>
     * By default, this operation returns as many log events as can fit in a response size of 1MB (up to 10,000 log
     * events). You can get additional log events by specifying one of the tokens in a subsequent call.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getLogEvents(software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.GetLogEventsPublisher publisher = client.getLogEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatchlogs.paginators.GetLogEventsPublisher publisher = client.getLogEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getLogEvents(software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsRequest)} operation.</b>
     * </p>
     *
     * @param getLogEventsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.GetLogEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/GetLogEvents" target="_top">AWS API
     *      Documentation</a>
     */
    public GetLogEventsPublisher getLogEventsPaginator(GetLogEventsRequest getLogEventsRequest) {
        return new GetLogEventsPublisher(this, applyPaginatorUserAgent(getLogEventsRequest));
    }

    /**
     * <p>
     * Returns a list of the fields that are included in log events in the specified log group, along with the
     * percentage of log events that contain each field. The search is limited to a time period that you specify.
     * </p>
     * <p>
     * In the results, fields that start with @ are fields generated by CloudWatch Logs. For example,
     * <code>@timestamp</code> is the timestamp of each log event.
     * </p>
     * <p>
     * The response results are sorted by the frequency percentage, starting with the highest percentage.
     * </p>
     *
     * @param getLogGroupFieldsRequest
     * @return A Java Future containing the result of the GetLogGroupFields operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.GetLogGroupFields
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/GetLogGroupFields" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLogGroupFieldsResponse> getLogGroupFields(GetLogGroupFieldsRequest getLogGroupFieldsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLogGroupFieldsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLogGroupFieldsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLogGroupFieldsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLogGroupFieldsRequest, GetLogGroupFieldsResponse>()
                            .withOperationName("GetLogGroupFields")
                            .withMarshaller(new GetLogGroupFieldsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getLogGroupFieldsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all the fields and values of a single log event. All fields are retrieved, even if the original query
     * that produced the <code>logRecordPointer</code> retrieved only a subset of fields. Fields are returned as field
     * name/field value pairs.
     * </p>
     * <p>
     * Additionally, the entire unparsed log event is returned within <code>@message</code>.
     * </p>
     *
     * @param getLogRecordRequest
     * @return A Java Future containing the result of the GetLogRecord operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.GetLogRecord
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/GetLogRecord" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLogRecordResponse> getLogRecord(GetLogRecordRequest getLogRecordRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLogRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLogRecordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLogRecordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLogRecordRequest, GetLogRecordResponse>()
                            .withOperationName("GetLogRecord").withMarshaller(new GetLogRecordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getLogRecordRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the results from the specified query.
     * </p>
     * <p>
     * Only the fields requested in the query are returned, along with a <code>@ptr</code> field which is the identifier
     * for the log record. You can use the value of <code>@ptr</code> in a operation to get the full log record.
     * </p>
     * <p>
     * <code>GetQueryResults</code> does not start a query execution. To run a query, use .
     * </p>
     * <p>
     * If the value of the <code>Status</code> field in the output is <code>Running</code>, this operation returns only
     * partial results. If you see a value of <code>Scheduled</code> or <code>Running</code> for the status, you can
     * retry the operation later to see the final results.
     * </p>
     *
     * @param getQueryResultsRequest
     * @return A Java Future containing the result of the GetQueryResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.GetQueryResults
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/GetQueryResults" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryResultsResponse> getQueryResults(GetQueryResultsRequest getQueryResultsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQueryResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQueryResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsRequest, GetQueryResultsResponse>()
                            .withOperationName("GetQueryResults")
                            .withMarshaller(new GetQueryResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getQueryResultsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified log group.
     * </p>
     *
     * @param listTagsLogGroupRequest
     * @return A Java Future containing the result of the ListTagsLogGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.ListTagsLogGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/ListTagsLogGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsLogGroupResponse> listTagsLogGroup(ListTagsLogGroupRequest listTagsLogGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsLogGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsLogGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsLogGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsLogGroupRequest, ListTagsLogGroupResponse>()
                            .withOperationName("ListTagsLogGroup")
                            .withMarshaller(new ListTagsLogGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsLogGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a destination. This operation is used only to create destinations for cross-account
     * subscriptions.
     * </p>
     * <p>
     * A destination encapsulates a physical resource (such as an Amazon Kinesis stream) and enables you to subscribe to
     * a real-time stream of log events for a different account, ingested using <a>PutLogEvents</a>.
     * </p>
     * <p>
     * Through an access policy, a destination controls what is written to it. By default, <code>PutDestination</code>
     * does not set any access policy with the destination, which means a cross-account user cannot call
     * <a>PutSubscriptionFilter</a> against this destination. To enable this, the destination owner must call
     * <a>PutDestinationPolicy</a> after <code>PutDestination</code>.
     * </p>
     *
     * @param putDestinationRequest
     * @return A Java Future containing the result of the PutDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutDestination
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutDestination" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutDestinationResponse> putDestination(PutDestinationRequest putDestinationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDestinationRequest, PutDestinationResponse>()
                            .withOperationName("PutDestination")
                            .withMarshaller(new PutDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putDestinationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates an access policy associated with an existing destination. An access policy is an <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html">IAM policy document</a> that is
     * used to authorize claims to register a subscription filter against a given destination.
     * </p>
     *
     * @param putDestinationPolicyRequest
     * @return A Java Future containing the result of the PutDestinationPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutDestinationPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutDestinationPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutDestinationPolicyResponse> putDestinationPolicy(
            PutDestinationPolicyRequest putDestinationPolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDestinationPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDestinationPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDestinationPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDestinationPolicyRequest, PutDestinationPolicyResponse>()
                            .withOperationName("PutDestinationPolicy")
                            .withMarshaller(new PutDestinationPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putDestinationPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads a batch of log events to the specified log stream.
     * </p>
     * <p>
     * You must include the sequence token obtained from the response of the previous call. An upload in a newly created
     * log stream does not require a sequence token. You can also get the sequence token using
     * <a>DescribeLogStreams</a>. If you call <code>PutLogEvents</code> twice within a narrow time period using the same
     * value for <code>sequenceToken</code>, both calls may be successful, or one may be rejected.
     * </p>
     * <p>
     * The batch of events must satisfy the following constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The maximum batch size is 1,048,576 bytes, and this size is calculated as the sum of all event messages in UTF-8,
     * plus 26 bytes for each log event.
     * </p>
     * </li>
     * <li>
     * <p>
     * None of the log events in the batch can be more than 2 hours in the future.
     * </p>
     * </li>
     * <li>
     * <p>
     * None of the log events in the batch can be older than 14 days or older than the retention period of the log
     * group.
     * </p>
     * </li>
     * <li>
     * <p>
     * The log events in the batch must be in chronological ordered by their timestamp. The timestamp is the time the
     * event occurred, expressed as the number of milliseconds after Jan 1, 1970 00:00:00 UTC. (In AWS Tools for
     * PowerShell and the AWS SDK for .NET, the timestamp is specified in .NET format: yyyy-mm-ddThh:mm:ss. For example,
     * 2017-09-15T13:45:30.)
     * </p>
     * </li>
     * <li>
     * <p>
     * The maximum number of log events in a batch is 10,000.
     * </p>
     * </li>
     * <li>
     * <p>
     * A batch of log events in a single request cannot span more than 24 hours. Otherwise, the operation fails.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If a call to PutLogEvents returns "UnrecognizedClientException" the most likely cause is an invalid AWS access
     * key ID or secret key.
     * </p>
     *
     * @param putLogEventsRequest
     * @return A Java Future containing the result of the PutLogEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>InvalidSequenceTokenException The sequence token is not valid.</li>
     *         <li>DataAlreadyAcceptedException The event was already logged.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>UnrecognizedClientException The most likely cause is an invalid AWS access key ID or secret key.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutLogEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutLogEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutLogEventsResponse> putLogEvents(PutLogEventsRequest putLogEventsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLogEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutLogEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLogEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLogEventsRequest, PutLogEventsResponse>()
                            .withOperationName("PutLogEvents").withMarshaller(new PutLogEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putLogEventsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a metric filter and associates it with the specified log group. Metric filters allow you to
     * configure rules to extract metric data from log events ingested through <a>PutLogEvents</a>.
     * </p>
     * <p>
     * The maximum number of metric filters that can be associated with a log group is 100.
     * </p>
     *
     * @param putMetricFilterRequest
     * @return A Java Future containing the result of the PutMetricFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutMetricFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutMetricFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutMetricFilterResponse> putMetricFilter(PutMetricFilterRequest putMetricFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutMetricFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutMetricFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutMetricFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutMetricFilterRequest, PutMetricFilterResponse>()
                            .withOperationName("PutMetricFilter")
                            .withMarshaller(new PutMetricFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putMetricFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a resource policy allowing other AWS services to put log events to this account, such as
     * Amazon Route 53. An account can have up to 10 resource policies per region.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutResourcePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy")
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putResourcePolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the retention of the specified log group. A retention policy allows you to configure the number of days for
     * which to retain log events in the specified log group.
     * </p>
     *
     * @param putRetentionPolicyRequest
     * @return A Java Future containing the result of the PutRetentionPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutRetentionPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutRetentionPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutRetentionPolicyResponse> putRetentionPolicy(PutRetentionPolicyRequest putRetentionPolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRetentionPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRetentionPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRetentionPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRetentionPolicyRequest, PutRetentionPolicyResponse>()
                            .withOperationName("PutRetentionPolicy")
                            .withMarshaller(new PutRetentionPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putRetentionPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a subscription filter and associates it with the specified log group. Subscription filters
     * allow you to subscribe to a real-time stream of log events ingested through <a>PutLogEvents</a> and have them
     * delivered to a specific destination. Currently, the supported destinations are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon Kinesis stream belonging to the same account as the subscription filter, for same-account delivery.
     * </p>
     * </li>
     * <li>
     * <p>
     * A logical destination that belongs to a different account, for cross-account delivery.
     * </p>
     * </li>
     * <li>
     * <p>
     * An Amazon Kinesis Firehose delivery stream that belongs to the same account as the subscription filter, for
     * same-account delivery.
     * </p>
     * </li>
     * <li>
     * <p>
     * An AWS Lambda function that belongs to the same account as the subscription filter, for same-account delivery.
     * </p>
     * </li>
     * </ul>
     * <p>
     * There can only be one subscription filter associated with a log group. If you are updating an existing filter,
     * you must specify the correct name in <code>filterName</code>. Otherwise, the call fails because you cannot
     * associate a second filter with a log group.
     * </p>
     *
     * @param putSubscriptionFilterRequest
     * @return A Java Future containing the result of the PutSubscriptionFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>OperationAbortedException Multiple requests to update the same resource were in conflict.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.PutSubscriptionFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/PutSubscriptionFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutSubscriptionFilterResponse> putSubscriptionFilter(
            PutSubscriptionFilterRequest putSubscriptionFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutSubscriptionFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutSubscriptionFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutSubscriptionFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutSubscriptionFilterRequest, PutSubscriptionFilterResponse>()
                            .withOperationName("PutSubscriptionFilter")
                            .withMarshaller(new PutSubscriptionFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putSubscriptionFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Schedules a query of a log group using CloudWatch Logs Insights. You specify the log group and time range to
     * query, and the query string to use.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL_QuerySyntax.html">CloudWatch Logs Insights
     * Query Syntax</a>.
     * </p>
     * <p>
     * Queries time out after 15 minutes of execution. If your queries are timing out, reduce the time range being
     * searched, or partition your query into a number of queries.
     * </p>
     *
     * @param startQueryRequest
     * @return A Java Future containing the result of the StartQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MalformedQueryException The query string is not valid. Details about this error are displayed in a
     *         <code>QueryCompileError</code> object. For more information, see .</p>
     *         <p>
     *         For more information about valid query syntax, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL_QuerySyntax.html">CloudWatch Logs
     *         Insights Query Syntax</a>.</li>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>LimitExceededException You have reached the maximum number of resources that can be created.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.StartQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/StartQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryResponse> startQuery(StartQueryRequest startQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryRequest, StartQueryResponse>().withOperationName("StartQuery")
                            .withMarshaller(new StartQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a CloudWatch Logs Insights query that is in progress. If the query has already ended, the operation returns
     * an error indicating that the specified query is not running.
     * </p>
     *
     * @param stopQueryRequest
     * @return A Java Future containing the result of the StopQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.StopQuery
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/StopQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopQueryResponse> stopQuery(StopQueryRequest stopQueryRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopQueryRequest, StopQueryResponse>().withOperationName("StopQuery")
                            .withMarshaller(new StopQueryRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(stopQueryRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates the specified tags for the specified log group.
     * </p>
     * <p>
     * To list the tags for a log group, use <a>ListTagsLogGroup</a>. To remove tags, use <a>UntagLogGroup</a>.
     * </p>
     * <p>
     * For more information about tags, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/log-group-tagging.html">Tag Log Groups in Amazon
     * CloudWatch Logs</a> in the <i>Amazon CloudWatch Logs User Guide</i>.
     * </p>
     *
     * @param tagLogGroupRequest
     * @return A Java Future containing the result of the TagLogGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.TagLogGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/TagLogGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagLogGroupResponse> tagLogGroup(TagLogGroupRequest tagLogGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagLogGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagLogGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagLogGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagLogGroupRequest, TagLogGroupResponse>()
                            .withOperationName("TagLogGroup").withMarshaller(new TagLogGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagLogGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tests the filter pattern of a metric filter against a sample of log event messages. You can use this operation to
     * validate the correctness of a metric filter pattern.
     * </p>
     *
     * @param testMetricFilterRequest
     * @return A Java Future containing the result of the TestMetricFilter operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException A parameter is specified incorrectly.</li>
     *         <li>ServiceUnavailableException The service cannot complete the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.TestMetricFilter
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/TestMetricFilter" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TestMetricFilterResponse> testMetricFilter(TestMetricFilterRequest testMetricFilterRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestMetricFilterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestMetricFilterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestMetricFilterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestMetricFilterRequest, TestMetricFilterResponse>()
                            .withOperationName("TestMetricFilter")
                            .withMarshaller(new TestMetricFilterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(testMetricFilterRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified log group.
     * </p>
     * <p>
     * To list the tags for a log group, use <a>ListTagsLogGroup</a>. To add tags, use <a>UntagLogGroup</a>.
     * </p>
     *
     * @param untagLogGroupRequest
     * @return A Java Future containing the result of the UntagLogGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudWatchLogsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CloudWatchLogsAsyncClient.UntagLogGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/UntagLogGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagLogGroupResponse> untagLogGroup(UntagLogGroupRequest untagLogGroupRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagLogGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagLogGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagLogGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagLogGroupRequest, UntagLogGroupResponse>()
                            .withOperationName("UntagLogGroup")
                            .withMarshaller(new UntagLogGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagLogGroupRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudWatchLogsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationAbortedException")
                                .exceptionBuilderSupplier(OperationAbortedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnrecognizedClientException")
                                .exceptionBuilderSupplier(UnrecognizedClientException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedQueryException")
                                .exceptionBuilderSupplier(MalformedQueryException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataAlreadyAcceptedException")
                                .exceptionBuilderSupplier(DataAlreadyAcceptedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSequenceTokenException")
                                .exceptionBuilderSupplier(InvalidSequenceTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
    }

    private <T extends CloudWatchLogsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
