/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains details about a saved CloudWatch Logs Insights query definition.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryDefinition implements SdkPojo, Serializable, ToCopyableBuilder<QueryDefinition.Builder, QueryDefinition> {
    private static final SdkField<String> QUERY_DEFINITION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("queryDefinitionId").getter(getter(QueryDefinition::queryDefinitionId))
            .setter(setter(Builder::queryDefinitionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("queryDefinitionId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(QueryDefinition::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> QUERY_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("queryString").getter(getter(QueryDefinition::queryString)).setter(setter(Builder::queryString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("queryString").build()).build();

    private static final SdkField<Long> LAST_MODIFIED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("lastModified").getter(getter(QueryDefinition::lastModified)).setter(setter(Builder::lastModified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModified").build()).build();

    private static final SdkField<List<String>> LOG_GROUP_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("logGroupNames")
            .getter(getter(QueryDefinition::logGroupNames))
            .setter(setter(Builder::logGroupNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logGroupNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUERY_DEFINITION_ID_FIELD,
            NAME_FIELD, QUERY_STRING_FIELD, LAST_MODIFIED_FIELD, LOG_GROUP_NAMES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String queryDefinitionId;

    private final String name;

    private final String queryString;

    private final Long lastModified;

    private final List<String> logGroupNames;

    private QueryDefinition(BuilderImpl builder) {
        this.queryDefinitionId = builder.queryDefinitionId;
        this.name = builder.name;
        this.queryString = builder.queryString;
        this.lastModified = builder.lastModified;
        this.logGroupNames = builder.logGroupNames;
    }

    /**
     * <p>
     * The unique ID of the query definition.
     * </p>
     * 
     * @return The unique ID of the query definition.
     */
    public final String queryDefinitionId() {
        return queryDefinitionId;
    }

    /**
     * <p>
     * The name of the query definition.
     * </p>
     * 
     * @return The name of the query definition.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The query string to use for this definition. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL_QuerySyntax.html">CloudWatch Logs Insights
     * Query Syntax</a>.
     * </p>
     * 
     * @return The query string to use for this definition. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL_QuerySyntax.html">CloudWatch Logs
     *         Insights Query Syntax</a>.
     */
    public final String queryString() {
        return queryString;
    }

    /**
     * <p>
     * The date that the query definition was most recently modified.
     * </p>
     * 
     * @return The date that the query definition was most recently modified.
     */
    public final Long lastModified() {
        return lastModified;
    }

    /**
     * Returns true if the LogGroupNames property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasLogGroupNames() {
        return logGroupNames != null && !(logGroupNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If this query definition contains a list of log groups that it is limited to, that list appears here.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogGroupNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If this query definition contains a list of log groups that it is limited to, that list appears here.
     */
    public final List<String> logGroupNames() {
        return logGroupNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(queryDefinitionId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(queryString());
        hashCode = 31 * hashCode + Objects.hashCode(lastModified());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogGroupNames() ? logGroupNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryDefinition)) {
            return false;
        }
        QueryDefinition other = (QueryDefinition) obj;
        return Objects.equals(queryDefinitionId(), other.queryDefinitionId()) && Objects.equals(name(), other.name())
                && Objects.equals(queryString(), other.queryString()) && Objects.equals(lastModified(), other.lastModified())
                && hasLogGroupNames() == other.hasLogGroupNames() && Objects.equals(logGroupNames(), other.logGroupNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QueryDefinition").add("QueryDefinitionId", queryDefinitionId()).add("Name", name())
                .add("QueryString", queryString()).add("LastModified", lastModified())
                .add("LogGroupNames", hasLogGroupNames() ? logGroupNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "queryDefinitionId":
            return Optional.ofNullable(clazz.cast(queryDefinitionId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "queryString":
            return Optional.ofNullable(clazz.cast(queryString()));
        case "lastModified":
            return Optional.ofNullable(clazz.cast(lastModified()));
        case "logGroupNames":
            return Optional.ofNullable(clazz.cast(logGroupNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QueryDefinition, T> g) {
        return obj -> g.apply((QueryDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryDefinition> {
        /**
         * <p>
         * The unique ID of the query definition.
         * </p>
         * 
         * @param queryDefinitionId
         *        The unique ID of the query definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryDefinitionId(String queryDefinitionId);

        /**
         * <p>
         * The name of the query definition.
         * </p>
         * 
         * @param name
         *        The name of the query definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The query string to use for this definition. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL_QuerySyntax.html">CloudWatch Logs Insights
         * Query Syntax</a>.
         * </p>
         * 
         * @param queryString
         *        The query string to use for this definition. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CWL_QuerySyntax.html">CloudWatch Logs
         *        Insights Query Syntax</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryString(String queryString);

        /**
         * <p>
         * The date that the query definition was most recently modified.
         * </p>
         * 
         * @param lastModified
         *        The date that the query definition was most recently modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModified(Long lastModified);

        /**
         * <p>
         * If this query definition contains a list of log groups that it is limited to, that list appears here.
         * </p>
         * 
         * @param logGroupNames
         *        If this query definition contains a list of log groups that it is limited to, that list appears here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupNames(Collection<String> logGroupNames);

        /**
         * <p>
         * If this query definition contains a list of log groups that it is limited to, that list appears here.
         * </p>
         * 
         * @param logGroupNames
         *        If this query definition contains a list of log groups that it is limited to, that list appears here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupNames(String... logGroupNames);
    }

    static final class BuilderImpl implements Builder {
        private String queryDefinitionId;

        private String name;

        private String queryString;

        private Long lastModified;

        private List<String> logGroupNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(QueryDefinition model) {
            queryDefinitionId(model.queryDefinitionId);
            name(model.name);
            queryString(model.queryString);
            lastModified(model.lastModified);
            logGroupNames(model.logGroupNames);
        }

        public final String getQueryDefinitionId() {
            return queryDefinitionId;
        }

        public final void setQueryDefinitionId(String queryDefinitionId) {
            this.queryDefinitionId = queryDefinitionId;
        }

        @Override
        @Transient
        public final Builder queryDefinitionId(String queryDefinitionId) {
            this.queryDefinitionId = queryDefinitionId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getQueryString() {
            return queryString;
        }

        public final void setQueryString(String queryString) {
            this.queryString = queryString;
        }

        @Override
        @Transient
        public final Builder queryString(String queryString) {
            this.queryString = queryString;
            return this;
        }

        public final Long getLastModified() {
            return lastModified;
        }

        public final void setLastModified(Long lastModified) {
            this.lastModified = lastModified;
        }

        @Override
        @Transient
        public final Builder lastModified(Long lastModified) {
            this.lastModified = lastModified;
            return this;
        }

        public final Collection<String> getLogGroupNames() {
            if (logGroupNames instanceof SdkAutoConstructList) {
                return null;
            }
            return logGroupNames;
        }

        public final void setLogGroupNames(Collection<String> logGroupNames) {
            this.logGroupNames = LogGroupNamesCopier.copy(logGroupNames);
        }

        @Override
        @Transient
        public final Builder logGroupNames(Collection<String> logGroupNames) {
            this.logGroupNames = LogGroupNamesCopier.copy(logGroupNames);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder logGroupNames(String... logGroupNames) {
            logGroupNames(Arrays.asList(logGroupNames));
            return this;
        }

        @Override
        public QueryDefinition build() {
            return new QueryDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
