/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about one <i>delivery source</i> in your account. A delivery source is an Amazon
 * Web Services resource that sends logs to an Amazon Web Services destination. The destination can be CloudWatch Logs,
 * Amazon S3, or Kinesis Data Firehose.
 * </p>
 * <p>
 * Only some Amazon Web Services services support being configured as a delivery source. These services are listed as
 * <b>Supported [V2 Permissions]</b> in the table at <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html">Enabling logging
 * from Amazon Web Services services.</a>
 * </p>
 * <p>
 * To configure logs delivery between a supported Amazon Web Services service and a destination, you must do the
 * following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Create a delivery source, which is a logical object that represents the resource that is actually sending the logs.
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliverySource.html"
 * >PutDeliverySource</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Create a <i>delivery destination</i>, which is a logical object that represents the actual delivery destination. For
 * more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliveryDestination.html"
 * >PutDeliveryDestination</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you are delivering logs cross-account, you must use <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliveryDestinationolicy.html"
 * >PutDeliveryDestinationPolicy</a> in the destination account to assign an IAM policy to the destination. This policy
 * allows delivery to that destination.
 * </p>
 * </li>
 * <li>
 * <p>
 * Create a <i>delivery</i> by pairing exactly one delivery source and one delivery destination. For more information,
 * see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateDelivery.html">CreateDelivery
 * </a>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can configure a single delivery source to send logs to multiple destinations by creating multiple deliveries. You
 * can also create multiple deliveries to configure multiple delivery sources to send logs to the same delivery
 * destination.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeliverySource implements SdkPojo, Serializable, ToCopyableBuilder<DeliverySource.Builder, DeliverySource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(DeliverySource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(DeliverySource::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<List<String>> RESOURCE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceArns")
            .getter(getter(DeliverySource::resourceArns))
            .setter(setter(Builder::resourceArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("service")
            .getter(getter(DeliverySource::service)).setter(setter(Builder::service))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("service").build()).build();

    private static final SdkField<String> LOG_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("logType").getter(getter(DeliverySource::logType)).setter(setter(Builder::logType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logType").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(DeliverySource::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            RESOURCE_ARNS_FIELD, SERVICE_FIELD, LOG_TYPE_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final List<String> resourceArns;

    private final String service;

    private final String logType;

    private final Map<String, String> tags;

    private DeliverySource(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.resourceArns = builder.resourceArns;
        this.service = builder.service;
        this.logType = builder.logType;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The unique name of the delivery source.
     * </p>
     * 
     * @return The unique name of the delivery source.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies this delivery source.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies this delivery source.
     */
    public final String arn() {
        return arn;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceArns property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceArns() {
        return resourceArns != null && !(resourceArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This array contains the ARN of the Amazon Web Services resource that sends logs and is represented by this
     * delivery source. Currently, only one ARN can be in the array.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceArns} method.
     * </p>
     * 
     * @return This array contains the ARN of the Amazon Web Services resource that sends logs and is represented by
     *         this delivery source. Currently, only one ARN can be in the array.
     */
    public final List<String> resourceArns() {
        return resourceArns;
    }

    /**
     * <p>
     * The Amazon Web Services service that is sending logs.
     * </p>
     * 
     * @return The Amazon Web Services service that is sending logs.
     */
    public final String service() {
        return service;
    }

    /**
     * <p>
     * The type of log that the source is sending. For valid values for this parameter, see the documentation for the
     * source service.
     * </p>
     * 
     * @return The type of log that the source is sending. For valid values for this parameter, see the documentation
     *         for the source service.
     */
    public final String logType() {
        return logType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that have been assigned to this delivery source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that have been assigned to this delivery source.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceArns() ? resourceArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(service());
        hashCode = 31 * hashCode + Objects.hashCode(logType());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeliverySource)) {
            return false;
        }
        DeliverySource other = (DeliverySource) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && hasResourceArns() == other.hasResourceArns() && Objects.equals(resourceArns(), other.resourceArns())
                && Objects.equals(service(), other.service()) && Objects.equals(logType(), other.logType())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeliverySource").add("Name", name()).add("Arn", arn())
                .add("ResourceArns", hasResourceArns() ? resourceArns() : null).add("Service", service())
                .add("LogType", logType()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "resourceArns":
            return Optional.ofNullable(clazz.cast(resourceArns()));
        case "service":
            return Optional.ofNullable(clazz.cast(service()));
        case "logType":
            return Optional.ofNullable(clazz.cast(logType()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeliverySource, T> g) {
        return obj -> g.apply((DeliverySource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeliverySource> {
        /**
         * <p>
         * The unique name of the delivery source.
         * </p>
         * 
         * @param name
         *        The unique name of the delivery source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies this delivery source.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) that uniquely identifies this delivery source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * This array contains the ARN of the Amazon Web Services resource that sends logs and is represented by this
         * delivery source. Currently, only one ARN can be in the array.
         * </p>
         * 
         * @param resourceArns
         *        This array contains the ARN of the Amazon Web Services resource that sends logs and is represented by
         *        this delivery source. Currently, only one ARN can be in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArns(Collection<String> resourceArns);

        /**
         * <p>
         * This array contains the ARN of the Amazon Web Services resource that sends logs and is represented by this
         * delivery source. Currently, only one ARN can be in the array.
         * </p>
         * 
         * @param resourceArns
         *        This array contains the ARN of the Amazon Web Services resource that sends logs and is represented by
         *        this delivery source. Currently, only one ARN can be in the array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArns(String... resourceArns);

        /**
         * <p>
         * The Amazon Web Services service that is sending logs.
         * </p>
         * 
         * @param service
         *        The Amazon Web Services service that is sending logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder service(String service);

        /**
         * <p>
         * The type of log that the source is sending. For valid values for this parameter, see the documentation for
         * the source service.
         * </p>
         * 
         * @param logType
         *        The type of log that the source is sending. For valid values for this parameter, see the documentation
         *        for the source service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logType(String logType);

        /**
         * <p>
         * The tags that have been assigned to this delivery source.
         * </p>
         * 
         * @param tags
         *        The tags that have been assigned to this delivery source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private List<String> resourceArns = DefaultSdkAutoConstructList.getInstance();

        private String service;

        private String logType;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DeliverySource model) {
            name(model.name);
            arn(model.arn);
            resourceArns(model.resourceArns);
            service(model.service);
            logType(model.logType);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Collection<String> getResourceArns() {
            if (resourceArns instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceArns;
        }

        public final void setResourceArns(Collection<String> resourceArns) {
            this.resourceArns = ResourceArnsCopier.copy(resourceArns);
        }

        @Override
        public final Builder resourceArns(Collection<String> resourceArns) {
            this.resourceArns = ResourceArnsCopier.copy(resourceArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceArns(String... resourceArns) {
            resourceArns(Arrays.asList(resourceArns));
            return this;
        }

        public final String getService() {
            return service;
        }

        public final void setService(String service) {
            this.service = service;
        }

        @Override
        public final Builder service(String service) {
            this.service = service;
            return this;
        }

        public final String getLogType() {
            return logType;
        }

        public final void setLogType(String logType) {
            this.logType = logType;
        }

        @Override
        public final Builder logType(String logType) {
            this.logType = logType;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        public DeliverySource build() {
            return new DeliverySource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
