/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about one <i>delivery</i> in your account.
 * </p>
 * <p>
 * A delivery is a connection between a logical <i>delivery source</i> and a logical <i>delivery destination</i>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateDelivery.html"
 * >CreateDelivery</a>.
 * </p>
 * <p>
 * You can't update an existing delivery. You can only create and delete deliveries.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Delivery implements SdkPojo, Serializable, ToCopyableBuilder<Delivery.Builder, Delivery> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Delivery::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Delivery::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> DELIVERY_SOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deliverySourceName").getter(getter(Delivery::deliverySourceName))
            .setter(setter(Builder::deliverySourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deliverySourceName").build())
            .build();

    private static final SdkField<String> DELIVERY_DESTINATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deliveryDestinationArn").getter(getter(Delivery::deliveryDestinationArn))
            .setter(setter(Builder::deliveryDestinationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deliveryDestinationArn").build())
            .build();

    private static final SdkField<String> DELIVERY_DESTINATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deliveryDestinationType").getter(getter(Delivery::deliveryDestinationTypeAsString))
            .setter(setter(Builder::deliveryDestinationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deliveryDestinationType").build())
            .build();

    private static final SdkField<List<String>> RECORD_FIELDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recordFields")
            .getter(getter(Delivery::recordFields))
            .setter(setter(Builder::recordFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordFields").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FIELD_DELIMITER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("fieldDelimiter").getter(getter(Delivery::fieldDelimiter)).setter(setter(Builder::fieldDelimiter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fieldDelimiter").build()).build();

    private static final SdkField<S3DeliveryConfiguration> S3_DELIVERY_CONFIGURATION_FIELD = SdkField
            .<S3DeliveryConfiguration> builder(MarshallingType.SDK_POJO).memberName("s3DeliveryConfiguration")
            .getter(getter(Delivery::s3DeliveryConfiguration)).setter(setter(Builder::s3DeliveryConfiguration))
            .constructor(S3DeliveryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3DeliveryConfiguration").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Delivery::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            DELIVERY_SOURCE_NAME_FIELD, DELIVERY_DESTINATION_ARN_FIELD, DELIVERY_DESTINATION_TYPE_FIELD, RECORD_FIELDS_FIELD,
            FIELD_DELIMITER_FIELD, S3_DELIVERY_CONFIGURATION_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final String deliverySourceName;

    private final String deliveryDestinationArn;

    private final String deliveryDestinationType;

    private final List<String> recordFields;

    private final String fieldDelimiter;

    private final S3DeliveryConfiguration s3DeliveryConfiguration;

    private final Map<String, String> tags;

    private Delivery(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.deliverySourceName = builder.deliverySourceName;
        this.deliveryDestinationArn = builder.deliveryDestinationArn;
        this.deliveryDestinationType = builder.deliveryDestinationType;
        this.recordFields = builder.recordFields;
        this.fieldDelimiter = builder.fieldDelimiter;
        this.s3DeliveryConfiguration = builder.s3DeliveryConfiguration;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The unique ID that identifies this delivery in your account.
     * </p>
     * 
     * @return The unique ID that identifies this delivery in your account.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies this delivery.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies this delivery.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the delivery source that is associated with this delivery.
     * </p>
     * 
     * @return The name of the delivery source that is associated with this delivery.
     */
    public final String deliverySourceName() {
        return deliverySourceName;
    }

    /**
     * <p>
     * The ARN of the delivery destination that is associated with this delivery.
     * </p>
     * 
     * @return The ARN of the delivery destination that is associated with this delivery.
     */
    public final String deliveryDestinationArn() {
        return deliveryDestinationArn;
    }

    /**
     * <p>
     * Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3, or
     * Firehose.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryDestinationType} will return {@link DeliveryDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #deliveryDestinationTypeAsString}.
     * </p>
     * 
     * @return Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3, or
     *         Firehose.
     * @see DeliveryDestinationType
     */
    public final DeliveryDestinationType deliveryDestinationType() {
        return DeliveryDestinationType.fromValue(deliveryDestinationType);
    }

    /**
     * <p>
     * Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3, or
     * Firehose.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryDestinationType} will return {@link DeliveryDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #deliveryDestinationTypeAsString}.
     * </p>
     * 
     * @return Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3, or
     *         Firehose.
     * @see DeliveryDestinationType
     */
    public final String deliveryDestinationTypeAsString() {
        return deliveryDestinationType;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecordFields property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecordFields() {
        return recordFields != null && !(recordFields instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The record fields used in this delivery.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecordFields} method.
     * </p>
     * 
     * @return The record fields used in this delivery.
     */
    public final List<String> recordFields() {
        return recordFields;
    }

    /**
     * <p>
     * The field delimiter that is used between record fields when the final output format of a delivery is in
     * <code>Plain</code>, <code>W3C</code>, or <code>Raw</code> format.
     * </p>
     * 
     * @return The field delimiter that is used between record fields when the final output format of a delivery is in
     *         <code>Plain</code>, <code>W3C</code>, or <code>Raw</code> format.
     */
    public final String fieldDelimiter() {
        return fieldDelimiter;
    }

    /**
     * <p>
     * This structure contains delivery configurations that apply only when the delivery destination resource is an S3
     * bucket.
     * </p>
     * 
     * @return This structure contains delivery configurations that apply only when the delivery destination resource is
     *         an S3 bucket.
     */
    public final S3DeliveryConfiguration s3DeliveryConfiguration() {
        return s3DeliveryConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that have been assigned to this delivery.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that have been assigned to this delivery.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(deliverySourceName());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryDestinationArn());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryDestinationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecordFields() ? recordFields() : null);
        hashCode = 31 * hashCode + Objects.hashCode(fieldDelimiter());
        hashCode = 31 * hashCode + Objects.hashCode(s3DeliveryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Delivery)) {
            return false;
        }
        Delivery other = (Delivery) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(deliverySourceName(), other.deliverySourceName())
                && Objects.equals(deliveryDestinationArn(), other.deliveryDestinationArn())
                && Objects.equals(deliveryDestinationTypeAsString(), other.deliveryDestinationTypeAsString())
                && hasRecordFields() == other.hasRecordFields() && Objects.equals(recordFields(), other.recordFields())
                && Objects.equals(fieldDelimiter(), other.fieldDelimiter())
                && Objects.equals(s3DeliveryConfiguration(), other.s3DeliveryConfiguration()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Delivery").add("Id", id()).add("Arn", arn()).add("DeliverySourceName", deliverySourceName())
                .add("DeliveryDestinationArn", deliveryDestinationArn())
                .add("DeliveryDestinationType", deliveryDestinationTypeAsString())
                .add("RecordFields", hasRecordFields() ? recordFields() : null).add("FieldDelimiter", fieldDelimiter())
                .add("S3DeliveryConfiguration", s3DeliveryConfiguration()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "deliverySourceName":
            return Optional.ofNullable(clazz.cast(deliverySourceName()));
        case "deliveryDestinationArn":
            return Optional.ofNullable(clazz.cast(deliveryDestinationArn()));
        case "deliveryDestinationType":
            return Optional.ofNullable(clazz.cast(deliveryDestinationTypeAsString()));
        case "recordFields":
            return Optional.ofNullable(clazz.cast(recordFields()));
        case "fieldDelimiter":
            return Optional.ofNullable(clazz.cast(fieldDelimiter()));
        case "s3DeliveryConfiguration":
            return Optional.ofNullable(clazz.cast(s3DeliveryConfiguration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Delivery, T> g) {
        return obj -> g.apply((Delivery) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Delivery> {
        /**
         * <p>
         * The unique ID that identifies this delivery in your account.
         * </p>
         * 
         * @param id
         *        The unique ID that identifies this delivery in your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies this delivery.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) that uniquely identifies this delivery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the delivery source that is associated with this delivery.
         * </p>
         * 
         * @param deliverySourceName
         *        The name of the delivery source that is associated with this delivery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverySourceName(String deliverySourceName);

        /**
         * <p>
         * The ARN of the delivery destination that is associated with this delivery.
         * </p>
         * 
         * @param deliveryDestinationArn
         *        The ARN of the delivery destination that is associated with this delivery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryDestinationArn(String deliveryDestinationArn);

        /**
         * <p>
         * Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3, or
         * Firehose.
         * </p>
         * 
         * @param deliveryDestinationType
         *        Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3,
         *        or Firehose.
         * @see DeliveryDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryDestinationType
         */
        Builder deliveryDestinationType(String deliveryDestinationType);

        /**
         * <p>
         * Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3, or
         * Firehose.
         * </p>
         * 
         * @param deliveryDestinationType
         *        Displays whether the delivery destination associated with this delivery is CloudWatch Logs, Amazon S3,
         *        or Firehose.
         * @see DeliveryDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryDestinationType
         */
        Builder deliveryDestinationType(DeliveryDestinationType deliveryDestinationType);

        /**
         * <p>
         * The record fields used in this delivery.
         * </p>
         * 
         * @param recordFields
         *        The record fields used in this delivery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordFields(Collection<String> recordFields);

        /**
         * <p>
         * The record fields used in this delivery.
         * </p>
         * 
         * @param recordFields
         *        The record fields used in this delivery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordFields(String... recordFields);

        /**
         * <p>
         * The field delimiter that is used between record fields when the final output format of a delivery is in
         * <code>Plain</code>, <code>W3C</code>, or <code>Raw</code> format.
         * </p>
         * 
         * @param fieldDelimiter
         *        The field delimiter that is used between record fields when the final output format of a delivery is
         *        in <code>Plain</code>, <code>W3C</code>, or <code>Raw</code> format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldDelimiter(String fieldDelimiter);

        /**
         * <p>
         * This structure contains delivery configurations that apply only when the delivery destination resource is an
         * S3 bucket.
         * </p>
         * 
         * @param s3DeliveryConfiguration
         *        This structure contains delivery configurations that apply only when the delivery destination resource
         *        is an S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3DeliveryConfiguration(S3DeliveryConfiguration s3DeliveryConfiguration);

        /**
         * <p>
         * This structure contains delivery configurations that apply only when the delivery destination resource is an
         * S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3DeliveryConfiguration.Builder} avoiding
         * the need to create one manually via {@link S3DeliveryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3DeliveryConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #s3DeliveryConfiguration(S3DeliveryConfiguration)}.
         * 
         * @param s3DeliveryConfiguration
         *        a consumer that will call methods on {@link S3DeliveryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3DeliveryConfiguration(S3DeliveryConfiguration)
         */
        default Builder s3DeliveryConfiguration(Consumer<S3DeliveryConfiguration.Builder> s3DeliveryConfiguration) {
            return s3DeliveryConfiguration(S3DeliveryConfiguration.builder().applyMutation(s3DeliveryConfiguration).build());
        }

        /**
         * <p>
         * The tags that have been assigned to this delivery.
         * </p>
         * 
         * @param tags
         *        The tags that have been assigned to this delivery.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private String deliverySourceName;

        private String deliveryDestinationArn;

        private String deliveryDestinationType;

        private List<String> recordFields = DefaultSdkAutoConstructList.getInstance();

        private String fieldDelimiter;

        private S3DeliveryConfiguration s3DeliveryConfiguration;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Delivery model) {
            id(model.id);
            arn(model.arn);
            deliverySourceName(model.deliverySourceName);
            deliveryDestinationArn(model.deliveryDestinationArn);
            deliveryDestinationType(model.deliveryDestinationType);
            recordFields(model.recordFields);
            fieldDelimiter(model.fieldDelimiter);
            s3DeliveryConfiguration(model.s3DeliveryConfiguration);
            tags(model.tags);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDeliverySourceName() {
            return deliverySourceName;
        }

        public final void setDeliverySourceName(String deliverySourceName) {
            this.deliverySourceName = deliverySourceName;
        }

        @Override
        public final Builder deliverySourceName(String deliverySourceName) {
            this.deliverySourceName = deliverySourceName;
            return this;
        }

        public final String getDeliveryDestinationArn() {
            return deliveryDestinationArn;
        }

        public final void setDeliveryDestinationArn(String deliveryDestinationArn) {
            this.deliveryDestinationArn = deliveryDestinationArn;
        }

        @Override
        public final Builder deliveryDestinationArn(String deliveryDestinationArn) {
            this.deliveryDestinationArn = deliveryDestinationArn;
            return this;
        }

        public final String getDeliveryDestinationType() {
            return deliveryDestinationType;
        }

        public final void setDeliveryDestinationType(String deliveryDestinationType) {
            this.deliveryDestinationType = deliveryDestinationType;
        }

        @Override
        public final Builder deliveryDestinationType(String deliveryDestinationType) {
            this.deliveryDestinationType = deliveryDestinationType;
            return this;
        }

        @Override
        public final Builder deliveryDestinationType(DeliveryDestinationType deliveryDestinationType) {
            this.deliveryDestinationType(deliveryDestinationType == null ? null : deliveryDestinationType.toString());
            return this;
        }

        public final Collection<String> getRecordFields() {
            if (recordFields instanceof SdkAutoConstructList) {
                return null;
            }
            return recordFields;
        }

        public final void setRecordFields(Collection<String> recordFields) {
            this.recordFields = RecordFieldsCopier.copy(recordFields);
        }

        @Override
        public final Builder recordFields(Collection<String> recordFields) {
            this.recordFields = RecordFieldsCopier.copy(recordFields);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recordFields(String... recordFields) {
            recordFields(Arrays.asList(recordFields));
            return this;
        }

        public final String getFieldDelimiter() {
            return fieldDelimiter;
        }

        public final void setFieldDelimiter(String fieldDelimiter) {
            this.fieldDelimiter = fieldDelimiter;
        }

        @Override
        public final Builder fieldDelimiter(String fieldDelimiter) {
            this.fieldDelimiter = fieldDelimiter;
            return this;
        }

        public final S3DeliveryConfiguration.Builder getS3DeliveryConfiguration() {
            return s3DeliveryConfiguration != null ? s3DeliveryConfiguration.toBuilder() : null;
        }

        public final void setS3DeliveryConfiguration(S3DeliveryConfiguration.BuilderImpl s3DeliveryConfiguration) {
            this.s3DeliveryConfiguration = s3DeliveryConfiguration != null ? s3DeliveryConfiguration.build() : null;
        }

        @Override
        public final Builder s3DeliveryConfiguration(S3DeliveryConfiguration s3DeliveryConfiguration) {
            this.s3DeliveryConfiguration = s3DeliveryConfiguration;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        public Delivery build() {
            return new Delivery(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
