/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The <code>CSV</code> processor parses comma-separated values (CSV) from the log events into columns.
 * </p>
 * <p>
 * For more information about this processor including examples, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation.html#CloudWatch-Logs-Transformation-csv"
 * > csv</a> in the <i>CloudWatch Logs User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CSV implements SdkPojo, Serializable, ToCopyableBuilder<CSV.Builder, CSV> {
    private static final SdkField<String> QUOTE_CHARACTER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("quoteCharacter").getter(getter(CSV::quoteCharacter)).setter(setter(Builder::quoteCharacter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("quoteCharacter").build()).build();

    private static final SdkField<String> DELIMITER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("delimiter").getter(getter(CSV::delimiter)).setter(setter(Builder::delimiter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("delimiter").build()).build();

    private static final SdkField<List<String>> COLUMNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("columns")
            .getter(getter(CSV::columns))
            .setter(setter(Builder::columns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("columns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("source")
            .getter(getter(CSV::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUOTE_CHARACTER_FIELD,
            DELIMITER_FIELD, COLUMNS_FIELD, SOURCE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("quoteCharacter", QUOTE_CHARACTER_FIELD);
                    put("delimiter", DELIMITER_FIELD);
                    put("columns", COLUMNS_FIELD);
                    put("source", SOURCE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String quoteCharacter;

    private final String delimiter;

    private final List<String> columns;

    private final String source;

    private CSV(BuilderImpl builder) {
        this.quoteCharacter = builder.quoteCharacter;
        this.delimiter = builder.delimiter;
        this.columns = builder.columns;
        this.source = builder.source;
    }

    /**
     * <p>
     * The character used used as a text qualifier for a single column of data. If you omit this, the double quotation
     * mark <code>"</code> character is used.
     * </p>
     * 
     * @return The character used used as a text qualifier for a single column of data. If you omit this, the double
     *         quotation mark <code>"</code> character is used.
     */
    public final String quoteCharacter() {
        return quoteCharacter;
    }

    /**
     * <p>
     * The character used to separate each column in the original comma-separated value log event. If you omit this, the
     * processor looks for the comma <code>,</code> character as the delimiter.
     * </p>
     * 
     * @return The character used to separate each column in the original comma-separated value log event. If you omit
     *         this, the processor looks for the comma <code>,</code> character as the delimiter.
     */
    public final String delimiter() {
        return delimiter;
    }

    /**
     * For responses, this returns true if the service returned a value for the Columns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasColumns() {
        return columns != null && !(columns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of names to use for the columns in the transformed log event.
     * </p>
     * <p>
     * If you omit this, default column names (<code>[column_1, column_2 ...]</code>) are used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasColumns} method.
     * </p>
     * 
     * @return An array of names to use for the columns in the transformed log event.</p>
     *         <p>
     *         If you omit this, default column names (<code>[column_1, column_2 ...]</code>) are used.
     */
    public final List<String> columns() {
        return columns;
    }

    /**
     * <p>
     * The path to the field in the log event that has the comma separated values to be parsed. If you omit this value,
     * the whole log message is processed.
     * </p>
     * 
     * @return The path to the field in the log event that has the comma separated values to be parsed. If you omit this
     *         value, the whole log message is processed.
     */
    public final String source() {
        return source;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quoteCharacter());
        hashCode = 31 * hashCode + Objects.hashCode(delimiter());
        hashCode = 31 * hashCode + Objects.hashCode(hasColumns() ? columns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(source());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CSV)) {
            return false;
        }
        CSV other = (CSV) obj;
        return Objects.equals(quoteCharacter(), other.quoteCharacter()) && Objects.equals(delimiter(), other.delimiter())
                && hasColumns() == other.hasColumns() && Objects.equals(columns(), other.columns())
                && Objects.equals(source(), other.source());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CSV").add("QuoteCharacter", quoteCharacter()).add("Delimiter", delimiter())
                .add("Columns", hasColumns() ? columns() : null).add("Source", source()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "quoteCharacter":
            return Optional.ofNullable(clazz.cast(quoteCharacter()));
        case "delimiter":
            return Optional.ofNullable(clazz.cast(delimiter()));
        case "columns":
            return Optional.ofNullable(clazz.cast(columns()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CSV, T> g) {
        return obj -> g.apply((CSV) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CSV> {
        /**
         * <p>
         * The character used used as a text qualifier for a single column of data. If you omit this, the double
         * quotation mark <code>"</code> character is used.
         * </p>
         * 
         * @param quoteCharacter
         *        The character used used as a text qualifier for a single column of data. If you omit this, the double
         *        quotation mark <code>"</code> character is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quoteCharacter(String quoteCharacter);

        /**
         * <p>
         * The character used to separate each column in the original comma-separated value log event. If you omit this,
         * the processor looks for the comma <code>,</code> character as the delimiter.
         * </p>
         * 
         * @param delimiter
         *        The character used to separate each column in the original comma-separated value log event. If you
         *        omit this, the processor looks for the comma <code>,</code> character as the delimiter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delimiter(String delimiter);

        /**
         * <p>
         * An array of names to use for the columns in the transformed log event.
         * </p>
         * <p>
         * If you omit this, default column names (<code>[column_1, column_2 ...]</code>) are used.
         * </p>
         * 
         * @param columns
         *        An array of names to use for the columns in the transformed log event.</p>
         *        <p>
         *        If you omit this, default column names (<code>[column_1, column_2 ...]</code>) are used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columns(Collection<String> columns);

        /**
         * <p>
         * An array of names to use for the columns in the transformed log event.
         * </p>
         * <p>
         * If you omit this, default column names (<code>[column_1, column_2 ...]</code>) are used.
         * </p>
         * 
         * @param columns
         *        An array of names to use for the columns in the transformed log event.</p>
         *        <p>
         *        If you omit this, default column names (<code>[column_1, column_2 ...]</code>) are used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columns(String... columns);

        /**
         * <p>
         * The path to the field in the log event that has the comma separated values to be parsed. If you omit this
         * value, the whole log message is processed.
         * </p>
         * 
         * @param source
         *        The path to the field in the log event that has the comma separated values to be parsed. If you omit
         *        this value, the whole log message is processed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);
    }

    static final class BuilderImpl implements Builder {
        private String quoteCharacter;

        private String delimiter;

        private List<String> columns = DefaultSdkAutoConstructList.getInstance();

        private String source;

        private BuilderImpl() {
        }

        private BuilderImpl(CSV model) {
            quoteCharacter(model.quoteCharacter);
            delimiter(model.delimiter);
            columns(model.columns);
            source(model.source);
        }

        public final String getQuoteCharacter() {
            return quoteCharacter;
        }

        public final void setQuoteCharacter(String quoteCharacter) {
            this.quoteCharacter = quoteCharacter;
        }

        @Override
        public final Builder quoteCharacter(String quoteCharacter) {
            this.quoteCharacter = quoteCharacter;
            return this;
        }

        public final String getDelimiter() {
            return delimiter;
        }

        public final void setDelimiter(String delimiter) {
            this.delimiter = delimiter;
        }

        @Override
        public final Builder delimiter(String delimiter) {
            this.delimiter = delimiter;
            return this;
        }

        public final Collection<String> getColumns() {
            if (columns instanceof SdkAutoConstructList) {
                return null;
            }
            return columns;
        }

        public final void setColumns(Collection<String> columns) {
            this.columns = ColumnsCopier.copy(columns);
        }

        @Override
        public final Builder columns(Collection<String> columns) {
            this.columns = ColumnsCopier.copy(columns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder columns(String... columns) {
            columns(Arrays.asList(columns));
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        @Override
        public CSV build() {
            return new CSV(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
