/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about the OpenSearch Service data source used for this integration. This data
 * source was created as part of the integration setup. An OpenSearch Service data source defines the source and
 * destination for OpenSearch Service queries. It includes the role required to execute queries and write to
 * collections.
 * </p>
 * <p>
 * For more information about OpenSearch Service data sources , see <a
 * href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/direct-query-s3-creating.html">Creating
 * OpenSearch Service data source integrations with Amazon S3.</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpenSearchDataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<OpenSearchDataSource.Builder, OpenSearchDataSource> {
    private static final SdkField<String> DATA_SOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dataSourceName").getter(getter(OpenSearchDataSource::dataSourceName))
            .setter(setter(Builder::dataSourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSourceName").build()).build();

    private static final SdkField<OpenSearchResourceStatus> STATUS_FIELD = SdkField
            .<OpenSearchResourceStatus> builder(MarshallingType.SDK_POJO).memberName("status")
            .getter(getter(OpenSearchDataSource::status)).setter(setter(Builder::status))
            .constructor(OpenSearchResourceStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_SOURCE_NAME_FIELD,
            STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("dataSourceName", DATA_SOURCE_NAME_FIELD);
                    put("status", STATUS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String dataSourceName;

    private final OpenSearchResourceStatus status;

    private OpenSearchDataSource(BuilderImpl builder) {
        this.dataSourceName = builder.dataSourceName;
        this.status = builder.status;
    }

    /**
     * <p>
     * The name of the OpenSearch Service data source.
     * </p>
     * 
     * @return The name of the OpenSearch Service data source.
     */
    public final String dataSourceName() {
        return dataSourceName;
    }

    /**
     * <p>
     * This structure contains information about the status of this OpenSearch Service resource.
     * </p>
     * 
     * @return This structure contains information about the status of this OpenSearch Service resource.
     */
    public final OpenSearchResourceStatus status() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceName());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpenSearchDataSource)) {
            return false;
        }
        OpenSearchDataSource other = (OpenSearchDataSource) obj;
        return Objects.equals(dataSourceName(), other.dataSourceName()) && Objects.equals(status(), other.status());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpenSearchDataSource").add("DataSourceName", dataSourceName()).add("Status", status()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "dataSourceName":
            return Optional.ofNullable(clazz.cast(dataSourceName()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<OpenSearchDataSource, T> g) {
        return obj -> g.apply((OpenSearchDataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpenSearchDataSource> {
        /**
         * <p>
         * The name of the OpenSearch Service data source.
         * </p>
         * 
         * @param dataSourceName
         *        The name of the OpenSearch Service data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceName(String dataSourceName);

        /**
         * <p>
         * This structure contains information about the status of this OpenSearch Service resource.
         * </p>
         * 
         * @param status
         *        This structure contains information about the status of this OpenSearch Service resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(OpenSearchResourceStatus status);

        /**
         * <p>
         * This structure contains information about the status of this OpenSearch Service resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link OpenSearchResourceStatus.Builder}
         * avoiding the need to create one manually via {@link OpenSearchResourceStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OpenSearchResourceStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #status(OpenSearchResourceStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link OpenSearchResourceStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(OpenSearchResourceStatus)
         */
        default Builder status(Consumer<OpenSearchResourceStatus.Builder> status) {
            return status(OpenSearchResourceStatus.builder().applyMutation(status).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String dataSourceName;

        private OpenSearchResourceStatus status;

        private BuilderImpl() {
        }

        private BuilderImpl(OpenSearchDataSource model) {
            dataSourceName(model.dataSourceName);
            status(model.status);
        }

        public final String getDataSourceName() {
            return dataSourceName;
        }

        public final void setDataSourceName(String dataSourceName) {
            this.dataSourceName = dataSourceName;
        }

        @Override
        public final Builder dataSourceName(String dataSourceName) {
            this.dataSourceName = dataSourceName;
            return this;
        }

        public final OpenSearchResourceStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(OpenSearchResourceStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(OpenSearchResourceStatus status) {
            this.status = status;
            return this;
        }

        @Override
        public OpenSearchDataSource build() {
            return new OpenSearchDataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
