/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about one <i>delivery destination</i> in your account. A delivery destination is
 * an Amazon Web Services resource that represents an Amazon Web Services service that logs can be sent to. CloudWatch
 * Logs, Amazon S3, are supported as Firehose delivery destinations.
 * </p>
 * <p>
 * To configure logs delivery between a supported Amazon Web Services service and a destination, you must do the
 * following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Create a delivery source, which is a logical object that represents the resource that is actually sending the logs.
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliverySource.html"
 * >PutDeliverySource</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Create a <i>delivery destination</i>, which is a logical object that represents the actual delivery destination.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you are delivering logs cross-account, you must use <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliveryDestinationPolicy.html"
 * >PutDeliveryDestinationPolicy</a> in the destination account to assign an IAM policy to the destination. This policy
 * allows delivery to that destination.
 * </p>
 * </li>
 * <li>
 * <p>
 * Create a <i>delivery</i> by pairing exactly one delivery source and one delivery destination. For more information,
 * see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateDelivery.html">CreateDelivery
 * </a>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can configure a single delivery source to send logs to multiple destinations by creating multiple deliveries. You
 * can also create multiple deliveries to configure multiple delivery sources to send logs to the same delivery
 * destination.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeliveryDestination implements SdkPojo, Serializable,
        ToCopyableBuilder<DeliveryDestination.Builder, DeliveryDestination> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(DeliveryDestination::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(DeliveryDestination::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> DELIVERY_DESTINATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deliveryDestinationType").getter(getter(DeliveryDestination::deliveryDestinationTypeAsString))
            .setter(setter(Builder::deliveryDestinationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deliveryDestinationType").build())
            .build();

    private static final SdkField<String> OUTPUT_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("outputFormat").getter(getter(DeliveryDestination::outputFormatAsString))
            .setter(setter(Builder::outputFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputFormat").build()).build();

    private static final SdkField<DeliveryDestinationConfiguration> DELIVERY_DESTINATION_CONFIGURATION_FIELD = SdkField
            .<DeliveryDestinationConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("deliveryDestinationConfiguration")
            .getter(getter(DeliveryDestination::deliveryDestinationConfiguration))
            .setter(setter(Builder::deliveryDestinationConfiguration))
            .constructor(DeliveryDestinationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deliveryDestinationConfiguration")
                    .build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(DeliveryDestination::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            DELIVERY_DESTINATION_TYPE_FIELD, OUTPUT_FORMAT_FIELD, DELIVERY_DESTINATION_CONFIGURATION_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String deliveryDestinationType;

    private final String outputFormat;

    private final DeliveryDestinationConfiguration deliveryDestinationConfiguration;

    private final Map<String, String> tags;

    private DeliveryDestination(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.deliveryDestinationType = builder.deliveryDestinationType;
        this.outputFormat = builder.outputFormat;
        this.deliveryDestinationConfiguration = builder.deliveryDestinationConfiguration;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of this delivery destination.
     * </p>
     * 
     * @return The name of this delivery destination.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies this delivery destination.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies this delivery destination.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryDestinationType} will return {@link DeliveryDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #deliveryDestinationTypeAsString}.
     * </p>
     * 
     * @return Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
     * @see DeliveryDestinationType
     */
    public final DeliveryDestinationType deliveryDestinationType() {
        return DeliveryDestinationType.fromValue(deliveryDestinationType);
    }

    /**
     * <p>
     * Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deliveryDestinationType} will return {@link DeliveryDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #deliveryDestinationTypeAsString}.
     * </p>
     * 
     * @return Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
     * @see DeliveryDestinationType
     */
    public final String deliveryDestinationTypeAsString() {
        return deliveryDestinationType;
    }

    /**
     * <p>
     * The format of the logs that are sent to this delivery destination.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputFormat} will
     * return {@link OutputFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #outputFormatAsString}.
     * </p>
     * 
     * @return The format of the logs that are sent to this delivery destination.
     * @see OutputFormat
     */
    public final OutputFormat outputFormat() {
        return OutputFormat.fromValue(outputFormat);
    }

    /**
     * <p>
     * The format of the logs that are sent to this delivery destination.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputFormat} will
     * return {@link OutputFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #outputFormatAsString}.
     * </p>
     * 
     * @return The format of the logs that are sent to this delivery destination.
     * @see OutputFormat
     */
    public final String outputFormatAsString() {
        return outputFormat;
    }

    /**
     * <p>
     * A structure that contains the ARN of the Amazon Web Services resource that will receive the logs.
     * </p>
     * 
     * @return A structure that contains the ARN of the Amazon Web Services resource that will receive the logs.
     */
    public final DeliveryDestinationConfiguration deliveryDestinationConfiguration() {
        return deliveryDestinationConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that have been assigned to this delivery destination.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that have been assigned to this delivery destination.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryDestinationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(outputFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deliveryDestinationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeliveryDestination)) {
            return false;
        }
        DeliveryDestination other = (DeliveryDestination) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(deliveryDestinationTypeAsString(), other.deliveryDestinationTypeAsString())
                && Objects.equals(outputFormatAsString(), other.outputFormatAsString())
                && Objects.equals(deliveryDestinationConfiguration(), other.deliveryDestinationConfiguration())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeliveryDestination").add("Name", name()).add("Arn", arn())
                .add("DeliveryDestinationType", deliveryDestinationTypeAsString()).add("OutputFormat", outputFormatAsString())
                .add("DeliveryDestinationConfiguration", deliveryDestinationConfiguration())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "deliveryDestinationType":
            return Optional.ofNullable(clazz.cast(deliveryDestinationTypeAsString()));
        case "outputFormat":
            return Optional.ofNullable(clazz.cast(outputFormatAsString()));
        case "deliveryDestinationConfiguration":
            return Optional.ofNullable(clazz.cast(deliveryDestinationConfiguration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("deliveryDestinationType", DELIVERY_DESTINATION_TYPE_FIELD);
        map.put("outputFormat", OUTPUT_FORMAT_FIELD);
        map.put("deliveryDestinationConfiguration", DELIVERY_DESTINATION_CONFIGURATION_FIELD);
        map.put("tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeliveryDestination, T> g) {
        return obj -> g.apply((DeliveryDestination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeliveryDestination> {
        /**
         * <p>
         * The name of this delivery destination.
         * </p>
         * 
         * @param name
         *        The name of this delivery destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies this delivery destination.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) that uniquely identifies this delivery destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
         * </p>
         * 
         * @param deliveryDestinationType
         *        Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
         * @see DeliveryDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryDestinationType
         */
        Builder deliveryDestinationType(String deliveryDestinationType);

        /**
         * <p>
         * Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
         * </p>
         * 
         * @param deliveryDestinationType
         *        Displays whether this delivery destination is CloudWatch Logs, Amazon S3, or Firehose.
         * @see DeliveryDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeliveryDestinationType
         */
        Builder deliveryDestinationType(DeliveryDestinationType deliveryDestinationType);

        /**
         * <p>
         * The format of the logs that are sent to this delivery destination.
         * </p>
         * 
         * @param outputFormat
         *        The format of the logs that are sent to this delivery destination.
         * @see OutputFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputFormat
         */
        Builder outputFormat(String outputFormat);

        /**
         * <p>
         * The format of the logs that are sent to this delivery destination.
         * </p>
         * 
         * @param outputFormat
         *        The format of the logs that are sent to this delivery destination.
         * @see OutputFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputFormat
         */
        Builder outputFormat(OutputFormat outputFormat);

        /**
         * <p>
         * A structure that contains the ARN of the Amazon Web Services resource that will receive the logs.
         * </p>
         * 
         * @param deliveryDestinationConfiguration
         *        A structure that contains the ARN of the Amazon Web Services resource that will receive the logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliveryDestinationConfiguration(DeliveryDestinationConfiguration deliveryDestinationConfiguration);

        /**
         * <p>
         * A structure that contains the ARN of the Amazon Web Services resource that will receive the logs.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeliveryDestinationConfiguration.Builder}
         * avoiding the need to create one manually via {@link DeliveryDestinationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeliveryDestinationConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #deliveryDestinationConfiguration(DeliveryDestinationConfiguration)}.
         * 
         * @param deliveryDestinationConfiguration
         *        a consumer that will call methods on {@link DeliveryDestinationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deliveryDestinationConfiguration(DeliveryDestinationConfiguration)
         */
        default Builder deliveryDestinationConfiguration(
                Consumer<DeliveryDestinationConfiguration.Builder> deliveryDestinationConfiguration) {
            return deliveryDestinationConfiguration(DeliveryDestinationConfiguration.builder()
                    .applyMutation(deliveryDestinationConfiguration).build());
        }

        /**
         * <p>
         * The tags that have been assigned to this delivery destination.
         * </p>
         * 
         * @param tags
         *        The tags that have been assigned to this delivery destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String deliveryDestinationType;

        private String outputFormat;

        private DeliveryDestinationConfiguration deliveryDestinationConfiguration;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DeliveryDestination model) {
            name(model.name);
            arn(model.arn);
            deliveryDestinationType(model.deliveryDestinationType);
            outputFormat(model.outputFormat);
            deliveryDestinationConfiguration(model.deliveryDestinationConfiguration);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDeliveryDestinationType() {
            return deliveryDestinationType;
        }

        public final void setDeliveryDestinationType(String deliveryDestinationType) {
            this.deliveryDestinationType = deliveryDestinationType;
        }

        @Override
        public final Builder deliveryDestinationType(String deliveryDestinationType) {
            this.deliveryDestinationType = deliveryDestinationType;
            return this;
        }

        @Override
        public final Builder deliveryDestinationType(DeliveryDestinationType deliveryDestinationType) {
            this.deliveryDestinationType(deliveryDestinationType == null ? null : deliveryDestinationType.toString());
            return this;
        }

        public final String getOutputFormat() {
            return outputFormat;
        }

        public final void setOutputFormat(String outputFormat) {
            this.outputFormat = outputFormat;
        }

        @Override
        public final Builder outputFormat(String outputFormat) {
            this.outputFormat = outputFormat;
            return this;
        }

        @Override
        public final Builder outputFormat(OutputFormat outputFormat) {
            this.outputFormat(outputFormat == null ? null : outputFormat.toString());
            return this;
        }

        public final DeliveryDestinationConfiguration.Builder getDeliveryDestinationConfiguration() {
            return deliveryDestinationConfiguration != null ? deliveryDestinationConfiguration.toBuilder() : null;
        }

        public final void setDeliveryDestinationConfiguration(
                DeliveryDestinationConfiguration.BuilderImpl deliveryDestinationConfiguration) {
            this.deliveryDestinationConfiguration = deliveryDestinationConfiguration != null ? deliveryDestinationConfiguration
                    .build() : null;
        }

        @Override
        public final Builder deliveryDestinationConfiguration(DeliveryDestinationConfiguration deliveryDestinationConfiguration) {
            this.deliveryDestinationConfiguration = deliveryDestinationConfiguration;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        public DeliveryDestination build() {
            return new DeliveryDestination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
