/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure represents one anomaly that has been found by a logs anomaly detector.
 * </p>
 * <p>
 * For more information about patterns and anomalies, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogAnomalyDetector.html"
 * >CreateLogAnomalyDetector</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Anomaly implements SdkPojo, Serializable, ToCopyableBuilder<Anomaly.Builder, Anomaly> {
    private static final SdkField<String> ANOMALY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("anomalyId").getter(getter(Anomaly::anomalyId)).setter(setter(Builder::anomalyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("anomalyId").build()).build();

    private static final SdkField<String> PATTERN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("patternId").getter(getter(Anomaly::patternId)).setter(setter(Builder::patternId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("patternId").build()).build();

    private static final SdkField<String> ANOMALY_DETECTOR_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("anomalyDetectorArn").getter(getter(Anomaly::anomalyDetectorArn))
            .setter(setter(Builder::anomalyDetectorArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("anomalyDetectorArn").build())
            .build();

    private static final SdkField<String> PATTERN_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("patternString").getter(getter(Anomaly::patternString)).setter(setter(Builder::patternString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("patternString").build()).build();

    private static final SdkField<String> PATTERN_REGEX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("patternRegex").getter(getter(Anomaly::patternRegex)).setter(setter(Builder::patternRegex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("patternRegex").build()).build();

    private static final SdkField<String> PRIORITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("priority").getter(getter(Anomaly::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("priority").build()).build();

    private static final SdkField<Long> FIRST_SEEN_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("firstSeen")
            .getter(getter(Anomaly::firstSeen)).setter(setter(Builder::firstSeen))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("firstSeen").build()).build();

    private static final SdkField<Long> LAST_SEEN_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("lastSeen")
            .getter(getter(Anomaly::lastSeen)).setter(setter(Builder::lastSeen))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastSeen").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Anomaly::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Boolean> ACTIVE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("active").getter(getter(Anomaly::active)).setter(setter(Builder::active))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("active").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Anomaly::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<Map<String, Long>> HISTOGRAM_FIELD = SdkField
            .<Map<String, Long>> builder(MarshallingType.MAP)
            .memberName("histogram")
            .getter(getter(Anomaly::histogram))
            .setter(setter(Builder::histogram))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("histogram").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Long> builder(MarshallingType.LONG)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<LogEvent>> LOG_SAMPLES_FIELD = SdkField
            .<List<LogEvent>> builder(MarshallingType.LIST)
            .memberName("logSamples")
            .getter(getter(Anomaly::logSamples))
            .setter(setter(Builder::logSamples))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logSamples").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LogEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(LogEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PatternToken>> PATTERN_TOKENS_FIELD = SdkField
            .<List<PatternToken>> builder(MarshallingType.LIST)
            .memberName("patternTokens")
            .getter(getter(Anomaly::patternTokens))
            .setter(setter(Builder::patternTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("patternTokens").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PatternToken> builder(MarshallingType.SDK_POJO)
                                            .constructor(PatternToken::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> LOG_GROUP_ARN_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("logGroupArnList")
            .getter(getter(Anomaly::logGroupArnList))
            .setter(setter(Builder::logGroupArnList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logGroupArnList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SUPPRESSED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("suppressed").getter(getter(Anomaly::suppressed)).setter(setter(Builder::suppressed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("suppressed").build()).build();

    private static final SdkField<Long> SUPPRESSED_DATE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("suppressedDate").getter(getter(Anomaly::suppressedDate)).setter(setter(Builder::suppressedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("suppressedDate").build()).build();

    private static final SdkField<Long> SUPPRESSED_UNTIL_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("suppressedUntil").getter(getter(Anomaly::suppressedUntil)).setter(setter(Builder::suppressedUntil))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("suppressedUntil").build()).build();

    private static final SdkField<Boolean> IS_PATTERN_LEVEL_SUPPRESSION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("isPatternLevelSuppression")
            .getter(getter(Anomaly::isPatternLevelSuppression)).setter(setter(Builder::isPatternLevelSuppression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isPatternLevelSuppression").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ANOMALY_ID_FIELD,
            PATTERN_ID_FIELD, ANOMALY_DETECTOR_ARN_FIELD, PATTERN_STRING_FIELD, PATTERN_REGEX_FIELD, PRIORITY_FIELD,
            FIRST_SEEN_FIELD, LAST_SEEN_FIELD, DESCRIPTION_FIELD, ACTIVE_FIELD, STATE_FIELD, HISTOGRAM_FIELD, LOG_SAMPLES_FIELD,
            PATTERN_TOKENS_FIELD, LOG_GROUP_ARN_LIST_FIELD, SUPPRESSED_FIELD, SUPPRESSED_DATE_FIELD, SUPPRESSED_UNTIL_FIELD,
            IS_PATTERN_LEVEL_SUPPRESSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String anomalyId;

    private final String patternId;

    private final String anomalyDetectorArn;

    private final String patternString;

    private final String patternRegex;

    private final String priority;

    private final Long firstSeen;

    private final Long lastSeen;

    private final String description;

    private final Boolean active;

    private final String state;

    private final Map<String, Long> histogram;

    private final List<LogEvent> logSamples;

    private final List<PatternToken> patternTokens;

    private final List<String> logGroupArnList;

    private final Boolean suppressed;

    private final Long suppressedDate;

    private final Long suppressedUntil;

    private final Boolean isPatternLevelSuppression;

    private Anomaly(BuilderImpl builder) {
        this.anomalyId = builder.anomalyId;
        this.patternId = builder.patternId;
        this.anomalyDetectorArn = builder.anomalyDetectorArn;
        this.patternString = builder.patternString;
        this.patternRegex = builder.patternRegex;
        this.priority = builder.priority;
        this.firstSeen = builder.firstSeen;
        this.lastSeen = builder.lastSeen;
        this.description = builder.description;
        this.active = builder.active;
        this.state = builder.state;
        this.histogram = builder.histogram;
        this.logSamples = builder.logSamples;
        this.patternTokens = builder.patternTokens;
        this.logGroupArnList = builder.logGroupArnList;
        this.suppressed = builder.suppressed;
        this.suppressedDate = builder.suppressedDate;
        this.suppressedUntil = builder.suppressedUntil;
        this.isPatternLevelSuppression = builder.isPatternLevelSuppression;
    }

    /**
     * <p>
     * The unique ID that CloudWatch Logs assigned to this anomaly.
     * </p>
     * 
     * @return The unique ID that CloudWatch Logs assigned to this anomaly.
     */
    public final String anomalyId() {
        return anomalyId;
    }

    /**
     * <p>
     * The ID of the pattern used to help identify this anomaly.
     * </p>
     * 
     * @return The ID of the pattern used to help identify this anomaly.
     */
    public final String patternId() {
        return patternId;
    }

    /**
     * <p>
     * The ARN of the anomaly detector that identified this anomaly.
     * </p>
     * 
     * @return The ARN of the anomaly detector that identified this anomaly.
     */
    public final String anomalyDetectorArn() {
        return anomalyDetectorArn;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in string format.
     * </p>
     * 
     * @return The pattern used to help identify this anomaly, in string format.
     */
    public final String patternString() {
        return patternString;
    }

    /**
     * <p>
     * The pattern used to help identify this anomaly, in regular expression format.
     * </p>
     * 
     * @return The pattern used to help identify this anomaly, in regular expression format.
     */
    public final String patternRegex() {
        return patternRegex;
    }

    /**
     * <p>
     * The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log severity
     * labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the baseline. Possible
     * values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     * </p>
     * 
     * @return The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log
     *         severity labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the
     *         baseline. Possible values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
     */
    public final String priority() {
        return priority;
    }

    /**
     * <p>
     * The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which is the
     * number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @return The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which
     *         is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */
    public final Long firstSeen() {
        return firstSeen;
    }

    /**
     * <p>
     * The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time, which
     * is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * 
     * @return The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch
     *         time, which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */
    public final Long lastSeen() {
        return lastSeen;
    }

    /**
     * <p>
     * A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     * </p>
     * 
     * @return A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Specifies whether this anomaly is still ongoing.
     * </p>
     * 
     * @return Specifies whether this anomaly is still ongoing.
     */
    public final Boolean active() {
        return active;
    }

    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     *         <code>Active</code>. If you have suppressed this anomaly by using the <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *         >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to
     *         be normal, the value is <code>Baseline</code>.
     * @see State
     */
    public final State state() {
        return State.fromValue(state);
    }

    /**
     * <p>
     * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     * <code>Active</code>. If you have suppressed this anomaly by using the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
     * normal, the value is <code>Baseline</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link State#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
     *         <code>Active</code>. If you have suppressed this anomaly by using the <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *         >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to
     *         be normal, the value is <code>Baseline</code>.
     * @see State
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Histogram property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasHistogram() {
        return histogram != null && !(histogram instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
     * detected at each of those runs. The times are specified in epoch time, which is the number of seconds since
     * <code>January 1, 1970, 00:00:00 UTC</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHistogram} method.
     * </p>
     * 
     * @return A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that
     *         were detected at each of those runs. The times are specified in epoch time, which is the number of
     *         seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
     */
    public final Map<String, Long> histogram() {
        return histogram;
    }

    /**
     * For responses, this returns true if the service returned a value for the LogSamples property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLogSamples() {
        return logSamples != null && !(logSamples instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of sample log event messages that are considered to be part of this anomaly.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogSamples} method.
     * </p>
     * 
     * @return An array of sample log event messages that are considered to be part of this anomaly.
     */
    public final List<LogEvent> logSamples() {
        return logSamples;
    }

    /**
     * For responses, this returns true if the service returned a value for the PatternTokens property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPatternTokens() {
        return patternTokens != null && !(patternTokens instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of structures where each structure contains information about one token that makes up the pattern.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPatternTokens} method.
     * </p>
     * 
     * @return An array of structures where each structure contains information about one token that makes up the
     *         pattern.
     */
    public final List<PatternToken> patternTokens() {
        return patternTokens;
    }

    /**
     * For responses, this returns true if the service returned a value for the LogGroupArnList property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLogGroupArnList() {
        return logGroupArnList != null && !(logGroupArnList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogGroupArnList} method.
     * </p>
     * 
     * @return An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
     */
    public final List<String> logGroupArnList() {
        return logGroupArnList;
    }

    /**
     * <p>
     * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     * >UpdateAnomaly</a>.
     * </p>
     * 
     * @return Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
     *         >UpdateAnomaly</a>.
     */
    public final Boolean suppressed() {
        return suppressed;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when it was suppressed.
     * </p>
     * 
     * @return If the anomaly is suppressed, this indicates when it was suppressed.
     */
    public final Long suppressedDate() {
        return suppressedDate;
    }

    /**
     * <p>
     * If the anomaly is suppressed, this indicates when the suppression will end. If this value is <code>0</code>, the
     * anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     * </p>
     * 
     * @return If the anomaly is suppressed, this indicates when the suppression will end. If this value is
     *         <code>0</code>, the anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
     */
    public final Long suppressedUntil() {
        return suppressedUntil;
    }

    /**
     * <p>
     * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
     * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     * </p>
     * 
     * @return If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern
     *         is suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
     */
    public final Boolean isPatternLevelSuppression() {
        return isPatternLevelSuppression;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(anomalyId());
        hashCode = 31 * hashCode + Objects.hashCode(patternId());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyDetectorArn());
        hashCode = 31 * hashCode + Objects.hashCode(patternString());
        hashCode = 31 * hashCode + Objects.hashCode(patternRegex());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(firstSeen());
        hashCode = 31 * hashCode + Objects.hashCode(lastSeen());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(active());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasHistogram() ? histogram() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLogSamples() ? logSamples() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPatternTokens() ? patternTokens() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasLogGroupArnList() ? logGroupArnList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(suppressed());
        hashCode = 31 * hashCode + Objects.hashCode(suppressedDate());
        hashCode = 31 * hashCode + Objects.hashCode(suppressedUntil());
        hashCode = 31 * hashCode + Objects.hashCode(isPatternLevelSuppression());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Anomaly)) {
            return false;
        }
        Anomaly other = (Anomaly) obj;
        return Objects.equals(anomalyId(), other.anomalyId()) && Objects.equals(patternId(), other.patternId())
                && Objects.equals(anomalyDetectorArn(), other.anomalyDetectorArn())
                && Objects.equals(patternString(), other.patternString()) && Objects.equals(patternRegex(), other.patternRegex())
                && Objects.equals(priority(), other.priority()) && Objects.equals(firstSeen(), other.firstSeen())
                && Objects.equals(lastSeen(), other.lastSeen()) && Objects.equals(description(), other.description())
                && Objects.equals(active(), other.active()) && Objects.equals(stateAsString(), other.stateAsString())
                && hasHistogram() == other.hasHistogram() && Objects.equals(histogram(), other.histogram())
                && hasLogSamples() == other.hasLogSamples() && Objects.equals(logSamples(), other.logSamples())
                && hasPatternTokens() == other.hasPatternTokens() && Objects.equals(patternTokens(), other.patternTokens())
                && hasLogGroupArnList() == other.hasLogGroupArnList()
                && Objects.equals(logGroupArnList(), other.logGroupArnList()) && Objects.equals(suppressed(), other.suppressed())
                && Objects.equals(suppressedDate(), other.suppressedDate())
                && Objects.equals(suppressedUntil(), other.suppressedUntil())
                && Objects.equals(isPatternLevelSuppression(), other.isPatternLevelSuppression());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Anomaly").add("AnomalyId", anomalyId()).add("PatternId", patternId())
                .add("AnomalyDetectorArn", anomalyDetectorArn()).add("PatternString", patternString())
                .add("PatternRegex", patternRegex()).add("Priority", priority()).add("FirstSeen", firstSeen())
                .add("LastSeen", lastSeen()).add("Description", description()).add("Active", active())
                .add("State", stateAsString()).add("Histogram", hasHistogram() ? histogram() : null)
                .add("LogSamples", hasLogSamples() ? logSamples() : null)
                .add("PatternTokens", hasPatternTokens() ? patternTokens() : null)
                .add("LogGroupArnList", hasLogGroupArnList() ? logGroupArnList() : null).add("Suppressed", suppressed())
                .add("SuppressedDate", suppressedDate()).add("SuppressedUntil", suppressedUntil())
                .add("IsPatternLevelSuppression", isPatternLevelSuppression()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "anomalyId":
            return Optional.ofNullable(clazz.cast(anomalyId()));
        case "patternId":
            return Optional.ofNullable(clazz.cast(patternId()));
        case "anomalyDetectorArn":
            return Optional.ofNullable(clazz.cast(anomalyDetectorArn()));
        case "patternString":
            return Optional.ofNullable(clazz.cast(patternString()));
        case "patternRegex":
            return Optional.ofNullable(clazz.cast(patternRegex()));
        case "priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "firstSeen":
            return Optional.ofNullable(clazz.cast(firstSeen()));
        case "lastSeen":
            return Optional.ofNullable(clazz.cast(lastSeen()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "active":
            return Optional.ofNullable(clazz.cast(active()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "histogram":
            return Optional.ofNullable(clazz.cast(histogram()));
        case "logSamples":
            return Optional.ofNullable(clazz.cast(logSamples()));
        case "patternTokens":
            return Optional.ofNullable(clazz.cast(patternTokens()));
        case "logGroupArnList":
            return Optional.ofNullable(clazz.cast(logGroupArnList()));
        case "suppressed":
            return Optional.ofNullable(clazz.cast(suppressed()));
        case "suppressedDate":
            return Optional.ofNullable(clazz.cast(suppressedDate()));
        case "suppressedUntil":
            return Optional.ofNullable(clazz.cast(suppressedUntil()));
        case "isPatternLevelSuppression":
            return Optional.ofNullable(clazz.cast(isPatternLevelSuppression()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("anomalyId", ANOMALY_ID_FIELD);
        map.put("patternId", PATTERN_ID_FIELD);
        map.put("anomalyDetectorArn", ANOMALY_DETECTOR_ARN_FIELD);
        map.put("patternString", PATTERN_STRING_FIELD);
        map.put("patternRegex", PATTERN_REGEX_FIELD);
        map.put("priority", PRIORITY_FIELD);
        map.put("firstSeen", FIRST_SEEN_FIELD);
        map.put("lastSeen", LAST_SEEN_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("active", ACTIVE_FIELD);
        map.put("state", STATE_FIELD);
        map.put("histogram", HISTOGRAM_FIELD);
        map.put("logSamples", LOG_SAMPLES_FIELD);
        map.put("patternTokens", PATTERN_TOKENS_FIELD);
        map.put("logGroupArnList", LOG_GROUP_ARN_LIST_FIELD);
        map.put("suppressed", SUPPRESSED_FIELD);
        map.put("suppressedDate", SUPPRESSED_DATE_FIELD);
        map.put("suppressedUntil", SUPPRESSED_UNTIL_FIELD);
        map.put("isPatternLevelSuppression", IS_PATTERN_LEVEL_SUPPRESSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Anomaly, T> g) {
        return obj -> g.apply((Anomaly) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Anomaly> {
        /**
         * <p>
         * The unique ID that CloudWatch Logs assigned to this anomaly.
         * </p>
         * 
         * @param anomalyId
         *        The unique ID that CloudWatch Logs assigned to this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyId(String anomalyId);

        /**
         * <p>
         * The ID of the pattern used to help identify this anomaly.
         * </p>
         * 
         * @param patternId
         *        The ID of the pattern used to help identify this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patternId(String patternId);

        /**
         * <p>
         * The ARN of the anomaly detector that identified this anomaly.
         * </p>
         * 
         * @param anomalyDetectorArn
         *        The ARN of the anomaly detector that identified this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyDetectorArn(String anomalyDetectorArn);

        /**
         * <p>
         * The pattern used to help identify this anomaly, in string format.
         * </p>
         * 
         * @param patternString
         *        The pattern used to help identify this anomaly, in string format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patternString(String patternString);

        /**
         * <p>
         * The pattern used to help identify this anomaly, in regular expression format.
         * </p>
         * 
         * @param patternRegex
         *        The pattern used to help identify this anomaly, in regular expression format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patternRegex(String patternRegex);

        /**
         * <p>
         * The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on log
         * severity labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from the
         * baseline. Possible values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
         * </p>
         * 
         * @param priority
         *        The priority level of this anomaly, as determined by CloudWatch Logs. Priority is computed based on
         *        log severity labels such as <code>FATAL</code> and <code>ERROR</code> and the amount of deviation from
         *        the baseline. Possible values are <code>HIGH</code>, <code>MEDIUM</code>, and <code>LOW</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(String priority);

        /**
         * <p>
         * The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time, which is
         * the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
         * </p>
         * 
         * @param firstSeen
         *        The date and time when the anomaly detector first saw this anomaly. It is specified as epoch time,
         *        which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firstSeen(Long firstSeen);

        /**
         * <p>
         * The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch time,
         * which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
         * </p>
         * 
         * @param lastSeen
         *        The date and time when the anomaly detector most recently saw this anomaly. It is specified as epoch
         *        time, which is the number of seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSeen(Long lastSeen);

        /**
         * <p>
         * A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
         * </p>
         * 
         * @param description
         *        A human-readable description of the anomaly. This description is generated by CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Specifies whether this anomaly is still ongoing.
         * </p>
         * 
         * @param active
         *        Specifies whether this anomaly is still ongoing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder active(Boolean active);

        /**
         * <p>
         * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
         * <code>Active</code>. If you have suppressed this anomaly by using the <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
         * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
         * normal, the value is <code>Baseline</code>.
         * </p>
         * 
         * @param state
         *        Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
         *        <code>Active</code>. If you have suppressed this anomaly by using the <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
         *        >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered
         *        to be normal, the value is <code>Baseline</code>.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder state(String state);

        /**
         * <p>
         * Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
         * <code>Active</code>. If you have suppressed this anomaly by using the <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
         * >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered to be
         * normal, the value is <code>Baseline</code>.
         * </p>
         * 
         * @param state
         *        Indicates the current state of this anomaly. If it is still being treated as an anomaly, the value is
         *        <code>Active</code>. If you have suppressed this anomaly by using the <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
         *        >UpdateAnomaly</a> operation, the value is <code>Suppressed</code>. If this behavior is now considered
         *        to be normal, the value is <code>Baseline</code>.
         * @see State
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see State
         */
        Builder state(State state);

        /**
         * <p>
         * A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that were
         * detected at each of those runs. The times are specified in epoch time, which is the number of seconds since
         * <code>January 1, 1970, 00:00:00 UTC</code>.
         * </p>
         * 
         * @param histogram
         *        A map showing times when the anomaly detector ran, and the number of occurrences of this anomaly that
         *        were detected at each of those runs. The times are specified in epoch time, which is the number of
         *        seconds since <code>January 1, 1970, 00:00:00 UTC</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder histogram(Map<String, Long> histogram);

        /**
         * <p>
         * An array of sample log event messages that are considered to be part of this anomaly.
         * </p>
         * 
         * @param logSamples
         *        An array of sample log event messages that are considered to be part of this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logSamples(Collection<LogEvent> logSamples);

        /**
         * <p>
         * An array of sample log event messages that are considered to be part of this anomaly.
         * </p>
         * 
         * @param logSamples
         *        An array of sample log event messages that are considered to be part of this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logSamples(LogEvent... logSamples);

        /**
         * <p>
         * An array of sample log event messages that are considered to be part of this anomaly.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudwatchlogs.model.LogEvent.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.cloudwatchlogs.model.LogEvent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudwatchlogs.model.LogEvent.Builder#build()} is called immediately
         * and its result is passed to {@link #logSamples(List<LogEvent>)}.
         * 
         * @param logSamples
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudwatchlogs.model.LogEvent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logSamples(java.util.Collection<LogEvent>)
         */
        Builder logSamples(Consumer<LogEvent.Builder>... logSamples);

        /**
         * <p>
         * An array of structures where each structure contains information about one token that makes up the pattern.
         * </p>
         * 
         * @param patternTokens
         *        An array of structures where each structure contains information about one token that makes up the
         *        pattern.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patternTokens(Collection<PatternToken> patternTokens);

        /**
         * <p>
         * An array of structures where each structure contains information about one token that makes up the pattern.
         * </p>
         * 
         * @param patternTokens
         *        An array of structures where each structure contains information about one token that makes up the
         *        pattern.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder patternTokens(PatternToken... patternTokens);

        /**
         * <p>
         * An array of structures where each structure contains information about one token that makes up the pattern.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudwatchlogs.model.PatternToken.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.cloudwatchlogs.model.PatternToken#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudwatchlogs.model.PatternToken.Builder#build()} is called
         * immediately and its result is passed to {@link #patternTokens(List<PatternToken>)}.
         * 
         * @param patternTokens
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudwatchlogs.model.PatternToken.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #patternTokens(java.util.Collection<PatternToken>)
         */
        Builder patternTokens(Consumer<PatternToken.Builder>... patternTokens);

        /**
         * <p>
         * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
         * </p>
         * 
         * @param logGroupArnList
         *        An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupArnList(Collection<String> logGroupArnList);

        /**
         * <p>
         * An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
         * </p>
         * 
         * @param logGroupArnList
         *        An array of ARNS of the log groups that contained log events considered to be part of this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupArnList(String... logGroupArnList);

        /**
         * <p>
         * Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
         * >UpdateAnomaly</a>.
         * </p>
         * 
         * @param suppressed
         *        Indicates whether this anomaly is currently suppressed. To suppress an anomaly, use <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html"
         *        >UpdateAnomaly</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suppressed(Boolean suppressed);

        /**
         * <p>
         * If the anomaly is suppressed, this indicates when it was suppressed.
         * </p>
         * 
         * @param suppressedDate
         *        If the anomaly is suppressed, this indicates when it was suppressed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suppressedDate(Long suppressedDate);

        /**
         * <p>
         * If the anomaly is suppressed, this indicates when the suppression will end. If this value is <code>0</code>,
         * the anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
         * </p>
         * 
         * @param suppressedUntil
         *        If the anomaly is suppressed, this indicates when the suppression will end. If this value is
         *        <code>0</code>, the anomaly was suppressed with no expiration, with the <code>INFINITE</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suppressedUntil(Long suppressedUntil);

        /**
         * <p>
         * If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the pattern is
         * suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
         * </p>
         * 
         * @param isPatternLevelSuppression
         *        If this anomaly is suppressed, this field is <code>true</code> if the suppression is because the
         *        pattern is suppressed. If <code>false</code>, then only this particular anomaly is suppressed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isPatternLevelSuppression(Boolean isPatternLevelSuppression);
    }

    static final class BuilderImpl implements Builder {
        private String anomalyId;

        private String patternId;

        private String anomalyDetectorArn;

        private String patternString;

        private String patternRegex;

        private String priority;

        private Long firstSeen;

        private Long lastSeen;

        private String description;

        private Boolean active;

        private String state;

        private Map<String, Long> histogram = DefaultSdkAutoConstructMap.getInstance();

        private List<LogEvent> logSamples = DefaultSdkAutoConstructList.getInstance();

        private List<PatternToken> patternTokens = DefaultSdkAutoConstructList.getInstance();

        private List<String> logGroupArnList = DefaultSdkAutoConstructList.getInstance();

        private Boolean suppressed;

        private Long suppressedDate;

        private Long suppressedUntil;

        private Boolean isPatternLevelSuppression;

        private BuilderImpl() {
        }

        private BuilderImpl(Anomaly model) {
            anomalyId(model.anomalyId);
            patternId(model.patternId);
            anomalyDetectorArn(model.anomalyDetectorArn);
            patternString(model.patternString);
            patternRegex(model.patternRegex);
            priority(model.priority);
            firstSeen(model.firstSeen);
            lastSeen(model.lastSeen);
            description(model.description);
            active(model.active);
            state(model.state);
            histogram(model.histogram);
            logSamples(model.logSamples);
            patternTokens(model.patternTokens);
            logGroupArnList(model.logGroupArnList);
            suppressed(model.suppressed);
            suppressedDate(model.suppressedDate);
            suppressedUntil(model.suppressedUntil);
            isPatternLevelSuppression(model.isPatternLevelSuppression);
        }

        public final String getAnomalyId() {
            return anomalyId;
        }

        public final void setAnomalyId(String anomalyId) {
            this.anomalyId = anomalyId;
        }

        @Override
        public final Builder anomalyId(String anomalyId) {
            this.anomalyId = anomalyId;
            return this;
        }

        public final String getPatternId() {
            return patternId;
        }

        public final void setPatternId(String patternId) {
            this.patternId = patternId;
        }

        @Override
        public final Builder patternId(String patternId) {
            this.patternId = patternId;
            return this;
        }

        public final String getAnomalyDetectorArn() {
            return anomalyDetectorArn;
        }

        public final void setAnomalyDetectorArn(String anomalyDetectorArn) {
            this.anomalyDetectorArn = anomalyDetectorArn;
        }

        @Override
        public final Builder anomalyDetectorArn(String anomalyDetectorArn) {
            this.anomalyDetectorArn = anomalyDetectorArn;
            return this;
        }

        public final String getPatternString() {
            return patternString;
        }

        public final void setPatternString(String patternString) {
            this.patternString = patternString;
        }

        @Override
        public final Builder patternString(String patternString) {
            this.patternString = patternString;
            return this;
        }

        public final String getPatternRegex() {
            return patternRegex;
        }

        public final void setPatternRegex(String patternRegex) {
            this.patternRegex = patternRegex;
        }

        @Override
        public final Builder patternRegex(String patternRegex) {
            this.patternRegex = patternRegex;
            return this;
        }

        public final String getPriority() {
            return priority;
        }

        public final void setPriority(String priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(String priority) {
            this.priority = priority;
            return this;
        }

        public final Long getFirstSeen() {
            return firstSeen;
        }

        public final void setFirstSeen(Long firstSeen) {
            this.firstSeen = firstSeen;
        }

        @Override
        public final Builder firstSeen(Long firstSeen) {
            this.firstSeen = firstSeen;
            return this;
        }

        public final Long getLastSeen() {
            return lastSeen;
        }

        public final void setLastSeen(Long lastSeen) {
            this.lastSeen = lastSeen;
        }

        @Override
        public final Builder lastSeen(Long lastSeen) {
            this.lastSeen = lastSeen;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Boolean getActive() {
            return active;
        }

        public final void setActive(Boolean active) {
            this.active = active;
        }

        @Override
        public final Builder active(Boolean active) {
            this.active = active;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(State state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Map<String, Long> getHistogram() {
            if (histogram instanceof SdkAutoConstructMap) {
                return null;
            }
            return histogram;
        }

        public final void setHistogram(Map<String, Long> histogram) {
            this.histogram = HistogramCopier.copy(histogram);
        }

        @Override
        public final Builder histogram(Map<String, Long> histogram) {
            this.histogram = HistogramCopier.copy(histogram);
            return this;
        }

        public final List<LogEvent.Builder> getLogSamples() {
            List<LogEvent.Builder> result = LogSamplesCopier.copyToBuilder(this.logSamples);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLogSamples(Collection<LogEvent.BuilderImpl> logSamples) {
            this.logSamples = LogSamplesCopier.copyFromBuilder(logSamples);
        }

        @Override
        public final Builder logSamples(Collection<LogEvent> logSamples) {
            this.logSamples = LogSamplesCopier.copy(logSamples);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logSamples(LogEvent... logSamples) {
            logSamples(Arrays.asList(logSamples));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logSamples(Consumer<LogEvent.Builder>... logSamples) {
            logSamples(Stream.of(logSamples).map(c -> LogEvent.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<PatternToken.Builder> getPatternTokens() {
            List<PatternToken.Builder> result = PatternTokensCopier.copyToBuilder(this.patternTokens);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPatternTokens(Collection<PatternToken.BuilderImpl> patternTokens) {
            this.patternTokens = PatternTokensCopier.copyFromBuilder(patternTokens);
        }

        @Override
        public final Builder patternTokens(Collection<PatternToken> patternTokens) {
            this.patternTokens = PatternTokensCopier.copy(patternTokens);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder patternTokens(PatternToken... patternTokens) {
            patternTokens(Arrays.asList(patternTokens));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder patternTokens(Consumer<PatternToken.Builder>... patternTokens) {
            patternTokens(Stream.of(patternTokens).map(c -> PatternToken.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getLogGroupArnList() {
            if (logGroupArnList instanceof SdkAutoConstructList) {
                return null;
            }
            return logGroupArnList;
        }

        public final void setLogGroupArnList(Collection<String> logGroupArnList) {
            this.logGroupArnList = LogGroupArnListCopier.copy(logGroupArnList);
        }

        @Override
        public final Builder logGroupArnList(Collection<String> logGroupArnList) {
            this.logGroupArnList = LogGroupArnListCopier.copy(logGroupArnList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logGroupArnList(String... logGroupArnList) {
            logGroupArnList(Arrays.asList(logGroupArnList));
            return this;
        }

        public final Boolean getSuppressed() {
            return suppressed;
        }

        public final void setSuppressed(Boolean suppressed) {
            this.suppressed = suppressed;
        }

        @Override
        public final Builder suppressed(Boolean suppressed) {
            this.suppressed = suppressed;
            return this;
        }

        public final Long getSuppressedDate() {
            return suppressedDate;
        }

        public final void setSuppressedDate(Long suppressedDate) {
            this.suppressedDate = suppressedDate;
        }

        @Override
        public final Builder suppressedDate(Long suppressedDate) {
            this.suppressedDate = suppressedDate;
            return this;
        }

        public final Long getSuppressedUntil() {
            return suppressedUntil;
        }

        public final void setSuppressedUntil(Long suppressedUntil) {
            this.suppressedUntil = suppressedUntil;
        }

        @Override
        public final Builder suppressedUntil(Long suppressedUntil) {
            this.suppressedUntil = suppressedUntil;
            return this;
        }

        public final Boolean getIsPatternLevelSuppression() {
            return isPatternLevelSuppression;
        }

        public final void setIsPatternLevelSuppression(Boolean isPatternLevelSuppression) {
            this.isPatternLevelSuppression = isPatternLevelSuppression;
        }

        @Override
        public final Builder isPatternLevelSuppression(Boolean isPatternLevelSuppression) {
            this.isPatternLevelSuppression = isPatternLevelSuppression;
            return this;
        }

        @Override
        public Anomaly build() {
            return new Anomaly(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
