/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatchlogs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This processor converts logs into <a href="https://ocsf.io">Open Cybersecurity Schema Framework (OCSF)</a> events.
 * </p>
 * <p>
 * For more information about this processor including examples, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatch-Logs-Transformation.html#CloudWatch-Logs-Transformation-parseToOCSF"
 * >parseToOCSF</a> in the <i>CloudWatch Logs User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParseToOCSF implements SdkPojo, Serializable, ToCopyableBuilder<ParseToOCSF.Builder, ParseToOCSF> {
    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("source")
            .getter(getter(ParseToOCSF::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<String> EVENT_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("eventSource").getter(getter(ParseToOCSF::eventSourceAsString)).setter(setter(Builder::eventSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventSource").build()).build();

    private static final SdkField<String> OCSF_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ocsfVersion").getter(getter(ParseToOCSF::ocsfVersionAsString)).setter(setter(Builder::ocsfVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ocsfVersion").build()).build();

    private static final SdkField<String> MAPPING_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("mappingVersion").getter(getter(ParseToOCSF::mappingVersion)).setter(setter(Builder::mappingVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mappingVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD,
            EVENT_SOURCE_FIELD, OCSF_VERSION_FIELD, MAPPING_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String source;

    private final String eventSource;

    private final String ocsfVersion;

    private final String mappingVersion;

    private ParseToOCSF(BuilderImpl builder) {
        this.source = builder.source;
        this.eventSource = builder.eventSource;
        this.ocsfVersion = builder.ocsfVersion;
        this.mappingVersion = builder.mappingVersion;
    }

    /**
     * <p>
     * The path to the field in the log event that you want to parse. If you omit this value, the whole log message is
     * parsed.
     * </p>
     * 
     * @return The path to the field in the log event that you want to parse. If you omit this value, the whole log
     *         message is parsed.
     */
    public final String source() {
        return source;
    }

    /**
     * <p>
     * Specify the service or process that produces the log events that will be converted with this processor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventSource} will
     * return {@link EventSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventSourceAsString}.
     * </p>
     * 
     * @return Specify the service or process that produces the log events that will be converted with this processor.
     * @see EventSource
     */
    public final EventSource eventSource() {
        return EventSource.fromValue(eventSource);
    }

    /**
     * <p>
     * Specify the service or process that produces the log events that will be converted with this processor.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventSource} will
     * return {@link EventSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventSourceAsString}.
     * </p>
     * 
     * @return Specify the service or process that produces the log events that will be converted with this processor.
     * @see EventSource
     */
    public final String eventSourceAsString() {
        return eventSource;
    }

    /**
     * <p>
     * Specify which version of the OCSF schema to use for the transformed log events.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ocsfVersion} will
     * return {@link OCSFVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ocsfVersionAsString}.
     * </p>
     * 
     * @return Specify which version of the OCSF schema to use for the transformed log events.
     * @see OCSFVersion
     */
    public final OCSFVersion ocsfVersion() {
        return OCSFVersion.fromValue(ocsfVersion);
    }

    /**
     * <p>
     * Specify which version of the OCSF schema to use for the transformed log events.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ocsfVersion} will
     * return {@link OCSFVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ocsfVersionAsString}.
     * </p>
     * 
     * @return Specify which version of the OCSF schema to use for the transformed log events.
     * @see OCSFVersion
     */
    public final String ocsfVersionAsString() {
        return ocsfVersion;
    }

    /**
     * <p>
     * Identifies the specific release of the Open Cybersecurity Schema Framework (OCSF) transformer being used to parse
     * OCSF data. Defaults to the latest version if not specified. Does not automatically update.
     * </p>
     * 
     * @return Identifies the specific release of the Open Cybersecurity Schema Framework (OCSF) transformer being used
     *         to parse OCSF data. Defaults to the latest version if not specified. Does not automatically update.
     */
    public final String mappingVersion() {
        return mappingVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(eventSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ocsfVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mappingVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParseToOCSF)) {
            return false;
        }
        ParseToOCSF other = (ParseToOCSF) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(eventSourceAsString(), other.eventSourceAsString())
                && Objects.equals(ocsfVersionAsString(), other.ocsfVersionAsString())
                && Objects.equals(mappingVersion(), other.mappingVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ParseToOCSF").add("Source", source()).add("EventSource", eventSourceAsString())
                .add("OcsfVersion", ocsfVersionAsString()).add("MappingVersion", mappingVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "eventSource":
            return Optional.ofNullable(clazz.cast(eventSourceAsString()));
        case "ocsfVersion":
            return Optional.ofNullable(clazz.cast(ocsfVersionAsString()));
        case "mappingVersion":
            return Optional.ofNullable(clazz.cast(mappingVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("source", SOURCE_FIELD);
        map.put("eventSource", EVENT_SOURCE_FIELD);
        map.put("ocsfVersion", OCSF_VERSION_FIELD);
        map.put("mappingVersion", MAPPING_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ParseToOCSF, T> g) {
        return obj -> g.apply((ParseToOCSF) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParseToOCSF> {
        /**
         * <p>
         * The path to the field in the log event that you want to parse. If you omit this value, the whole log message
         * is parsed.
         * </p>
         * 
         * @param source
         *        The path to the field in the log event that you want to parse. If you omit this value, the whole log
         *        message is parsed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * Specify the service or process that produces the log events that will be converted with this processor.
         * </p>
         * 
         * @param eventSource
         *        Specify the service or process that produces the log events that will be converted with this
         *        processor.
         * @see EventSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventSource
         */
        Builder eventSource(String eventSource);

        /**
         * <p>
         * Specify the service or process that produces the log events that will be converted with this processor.
         * </p>
         * 
         * @param eventSource
         *        Specify the service or process that produces the log events that will be converted with this
         *        processor.
         * @see EventSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventSource
         */
        Builder eventSource(EventSource eventSource);

        /**
         * <p>
         * Specify which version of the OCSF schema to use for the transformed log events.
         * </p>
         * 
         * @param ocsfVersion
         *        Specify which version of the OCSF schema to use for the transformed log events.
         * @see OCSFVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OCSFVersion
         */
        Builder ocsfVersion(String ocsfVersion);

        /**
         * <p>
         * Specify which version of the OCSF schema to use for the transformed log events.
         * </p>
         * 
         * @param ocsfVersion
         *        Specify which version of the OCSF schema to use for the transformed log events.
         * @see OCSFVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OCSFVersion
         */
        Builder ocsfVersion(OCSFVersion ocsfVersion);

        /**
         * <p>
         * Identifies the specific release of the Open Cybersecurity Schema Framework (OCSF) transformer being used to
         * parse OCSF data. Defaults to the latest version if not specified. Does not automatically update.
         * </p>
         * 
         * @param mappingVersion
         *        Identifies the specific release of the Open Cybersecurity Schema Framework (OCSF) transformer being
         *        used to parse OCSF data. Defaults to the latest version if not specified. Does not automatically
         *        update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mappingVersion(String mappingVersion);
    }

    static final class BuilderImpl implements Builder {
        private String source;

        private String eventSource;

        private String ocsfVersion;

        private String mappingVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(ParseToOCSF model) {
            source(model.source);
            eventSource(model.eventSource);
            ocsfVersion(model.ocsfVersion);
            mappingVersion(model.mappingVersion);
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final String getEventSource() {
            return eventSource;
        }

        public final void setEventSource(String eventSource) {
            this.eventSource = eventSource;
        }

        @Override
        public final Builder eventSource(String eventSource) {
            this.eventSource = eventSource;
            return this;
        }

        @Override
        public final Builder eventSource(EventSource eventSource) {
            this.eventSource(eventSource == null ? null : eventSource.toString());
            return this;
        }

        public final String getOcsfVersion() {
            return ocsfVersion;
        }

        public final void setOcsfVersion(String ocsfVersion) {
            this.ocsfVersion = ocsfVersion;
        }

        @Override
        public final Builder ocsfVersion(String ocsfVersion) {
            this.ocsfVersion = ocsfVersion;
            return this;
        }

        @Override
        public final Builder ocsfVersion(OCSFVersion ocsfVersion) {
            this.ocsfVersion(ocsfVersion == null ? null : ocsfVersion.toString());
            return this;
        }

        public final String getMappingVersion() {
            return mappingVersion;
        }

        public final void setMappingVersion(String mappingVersion) {
            this.mappingVersion = mappingVersion;
        }

        @Override
        public final Builder mappingVersion(String mappingVersion) {
            this.mappingVersion = mappingVersion;
            return this;
        }

        @Override
        public ParseToOCSF build() {
            return new ParseToOCSF(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
