/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeartifact;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codeartifact.model.AccessDeniedException;
import software.amazon.awssdk.services.codeartifact.model.AssociateExternalConnectionRequest;
import software.amazon.awssdk.services.codeartifact.model.AssociateExternalConnectionResponse;
import software.amazon.awssdk.services.codeartifact.model.CodeartifactException;
import software.amazon.awssdk.services.codeartifact.model.CodeartifactRequest;
import software.amazon.awssdk.services.codeartifact.model.ConflictException;
import software.amazon.awssdk.services.codeartifact.model.CopyPackageVersionsRequest;
import software.amazon.awssdk.services.codeartifact.model.CopyPackageVersionsResponse;
import software.amazon.awssdk.services.codeartifact.model.CreateDomainRequest;
import software.amazon.awssdk.services.codeartifact.model.CreateDomainResponse;
import software.amazon.awssdk.services.codeartifact.model.CreateRepositoryRequest;
import software.amazon.awssdk.services.codeartifact.model.CreateRepositoryResponse;
import software.amazon.awssdk.services.codeartifact.model.DeleteDomainPermissionsPolicyRequest;
import software.amazon.awssdk.services.codeartifact.model.DeleteDomainPermissionsPolicyResponse;
import software.amazon.awssdk.services.codeartifact.model.DeleteDomainRequest;
import software.amazon.awssdk.services.codeartifact.model.DeleteDomainResponse;
import software.amazon.awssdk.services.codeartifact.model.DeletePackageVersionsRequest;
import software.amazon.awssdk.services.codeartifact.model.DeletePackageVersionsResponse;
import software.amazon.awssdk.services.codeartifact.model.DeleteRepositoryPermissionsPolicyRequest;
import software.amazon.awssdk.services.codeartifact.model.DeleteRepositoryPermissionsPolicyResponse;
import software.amazon.awssdk.services.codeartifact.model.DeleteRepositoryRequest;
import software.amazon.awssdk.services.codeartifact.model.DeleteRepositoryResponse;
import software.amazon.awssdk.services.codeartifact.model.DescribeDomainRequest;
import software.amazon.awssdk.services.codeartifact.model.DescribeDomainResponse;
import software.amazon.awssdk.services.codeartifact.model.DescribePackageVersionRequest;
import software.amazon.awssdk.services.codeartifact.model.DescribePackageVersionResponse;
import software.amazon.awssdk.services.codeartifact.model.DescribeRepositoryRequest;
import software.amazon.awssdk.services.codeartifact.model.DescribeRepositoryResponse;
import software.amazon.awssdk.services.codeartifact.model.DisassociateExternalConnectionRequest;
import software.amazon.awssdk.services.codeartifact.model.DisassociateExternalConnectionResponse;
import software.amazon.awssdk.services.codeartifact.model.DisposePackageVersionsRequest;
import software.amazon.awssdk.services.codeartifact.model.DisposePackageVersionsResponse;
import software.amazon.awssdk.services.codeartifact.model.GetAuthorizationTokenRequest;
import software.amazon.awssdk.services.codeartifact.model.GetAuthorizationTokenResponse;
import software.amazon.awssdk.services.codeartifact.model.GetDomainPermissionsPolicyRequest;
import software.amazon.awssdk.services.codeartifact.model.GetDomainPermissionsPolicyResponse;
import software.amazon.awssdk.services.codeartifact.model.GetPackageVersionAssetRequest;
import software.amazon.awssdk.services.codeartifact.model.GetPackageVersionAssetResponse;
import software.amazon.awssdk.services.codeartifact.model.GetPackageVersionReadmeRequest;
import software.amazon.awssdk.services.codeartifact.model.GetPackageVersionReadmeResponse;
import software.amazon.awssdk.services.codeartifact.model.GetRepositoryEndpointRequest;
import software.amazon.awssdk.services.codeartifact.model.GetRepositoryEndpointResponse;
import software.amazon.awssdk.services.codeartifact.model.GetRepositoryPermissionsPolicyRequest;
import software.amazon.awssdk.services.codeartifact.model.GetRepositoryPermissionsPolicyResponse;
import software.amazon.awssdk.services.codeartifact.model.InternalServerException;
import software.amazon.awssdk.services.codeartifact.model.ListDomainsRequest;
import software.amazon.awssdk.services.codeartifact.model.ListDomainsResponse;
import software.amazon.awssdk.services.codeartifact.model.ListPackageVersionAssetsRequest;
import software.amazon.awssdk.services.codeartifact.model.ListPackageVersionAssetsResponse;
import software.amazon.awssdk.services.codeartifact.model.ListPackageVersionDependenciesRequest;
import software.amazon.awssdk.services.codeartifact.model.ListPackageVersionDependenciesResponse;
import software.amazon.awssdk.services.codeartifact.model.ListPackageVersionsRequest;
import software.amazon.awssdk.services.codeartifact.model.ListPackageVersionsResponse;
import software.amazon.awssdk.services.codeartifact.model.ListPackagesRequest;
import software.amazon.awssdk.services.codeartifact.model.ListPackagesResponse;
import software.amazon.awssdk.services.codeartifact.model.ListRepositoriesInDomainRequest;
import software.amazon.awssdk.services.codeartifact.model.ListRepositoriesInDomainResponse;
import software.amazon.awssdk.services.codeartifact.model.ListRepositoriesRequest;
import software.amazon.awssdk.services.codeartifact.model.ListRepositoriesResponse;
import software.amazon.awssdk.services.codeartifact.model.PutDomainPermissionsPolicyRequest;
import software.amazon.awssdk.services.codeartifact.model.PutDomainPermissionsPolicyResponse;
import software.amazon.awssdk.services.codeartifact.model.PutRepositoryPermissionsPolicyRequest;
import software.amazon.awssdk.services.codeartifact.model.PutRepositoryPermissionsPolicyResponse;
import software.amazon.awssdk.services.codeartifact.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codeartifact.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.codeartifact.model.ThrottlingException;
import software.amazon.awssdk.services.codeartifact.model.UpdatePackageVersionsStatusRequest;
import software.amazon.awssdk.services.codeartifact.model.UpdatePackageVersionsStatusResponse;
import software.amazon.awssdk.services.codeartifact.model.UpdateRepositoryRequest;
import software.amazon.awssdk.services.codeartifact.model.UpdateRepositoryResponse;
import software.amazon.awssdk.services.codeartifact.model.ValidationException;
import software.amazon.awssdk.services.codeartifact.paginators.ListDomainsPublisher;
import software.amazon.awssdk.services.codeartifact.paginators.ListPackageVersionAssetsPublisher;
import software.amazon.awssdk.services.codeartifact.paginators.ListPackageVersionsPublisher;
import software.amazon.awssdk.services.codeartifact.paginators.ListPackagesPublisher;
import software.amazon.awssdk.services.codeartifact.paginators.ListRepositoriesInDomainPublisher;
import software.amazon.awssdk.services.codeartifact.paginators.ListRepositoriesPublisher;
import software.amazon.awssdk.services.codeartifact.transform.AssociateExternalConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.CopyPackageVersionsRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.CreateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DeleteDomainPermissionsPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DeletePackageVersionsRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DeleteRepositoryPermissionsPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DeleteRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DescribeDomainRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DescribePackageVersionRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DescribeRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DisassociateExternalConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.DisposePackageVersionsRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.GetAuthorizationTokenRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.GetDomainPermissionsPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.GetPackageVersionAssetRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.GetPackageVersionReadmeRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.GetRepositoryEndpointRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.GetRepositoryPermissionsPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListPackageVersionAssetsRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListPackageVersionDependenciesRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListPackageVersionsRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListPackagesRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListRepositoriesInDomainRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.ListRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.PutDomainPermissionsPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.PutRepositoryPermissionsPolicyRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.UpdatePackageVersionsStatusRequestMarshaller;
import software.amazon.awssdk.services.codeartifact.transform.UpdateRepositoryRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodeartifactAsyncClient}.
 *
 * @see CodeartifactAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeartifactAsyncClient implements CodeartifactAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodeartifactAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeartifactAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds an existing external connection to a repository. One external connection is allowed per repository.
     * </p>
     * <note>
     * <p>
     * A repository can have one or more upstream repositories, or an external connection.
     * </p>
     * </note>
     *
     * @param associateExternalConnectionRequest
     * @return A Java Future containing the result of the AssociateExternalConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.AssociateExternalConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/AssociateExternalConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateExternalConnectionResponse> associateExternalConnection(
            AssociateExternalConnectionRequest associateExternalConnectionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateExternalConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateExternalConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateExternalConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateExternalConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateExternalConnectionRequest, AssociateExternalConnectionResponse>()
                            .withOperationName("AssociateExternalConnection")
                            .withMarshaller(new AssociateExternalConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateExternalConnectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateExternalConnectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<AssociateExternalConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    associateExternalConnectionRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Copies package versions from one repository to another repository in the same domain.
     * </p>
     * <note>
     * <p>
     * You must specify <code>versions</code> or <code>versionRevisions</code>. You cannot specify both.
     * </p>
     * </note>
     *
     * @param copyPackageVersionsRequest
     * @return A Java Future containing the result of the CopyPackageVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.CopyPackageVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/CopyPackageVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CopyPackageVersionsResponse> copyPackageVersions(
            CopyPackageVersionsRequest copyPackageVersionsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyPackageVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CopyPackageVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CopyPackageVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CopyPackageVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CopyPackageVersionsRequest, CopyPackageVersionsResponse>()
                            .withOperationName("CopyPackageVersions")
                            .withMarshaller(new CopyPackageVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(copyPackageVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = copyPackageVersionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CopyPackageVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyPackageVersionsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a domain. CodeArtifact <i>domains</i> make it easier to manage multiple repositories across an
     * organization. You can use a domain to apply permissions across many repositories owned by different AWS accounts.
     * An asset is stored only once in a domain, even if it's in multiple repositories.
     * </p>
     * <p>
     * Although you can have multiple domains, we recommend a single production domain that contains all published
     * artifacts so that your development teams can find and share packages. You can use a second pre-production domain
     * to test changes to the production domain configuration.
     * </p>
     *
     * @param createDomainRequest
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.CreateDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/CreateDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a repository.
     * </p>
     *
     * @param createRepositoryRequest
     * @return A Java Future containing the result of the CreateRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.CreateRepository
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/CreateRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRepositoryResponse> createRepository(CreateRepositoryRequest createRepositoryRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryRequest, CreateRepositoryResponse>()
                            .withOperationName("CreateRepository")
                            .withMarshaller(new CreateRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createRepositoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createRepositoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a domain. You cannot delete a domain that contains repositories. If you want to delete a domain with
     * repositories, first delete its repositories.
     * </p>
     *
     * @param deleteDomainRequest
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DeleteDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DeleteDomain" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the resource policy set on a domain.
     * </p>
     *
     * @param deleteDomainPermissionsPolicyRequest
     * @return A Java Future containing the result of the DeleteDomainPermissionsPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DeleteDomainPermissionsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DeleteDomainPermissionsPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainPermissionsPolicyResponse> deleteDomainPermissionsPolicy(
            DeleteDomainPermissionsPolicyRequest deleteDomainPermissionsPolicyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomainPermissionsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainPermissionsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDomainPermissionsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainPermissionsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainPermissionsPolicyRequest, DeleteDomainPermissionsPolicyResponse>()
                            .withOperationName("DeleteDomainPermissionsPolicy")
                            .withMarshaller(new DeleteDomainPermissionsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainPermissionsPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDomainPermissionsPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteDomainPermissionsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    deleteDomainPermissionsPolicyRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes one or more versions of a package. A deleted package version cannot be restored in your repository. If
     * you want to remove a package version from your repository and be able to restore it later, set its status to
     * <code>Archived</code>. Archived packages cannot be downloaded from a repository and don't show up with list
     * package APIs (for example,
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html">ListackageVersions</a> </code>
     * ), but you can restore them using
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_UpdatePackageVersionsStatus.html">UpdatePackageVersionsStatus</a> </code>
     * .
     * </p>
     *
     * @param deletePackageVersionsRequest
     * @return A Java Future containing the result of the DeletePackageVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DeletePackageVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DeletePackageVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePackageVersionsResponse> deletePackageVersions(
            DeletePackageVersionsRequest deletePackageVersionsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePackageVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePackageVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePackageVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePackageVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePackageVersionsRequest, DeletePackageVersionsResponse>()
                            .withOperationName("DeletePackageVersions")
                            .withMarshaller(new DeletePackageVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePackageVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePackageVersionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeletePackageVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePackageVersionsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a repository.
     * </p>
     *
     * @param deleteRepositoryRequest
     * @return A Java Future containing the result of the DeleteRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DeleteRepository
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DeleteRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryResponse> deleteRepository(DeleteRepositoryRequest deleteRepositoryRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryRequest, DeleteRepositoryResponse>()
                            .withOperationName("DeleteRepository")
                            .withMarshaller(new DeleteRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRepositoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteRepositoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the resource policy that is set on a repository. After a resource policy is deleted, the permissions
     * allowed and denied by the deleted policy are removed. The effect of deleting a resource policy might not be
     * immediate.
     * </p>
     * <important>
     * <p>
     * Use <code>DeleteRepositoryPermissionsPolicy</code> with caution. After a policy is deleted, AWS users, roles, and
     * accounts lose permissions to perform the repository actions granted by the deleted policy.
     * </p>
     * </important>
     *
     * @param deleteRepositoryPermissionsPolicyRequest
     * @return A Java Future containing the result of the DeleteRepositoryPermissionsPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DeleteRepositoryPermissionsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DeleteRepositoryPermissionsPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryPermissionsPolicyResponse> deleteRepositoryPermissionsPolicy(
            DeleteRepositoryPermissionsPolicyRequest deleteRepositoryPermissionsPolicyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryPermissionsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryPermissionsPolicyResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteRepositoryPermissionsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryPermissionsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryPermissionsPolicyRequest, DeleteRepositoryPermissionsPolicyResponse>()
                            .withOperationName("DeleteRepositoryPermissionsPolicy")
                            .withMarshaller(new DeleteRepositoryPermissionsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteRepositoryPermissionsPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteRepositoryPermissionsPolicyRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteRepositoryPermissionsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    deleteRepositoryPermissionsPolicyRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_DomainDescription.html">
     * <code>DomainDescription</code> </a> object that contains information about the requested domain.
     * </p>
     *
     * @param describeDomainRequest
     * @return A Java Future containing the result of the DescribeDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DescribeDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DescribeDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDomainResponse> describeDomain(DescribeDomainRequest describeDomainRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainRequest, DescribeDomainResponse>()
                            .withOperationName("DescribeDomain")
                            .withMarshaller(new DescribeDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionDescription.html">
     * <code>PackageVersionDescription</code> </a> object that contains information about the requested package version.
     * </p>
     *
     * @param describePackageVersionRequest
     * @return A Java Future containing the result of the DescribePackageVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DescribePackageVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DescribePackageVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePackageVersionResponse> describePackageVersion(
            DescribePackageVersionRequest describePackageVersionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePackageVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePackageVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePackageVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePackageVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePackageVersionRequest, DescribePackageVersionResponse>()
                            .withOperationName("DescribePackageVersion")
                            .withMarshaller(new DescribePackageVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePackageVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describePackageVersionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribePackageVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePackageVersionRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <code>RepositoryDescription</code> object that contains detailed information about the requested
     * repository.
     * </p>
     *
     * @param describeRepositoryRequest
     * @return A Java Future containing the result of the DescribeRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DescribeRepository
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DescribeRepository"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRepositoryResponse> describeRepository(DescribeRepositoryRequest describeRepositoryRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRepositoryRequest, DescribeRepositoryResponse>()
                            .withOperationName("DescribeRepository")
                            .withMarshaller(new DescribeRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRepositoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeRepositoryRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRepositoryRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an existing external connection from a repository.
     * </p>
     *
     * @param disassociateExternalConnectionRequest
     * @return A Java Future containing the result of the DisassociateExternalConnection operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DisassociateExternalConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DisassociateExternalConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateExternalConnectionResponse> disassociateExternalConnection(
            DisassociateExternalConnectionRequest disassociateExternalConnectionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateExternalConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateExternalConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateExternalConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateExternalConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateExternalConnectionRequest, DisassociateExternalConnectionResponse>()
                            .withOperationName("DisassociateExternalConnection")
                            .withMarshaller(new DisassociateExternalConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateExternalConnectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateExternalConnectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DisassociateExternalConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    disassociateExternalConnectionRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the assets in package versions and sets the package versions' status to <code>Disposed</code>. A disposed
     * package version cannot be restored in your repository because its assets are deleted.
     * </p>
     * <p>
     * To view all disposed package versions in a repository, use
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html">ListackageVersions</a> </code>
     * and set the
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html#API_ListPackageVersions_RequestSyntax">status</a> </code>
     * parameter to <code>Disposed</code>.
     * </p>
     * <p>
     * To view information about a disposed package version, use
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html">ListPackageVersions</a> </code>
     * and set the
     * <code> <a href="https://docs.aws.amazon.com/API_ListPackageVersions.html#codeartifact-ListPackageVersions-response-status">status</a> </code>
     * parameter to <code>Disposed</code>.
     * </p>
     *
     * @param disposePackageVersionsRequest
     * @return A Java Future containing the result of the DisposePackageVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.DisposePackageVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/DisposePackageVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisposePackageVersionsResponse> disposePackageVersions(
            DisposePackageVersionsRequest disposePackageVersionsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisposePackageVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisposePackageVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisposePackageVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisposePackageVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisposePackageVersionsRequest, DisposePackageVersionsResponse>()
                            .withOperationName("DisposePackageVersions")
                            .withMarshaller(new DisposePackageVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disposePackageVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disposePackageVersionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DisposePackageVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disposePackageVersionsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a temporary authentication token for accessing repositories in the domain. This API requires the
     * <code>codeartifact:GetAuthorizationToken</code> and <code>sts:GetServiceBearerToken</code> permissions.
     * </p>
     * <note>
     * <p>
     * CodeArtifact authorization tokens are valid for a period of 12 hours when created with the <code>login</code>
     * command. You can call <code>login</code> periodically to refresh the token. When you create an authorization
     * token with the <code>GetAuthorizationToken</code> API, you can set a custom authorization period, up to a maximum
     * of 12 hours, with the <code>durationSeconds</code> parameter.
     * </p>
     * <p>
     * The authorization period begins after <code>login</code> or <code>GetAuthorizationToken</code> is called. If
     * <code>login</code> or <code>GetAuthorizationToken</code> is called while assuming a role, the token lifetime is
     * independent of the maximum session duration of the role. For example, if you call <code>sts assume-role</code>
     * and specify a session duration of 15 minutes, then generate a CodeArtifact authorization token, the token will be
     * valid for the full authorization period even though this is longer than the 15-minute session duration.
     * </p>
     * <p>
     * See <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use.html">Using IAM Roles</a> for more
     * information on controlling session duration.
     * </p>
     * </note>
     *
     * @param getAuthorizationTokenRequest
     * @return A Java Future containing the result of the GetAuthorizationToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.GetAuthorizationToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/GetAuthorizationToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAuthorizationTokenResponse> getAuthorizationToken(
            GetAuthorizationTokenRequest getAuthorizationTokenRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAuthorizationToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAuthorizationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAuthorizationTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAuthorizationTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAuthorizationTokenRequest, GetAuthorizationTokenResponse>()
                            .withOperationName("GetAuthorizationToken")
                            .withMarshaller(new GetAuthorizationTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAuthorizationTokenRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAuthorizationTokenRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetAuthorizationTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAuthorizationTokenRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the resource policy attached to the specified domain.
     * </p>
     * <note>
     * <p>
     * The policy is a resource-based policy, not an identity-based policy. For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_identity-vs-resource.html">Identity-based
     * policies and resource-based policies </a> in the <i>AWS Identity and Access Management User Guide</i>.
     * </p>
     * </note>
     *
     * @param getDomainPermissionsPolicyRequest
     * @return A Java Future containing the result of the GetDomainPermissionsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.GetDomainPermissionsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/GetDomainPermissionsPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainPermissionsPolicyResponse> getDomainPermissionsPolicy(
            GetDomainPermissionsPolicyRequest getDomainPermissionsPolicyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainPermissionsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainPermissionsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainPermissionsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainPermissionsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainPermissionsPolicyRequest, GetDomainPermissionsPolicyResponse>()
                            .withOperationName("GetDomainPermissionsPolicy")
                            .withMarshaller(new GetDomainPermissionsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDomainPermissionsPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDomainPermissionsPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetDomainPermissionsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    getDomainPermissionsPolicyRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an asset (or file) that is in a package. For example, for a Maven package version, use
     * <code>GetPackageVersionAsset</code> to download a <code>JAR</code> file, a <code>POM</code> file, or any other
     * assets in the package version.
     * </p>
     *
     * @param getPackageVersionAssetRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The binary file, or asset, that is downloaded.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.GetPackageVersionAsset
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/GetPackageVersionAsset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getPackageVersionAsset(
            GetPackageVersionAssetRequest getPackageVersionAssetRequest,
            AsyncResponseTransformer<GetPackageVersionAssetResponse, ReturnT> asyncResponseTransformer) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPackageVersionAsset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetPackageVersionAssetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPackageVersionAssetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetPackageVersionAssetRequest, GetPackageVersionAssetResponse>()
                            .withOperationName("GetPackageVersionAsset")
                            .withMarshaller(new GetPackageVersionAssetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPackageVersionAssetRequest),
                    asyncResponseTransformer);
            AwsRequestOverrideConfiguration requestOverrideConfig = getPackageVersionAssetRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> asyncResponseTransformer.exceptionOccurred(e));
                }
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseTransformer.exceptionOccurred(t));
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPackageVersionAssetRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the readme file or descriptive text for a package version. For packages that do not contain a readme file,
     * CodeArtifact extracts a description from a metadata file. For example, from the <code>&lt;description&gt;</code>
     * element in the <code>pom.xml</code> file of a Maven package.
     * </p>
     * <p>
     * The returned text might contain formatting. For example, it might contain formatting for Markdown or
     * reStructuredText.
     * </p>
     *
     * @param getPackageVersionReadmeRequest
     * @return A Java Future containing the result of the GetPackageVersionReadme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.GetPackageVersionReadme
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/GetPackageVersionReadme"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPackageVersionReadmeResponse> getPackageVersionReadme(
            GetPackageVersionReadmeRequest getPackageVersionReadmeRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPackageVersionReadme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPackageVersionReadmeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPackageVersionReadmeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPackageVersionReadmeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPackageVersionReadmeRequest, GetPackageVersionReadmeResponse>()
                            .withOperationName("GetPackageVersionReadme")
                            .withMarshaller(new GetPackageVersionReadmeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPackageVersionReadmeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getPackageVersionReadmeRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetPackageVersionReadmeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPackageVersionReadmeRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the endpoint of a repository for a specific package format. A repository has one endpoint for each
     * package format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>npm</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pypi</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>maven</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param getRepositoryEndpointRequest
     * @return A Java Future containing the result of the GetRepositoryEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.GetRepositoryEndpoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/GetRepositoryEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryEndpointResponse> getRepositoryEndpoint(
            GetRepositoryEndpointRequest getRepositoryEndpointRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryEndpointRequest, GetRepositoryEndpointResponse>()
                            .withOperationName("GetRepositoryEndpoint")
                            .withMarshaller(new GetRepositoryEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRepositoryEndpointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRepositoryEndpointRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetRepositoryEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryEndpointRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the resource policy that is set on a repository.
     * </p>
     *
     * @param getRepositoryPermissionsPolicyRequest
     * @return A Java Future containing the result of the GetRepositoryPermissionsPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.GetRepositoryPermissionsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/GetRepositoryPermissionsPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryPermissionsPolicyResponse> getRepositoryPermissionsPolicy(
            GetRepositoryPermissionsPolicyRequest getRepositoryPermissionsPolicyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryPermissionsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryPermissionsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryPermissionsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryPermissionsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryPermissionsPolicyRequest, GetRepositoryPermissionsPolicyResponse>()
                            .withOperationName("GetRepositoryPermissionsPolicy")
                            .withMarshaller(new GetRepositoryPermissionsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRepositoryPermissionsPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRepositoryPermissionsPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetRepositoryPermissionsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    getRepositoryPermissionsPolicyRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionDescription.html">DomainSummary</a> </code>
     * objects for all domains owned by the AWS account that makes this call. Each returned <code>DomainSummary</code>
     * object contains information about a domain.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDomainsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of
     * <code> <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionDescription.html">DomainSummary</a> </code>
     * objects for all domains owned by the AWS account that makes this call. Each returned <code>DomainSummary</code>
     * object contains information about a domain.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listDomains(software.amazon.awssdk.services.codeartifact.model.ListDomainsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListDomainsPublisher publisher = client.listDomainsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeartifact.model.ListDomainsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeartifact.model.ListDomainsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.codeartifact.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListDomains
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListDomains" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDomainsPublisher listDomainsPaginator(ListDomainsRequest listDomainsRequest) {
        return new ListDomainsPublisher(this, applyPaginatorUserAgent(listDomainsRequest));
    }

    /**
     * <p>
     * Returns a list of <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssetSummary.html">
     * <code>AssetSummary</code> </a> objects for assets in a package version.
     * </p>
     *
     * @param listPackageVersionAssetsRequest
     * @return A Java Future containing the result of the ListPackageVersionAssets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackageVersionAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackageVersionAssets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackageVersionAssetsResponse> listPackageVersionAssets(
            ListPackageVersionAssetsRequest listPackageVersionAssetsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackageVersionAssets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackageVersionAssetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackageVersionAssetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackageVersionAssetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackageVersionAssetsRequest, ListPackageVersionAssetsResponse>()
                            .withOperationName("ListPackageVersionAssets")
                            .withMarshaller(new ListPackageVersionAssetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackageVersionAssetsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPackageVersionAssetsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListPackageVersionAssetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackageVersionAssetsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_AssetSummary.html">
     * <code>AssetSummary</code> </a> objects for assets in a package version.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPackageVersionAssets(software.amazon.awssdk.services.codeartifact.model.ListPackageVersionAssetsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListPackageVersionAssetsPublisher publisher = client.listPackageVersionAssetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListPackageVersionAssetsPublisher publisher = client.listPackageVersionAssetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeartifact.model.ListPackageVersionAssetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeartifact.model.ListPackageVersionAssetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackageVersionAssets(software.amazon.awssdk.services.codeartifact.model.ListPackageVersionAssetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackageVersionAssetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackageVersionAssets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackageVersionAssets"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPackageVersionAssetsPublisher listPackageVersionAssetsPaginator(
            ListPackageVersionAssetsRequest listPackageVersionAssetsRequest) {
        return new ListPackageVersionAssetsPublisher(this, applyPaginatorUserAgent(listPackageVersionAssetsRequest));
    }

    /**
     * <p>
     * Returns the direct dependencies for a package version. The dependencies are returned as <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageDependency.html">
     * <code>PackageDependency</code> </a> objects. CodeArtifact extracts the dependencies for a package version from
     * the metadata file for the package format (for example, the <code>package.json</code> file for npm packages and
     * the <code>pom.xml</code> file for Maven). Any package version dependencies that are not listed in the
     * configuration file are not returned.
     * </p>
     *
     * @param listPackageVersionDependenciesRequest
     * @return A Java Future containing the result of the ListPackageVersionDependencies operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackageVersionDependencies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackageVersionDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackageVersionDependenciesResponse> listPackageVersionDependencies(
            ListPackageVersionDependenciesRequest listPackageVersionDependenciesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackageVersionDependencies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackageVersionDependenciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackageVersionDependenciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackageVersionDependenciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackageVersionDependenciesRequest, ListPackageVersionDependenciesResponse>()
                            .withOperationName("ListPackageVersionDependencies")
                            .withMarshaller(new ListPackageVersionDependenciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackageVersionDependenciesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPackageVersionDependenciesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListPackageVersionDependenciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listPackageVersionDependenciesRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionSummary.html">
     * <code>PackageVersionSummary</code> </a> objects for package versions in a repository that match the request
     * parameters.
     * </p>
     *
     * @param listPackageVersionsRequest
     * @return A Java Future containing the result of the ListPackageVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackageVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackageVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackageVersionsResponse> listPackageVersions(
            ListPackageVersionsRequest listPackageVersionsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackageVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackageVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPackageVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackageVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackageVersionsRequest, ListPackageVersionsResponse>()
                            .withOperationName("ListPackageVersions")
                            .withMarshaller(new ListPackageVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackageVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPackageVersionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListPackageVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackageVersionsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionSummary.html">
     * <code>PackageVersionSummary</code> </a> objects for package versions in a repository that match the request
     * parameters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPackageVersions(software.amazon.awssdk.services.codeartifact.model.ListPackageVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListPackageVersionsPublisher publisher = client.listPackageVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListPackageVersionsPublisher publisher = client.listPackageVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeartifact.model.ListPackageVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeartifact.model.ListPackageVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackageVersions(software.amazon.awssdk.services.codeartifact.model.ListPackageVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPackageVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackageVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackageVersions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListPackageVersionsPublisher listPackageVersionsPaginator(ListPackageVersionsRequest listPackageVersionsRequest) {
        return new ListPackageVersionsPublisher(this, applyPaginatorUserAgent(listPackageVersionsRequest));
    }

    /**
     * <p>
     * Returns a list of <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageSummary.html">
     * <code>PackageSummary</code> </a> objects for packages in a repository that match the request parameters.
     * </p>
     *
     * @param listPackagesRequest
     * @return A Java Future containing the result of the ListPackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackages
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPackagesResponse> listPackages(ListPackagesRequest listPackagesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPackagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPackagesRequest, ListPackagesResponse>()
                            .withOperationName("ListPackages").withMarshaller(new ListPackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPackagesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPackagesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListPackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPackagesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageSummary.html">
     * <code>PackageSummary</code> </a> objects for packages in a repository that match the request parameters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPackages(software.amazon.awssdk.services.codeartifact.model.ListPackagesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListPackagesPublisher publisher = client.listPackagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListPackagesPublisher publisher = client.listPackagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeartifact.model.ListPackagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeartifact.model.ListPackagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPackages(software.amazon.awssdk.services.codeartifact.model.ListPackagesRequest)} operation.</b>
     * </p>
     *
     * @param listPackagesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListPackages
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListPackages" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPackagesPublisher listPackagesPaginator(ListPackagesRequest listPackagesRequest) {
        return new ListPackagesPublisher(this, applyPaginatorUserAgent(listPackagesRequest));
    }

    /**
     * <p>
     * Returns a list of <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_RepositorySummary.html">
     * <code>RepositorySummary</code> </a> objects. Each <code>RepositorySummary</code> contains information about a
     * repository in the specified AWS account and that matches the input parameters.
     * </p>
     *
     * @param listRepositoriesRequest
     * @return A Java Future containing the result of the ListRepositories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListRepositories
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListRepositories" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRepositoriesResponse> listRepositories(ListRepositoriesRequest listRepositoriesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRepositoriesRequest, ListRepositoriesResponse>()
                            .withOperationName("ListRepositories")
                            .withMarshaller(new ListRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRepositoriesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRepositoriesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListRepositoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRepositoriesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_RepositorySummary.html">
     * <code>RepositorySummary</code> </a> objects. Each <code>RepositorySummary</code> contains information about a
     * repository in the specified domain and that matches the input parameters.
     * </p>
     *
     * @param listRepositoriesInDomainRequest
     * @return A Java Future containing the result of the ListRepositoriesInDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListRepositoriesInDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListRepositoriesInDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRepositoriesInDomainResponse> listRepositoriesInDomain(
            ListRepositoriesInDomainRequest listRepositoriesInDomainRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositoriesInDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRepositoriesInDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRepositoriesInDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRepositoriesInDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRepositoriesInDomainRequest, ListRepositoriesInDomainResponse>()
                            .withOperationName("ListRepositoriesInDomain")
                            .withMarshaller(new ListRepositoriesInDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRepositoriesInDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRepositoriesInDomainRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListRepositoriesInDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRepositoriesInDomainRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_RepositorySummary.html">
     * <code>RepositorySummary</code> </a> objects. Each <code>RepositorySummary</code> contains information about a
     * repository in the specified domain and that matches the input parameters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRepositoriesInDomain(software.amazon.awssdk.services.codeartifact.model.ListRepositoriesInDomainRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListRepositoriesInDomainPublisher publisher = client.listRepositoriesInDomainPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListRepositoriesInDomainPublisher publisher = client.listRepositoriesInDomainPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeartifact.model.ListRepositoriesInDomainResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeartifact.model.ListRepositoriesInDomainResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRepositoriesInDomain(software.amazon.awssdk.services.codeartifact.model.ListRepositoriesInDomainRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRepositoriesInDomainRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListRepositoriesInDomain
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListRepositoriesInDomain"
     *      target="_top">AWS API Documentation</a>
     */
    public ListRepositoriesInDomainPublisher listRepositoriesInDomainPaginator(
            ListRepositoriesInDomainRequest listRepositoriesInDomainRequest) {
        return new ListRepositoriesInDomainPublisher(this, applyPaginatorUserAgent(listRepositoriesInDomainRequest));
    }

    /**
     * <p>
     * Returns a list of <a
     * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_RepositorySummary.html">
     * <code>RepositorySummary</code> </a> objects. Each <code>RepositorySummary</code> contains information about a
     * repository in the specified AWS account and that matches the input parameters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRepositories(software.amazon.awssdk.services.codeartifact.model.ListRepositoriesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListRepositoriesPublisher publisher = client.listRepositoriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codeartifact.paginators.ListRepositoriesPublisher publisher = client.listRepositoriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codeartifact.model.ListRepositoriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codeartifact.model.ListRepositoriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRepositories(software.amazon.awssdk.services.codeartifact.model.ListRepositoriesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRepositoriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.ListRepositories
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/ListRepositories" target="_top">AWS
     *      API Documentation</a>
     */
    public ListRepositoriesPublisher listRepositoriesPaginator(ListRepositoriesRequest listRepositoriesRequest) {
        return new ListRepositoriesPublisher(this, applyPaginatorUserAgent(listRepositoriesRequest));
    }

    /**
     * <p>
     * Sets a resource policy on a domain that specifies permissions to access it.
     * </p>
     *
     * @param putDomainPermissionsPolicyRequest
     * @return A Java Future containing the result of the PutDomainPermissionsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.PutDomainPermissionsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/PutDomainPermissionsPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutDomainPermissionsPolicyResponse> putDomainPermissionsPolicy(
            PutDomainPermissionsPolicyRequest putDomainPermissionsPolicyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDomainPermissionsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDomainPermissionsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutDomainPermissionsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDomainPermissionsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDomainPermissionsPolicyRequest, PutDomainPermissionsPolicyResponse>()
                            .withOperationName("PutDomainPermissionsPolicy")
                            .withMarshaller(new PutDomainPermissionsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putDomainPermissionsPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putDomainPermissionsPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<PutDomainPermissionsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    putDomainPermissionsPolicyRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the resource policy on a repository that specifies permissions to access it.
     * </p>
     *
     * @param putRepositoryPermissionsPolicyRequest
     * @return A Java Future containing the result of the PutRepositoryPermissionsPolicy operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.PutRepositoryPermissionsPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/PutRepositoryPermissionsPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRepositoryPermissionsPolicyResponse> putRepositoryPermissionsPolicy(
            PutRepositoryPermissionsPolicyRequest putRepositoryPermissionsPolicyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRepositoryPermissionsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRepositoryPermissionsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRepositoryPermissionsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRepositoryPermissionsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRepositoryPermissionsPolicyRequest, PutRepositoryPermissionsPolicyResponse>()
                            .withOperationName("PutRepositoryPermissionsPolicy")
                            .withMarshaller(new PutRepositoryPermissionsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putRepositoryPermissionsPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putRepositoryPermissionsPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<PutRepositoryPermissionsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    putRepositoryPermissionsPolicyRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status of one or more versions of a package.
     * </p>
     *
     * @param updatePackageVersionsStatusRequest
     * @return A Java Future containing the result of the UpdatePackageVersionsStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.UpdatePackageVersionsStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/UpdatePackageVersionsStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePackageVersionsStatusResponse> updatePackageVersionsStatus(
            UpdatePackageVersionsStatusRequest updatePackageVersionsStatusRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePackageVersionsStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePackageVersionsStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePackageVersionsStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePackageVersionsStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePackageVersionsStatusRequest, UpdatePackageVersionsStatusResponse>()
                            .withOperationName("UpdatePackageVersionsStatus")
                            .withMarshaller(new UpdatePackageVersionsStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updatePackageVersionsStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updatePackageVersionsStatusRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdatePackageVersionsStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    updatePackageVersionsStatusRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the properties of a repository.
     * </p>
     *
     * @param updateRepositoryRequest
     * @return A Java Future containing the result of the UpdateRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The operation did not succeed because of an unauthorized access attempt.</li>
     *         <li>ConflictException The operation did not succeed because prerequisites are not met.</li>
     *         <li>InternalServerException The operation did not succeed because of an error that occurred inside AWS
     *         CodeArtifact.</li>
     *         <li>ResourceNotFoundException The operation did not succeed because the resource requested is not found
     *         in the service.</li>
     *         <li>ServiceQuotaExceededException The operation did not succeed because it would have exceeded a service
     *         limit for your account.</li>
     *         <li>ThrottlingException The operation did not succeed because too many requests are sent to the service.</li>
     *         <li>ValidationException The operation did not succeed because a parameter in the request was sent with an
     *         invalid value.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeartifactException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeartifactAsyncClient.UpdateRepository
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeartifact-2018-09-22/UpdateRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRepositoryResponse> updateRepository(UpdateRepositoryRequest updateRepositoryRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "codeartifact");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRepositoryRequest, UpdateRepositoryResponse>()
                            .withOperationName("UpdateRepository")
                            .withMarshaller(new UpdateRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRepositoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRepositoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRepositoryRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeartifactException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends CodeartifactRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
