/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeartifact.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of a repository stored in AWS CodeArtifact. A CodeArtifact repository contains a set of package versions,
 * each of which maps to a set of assets. Repositories are polyglot—a single repository can contain packages of any
 * supported type. Each repository exposes endpoints for fetching and publishing packages using tools like the
 * <code>npm</code> CLI, the Maven CLI (<code>mvn</code>), and <code>pip</code>. You can create up to 100 repositories
 * per AWS account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RepositoryDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<RepositoryDescription.Builder, RepositoryDescription> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(RepositoryDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ADMINISTRATOR_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("administratorAccount").getter(getter(RepositoryDescription::administratorAccount))
            .setter(setter(Builder::administratorAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("administratorAccount").build())
            .build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainName").getter(getter(RepositoryDescription::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainName").build()).build();

    private static final SdkField<String> DOMAIN_OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainOwner").getter(getter(RepositoryDescription::domainOwner)).setter(setter(Builder::domainOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainOwner").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(RepositoryDescription::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(RepositoryDescription::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<UpstreamRepositoryInfo>> UPSTREAMS_FIELD = SdkField
            .<List<UpstreamRepositoryInfo>> builder(MarshallingType.LIST)
            .memberName("upstreams")
            .getter(getter(RepositoryDescription::upstreams))
            .setter(setter(Builder::upstreams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("upstreams").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UpstreamRepositoryInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(UpstreamRepositoryInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RepositoryExternalConnectionInfo>> EXTERNAL_CONNECTIONS_FIELD = SdkField
            .<List<RepositoryExternalConnectionInfo>> builder(MarshallingType.LIST)
            .memberName("externalConnections")
            .getter(getter(RepositoryDescription::externalConnections))
            .setter(setter(Builder::externalConnections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("externalConnections").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RepositoryExternalConnectionInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(RepositoryExternalConnectionInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            ADMINISTRATOR_ACCOUNT_FIELD, DOMAIN_NAME_FIELD, DOMAIN_OWNER_FIELD, ARN_FIELD, DESCRIPTION_FIELD, UPSTREAMS_FIELD,
            EXTERNAL_CONNECTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String administratorAccount;

    private final String domainName;

    private final String domainOwner;

    private final String arn;

    private final String description;

    private final List<UpstreamRepositoryInfo> upstreams;

    private final List<RepositoryExternalConnectionInfo> externalConnections;

    private RepositoryDescription(BuilderImpl builder) {
        this.name = builder.name;
        this.administratorAccount = builder.administratorAccount;
        this.domainName = builder.domainName;
        this.domainOwner = builder.domainOwner;
        this.arn = builder.arn;
        this.description = builder.description;
        this.upstreams = builder.upstreams;
        this.externalConnections = builder.externalConnections;
    }

    /**
     * <p>
     * The name of the repository.
     * </p>
     * 
     * @return The name of the repository.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The 12-digit account number of the AWS account that manages the repository.
     * </p>
     * 
     * @return The 12-digit account number of the AWS account that manages the repository.
     */
    public String administratorAccount() {
        return administratorAccount;
    }

    /**
     * <p>
     * The name of the domain that contains the repository.
     * </p>
     * 
     * @return The name of the domain that contains the repository.
     */
    public String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The 12-digit account number of the AWS account that owns the domain that contains the repository. It does not
     * include dashes or spaces.
     * </p>
     * 
     * @return The 12-digit account number of the AWS account that owns the domain that contains the repository. It does
     *         not include dashes or spaces.
     */
    public String domainOwner() {
        return domainOwner;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the repository.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the repository.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * A text description of the repository.
     * </p>
     * 
     * @return A text description of the repository.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the Upstreams property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasUpstreams() {
        return upstreams != null && !(upstreams instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of upstream repositories to associate with the repository. The order of the upstream repositories in the
     * list determines their priority order when AWS CodeArtifact looks for a requested package version. For more
     * information, see <a href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with
     * upstream repositories</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUpstreams()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of upstream repositories to associate with the repository. The order of the upstream repositories
     *         in the list determines their priority order when AWS CodeArtifact looks for a requested package version.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with upstream
     *         repositories</a>.
     */
    public List<UpstreamRepositoryInfo> upstreams() {
        return upstreams;
    }

    /**
     * Returns true if the ExternalConnections property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasExternalConnections() {
        return externalConnections != null && !(externalConnections instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of external connections associated with the repository.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExternalConnections()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of external connections associated with the repository.
     */
    public List<RepositoryExternalConnectionInfo> externalConnections() {
        return externalConnections;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(administratorAccount());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(domainOwner());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasUpstreams() ? upstreams() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExternalConnections() ? externalConnections() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RepositoryDescription)) {
            return false;
        }
        RepositoryDescription other = (RepositoryDescription) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(administratorAccount(), other.administratorAccount())
                && Objects.equals(domainName(), other.domainName()) && Objects.equals(domainOwner(), other.domainOwner())
                && Objects.equals(arn(), other.arn()) && Objects.equals(description(), other.description())
                && hasUpstreams() == other.hasUpstreams() && Objects.equals(upstreams(), other.upstreams())
                && hasExternalConnections() == other.hasExternalConnections()
                && Objects.equals(externalConnections(), other.externalConnections());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RepositoryDescription").add("Name", name()).add("AdministratorAccount", administratorAccount())
                .add("DomainName", domainName()).add("DomainOwner", domainOwner()).add("Arn", arn())
                .add("Description", description()).add("Upstreams", hasUpstreams() ? upstreams() : null)
                .add("ExternalConnections", hasExternalConnections() ? externalConnections() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "administratorAccount":
            return Optional.ofNullable(clazz.cast(administratorAccount()));
        case "domainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "domainOwner":
            return Optional.ofNullable(clazz.cast(domainOwner()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "upstreams":
            return Optional.ofNullable(clazz.cast(upstreams()));
        case "externalConnections":
            return Optional.ofNullable(clazz.cast(externalConnections()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RepositoryDescription, T> g) {
        return obj -> g.apply((RepositoryDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RepositoryDescription> {
        /**
         * <p>
         * The name of the repository.
         * </p>
         * 
         * @param name
         *        The name of the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The 12-digit account number of the AWS account that manages the repository.
         * </p>
         * 
         * @param administratorAccount
         *        The 12-digit account number of the AWS account that manages the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administratorAccount(String administratorAccount);

        /**
         * <p>
         * The name of the domain that contains the repository.
         * </p>
         * 
         * @param domainName
         *        The name of the domain that contains the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The 12-digit account number of the AWS account that owns the domain that contains the repository. It does not
         * include dashes or spaces.
         * </p>
         * 
         * @param domainOwner
         *        The 12-digit account number of the AWS account that owns the domain that contains the repository. It
         *        does not include dashes or spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainOwner(String domainOwner);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the repository.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A text description of the repository.
         * </p>
         * 
         * @param description
         *        A text description of the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of upstream repositories to associate with the repository. The order of the upstream repositories in
         * the list determines their priority order when AWS CodeArtifact looks for a requested package version. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with upstream
         * repositories</a>.
         * </p>
         * 
         * @param upstreams
         *        A list of upstream repositories to associate with the repository. The order of the upstream
         *        repositories in the list determines their priority order when AWS CodeArtifact looks for a requested
         *        package version. For more information, see <a
         *        href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with upstream
         *        repositories</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upstreams(Collection<UpstreamRepositoryInfo> upstreams);

        /**
         * <p>
         * A list of upstream repositories to associate with the repository. The order of the upstream repositories in
         * the list determines their priority order when AWS CodeArtifact looks for a requested package version. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with upstream
         * repositories</a>.
         * </p>
         * 
         * @param upstreams
         *        A list of upstream repositories to associate with the repository. The order of the upstream
         *        repositories in the list determines their priority order when AWS CodeArtifact looks for a requested
         *        package version. For more information, see <a
         *        href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with upstream
         *        repositories</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upstreams(UpstreamRepositoryInfo... upstreams);

        /**
         * <p>
         * A list of upstream repositories to associate with the repository. The order of the upstream repositories in
         * the list determines their priority order when AWS CodeArtifact looks for a requested package version. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/codeartifact/latest/ug/repos-upstream.html">Working with upstream
         * repositories</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UpstreamRepositoryInfo>.Builder} avoiding
         * the need to create one manually via {@link List<UpstreamRepositoryInfo>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UpstreamRepositoryInfo>.Builder#build()} is called
         * immediately and its result is passed to {@link #upstreams(List<UpstreamRepositoryInfo>)}.
         * 
         * @param upstreams
         *        a consumer that will call methods on {@link List<UpstreamRepositoryInfo>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #upstreams(List<UpstreamRepositoryInfo>)
         */
        Builder upstreams(Consumer<UpstreamRepositoryInfo.Builder>... upstreams);

        /**
         * <p>
         * An array of external connections associated with the repository.
         * </p>
         * 
         * @param externalConnections
         *        An array of external connections associated with the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalConnections(Collection<RepositoryExternalConnectionInfo> externalConnections);

        /**
         * <p>
         * An array of external connections associated with the repository.
         * </p>
         * 
         * @param externalConnections
         *        An array of external connections associated with the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalConnections(RepositoryExternalConnectionInfo... externalConnections);

        /**
         * <p>
         * An array of external connections associated with the repository.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RepositoryExternalConnectionInfo>.Builder}
         * avoiding the need to create one manually via {@link List<RepositoryExternalConnectionInfo>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RepositoryExternalConnectionInfo>.Builder#build()} is called
         * immediately and its result is passed to {@link #externalConnections(List<RepositoryExternalConnectionInfo>)}.
         * 
         * @param externalConnections
         *        a consumer that will call methods on {@link List<RepositoryExternalConnectionInfo>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #externalConnections(List<RepositoryExternalConnectionInfo>)
         */
        Builder externalConnections(Consumer<RepositoryExternalConnectionInfo.Builder>... externalConnections);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String administratorAccount;

        private String domainName;

        private String domainOwner;

        private String arn;

        private String description;

        private List<UpstreamRepositoryInfo> upstreams = DefaultSdkAutoConstructList.getInstance();

        private List<RepositoryExternalConnectionInfo> externalConnections = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RepositoryDescription model) {
            name(model.name);
            administratorAccount(model.administratorAccount);
            domainName(model.domainName);
            domainOwner(model.domainOwner);
            arn(model.arn);
            description(model.description);
            upstreams(model.upstreams);
            externalConnections(model.externalConnections);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getAdministratorAccount() {
            return administratorAccount;
        }

        @Override
        public final Builder administratorAccount(String administratorAccount) {
            this.administratorAccount = administratorAccount;
            return this;
        }

        public final void setAdministratorAccount(String administratorAccount) {
            this.administratorAccount = administratorAccount;
        }

        public final String getDomainName() {
            return domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        public final String getDomainOwner() {
            return domainOwner;
        }

        @Override
        public final Builder domainOwner(String domainOwner) {
            this.domainOwner = domainOwner;
            return this;
        }

        public final void setDomainOwner(String domainOwner) {
            this.domainOwner = domainOwner;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<UpstreamRepositoryInfo.Builder> getUpstreams() {
            if (upstreams instanceof SdkAutoConstructList) {
                return null;
            }
            return upstreams != null ? upstreams.stream().map(UpstreamRepositoryInfo::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder upstreams(Collection<UpstreamRepositoryInfo> upstreams) {
            this.upstreams = UpstreamRepositoryInfoListCopier.copy(upstreams);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder upstreams(UpstreamRepositoryInfo... upstreams) {
            upstreams(Arrays.asList(upstreams));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder upstreams(Consumer<UpstreamRepositoryInfo.Builder>... upstreams) {
            upstreams(Stream.of(upstreams).map(c -> UpstreamRepositoryInfo.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUpstreams(Collection<UpstreamRepositoryInfo.BuilderImpl> upstreams) {
            this.upstreams = UpstreamRepositoryInfoListCopier.copyFromBuilder(upstreams);
        }

        public final Collection<RepositoryExternalConnectionInfo.Builder> getExternalConnections() {
            if (externalConnections instanceof SdkAutoConstructList) {
                return null;
            }
            return externalConnections != null ? externalConnections.stream().map(RepositoryExternalConnectionInfo::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder externalConnections(Collection<RepositoryExternalConnectionInfo> externalConnections) {
            this.externalConnections = RepositoryExternalConnectionInfoListCopier.copy(externalConnections);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder externalConnections(RepositoryExternalConnectionInfo... externalConnections) {
            externalConnections(Arrays.asList(externalConnections));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder externalConnections(Consumer<RepositoryExternalConnectionInfo.Builder>... externalConnections) {
            externalConnections(Stream.of(externalConnections)
                    .map(c -> RepositoryExternalConnectionInfo.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setExternalConnections(Collection<RepositoryExternalConnectionInfo.BuilderImpl> externalConnections) {
            this.externalConnections = RepositoryExternalConnectionInfoListCopier.copyFromBuilder(externalConnections);
        }

        @Override
        public RepositoryDescription build() {
            return new RepositoryDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
