/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeartifact.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a package version asset.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssetSummary implements SdkPojo, Serializable, ToCopyableBuilder<AssetSummary.Builder, AssetSummary> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(AssetSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Long> SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("size")
            .getter(getter(AssetSummary::size)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("size").build()).build();

    private static final SdkField<Map<String, String>> HASHES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("hashes")
            .getter(getter(AssetSummary::hashesAsStrings))
            .setter(setter(Builder::hashesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hashes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SIZE_FIELD,
            HASHES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Long size;

    private final Map<String, String> hashes;

    private AssetSummary(BuilderImpl builder) {
        this.name = builder.name;
        this.size = builder.size;
        this.hashes = builder.hashes;
    }

    /**
     * <p>
     * The name of the asset.
     * </p>
     * 
     * @return The name of the asset.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The size of the asset.
     * </p>
     * 
     * @return The size of the asset.
     */
    public final Long size() {
        return size;
    }

    /**
     * <p>
     * The hashes of the asset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasHashes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The hashes of the asset.
     */
    public final Map<HashAlgorithm, String> hashes() {
        return AssetHashesCopier.copyStringToEnum(hashes);
    }

    /**
     * Returns true if the Hashes property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasHashes() {
        return hashes != null && !(hashes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The hashes of the asset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasHashes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The hashes of the asset.
     */
    public final Map<String, String> hashesAsStrings() {
        return hashes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        hashCode = 31 * hashCode + Objects.hashCode(hasHashes() ? hashesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssetSummary)) {
            return false;
        }
        AssetSummary other = (AssetSummary) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(size(), other.size()) && hasHashes() == other.hasHashes()
                && Objects.equals(hashesAsStrings(), other.hashesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssetSummary").add("Name", name()).add("Size", size())
                .add("Hashes", hasHashes() ? hashesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "size":
            return Optional.ofNullable(clazz.cast(size()));
        case "hashes":
            return Optional.ofNullable(clazz.cast(hashesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssetSummary, T> g) {
        return obj -> g.apply((AssetSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssetSummary> {
        /**
         * <p>
         * The name of the asset.
         * </p>
         * 
         * @param name
         *        The name of the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The size of the asset.
         * </p>
         * 
         * @param size
         *        The size of the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Long size);

        /**
         * <p>
         * The hashes of the asset.
         * </p>
         * 
         * @param hashes
         *        The hashes of the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hashesWithStrings(Map<String, String> hashes);

        /**
         * <p>
         * The hashes of the asset.
         * </p>
         * 
         * @param hashes
         *        The hashes of the asset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hashes(Map<HashAlgorithm, String> hashes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Long size;

        private Map<String, String> hashes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AssetSummary model) {
            name(model.name);
            size(model.size);
            hashesWithStrings(model.hashes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Long getSize() {
            return size;
        }

        public final void setSize(Long size) {
            this.size = size;
        }

        @Override
        @Transient
        public final Builder size(Long size) {
            this.size = size;
            return this;
        }

        public final Map<String, String> getHashes() {
            if (hashes instanceof SdkAutoConstructMap) {
                return null;
            }
            return hashes;
        }

        public final void setHashes(Map<String, String> hashes) {
            this.hashes = AssetHashesCopier.copy(hashes);
        }

        @Override
        @Transient
        public final Builder hashesWithStrings(Map<String, String> hashes) {
            this.hashes = AssetHashesCopier.copy(hashes);
            return this;
        }

        @Override
        @Transient
        public final Builder hashes(Map<HashAlgorithm, String> hashes) {
            this.hashes = AssetHashesCopier.copyEnumToString(hashes);
            return this;
        }

        @Override
        public AssetSummary build() {
            return new AssetSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
