/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeartifact.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a package version, including its status, version, and revision. The <a
 * href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_ListPackageVersions.html"
 * >ListPackageVersions</a> operation returns a list of <code>PackageVersionSummary</code> objects.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PackageVersionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<PackageVersionSummary.Builder, PackageVersionSummary> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(PackageVersionSummary::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> REVISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("revision").getter(getter(PackageVersionSummary::revision)).setter(setter(Builder::revision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revision").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(PackageVersionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<PackageVersionOrigin> ORIGIN_FIELD = SdkField
            .<PackageVersionOrigin> builder(MarshallingType.SDK_POJO).memberName("origin")
            .getter(getter(PackageVersionSummary::origin)).setter(setter(Builder::origin))
            .constructor(PackageVersionOrigin::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("origin").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD, REVISION_FIELD,
            STATUS_FIELD, ORIGIN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String version;

    private final String revision;

    private final String status;

    private final PackageVersionOrigin origin;

    private PackageVersionSummary(BuilderImpl builder) {
        this.version = builder.version;
        this.revision = builder.revision;
        this.status = builder.status;
        this.origin = builder.origin;
    }

    /**
     * <p>
     * Information about a package version.
     * </p>
     * 
     * @return Information about a package version.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The revision associated with a package version.
     * </p>
     * 
     * @return The revision associated with a package version.
     */
    public final String revision() {
        return revision;
    }

    /**
     * <p>
     * A string that contains the status of the package version. It can be one of the following:
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PackageVersionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return A string that contains the status of the package version. It can be one of the following:
     * @see PackageVersionStatus
     */
    public final PackageVersionStatus status() {
        return PackageVersionStatus.fromValue(status);
    }

    /**
     * <p>
     * A string that contains the status of the package version. It can be one of the following:
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PackageVersionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return A string that contains the status of the package version. It can be one of the following:
     * @see PackageVersionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
     * PackageVersionOrigin</a> object that contains information about how the package version was added to the
     * repository.
     * </p>
     * 
     * @return A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
     *         PackageVersionOrigin</a> object that contains information about how the package version was added to the
     *         repository.
     */
    public final PackageVersionOrigin origin() {
        return origin;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(revision());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(origin());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PackageVersionSummary)) {
            return false;
        }
        PackageVersionSummary other = (PackageVersionSummary) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(revision(), other.revision())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(origin(), other.origin());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PackageVersionSummary").add("Version", version()).add("Revision", revision())
                .add("Status", statusAsString()).add("Origin", origin()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "revision":
            return Optional.ofNullable(clazz.cast(revision()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "origin":
            return Optional.ofNullable(clazz.cast(origin()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PackageVersionSummary, T> g) {
        return obj -> g.apply((PackageVersionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PackageVersionSummary> {
        /**
         * <p>
         * Information about a package version.
         * </p>
         * 
         * @param version
         *        Information about a package version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The revision associated with a package version.
         * </p>
         * 
         * @param revision
         *        The revision associated with a package version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revision(String revision);

        /**
         * <p>
         * A string that contains the status of the package version. It can be one of the following:
         * </p>
         * 
         * @param status
         *        A string that contains the status of the package version. It can be one of the following:
         * @see PackageVersionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackageVersionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * A string that contains the status of the package version. It can be one of the following:
         * </p>
         * 
         * @param status
         *        A string that contains the status of the package version. It can be one of the following:
         * @see PackageVersionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackageVersionStatus
         */
        Builder status(PackageVersionStatus status);

        /**
         * <p>
         * A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
         * PackageVersionOrigin</a> object that contains information about how the package version was added to the
         * repository.
         * </p>
         * 
         * @param origin
         *        A <a
         *        href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
         *        PackageVersionOrigin</a> object that contains information about how the package version was added to
         *        the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origin(PackageVersionOrigin origin);

        /**
         * <p>
         * A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
         * PackageVersionOrigin</a> object that contains information about how the package version was added to the
         * repository.
         * </p>
         * This is a convenience method that creates an instance of the {@link PackageVersionOrigin.Builder} avoiding
         * the need to create one manually via {@link PackageVersionOrigin#builder()}.
         *
         * When the {@link Consumer} completes, {@link PackageVersionOrigin.Builder#build()} is called immediately and
         * its result is passed to {@link #origin(PackageVersionOrigin)}.
         * 
         * @param origin
         *        a consumer that will call methods on {@link PackageVersionOrigin.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #origin(PackageVersionOrigin)
         */
        default Builder origin(Consumer<PackageVersionOrigin.Builder> origin) {
            return origin(PackageVersionOrigin.builder().applyMutation(origin).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private String revision;

        private String status;

        private PackageVersionOrigin origin;

        private BuilderImpl() {
        }

        private BuilderImpl(PackageVersionSummary model) {
            version(model.version);
            revision(model.revision);
            status(model.status);
            origin(model.origin);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getRevision() {
            return revision;
        }

        public final void setRevision(String revision) {
            this.revision = revision;
        }

        @Override
        public final Builder revision(String revision) {
            this.revision = revision;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PackageVersionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final PackageVersionOrigin.Builder getOrigin() {
            return origin != null ? origin.toBuilder() : null;
        }

        public final void setOrigin(PackageVersionOrigin.BuilderImpl origin) {
            this.origin = origin != null ? origin.build() : null;
        }

        @Override
        public final Builder origin(PackageVersionOrigin origin) {
            this.origin = origin;
            return this;
        }

        @Override
        public PackageVersionSummary build() {
            return new PackageVersionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
