/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeartifact.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a package version.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PackageVersionDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<PackageVersionDescription.Builder, PackageVersionDescription> {
    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(PackageVersionDescription::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(PackageVersionDescription::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<String> PACKAGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("packageName").getter(getter(PackageVersionDescription::packageName))
            .setter(setter(Builder::packageName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("packageName").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("displayName").getter(getter(PackageVersionDescription::displayName))
            .setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("displayName").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(PackageVersionDescription::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("summary")
            .getter(getter(PackageVersionDescription::summary)).setter(setter(Builder::summary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("summary").build()).build();

    private static final SdkField<String> HOME_PAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("homePage").getter(getter(PackageVersionDescription::homePage)).setter(setter(Builder::homePage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("homePage").build()).build();

    private static final SdkField<String> SOURCE_CODE_REPOSITORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceCodeRepository").getter(getter(PackageVersionDescription::sourceCodeRepository))
            .setter(setter(Builder::sourceCodeRepository))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceCodeRepository").build())
            .build();

    private static final SdkField<Instant> PUBLISHED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("publishedTime").getter(getter(PackageVersionDescription::publishedTime))
            .setter(setter(Builder::publishedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publishedTime").build()).build();

    private static final SdkField<List<LicenseInfo>> LICENSES_FIELD = SdkField
            .<List<LicenseInfo>> builder(MarshallingType.LIST)
            .memberName("licenses")
            .getter(getter(PackageVersionDescription::licenses))
            .setter(setter(Builder::licenses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("licenses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LicenseInfo> builder(MarshallingType.SDK_POJO)
                                            .constructor(LicenseInfo::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REVISION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("revision").getter(getter(PackageVersionDescription::revision)).setter(setter(Builder::revision))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revision").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(PackageVersionDescription::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<PackageVersionOrigin> ORIGIN_FIELD = SdkField
            .<PackageVersionOrigin> builder(MarshallingType.SDK_POJO).memberName("origin")
            .getter(getter(PackageVersionDescription::origin)).setter(setter(Builder::origin))
            .constructor(PackageVersionOrigin::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("origin").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORMAT_FIELD, NAMESPACE_FIELD,
            PACKAGE_NAME_FIELD, DISPLAY_NAME_FIELD, VERSION_FIELD, SUMMARY_FIELD, HOME_PAGE_FIELD, SOURCE_CODE_REPOSITORY_FIELD,
            PUBLISHED_TIME_FIELD, LICENSES_FIELD, REVISION_FIELD, STATUS_FIELD, ORIGIN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String format;

    private final String namespace;

    private final String packageName;

    private final String displayName;

    private final String version;

    private final String summary;

    private final String homePage;

    private final String sourceCodeRepository;

    private final Instant publishedTime;

    private final List<LicenseInfo> licenses;

    private final String revision;

    private final String status;

    private final PackageVersionOrigin origin;

    private PackageVersionDescription(BuilderImpl builder) {
        this.format = builder.format;
        this.namespace = builder.namespace;
        this.packageName = builder.packageName;
        this.displayName = builder.displayName;
        this.version = builder.version;
        this.summary = builder.summary;
        this.homePage = builder.homePage;
        this.sourceCodeRepository = builder.sourceCodeRepository;
        this.publishedTime = builder.publishedTime;
        this.licenses = builder.licenses;
        this.revision = builder.revision;
        this.status = builder.status;
        this.origin = builder.origin;
    }

    /**
     * <p>
     * The format of the package version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link PackageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the package version.
     * @see PackageFormat
     */
    public final PackageFormat format() {
        return PackageFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the package version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link PackageFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the package version.
     * @see PackageFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The namespace of the package version. The package component that specifies its namespace depends on its type. For
     * example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The namespace of a Maven package version is its <code>groupId</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The namespace of an npm or Swift package version is its <code>scope</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The namespace of a generic package is its <code>namespace</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Python, NuGet, Ruby, and Cargo package versions do not contain a corresponding component, package versions of
     * those formats do not have a namespace.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The namespace of the package version. The package component that specifies its namespace depends on its
     *         type. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The namespace of a Maven package version is its <code>groupId</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The namespace of an npm or Swift package version is its <code>scope</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The namespace of a generic package is its <code>namespace</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Python, NuGet, Ruby, and Cargo package versions do not contain a corresponding component, package
     *         versions of those formats do not have a namespace.
     *         </p>
     *         </li>
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the requested package.
     * </p>
     * 
     * @return The name of the requested package.
     */
    public final String packageName() {
        return packageName;
    }

    /**
     * <p>
     * The name of the package that is displayed. The <code>displayName</code> varies depending on the package version's
     * format. For example, if an npm package is named <code>ui</code>, is in the namespace <code>vue</code>, and has
     * the format <code>npm</code>, then the <code>displayName</code> is <code>@vue/ui</code>.
     * </p>
     * 
     * @return The name of the package that is displayed. The <code>displayName</code> varies depending on the package
     *         version's format. For example, if an npm package is named <code>ui</code>, is in the namespace
     *         <code>vue</code>, and has the format <code>npm</code>, then the <code>displayName</code> is
     *         <code>@vue/ui</code>.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The version of the package.
     * </p>
     * 
     * @return The version of the package.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * A summary of the package version. The summary is extracted from the package. The information in and detail level
     * of the summary depends on the package version's format.
     * </p>
     * 
     * @return A summary of the package version. The summary is extracted from the package. The information in and
     *         detail level of the summary depends on the package version's format.
     */
    public final String summary() {
        return summary;
    }

    /**
     * <p>
     * The homepage associated with the package.
     * </p>
     * 
     * @return The homepage associated with the package.
     */
    public final String homePage() {
        return homePage;
    }

    /**
     * <p>
     * The repository for the source code in the package version, or the source code used to build it.
     * </p>
     * 
     * @return The repository for the source code in the package version, or the source code used to build it.
     */
    public final String sourceCodeRepository() {
        return sourceCodeRepository;
    }

    /**
     * <p>
     * A timestamp that contains the date and time the package version was published.
     * </p>
     * 
     * @return A timestamp that contains the date and time the package version was published.
     */
    public final Instant publishedTime() {
        return publishedTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Licenses property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLicenses() {
        return licenses != null && !(licenses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about licenses associated with the package version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLicenses} method.
     * </p>
     * 
     * @return Information about licenses associated with the package version.
     */
    public final List<LicenseInfo> licenses() {
        return licenses;
    }

    /**
     * <p>
     * The revision of the package version.
     * </p>
     * 
     * @return The revision of the package version.
     */
    public final String revision() {
        return revision;
    }

    /**
     * <p>
     * A string that contains the status of the package version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PackageVersionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return A string that contains the status of the package version.
     * @see PackageVersionStatus
     */
    public final PackageVersionStatus status() {
        return PackageVersionStatus.fromValue(status);
    }

    /**
     * <p>
     * A string that contains the status of the package version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PackageVersionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return A string that contains the status of the package version.
     * @see PackageVersionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
     * PackageVersionOrigin</a> object that contains information about how the package version was added to the
     * repository.
     * </p>
     * 
     * @return A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
     *         PackageVersionOrigin</a> object that contains information about how the package version was added to the
     *         repository.
     */
    public final PackageVersionOrigin origin() {
        return origin;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(packageName());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(summary());
        hashCode = 31 * hashCode + Objects.hashCode(homePage());
        hashCode = 31 * hashCode + Objects.hashCode(sourceCodeRepository());
        hashCode = 31 * hashCode + Objects.hashCode(publishedTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasLicenses() ? licenses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(revision());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(origin());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PackageVersionDescription)) {
            return false;
        }
        PackageVersionDescription other = (PackageVersionDescription) obj;
        return Objects.equals(formatAsString(), other.formatAsString()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(packageName(), other.packageName()) && Objects.equals(displayName(), other.displayName())
                && Objects.equals(version(), other.version()) && Objects.equals(summary(), other.summary())
                && Objects.equals(homePage(), other.homePage())
                && Objects.equals(sourceCodeRepository(), other.sourceCodeRepository())
                && Objects.equals(publishedTime(), other.publishedTime()) && hasLicenses() == other.hasLicenses()
                && Objects.equals(licenses(), other.licenses()) && Objects.equals(revision(), other.revision())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(origin(), other.origin());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PackageVersionDescription").add("Format", formatAsString()).add("Namespace", namespace())
                .add("PackageName", packageName()).add("DisplayName", displayName()).add("Version", version())
                .add("Summary", summary()).add("HomePage", homePage()).add("SourceCodeRepository", sourceCodeRepository())
                .add("PublishedTime", publishedTime()).add("Licenses", hasLicenses() ? licenses() : null)
                .add("Revision", revision()).add("Status", statusAsString()).add("Origin", origin()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "packageName":
            return Optional.ofNullable(clazz.cast(packageName()));
        case "displayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "summary":
            return Optional.ofNullable(clazz.cast(summary()));
        case "homePage":
            return Optional.ofNullable(clazz.cast(homePage()));
        case "sourceCodeRepository":
            return Optional.ofNullable(clazz.cast(sourceCodeRepository()));
        case "publishedTime":
            return Optional.ofNullable(clazz.cast(publishedTime()));
        case "licenses":
            return Optional.ofNullable(clazz.cast(licenses()));
        case "revision":
            return Optional.ofNullable(clazz.cast(revision()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "origin":
            return Optional.ofNullable(clazz.cast(origin()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PackageVersionDescription, T> g) {
        return obj -> g.apply((PackageVersionDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PackageVersionDescription> {
        /**
         * <p>
         * The format of the package version.
         * </p>
         * 
         * @param format
         *        The format of the package version.
         * @see PackageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackageFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the package version.
         * </p>
         * 
         * @param format
         *        The format of the package version.
         * @see PackageFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackageFormat
         */
        Builder format(PackageFormat format);

        /**
         * <p>
         * The namespace of the package version. The package component that specifies its namespace depends on its type.
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The namespace of a Maven package version is its <code>groupId</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The namespace of an npm or Swift package version is its <code>scope</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * The namespace of a generic package is its <code>namespace</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Python, NuGet, Ruby, and Cargo package versions do not contain a corresponding component, package versions of
         * those formats do not have a namespace.
         * </p>
         * </li>
         * </ul>
         * 
         * @param namespace
         *        The namespace of the package version. The package component that specifies its namespace depends on
         *        its type. For example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The namespace of a Maven package version is its <code>groupId</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The namespace of an npm or Swift package version is its <code>scope</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The namespace of a generic package is its <code>namespace</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Python, NuGet, Ruby, and Cargo package versions do not contain a corresponding component, package
         *        versions of those formats do not have a namespace.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name of the requested package.
         * </p>
         * 
         * @param packageName
         *        The name of the requested package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder packageName(String packageName);

        /**
         * <p>
         * The name of the package that is displayed. The <code>displayName</code> varies depending on the package
         * version's format. For example, if an npm package is named <code>ui</code>, is in the namespace
         * <code>vue</code>, and has the format <code>npm</code>, then the <code>displayName</code> is
         * <code>@vue/ui</code>.
         * </p>
         * 
         * @param displayName
         *        The name of the package that is displayed. The <code>displayName</code> varies depending on the
         *        package version's format. For example, if an npm package is named <code>ui</code>, is in the namespace
         *        <code>vue</code>, and has the format <code>npm</code>, then the <code>displayName</code> is
         *        <code>@vue/ui</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The version of the package.
         * </p>
         * 
         * @param version
         *        The version of the package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * A summary of the package version. The summary is extracted from the package. The information in and detail
         * level of the summary depends on the package version's format.
         * </p>
         * 
         * @param summary
         *        A summary of the package version. The summary is extracted from the package. The information in and
         *        detail level of the summary depends on the package version's format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summary(String summary);

        /**
         * <p>
         * The homepage associated with the package.
         * </p>
         * 
         * @param homePage
         *        The homepage associated with the package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder homePage(String homePage);

        /**
         * <p>
         * The repository for the source code in the package version, or the source code used to build it.
         * </p>
         * 
         * @param sourceCodeRepository
         *        The repository for the source code in the package version, or the source code used to build it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceCodeRepository(String sourceCodeRepository);

        /**
         * <p>
         * A timestamp that contains the date and time the package version was published.
         * </p>
         * 
         * @param publishedTime
         *        A timestamp that contains the date and time the package version was published.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publishedTime(Instant publishedTime);

        /**
         * <p>
         * Information about licenses associated with the package version.
         * </p>
         * 
         * @param licenses
         *        Information about licenses associated with the package version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenses(Collection<LicenseInfo> licenses);

        /**
         * <p>
         * Information about licenses associated with the package version.
         * </p>
         * 
         * @param licenses
         *        Information about licenses associated with the package version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licenses(LicenseInfo... licenses);

        /**
         * <p>
         * Information about licenses associated with the package version.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codeartifact.model.LicenseInfo.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codeartifact.model.LicenseInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codeartifact.model.LicenseInfo.Builder#build()} is called immediately
         * and its result is passed to {@link #licenses(List<LicenseInfo>)}.
         * 
         * @param licenses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codeartifact.model.LicenseInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #licenses(java.util.Collection<LicenseInfo>)
         */
        Builder licenses(Consumer<LicenseInfo.Builder>... licenses);

        /**
         * <p>
         * The revision of the package version.
         * </p>
         * 
         * @param revision
         *        The revision of the package version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revision(String revision);

        /**
         * <p>
         * A string that contains the status of the package version.
         * </p>
         * 
         * @param status
         *        A string that contains the status of the package version.
         * @see PackageVersionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackageVersionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * A string that contains the status of the package version.
         * </p>
         * 
         * @param status
         *        A string that contains the status of the package version.
         * @see PackageVersionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackageVersionStatus
         */
        Builder status(PackageVersionStatus status);

        /**
         * <p>
         * A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
         * PackageVersionOrigin</a> object that contains information about how the package version was added to the
         * repository.
         * </p>
         * 
         * @param origin
         *        A <a
         *        href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
         *        PackageVersionOrigin</a> object that contains information about how the package version was added to
         *        the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origin(PackageVersionOrigin origin);

        /**
         * <p>
         * A <a href="https://docs.aws.amazon.com/codeartifact/latest/APIReference/API_PackageVersionOrigin.html">
         * PackageVersionOrigin</a> object that contains information about how the package version was added to the
         * repository.
         * </p>
         * This is a convenience method that creates an instance of the {@link PackageVersionOrigin.Builder} avoiding
         * the need to create one manually via {@link PackageVersionOrigin#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PackageVersionOrigin.Builder#build()} is called immediately and
         * its result is passed to {@link #origin(PackageVersionOrigin)}.
         * 
         * @param origin
         *        a consumer that will call methods on {@link PackageVersionOrigin.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #origin(PackageVersionOrigin)
         */
        default Builder origin(Consumer<PackageVersionOrigin.Builder> origin) {
            return origin(PackageVersionOrigin.builder().applyMutation(origin).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String format;

        private String namespace;

        private String packageName;

        private String displayName;

        private String version;

        private String summary;

        private String homePage;

        private String sourceCodeRepository;

        private Instant publishedTime;

        private List<LicenseInfo> licenses = DefaultSdkAutoConstructList.getInstance();

        private String revision;

        private String status;

        private PackageVersionOrigin origin;

        private BuilderImpl() {
        }

        private BuilderImpl(PackageVersionDescription model) {
            format(model.format);
            namespace(model.namespace);
            packageName(model.packageName);
            displayName(model.displayName);
            version(model.version);
            summary(model.summary);
            homePage(model.homePage);
            sourceCodeRepository(model.sourceCodeRepository);
            publishedTime(model.publishedTime);
            licenses(model.licenses);
            revision(model.revision);
            status(model.status);
            origin(model.origin);
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(PackageFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getPackageName() {
            return packageName;
        }

        public final void setPackageName(String packageName) {
            this.packageName = packageName;
        }

        @Override
        public final Builder packageName(String packageName) {
            this.packageName = packageName;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getSummary() {
            return summary;
        }

        public final void setSummary(String summary) {
            this.summary = summary;
        }

        @Override
        public final Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public final String getHomePage() {
            return homePage;
        }

        public final void setHomePage(String homePage) {
            this.homePage = homePage;
        }

        @Override
        public final Builder homePage(String homePage) {
            this.homePage = homePage;
            return this;
        }

        public final String getSourceCodeRepository() {
            return sourceCodeRepository;
        }

        public final void setSourceCodeRepository(String sourceCodeRepository) {
            this.sourceCodeRepository = sourceCodeRepository;
        }

        @Override
        public final Builder sourceCodeRepository(String sourceCodeRepository) {
            this.sourceCodeRepository = sourceCodeRepository;
            return this;
        }

        public final Instant getPublishedTime() {
            return publishedTime;
        }

        public final void setPublishedTime(Instant publishedTime) {
            this.publishedTime = publishedTime;
        }

        @Override
        public final Builder publishedTime(Instant publishedTime) {
            this.publishedTime = publishedTime;
            return this;
        }

        public final List<LicenseInfo.Builder> getLicenses() {
            List<LicenseInfo.Builder> result = LicenseInfoListCopier.copyToBuilder(this.licenses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLicenses(Collection<LicenseInfo.BuilderImpl> licenses) {
            this.licenses = LicenseInfoListCopier.copyFromBuilder(licenses);
        }

        @Override
        public final Builder licenses(Collection<LicenseInfo> licenses) {
            this.licenses = LicenseInfoListCopier.copy(licenses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenses(LicenseInfo... licenses) {
            licenses(Arrays.asList(licenses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder licenses(Consumer<LicenseInfo.Builder>... licenses) {
            licenses(Stream.of(licenses).map(c -> LicenseInfo.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getRevision() {
            return revision;
        }

        public final void setRevision(String revision) {
            this.revision = revision;
        }

        @Override
        public final Builder revision(String revision) {
            this.revision = revision;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PackageVersionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final PackageVersionOrigin.Builder getOrigin() {
            return origin != null ? origin.toBuilder() : null;
        }

        public final void setOrigin(PackageVersionOrigin.BuilderImpl origin) {
            this.origin = origin != null ? origin.build() : null;
        }

        @Override
        public final Builder origin(PackageVersionOrigin origin) {
            this.origin = origin;
            return this;
        }

        @Override
        public PackageVersionDescription build() {
            return new PackageVersionDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
