/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a batch build.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BuildBatch implements SdkPojo, Serializable, ToCopyableBuilder<BuildBatch.Builder, BuildBatch> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(BuildBatch::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(BuildBatch::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(BuildBatch::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(BuildBatch::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> CURRENT_PHASE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPhase").getter(getter(BuildBatch::currentPhase)).setter(setter(Builder::currentPhase))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPhase").build()).build();

    private static final SdkField<String> BUILD_BATCH_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildBatchStatus").getter(getter(BuildBatch::buildBatchStatusAsString))
            .setter(setter(Builder::buildBatchStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildBatchStatus").build()).build();

    private static final SdkField<String> SOURCE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceVersion").getter(getter(BuildBatch::sourceVersion)).setter(setter(Builder::sourceVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceVersion").build()).build();

    private static final SdkField<String> RESOLVED_SOURCE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resolvedSourceVersion").getter(getter(BuildBatch::resolvedSourceVersion))
            .setter(setter(Builder::resolvedSourceVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resolvedSourceVersion").build())
            .build();

    private static final SdkField<String> PROJECT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("projectName").getter(getter(BuildBatch::projectName)).setter(setter(Builder::projectName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectName").build()).build();

    private static final SdkField<List<BuildBatchPhase>> PHASES_FIELD = SdkField
            .<List<BuildBatchPhase>> builder(MarshallingType.LIST)
            .memberName("phases")
            .getter(getter(BuildBatch::phases))
            .setter(setter(Builder::phases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("phases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BuildBatchPhase> builder(MarshallingType.SDK_POJO)
                                            .constructor(BuildBatchPhase::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProjectSource> SOURCE_FIELD = SdkField.<ProjectSource> builder(MarshallingType.SDK_POJO)
            .memberName("source").getter(getter(BuildBatch::source)).setter(setter(Builder::source))
            .constructor(ProjectSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<List<ProjectSource>> SECONDARY_SOURCES_FIELD = SdkField
            .<List<ProjectSource>> builder(MarshallingType.LIST)
            .memberName("secondarySources")
            .getter(getter(BuildBatch::secondarySources))
            .setter(setter(Builder::secondarySources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondarySources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ProjectSourceVersion>> SECONDARY_SOURCE_VERSIONS_FIELD = SdkField
            .<List<ProjectSourceVersion>> builder(MarshallingType.LIST)
            .memberName("secondarySourceVersions")
            .getter(getter(BuildBatch::secondarySourceVersions))
            .setter(setter(Builder::secondarySourceVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondarySourceVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectSourceVersion> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectSourceVersion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<BuildArtifacts> ARTIFACTS_FIELD = SdkField.<BuildArtifacts> builder(MarshallingType.SDK_POJO)
            .memberName("artifacts").getter(getter(BuildBatch::artifacts)).setter(setter(Builder::artifacts))
            .constructor(BuildArtifacts::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifacts").build()).build();

    private static final SdkField<List<BuildArtifacts>> SECONDARY_ARTIFACTS_FIELD = SdkField
            .<List<BuildArtifacts>> builder(MarshallingType.LIST)
            .memberName("secondaryArtifacts")
            .getter(getter(BuildBatch::secondaryArtifacts))
            .setter(setter(Builder::secondaryArtifacts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secondaryArtifacts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BuildArtifacts> builder(MarshallingType.SDK_POJO)
                                            .constructor(BuildArtifacts::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProjectCache> CACHE_FIELD = SdkField.<ProjectCache> builder(MarshallingType.SDK_POJO)
            .memberName("cache").getter(getter(BuildBatch::cache)).setter(setter(Builder::cache))
            .constructor(ProjectCache::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cache").build()).build();

    private static final SdkField<ProjectEnvironment> ENVIRONMENT_FIELD = SdkField
            .<ProjectEnvironment> builder(MarshallingType.SDK_POJO).memberName("environment")
            .getter(getter(BuildBatch::environment)).setter(setter(Builder::environment))
            .constructor(ProjectEnvironment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceRole").getter(getter(BuildBatch::serviceRole)).setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRole").build()).build();

    private static final SdkField<LogsConfig> LOG_CONFIG_FIELD = SdkField.<LogsConfig> builder(MarshallingType.SDK_POJO)
            .memberName("logConfig").getter(getter(BuildBatch::logConfig)).setter(setter(Builder::logConfig))
            .constructor(LogsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logConfig").build()).build();

    private static final SdkField<Integer> BUILD_TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("buildTimeoutInMinutes").getter(getter(BuildBatch::buildTimeoutInMinutes))
            .setter(setter(Builder::buildTimeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildTimeoutInMinutes").build())
            .build();

    private static final SdkField<Integer> QUEUED_TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("queuedTimeoutInMinutes").getter(getter(BuildBatch::queuedTimeoutInMinutes))
            .setter(setter(Builder::queuedTimeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("queuedTimeoutInMinutes").build())
            .build();

    private static final SdkField<Boolean> COMPLETE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("complete").getter(getter(BuildBatch::complete)).setter(setter(Builder::complete))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("complete").build()).build();

    private static final SdkField<String> INITIATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("initiator").getter(getter(BuildBatch::initiator)).setter(setter(Builder::initiator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("initiator").build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .memberName("vpcConfig").getter(getter(BuildBatch::vpcConfig)).setter(setter(Builder::vpcConfig))
            .constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcConfig").build()).build();

    private static final SdkField<String> ENCRYPTION_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("encryptionKey").getter(getter(BuildBatch::encryptionKey)).setter(setter(Builder::encryptionKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionKey").build()).build();

    private static final SdkField<Long> BUILD_BATCH_NUMBER_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("buildBatchNumber").getter(getter(BuildBatch::buildBatchNumber))
            .setter(setter(Builder::buildBatchNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildBatchNumber").build()).build();

    private static final SdkField<List<ProjectFileSystemLocation>> FILE_SYSTEM_LOCATIONS_FIELD = SdkField
            .<List<ProjectFileSystemLocation>> builder(MarshallingType.LIST)
            .memberName("fileSystemLocations")
            .getter(getter(BuildBatch::fileSystemLocations))
            .setter(setter(Builder::fileSystemLocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileSystemLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProjectFileSystemLocation> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProjectFileSystemLocation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProjectBuildBatchConfig> BUILD_BATCH_CONFIG_FIELD = SdkField
            .<ProjectBuildBatchConfig> builder(MarshallingType.SDK_POJO).memberName("buildBatchConfig")
            .getter(getter(BuildBatch::buildBatchConfig)).setter(setter(Builder::buildBatchConfig))
            .constructor(ProjectBuildBatchConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildBatchConfig").build()).build();

    private static final SdkField<List<BuildGroup>> BUILD_GROUPS_FIELD = SdkField
            .<List<BuildGroup>> builder(MarshallingType.LIST)
            .memberName("buildGroups")
            .getter(getter(BuildBatch::buildGroups))
            .setter(setter(Builder::buildGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BuildGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(BuildGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> DEBUG_SESSION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("debugSessionEnabled").getter(getter(BuildBatch::debugSessionEnabled))
            .setter(setter(Builder::debugSessionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("debugSessionEnabled").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, CURRENT_PHASE_FIELD, BUILD_BATCH_STATUS_FIELD, SOURCE_VERSION_FIELD,
            RESOLVED_SOURCE_VERSION_FIELD, PROJECT_NAME_FIELD, PHASES_FIELD, SOURCE_FIELD, SECONDARY_SOURCES_FIELD,
            SECONDARY_SOURCE_VERSIONS_FIELD, ARTIFACTS_FIELD, SECONDARY_ARTIFACTS_FIELD, CACHE_FIELD, ENVIRONMENT_FIELD,
            SERVICE_ROLE_FIELD, LOG_CONFIG_FIELD, BUILD_TIMEOUT_IN_MINUTES_FIELD, QUEUED_TIMEOUT_IN_MINUTES_FIELD,
            COMPLETE_FIELD, INITIATOR_FIELD, VPC_CONFIG_FIELD, ENCRYPTION_KEY_FIELD, BUILD_BATCH_NUMBER_FIELD,
            FILE_SYSTEM_LOCATIONS_FIELD, BUILD_BATCH_CONFIG_FIELD, BUILD_GROUPS_FIELD, DEBUG_SESSION_ENABLED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final Instant startTime;

    private final Instant endTime;

    private final String currentPhase;

    private final String buildBatchStatus;

    private final String sourceVersion;

    private final String resolvedSourceVersion;

    private final String projectName;

    private final List<BuildBatchPhase> phases;

    private final ProjectSource source;

    private final List<ProjectSource> secondarySources;

    private final List<ProjectSourceVersion> secondarySourceVersions;

    private final BuildArtifacts artifacts;

    private final List<BuildArtifacts> secondaryArtifacts;

    private final ProjectCache cache;

    private final ProjectEnvironment environment;

    private final String serviceRole;

    private final LogsConfig logConfig;

    private final Integer buildTimeoutInMinutes;

    private final Integer queuedTimeoutInMinutes;

    private final Boolean complete;

    private final String initiator;

    private final VpcConfig vpcConfig;

    private final String encryptionKey;

    private final Long buildBatchNumber;

    private final List<ProjectFileSystemLocation> fileSystemLocations;

    private final ProjectBuildBatchConfig buildBatchConfig;

    private final List<BuildGroup> buildGroups;

    private final Boolean debugSessionEnabled;

    private BuildBatch(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.currentPhase = builder.currentPhase;
        this.buildBatchStatus = builder.buildBatchStatus;
        this.sourceVersion = builder.sourceVersion;
        this.resolvedSourceVersion = builder.resolvedSourceVersion;
        this.projectName = builder.projectName;
        this.phases = builder.phases;
        this.source = builder.source;
        this.secondarySources = builder.secondarySources;
        this.secondarySourceVersions = builder.secondarySourceVersions;
        this.artifacts = builder.artifacts;
        this.secondaryArtifacts = builder.secondaryArtifacts;
        this.cache = builder.cache;
        this.environment = builder.environment;
        this.serviceRole = builder.serviceRole;
        this.logConfig = builder.logConfig;
        this.buildTimeoutInMinutes = builder.buildTimeoutInMinutes;
        this.queuedTimeoutInMinutes = builder.queuedTimeoutInMinutes;
        this.complete = builder.complete;
        this.initiator = builder.initiator;
        this.vpcConfig = builder.vpcConfig;
        this.encryptionKey = builder.encryptionKey;
        this.buildBatchNumber = builder.buildBatchNumber;
        this.fileSystemLocations = builder.fileSystemLocations;
        this.buildBatchConfig = builder.buildBatchConfig;
        this.buildGroups = builder.buildGroups;
        this.debugSessionEnabled = builder.debugSessionEnabled;
    }

    /**
     * <p>
     * The identifier of the batch build.
     * </p>
     * 
     * @return The identifier of the batch build.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The ARN of the batch build.
     * </p>
     * 
     * @return The ARN of the batch build.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The date and time that the batch build started.
     * </p>
     * 
     * @return The date and time that the batch build started.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time that the batch build ended.
     * </p>
     * 
     * @return The date and time that the batch build ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The current phase of the batch build.
     * </p>
     * 
     * @return The current phase of the batch build.
     */
    public final String currentPhase() {
        return currentPhase;
    }

    /**
     * <p>
     * The status of the batch build.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildBatchStatus}
     * will return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildBatchStatusAsString}.
     * </p>
     * 
     * @return The status of the batch build.
     * @see StatusType
     */
    public final StatusType buildBatchStatus() {
        return StatusType.fromValue(buildBatchStatus);
    }

    /**
     * <p>
     * The status of the batch build.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #buildBatchStatus}
     * will return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #buildBatchStatusAsString}.
     * </p>
     * 
     * @return The status of the batch build.
     * @see StatusType
     */
    public final String buildBatchStatusAsString() {
        return buildBatchStatus;
    }

    /**
     * <p>
     * The identifier of the version of the source code to be built.
     * </p>
     * 
     * @return The identifier of the version of the source code to be built.
     */
    public final String sourceVersion() {
        return sourceVersion;
    }

    /**
     * <p>
     * The identifier of the resolved version of this batch build's source code.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For AWS CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * For AWS CodePipeline, the source revision provided by AWS CodePipeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon S3, this does not apply.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier of the resolved version of this batch build's source code.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For AWS CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For AWS CodePipeline, the source revision provided by AWS CodePipeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon S3, this does not apply.
     *         </p>
     *         </li>
     */
    public final String resolvedSourceVersion() {
        return resolvedSourceVersion;
    }

    /**
     * <p>
     * The name of the batch build project.
     * </p>
     * 
     * @return The name of the batch build project.
     */
    public final String projectName() {
        return projectName;
    }

    /**
     * Returns true if the Phases property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasPhases() {
        return phases != null && !(phases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPhases()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
     */
    public final List<BuildBatchPhase> phases() {
        return phases;
    }

    /**
     * Returns the value of the Source property for this object.
     * 
     * @return The value of the Source property for this object.
     */
    public final ProjectSource source() {
        return source;
    }

    /**
     * Returns true if the SecondarySources property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSecondarySources() {
        return secondarySources != null && !(secondarySources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectSource</code> objects that define the sources for the batch build.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecondarySources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>ProjectSource</code> objects that define the sources for the batch build.
     */
    public final List<ProjectSource> secondarySources() {
        return secondarySources;
    }

    /**
     * Returns true if the SecondarySourceVersions property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSecondarySourceVersions() {
        return secondarySourceVersions != null && !(secondarySourceVersions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For AWS CodeCommit: the commit ID, branch, or Git tag to use.
     * </p>
     * </li>
     * <li>
     * <p>
     * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
     * source code you want to build. If a pull request ID is specified, it must use the format
     * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the branch's
     * HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
     * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the default
     * branch's HEAD commit ID is used.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecondarySourceVersions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one
     *         of: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For AWS CodeCommit: the commit ID, branch, or Git tag to use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of
     *         the source code you want to build. If a pull request ID is specified, it must use the format
     *         <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
     *         branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code
     *         you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified,
     *         the default branch's HEAD commit ID is used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
     *         </p>
     *         </li>
     */
    public final List<ProjectSourceVersion> secondarySourceVersions() {
        return secondarySourceVersions;
    }

    /**
     * <p>
     * A <code>BuildArtifacts</code> object the defines the build artifacts for this batch build.
     * </p>
     * 
     * @return A <code>BuildArtifacts</code> object the defines the build artifacts for this batch build.
     */
    public final BuildArtifacts artifacts() {
        return artifacts;
    }

    /**
     * Returns true if the SecondaryArtifacts property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSecondaryArtifacts() {
        return secondaryArtifacts != null && !(secondaryArtifacts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecondaryArtifacts()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
     */
    public final List<BuildArtifacts> secondaryArtifacts() {
        return secondaryArtifacts;
    }

    /**
     * Returns the value of the Cache property for this object.
     * 
     * @return The value of the Cache property for this object.
     */
    public final ProjectCache cache() {
        return cache;
    }

    /**
     * Returns the value of the Environment property for this object.
     * 
     * @return The value of the Environment property for this object.
     */
    public final ProjectEnvironment environment() {
        return environment;
    }

    /**
     * <p>
     * The name of a service role used for builds in the batch.
     * </p>
     * 
     * @return The name of a service role used for builds in the batch.
     */
    public final String serviceRole() {
        return serviceRole;
    }

    /**
     * Returns the value of the LogConfig property for this object.
     * 
     * @return The value of the LogConfig property for this object.
     */
    public final LogsConfig logConfig() {
        return logConfig;
    }

    /**
     * <p>
     * Specifies the maximum amount of time, in minutes, that the build in a batch must be completed in.
     * </p>
     * 
     * @return Specifies the maximum amount of time, in minutes, that the build in a batch must be completed in.
     */
    public final Integer buildTimeoutInMinutes() {
        return buildTimeoutInMinutes;
    }

    /**
     * <p>
     * Specifies the amount of time, in minutes, that the batch build is allowed to be queued before it times out.
     * </p>
     * 
     * @return Specifies the amount of time, in minutes, that the batch build is allowed to be queued before it times
     *         out.
     */
    public final Integer queuedTimeoutInMinutes() {
        return queuedTimeoutInMinutes;
    }

    /**
     * <p>
     * Indicates if the batch build is complete.
     * </p>
     * 
     * @return Indicates if the batch build is complete.
     */
    public final Boolean complete() {
        return complete;
    }

    /**
     * <p>
     * The entity that started the batch build. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If AWS CodePipeline started the build, the pipeline's name (for example,
     * <code>codepipeline/my-demo-pipeline</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * If an AWS Identity and Access Management (IAM) user started the build, the user's name.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the Jenkins plugin for AWS CodeBuild started the build, the string <code>CodeBuild-Jenkins-Plugin</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The entity that started the batch build. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If AWS CodePipeline started the build, the pipeline's name (for example,
     *         <code>codepipeline/my-demo-pipeline</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If an AWS Identity and Access Management (IAM) user started the build, the user's name.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the Jenkins plugin for AWS CodeBuild started the build, the string
     *         <code>CodeBuild-Jenkins-Plugin</code>.
     *         </p>
     *         </li>
     */
    public final String initiator() {
        return initiator;
    }

    /**
     * Returns the value of the VpcConfig property for this object.
     * 
     * @return The value of the VpcConfig property for this object.
     */
    public final VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * The AWS Key Management Service (AWS KMS) customer master key (CMK) to be used for encrypting the batch build
     * output artifacts.
     * </p>
     * <note>
     * <p>
     * You can use a cross-account KMS key to encrypt the build output artifacts if your service role has permission to
     * that key.
     * </p>
     * </note>
     * <p>
     * You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using the
     * format <code>alias/&lt;alias-name&gt;</code>).
     * </p>
     * 
     * @return The AWS Key Management Service (AWS KMS) customer master key (CMK) to be used for encrypting the batch
     *         build output artifacts.</p> <note>
     *         <p>
     *         You can use a cross-account KMS key to encrypt the build output artifacts if your service role has
     *         permission to that key.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using
     *         the format <code>alias/&lt;alias-name&gt;</code>).
     */
    public final String encryptionKey() {
        return encryptionKey;
    }

    /**
     * <p>
     * The number of the batch build. For each project, the <code>buildBatchNumber</code> of its first batch build is
     * <code>1</code>. The <code>buildBatchNumber</code> of each subsequent batch build is incremented by <code>1</code>
     * . If a batch build is deleted, the <code>buildBatchNumber</code> of other batch builds does not change.
     * </p>
     * 
     * @return The number of the batch build. For each project, the <code>buildBatchNumber</code> of its first batch
     *         build is <code>1</code>. The <code>buildBatchNumber</code> of each subsequent batch build is incremented
     *         by <code>1</code>. If a batch build is deleted, the <code>buildBatchNumber</code> of other batch builds
     *         does not change.
     */
    public final Long buildBatchNumber() {
        return buildBatchNumber;
    }

    /**
     * Returns true if the FileSystemLocations property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasFileSystemLocations() {
        return fileSystemLocations != null && !(fileSystemLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
     * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
     * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using Amazon
     * Elastic File System.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFileSystemLocations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
     *         <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>,
     *         <code>location</code>, <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a
     *         file system created using Amazon Elastic File System.
     */
    public final List<ProjectFileSystemLocation> fileSystemLocations() {
        return fileSystemLocations;
    }

    /**
     * Returns the value of the BuildBatchConfig property for this object.
     * 
     * @return The value of the BuildBatchConfig property for this object.
     */
    public final ProjectBuildBatchConfig buildBatchConfig() {
        return buildBatchConfig;
    }

    /**
     * Returns true if the BuildGroups property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasBuildGroups() {
        return buildGroups != null && !(buildGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasBuildGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
     */
    public final List<BuildGroup> buildGroups() {
        return buildGroups;
    }

    /**
     * <p>
     * Specifies if session debugging is enabled for this batch build. For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/session-manager.html">Viewing a running build in
     * Session Manager</a>. Batch session debugging is not supported for matrix batch builds.
     * </p>
     * 
     * @return Specifies if session debugging is enabled for this batch build. For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/session-manager.html">Viewing a running
     *         build in Session Manager</a>. Batch session debugging is not supported for matrix batch builds.
     */
    public final Boolean debugSessionEnabled() {
        return debugSessionEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(currentPhase());
        hashCode = 31 * hashCode + Objects.hashCode(buildBatchStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceVersion());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedSourceVersion());
        hashCode = 31 * hashCode + Objects.hashCode(projectName());
        hashCode = 31 * hashCode + Objects.hashCode(hasPhases() ? phases() : null);
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondarySources() ? secondarySources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondarySourceVersions() ? secondarySourceVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(artifacts());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecondaryArtifacts() ? secondaryArtifacts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cache());
        hashCode = 31 * hashCode + Objects.hashCode(environment());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(logConfig());
        hashCode = 31 * hashCode + Objects.hashCode(buildTimeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(queuedTimeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(complete());
        hashCode = 31 * hashCode + Objects.hashCode(initiator());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKey());
        hashCode = 31 * hashCode + Objects.hashCode(buildBatchNumber());
        hashCode = 31 * hashCode + Objects.hashCode(hasFileSystemLocations() ? fileSystemLocations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(buildBatchConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasBuildGroups() ? buildGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(debugSessionEnabled());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BuildBatch)) {
            return false;
        }
        BuildBatch other = (BuildBatch) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(currentPhase(), other.currentPhase())
                && Objects.equals(buildBatchStatusAsString(), other.buildBatchStatusAsString())
                && Objects.equals(sourceVersion(), other.sourceVersion())
                && Objects.equals(resolvedSourceVersion(), other.resolvedSourceVersion())
                && Objects.equals(projectName(), other.projectName()) && hasPhases() == other.hasPhases()
                && Objects.equals(phases(), other.phases()) && Objects.equals(source(), other.source())
                && hasSecondarySources() == other.hasSecondarySources()
                && Objects.equals(secondarySources(), other.secondarySources())
                && hasSecondarySourceVersions() == other.hasSecondarySourceVersions()
                && Objects.equals(secondarySourceVersions(), other.secondarySourceVersions())
                && Objects.equals(artifacts(), other.artifacts()) && hasSecondaryArtifacts() == other.hasSecondaryArtifacts()
                && Objects.equals(secondaryArtifacts(), other.secondaryArtifacts()) && Objects.equals(cache(), other.cache())
                && Objects.equals(environment(), other.environment()) && Objects.equals(serviceRole(), other.serviceRole())
                && Objects.equals(logConfig(), other.logConfig())
                && Objects.equals(buildTimeoutInMinutes(), other.buildTimeoutInMinutes())
                && Objects.equals(queuedTimeoutInMinutes(), other.queuedTimeoutInMinutes())
                && Objects.equals(complete(), other.complete()) && Objects.equals(initiator(), other.initiator())
                && Objects.equals(vpcConfig(), other.vpcConfig()) && Objects.equals(encryptionKey(), other.encryptionKey())
                && Objects.equals(buildBatchNumber(), other.buildBatchNumber())
                && hasFileSystemLocations() == other.hasFileSystemLocations()
                && Objects.equals(fileSystemLocations(), other.fileSystemLocations())
                && Objects.equals(buildBatchConfig(), other.buildBatchConfig()) && hasBuildGroups() == other.hasBuildGroups()
                && Objects.equals(buildGroups(), other.buildGroups())
                && Objects.equals(debugSessionEnabled(), other.debugSessionEnabled());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BuildBatch").add("Id", id()).add("Arn", arn()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("CurrentPhase", currentPhase())
                .add("BuildBatchStatus", buildBatchStatusAsString()).add("SourceVersion", sourceVersion())
                .add("ResolvedSourceVersion", resolvedSourceVersion()).add("ProjectName", projectName())
                .add("Phases", hasPhases() ? phases() : null).add("Source", source())
                .add("SecondarySources", hasSecondarySources() ? secondarySources() : null)
                .add("SecondarySourceVersions", hasSecondarySourceVersions() ? secondarySourceVersions() : null)
                .add("Artifacts", artifacts()).add("SecondaryArtifacts", hasSecondaryArtifacts() ? secondaryArtifacts() : null)
                .add("Cache", cache()).add("Environment", environment()).add("ServiceRole", serviceRole())
                .add("LogConfig", logConfig()).add("BuildTimeoutInMinutes", buildTimeoutInMinutes())
                .add("QueuedTimeoutInMinutes", queuedTimeoutInMinutes()).add("Complete", complete())
                .add("Initiator", initiator()).add("VpcConfig", vpcConfig()).add("EncryptionKey", encryptionKey())
                .add("BuildBatchNumber", buildBatchNumber())
                .add("FileSystemLocations", hasFileSystemLocations() ? fileSystemLocations() : null)
                .add("BuildBatchConfig", buildBatchConfig()).add("BuildGroups", hasBuildGroups() ? buildGroups() : null)
                .add("DebugSessionEnabled", debugSessionEnabled()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "currentPhase":
            return Optional.ofNullable(clazz.cast(currentPhase()));
        case "buildBatchStatus":
            return Optional.ofNullable(clazz.cast(buildBatchStatusAsString()));
        case "sourceVersion":
            return Optional.ofNullable(clazz.cast(sourceVersion()));
        case "resolvedSourceVersion":
            return Optional.ofNullable(clazz.cast(resolvedSourceVersion()));
        case "projectName":
            return Optional.ofNullable(clazz.cast(projectName()));
        case "phases":
            return Optional.ofNullable(clazz.cast(phases()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "secondarySources":
            return Optional.ofNullable(clazz.cast(secondarySources()));
        case "secondarySourceVersions":
            return Optional.ofNullable(clazz.cast(secondarySourceVersions()));
        case "artifacts":
            return Optional.ofNullable(clazz.cast(artifacts()));
        case "secondaryArtifacts":
            return Optional.ofNullable(clazz.cast(secondaryArtifacts()));
        case "cache":
            return Optional.ofNullable(clazz.cast(cache()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "serviceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "logConfig":
            return Optional.ofNullable(clazz.cast(logConfig()));
        case "buildTimeoutInMinutes":
            return Optional.ofNullable(clazz.cast(buildTimeoutInMinutes()));
        case "queuedTimeoutInMinutes":
            return Optional.ofNullable(clazz.cast(queuedTimeoutInMinutes()));
        case "complete":
            return Optional.ofNullable(clazz.cast(complete()));
        case "initiator":
            return Optional.ofNullable(clazz.cast(initiator()));
        case "vpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "encryptionKey":
            return Optional.ofNullable(clazz.cast(encryptionKey()));
        case "buildBatchNumber":
            return Optional.ofNullable(clazz.cast(buildBatchNumber()));
        case "fileSystemLocations":
            return Optional.ofNullable(clazz.cast(fileSystemLocations()));
        case "buildBatchConfig":
            return Optional.ofNullable(clazz.cast(buildBatchConfig()));
        case "buildGroups":
            return Optional.ofNullable(clazz.cast(buildGroups()));
        case "debugSessionEnabled":
            return Optional.ofNullable(clazz.cast(debugSessionEnabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BuildBatch, T> g) {
        return obj -> g.apply((BuildBatch) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BuildBatch> {
        /**
         * <p>
         * The identifier of the batch build.
         * </p>
         * 
         * @param id
         *        The identifier of the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The ARN of the batch build.
         * </p>
         * 
         * @param arn
         *        The ARN of the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date and time that the batch build started.
         * </p>
         * 
         * @param startTime
         *        The date and time that the batch build started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time that the batch build ended.
         * </p>
         * 
         * @param endTime
         *        The date and time that the batch build ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The current phase of the batch build.
         * </p>
         * 
         * @param currentPhase
         *        The current phase of the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentPhase(String currentPhase);

        /**
         * <p>
         * The status of the batch build.
         * </p>
         * 
         * @param buildBatchStatus
         *        The status of the batch build.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder buildBatchStatus(String buildBatchStatus);

        /**
         * <p>
         * The status of the batch build.
         * </p>
         * 
         * @param buildBatchStatus
         *        The status of the batch build.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder buildBatchStatus(StatusType buildBatchStatus);

        /**
         * <p>
         * The identifier of the version of the source code to be built.
         * </p>
         * 
         * @param sourceVersion
         *        The identifier of the version of the source code to be built.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceVersion(String sourceVersion);

        /**
         * <p>
         * The identifier of the resolved version of this batch build's source code.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For AWS CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * For AWS CodePipeline, the source revision provided by AWS CodePipeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3, this does not apply.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resolvedSourceVersion
         *        The identifier of the resolved version of this batch build's source code.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For AWS CodeCommit, GitHub, GitHub Enterprise, and BitBucket, the commit ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For AWS CodePipeline, the source revision provided by AWS CodePipeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon S3, this does not apply.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedSourceVersion(String resolvedSourceVersion);

        /**
         * <p>
         * The name of the batch build project.
         * </p>
         * 
         * @param projectName
         *        The name of the batch build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectName(String projectName);

        /**
         * <p>
         * An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
         * </p>
         * 
         * @param phases
         *        An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phases(Collection<BuildBatchPhase> phases);

        /**
         * <p>
         * An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
         * </p>
         * 
         * @param phases
         *        An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder phases(BuildBatchPhase... phases);

        /**
         * <p>
         * An array of <code>BuildBatchPhase</code> objects the specify the phases of the batch build.
         * </p>
         * This is a convenience that creates an instance of the {@link List<BuildBatchPhase>.Builder} avoiding the need
         * to create one manually via {@link List<BuildBatchPhase>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BuildBatchPhase>.Builder#build()} is called immediately and
         * its result is passed to {@link #phases(List<BuildBatchPhase>)}.
         * 
         * @param phases
         *        a consumer that will call methods on {@link List<BuildBatchPhase>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #phases(List<BuildBatchPhase>)
         */
        Builder phases(Consumer<BuildBatchPhase.Builder>... phases);

        /**
         * Sets the value of the Source property for this object.
         *
         * @param source
         *        The new value for the Source property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(ProjectSource source);

        /**
         * Sets the value of the Source property for this object.
         *
         * This is a convenience that creates an instance of the {@link ProjectSource.Builder} avoiding the need to
         * create one manually via {@link ProjectSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProjectSource.Builder#build()} is called immediately and its
         * result is passed to {@link #source(ProjectSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link ProjectSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(ProjectSource)
         */
        default Builder source(Consumer<ProjectSource.Builder> source) {
            return source(ProjectSource.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * An array of <code>ProjectSource</code> objects that define the sources for the batch build.
         * </p>
         * 
         * @param secondarySources
         *        An array of <code>ProjectSource</code> objects that define the sources for the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySources(Collection<ProjectSource> secondarySources);

        /**
         * <p>
         * An array of <code>ProjectSource</code> objects that define the sources for the batch build.
         * </p>
         * 
         * @param secondarySources
         *        An array of <code>ProjectSource</code> objects that define the sources for the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySources(ProjectSource... secondarySources);

        /**
         * <p>
         * An array of <code>ProjectSource</code> objects that define the sources for the batch build.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ProjectSource>.Builder} avoiding the need
         * to create one manually via {@link List<ProjectSource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProjectSource>.Builder#build()} is called immediately and
         * its result is passed to {@link #secondarySources(List<ProjectSource>)}.
         * 
         * @param secondarySources
         *        a consumer that will call methods on {@link List<ProjectSource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondarySources(List<ProjectSource>)
         */
        Builder secondarySources(Consumer<ProjectSource.Builder>... secondarySources);

        /**
         * <p>
         * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For AWS CodeCommit: the commit ID, branch, or Git tag to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
         * source code you want to build. If a pull request ID is specified, it must use the format
         * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         * branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
         * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the
         * default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secondarySourceVersions
         *        An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be
         *        one of: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For AWS CodeCommit: the commit ID, branch, or Git tag to use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version
         *        of the source code you want to build. If a pull request ID is specified, it must use the format
         *        <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         *        branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source
         *        code you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not
         *        specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySourceVersions(Collection<ProjectSourceVersion> secondarySourceVersions);

        /**
         * <p>
         * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For AWS CodeCommit: the commit ID, branch, or Git tag to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
         * source code you want to build. If a pull request ID is specified, it must use the format
         * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         * branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
         * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the
         * default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secondarySourceVersions
         *        An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be
         *        one of: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For AWS CodeCommit: the commit ID, branch, or Git tag to use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version
         *        of the source code you want to build. If a pull request ID is specified, it must use the format
         *        <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         *        branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source
         *        code you want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not
         *        specified, the default branch's HEAD commit ID is used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondarySourceVersions(ProjectSourceVersion... secondarySourceVersions);

        /**
         * <p>
         * An array of <code>ProjectSourceVersion</code> objects. Each <code>ProjectSourceVersion</code> must be one of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For AWS CodeCommit: the commit ID, branch, or Git tag to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * For GitHub: the commit ID, pull request ID, branch name, or tag name that corresponds to the version of the
         * source code you want to build. If a pull request ID is specified, it must use the format
         * <code>pr/pull-request-ID</code> (for example, <code>pr/25</code>). If a branch name is specified, the
         * branch's HEAD commit ID is used. If not specified, the default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Bitbucket: the commit ID, branch name, or tag name that corresponds to the version of the source code you
         * want to build. If a branch name is specified, the branch's HEAD commit ID is used. If not specified, the
         * default branch's HEAD commit ID is used.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon S3: the version ID of the object that represents the build input ZIP file to use.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<ProjectSourceVersion>.Builder} avoiding the
         * need to create one manually via {@link List<ProjectSourceVersion>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProjectSourceVersion>.Builder#build()} is called immediately
         * and its result is passed to {@link #secondarySourceVersions(List<ProjectSourceVersion>)}.
         * 
         * @param secondarySourceVersions
         *        a consumer that will call methods on {@link List<ProjectSourceVersion>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondarySourceVersions(List<ProjectSourceVersion>)
         */
        Builder secondarySourceVersions(Consumer<ProjectSourceVersion.Builder>... secondarySourceVersions);

        /**
         * <p>
         * A <code>BuildArtifacts</code> object the defines the build artifacts for this batch build.
         * </p>
         * 
         * @param artifacts
         *        A <code>BuildArtifacts</code> object the defines the build artifacts for this batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifacts(BuildArtifacts artifacts);

        /**
         * <p>
         * A <code>BuildArtifacts</code> object the defines the build artifacts for this batch build.
         * </p>
         * This is a convenience that creates an instance of the {@link BuildArtifacts.Builder} avoiding the need to
         * create one manually via {@link BuildArtifacts#builder()}.
         *
         * When the {@link Consumer} completes, {@link BuildArtifacts.Builder#build()} is called immediately and its
         * result is passed to {@link #artifacts(BuildArtifacts)}.
         * 
         * @param artifacts
         *        a consumer that will call methods on {@link BuildArtifacts.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #artifacts(BuildArtifacts)
         */
        default Builder artifacts(Consumer<BuildArtifacts.Builder> artifacts) {
            return artifacts(BuildArtifacts.builder().applyMutation(artifacts).build());
        }

        /**
         * <p>
         * An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
         * </p>
         * 
         * @param secondaryArtifacts
         *        An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryArtifacts(Collection<BuildArtifacts> secondaryArtifacts);

        /**
         * <p>
         * An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
         * </p>
         * 
         * @param secondaryArtifacts
         *        An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryArtifacts(BuildArtifacts... secondaryArtifacts);

        /**
         * <p>
         * An array of <code>BuildArtifacts</code> objects the define the build artifacts for this batch build.
         * </p>
         * This is a convenience that creates an instance of the {@link List<BuildArtifacts>.Builder} avoiding the need
         * to create one manually via {@link List<BuildArtifacts>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BuildArtifacts>.Builder#build()} is called immediately and
         * its result is passed to {@link #secondaryArtifacts(List<BuildArtifacts>)}.
         * 
         * @param secondaryArtifacts
         *        a consumer that will call methods on {@link List<BuildArtifacts>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secondaryArtifacts(List<BuildArtifacts>)
         */
        Builder secondaryArtifacts(Consumer<BuildArtifacts.Builder>... secondaryArtifacts);

        /**
         * Sets the value of the Cache property for this object.
         *
         * @param cache
         *        The new value for the Cache property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cache(ProjectCache cache);

        /**
         * Sets the value of the Cache property for this object.
         *
         * This is a convenience that creates an instance of the {@link ProjectCache.Builder} avoiding the need to
         * create one manually via {@link ProjectCache#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProjectCache.Builder#build()} is called immediately and its
         * result is passed to {@link #cache(ProjectCache)}.
         * 
         * @param cache
         *        a consumer that will call methods on {@link ProjectCache.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cache(ProjectCache)
         */
        default Builder cache(Consumer<ProjectCache.Builder> cache) {
            return cache(ProjectCache.builder().applyMutation(cache).build());
        }

        /**
         * Sets the value of the Environment property for this object.
         *
         * @param environment
         *        The new value for the Environment property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(ProjectEnvironment environment);

        /**
         * Sets the value of the Environment property for this object.
         *
         * This is a convenience that creates an instance of the {@link ProjectEnvironment.Builder} avoiding the need to
         * create one manually via {@link ProjectEnvironment#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProjectEnvironment.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(ProjectEnvironment)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link ProjectEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(ProjectEnvironment)
         */
        default Builder environment(Consumer<ProjectEnvironment.Builder> environment) {
            return environment(ProjectEnvironment.builder().applyMutation(environment).build());
        }

        /**
         * <p>
         * The name of a service role used for builds in the batch.
         * </p>
         * 
         * @param serviceRole
         *        The name of a service role used for builds in the batch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * Sets the value of the LogConfig property for this object.
         *
         * @param logConfig
         *        The new value for the LogConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfig(LogsConfig logConfig);

        /**
         * Sets the value of the LogConfig property for this object.
         *
         * This is a convenience that creates an instance of the {@link LogsConfig.Builder} avoiding the need to create
         * one manually via {@link LogsConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link LogsConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #logConfig(LogsConfig)}.
         * 
         * @param logConfig
         *        a consumer that will call methods on {@link LogsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfig(LogsConfig)
         */
        default Builder logConfig(Consumer<LogsConfig.Builder> logConfig) {
            return logConfig(LogsConfig.builder().applyMutation(logConfig).build());
        }

        /**
         * <p>
         * Specifies the maximum amount of time, in minutes, that the build in a batch must be completed in.
         * </p>
         * 
         * @param buildTimeoutInMinutes
         *        Specifies the maximum amount of time, in minutes, that the build in a batch must be completed in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildTimeoutInMinutes(Integer buildTimeoutInMinutes);

        /**
         * <p>
         * Specifies the amount of time, in minutes, that the batch build is allowed to be queued before it times out.
         * </p>
         * 
         * @param queuedTimeoutInMinutes
         *        Specifies the amount of time, in minutes, that the batch build is allowed to be queued before it times
         *        out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queuedTimeoutInMinutes(Integer queuedTimeoutInMinutes);

        /**
         * <p>
         * Indicates if the batch build is complete.
         * </p>
         * 
         * @param complete
         *        Indicates if the batch build is complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complete(Boolean complete);

        /**
         * <p>
         * The entity that started the batch build. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If AWS CodePipeline started the build, the pipeline's name (for example,
         * <code>codepipeline/my-demo-pipeline</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * If an AWS Identity and Access Management (IAM) user started the build, the user's name.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the Jenkins plugin for AWS CodeBuild started the build, the string <code>CodeBuild-Jenkins-Plugin</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param initiator
         *        The entity that started the batch build. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If AWS CodePipeline started the build, the pipeline's name (for example,
         *        <code>codepipeline/my-demo-pipeline</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If an AWS Identity and Access Management (IAM) user started the build, the user's name.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the Jenkins plugin for AWS CodeBuild started the build, the string
         *        <code>CodeBuild-Jenkins-Plugin</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initiator(String initiator);

        /**
         * Sets the value of the VpcConfig property for this object.
         *
         * @param vpcConfig
         *        The new value for the VpcConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * Sets the value of the VpcConfig property for this object.
         *
         * This is a convenience that creates an instance of the {@link VpcConfig.Builder} avoiding the need to create
         * one manually via {@link VpcConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * The AWS Key Management Service (AWS KMS) customer master key (CMK) to be used for encrypting the batch build
         * output artifacts.
         * </p>
         * <note>
         * <p>
         * You can use a cross-account KMS key to encrypt the build output artifacts if your service role has permission
         * to that key.
         * </p>
         * </note>
         * <p>
         * You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias (using the
         * format <code>alias/&lt;alias-name&gt;</code>).
         * </p>
         * 
         * @param encryptionKey
         *        The AWS Key Management Service (AWS KMS) customer master key (CMK) to be used for encrypting the batch
         *        build output artifacts.</p> <note>
         *        <p>
         *        You can use a cross-account KMS key to encrypt the build output artifacts if your service role has
         *        permission to that key.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify either the Amazon Resource Name (ARN) of the CMK or, if available, the CMK's alias
         *        (using the format <code>alias/&lt;alias-name&gt;</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKey(String encryptionKey);

        /**
         * <p>
         * The number of the batch build. For each project, the <code>buildBatchNumber</code> of its first batch build
         * is <code>1</code>. The <code>buildBatchNumber</code> of each subsequent batch build is incremented by
         * <code>1</code>. If a batch build is deleted, the <code>buildBatchNumber</code> of other batch builds does not
         * change.
         * </p>
         * 
         * @param buildBatchNumber
         *        The number of the batch build. For each project, the <code>buildBatchNumber</code> of its first batch
         *        build is <code>1</code>. The <code>buildBatchNumber</code> of each subsequent batch build is
         *        incremented by <code>1</code>. If a batch build is deleted, the <code>buildBatchNumber</code> of other
         *        batch builds does not change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildBatchNumber(Long buildBatchNumber);

        /**
         * <p>
         * An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
         * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
         * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using
         * Amazon Elastic File System.
         * </p>
         * 
         * @param fileSystemLocations
         *        An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
         *        <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>,
         *        <code>location</code>, <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a
         *        file system created using Amazon Elastic File System.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemLocations(Collection<ProjectFileSystemLocation> fileSystemLocations);

        /**
         * <p>
         * An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
         * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
         * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using
         * Amazon Elastic File System.
         * </p>
         * 
         * @param fileSystemLocations
         *        An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
         *        <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>,
         *        <code>location</code>, <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a
         *        file system created using Amazon Elastic File System.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemLocations(ProjectFileSystemLocation... fileSystemLocations);

        /**
         * <p>
         * An array of <code>ProjectFileSystemLocation</code> objects for the batch build project. A
         * <code>ProjectFileSystemLocation</code> object specifies the <code>identifier</code>, <code>location</code>,
         * <code>mountOptions</code>, <code>mountPoint</code>, and <code>type</code> of a file system created using
         * Amazon Elastic File System.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ProjectFileSystemLocation>.Builder}
         * avoiding the need to create one manually via {@link List<ProjectFileSystemLocation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProjectFileSystemLocation>.Builder#build()} is called
         * immediately and its result is passed to {@link #fileSystemLocations(List<ProjectFileSystemLocation>)}.
         * 
         * @param fileSystemLocations
         *        a consumer that will call methods on {@link List<ProjectFileSystemLocation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fileSystemLocations(List<ProjectFileSystemLocation>)
         */
        Builder fileSystemLocations(Consumer<ProjectFileSystemLocation.Builder>... fileSystemLocations);

        /**
         * Sets the value of the BuildBatchConfig property for this object.
         *
         * @param buildBatchConfig
         *        The new value for the BuildBatchConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildBatchConfig(ProjectBuildBatchConfig buildBatchConfig);

        /**
         * Sets the value of the BuildBatchConfig property for this object.
         *
         * This is a convenience that creates an instance of the {@link ProjectBuildBatchConfig.Builder} avoiding the
         * need to create one manually via {@link ProjectBuildBatchConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProjectBuildBatchConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #buildBatchConfig(ProjectBuildBatchConfig)}.
         * 
         * @param buildBatchConfig
         *        a consumer that will call methods on {@link ProjectBuildBatchConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #buildBatchConfig(ProjectBuildBatchConfig)
         */
        default Builder buildBatchConfig(Consumer<ProjectBuildBatchConfig.Builder> buildBatchConfig) {
            return buildBatchConfig(ProjectBuildBatchConfig.builder().applyMutation(buildBatchConfig).build());
        }

        /**
         * <p>
         * An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
         * </p>
         * 
         * @param buildGroups
         *        An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildGroups(Collection<BuildGroup> buildGroups);

        /**
         * <p>
         * An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
         * </p>
         * 
         * @param buildGroups
         *        An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildGroups(BuildGroup... buildGroups);

        /**
         * <p>
         * An array of <code>BuildGroup</code> objects that define the build groups for the batch build.
         * </p>
         * This is a convenience that creates an instance of the {@link List<BuildGroup>.Builder} avoiding the need to
         * create one manually via {@link List<BuildGroup>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BuildGroup>.Builder#build()} is called immediately and its
         * result is passed to {@link #buildGroups(List<BuildGroup>)}.
         * 
         * @param buildGroups
         *        a consumer that will call methods on {@link List<BuildGroup>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #buildGroups(List<BuildGroup>)
         */
        Builder buildGroups(Consumer<BuildGroup.Builder>... buildGroups);

        /**
         * <p>
         * Specifies if session debugging is enabled for this batch build. For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/session-manager.html">Viewing a running build in
         * Session Manager</a>. Batch session debugging is not supported for matrix batch builds.
         * </p>
         * 
         * @param debugSessionEnabled
         *        Specifies if session debugging is enabled for this batch build. For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/session-manager.html">Viewing a running
         *        build in Session Manager</a>. Batch session debugging is not supported for matrix batch builds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder debugSessionEnabled(Boolean debugSessionEnabled);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private Instant startTime;

        private Instant endTime;

        private String currentPhase;

        private String buildBatchStatus;

        private String sourceVersion;

        private String resolvedSourceVersion;

        private String projectName;

        private List<BuildBatchPhase> phases = DefaultSdkAutoConstructList.getInstance();

        private ProjectSource source;

        private List<ProjectSource> secondarySources = DefaultSdkAutoConstructList.getInstance();

        private List<ProjectSourceVersion> secondarySourceVersions = DefaultSdkAutoConstructList.getInstance();

        private BuildArtifacts artifacts;

        private List<BuildArtifacts> secondaryArtifacts = DefaultSdkAutoConstructList.getInstance();

        private ProjectCache cache;

        private ProjectEnvironment environment;

        private String serviceRole;

        private LogsConfig logConfig;

        private Integer buildTimeoutInMinutes;

        private Integer queuedTimeoutInMinutes;

        private Boolean complete;

        private String initiator;

        private VpcConfig vpcConfig;

        private String encryptionKey;

        private Long buildBatchNumber;

        private List<ProjectFileSystemLocation> fileSystemLocations = DefaultSdkAutoConstructList.getInstance();

        private ProjectBuildBatchConfig buildBatchConfig;

        private List<BuildGroup> buildGroups = DefaultSdkAutoConstructList.getInstance();

        private Boolean debugSessionEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(BuildBatch model) {
            id(model.id);
            arn(model.arn);
            startTime(model.startTime);
            endTime(model.endTime);
            currentPhase(model.currentPhase);
            buildBatchStatus(model.buildBatchStatus);
            sourceVersion(model.sourceVersion);
            resolvedSourceVersion(model.resolvedSourceVersion);
            projectName(model.projectName);
            phases(model.phases);
            source(model.source);
            secondarySources(model.secondarySources);
            secondarySourceVersions(model.secondarySourceVersions);
            artifacts(model.artifacts);
            secondaryArtifacts(model.secondaryArtifacts);
            cache(model.cache);
            environment(model.environment);
            serviceRole(model.serviceRole);
            logConfig(model.logConfig);
            buildTimeoutInMinutes(model.buildTimeoutInMinutes);
            queuedTimeoutInMinutes(model.queuedTimeoutInMinutes);
            complete(model.complete);
            initiator(model.initiator);
            vpcConfig(model.vpcConfig);
            encryptionKey(model.encryptionKey);
            buildBatchNumber(model.buildBatchNumber);
            fileSystemLocations(model.fileSystemLocations);
            buildBatchConfig(model.buildBatchConfig);
            buildGroups(model.buildGroups);
            debugSessionEnabled(model.debugSessionEnabled);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getCurrentPhase() {
            return currentPhase;
        }

        @Override
        public final Builder currentPhase(String currentPhase) {
            this.currentPhase = currentPhase;
            return this;
        }

        public final void setCurrentPhase(String currentPhase) {
            this.currentPhase = currentPhase;
        }

        public final String getBuildBatchStatus() {
            return buildBatchStatus;
        }

        @Override
        public final Builder buildBatchStatus(String buildBatchStatus) {
            this.buildBatchStatus = buildBatchStatus;
            return this;
        }

        @Override
        public final Builder buildBatchStatus(StatusType buildBatchStatus) {
            this.buildBatchStatus(buildBatchStatus == null ? null : buildBatchStatus.toString());
            return this;
        }

        public final void setBuildBatchStatus(String buildBatchStatus) {
            this.buildBatchStatus = buildBatchStatus;
        }

        public final String getSourceVersion() {
            return sourceVersion;
        }

        @Override
        public final Builder sourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
            return this;
        }

        public final void setSourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
        }

        public final String getResolvedSourceVersion() {
            return resolvedSourceVersion;
        }

        @Override
        public final Builder resolvedSourceVersion(String resolvedSourceVersion) {
            this.resolvedSourceVersion = resolvedSourceVersion;
            return this;
        }

        public final void setResolvedSourceVersion(String resolvedSourceVersion) {
            this.resolvedSourceVersion = resolvedSourceVersion;
        }

        public final String getProjectName() {
            return projectName;
        }

        @Override
        public final Builder projectName(String projectName) {
            this.projectName = projectName;
            return this;
        }

        public final void setProjectName(String projectName) {
            this.projectName = projectName;
        }

        public final List<BuildBatchPhase.Builder> getPhases() {
            List<BuildBatchPhase.Builder> result = BuildBatchPhasesCopier.copyToBuilder(this.phases);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder phases(Collection<BuildBatchPhase> phases) {
            this.phases = BuildBatchPhasesCopier.copy(phases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder phases(BuildBatchPhase... phases) {
            phases(Arrays.asList(phases));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder phases(Consumer<BuildBatchPhase.Builder>... phases) {
            phases(Stream.of(phases).map(c -> BuildBatchPhase.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPhases(Collection<BuildBatchPhase.BuilderImpl> phases) {
            this.phases = BuildBatchPhasesCopier.copyFromBuilder(phases);
        }

        public final ProjectSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        @Override
        public final Builder source(ProjectSource source) {
            this.source = source;
            return this;
        }

        public final void setSource(ProjectSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        public final List<ProjectSource.Builder> getSecondarySources() {
            List<ProjectSource.Builder> result = ProjectSourcesCopier.copyToBuilder(this.secondarySources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder secondarySources(Collection<ProjectSource> secondarySources) {
            this.secondarySources = ProjectSourcesCopier.copy(secondarySources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySources(ProjectSource... secondarySources) {
            secondarySources(Arrays.asList(secondarySources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySources(Consumer<ProjectSource.Builder>... secondarySources) {
            secondarySources(Stream.of(secondarySources).map(c -> ProjectSource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSecondarySources(Collection<ProjectSource.BuilderImpl> secondarySources) {
            this.secondarySources = ProjectSourcesCopier.copyFromBuilder(secondarySources);
        }

        public final List<ProjectSourceVersion.Builder> getSecondarySourceVersions() {
            List<ProjectSourceVersion.Builder> result = ProjectSecondarySourceVersionsCopier
                    .copyToBuilder(this.secondarySourceVersions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder secondarySourceVersions(Collection<ProjectSourceVersion> secondarySourceVersions) {
            this.secondarySourceVersions = ProjectSecondarySourceVersionsCopier.copy(secondarySourceVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySourceVersions(ProjectSourceVersion... secondarySourceVersions) {
            secondarySourceVersions(Arrays.asList(secondarySourceVersions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondarySourceVersions(Consumer<ProjectSourceVersion.Builder>... secondarySourceVersions) {
            secondarySourceVersions(Stream.of(secondarySourceVersions)
                    .map(c -> ProjectSourceVersion.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSecondarySourceVersions(Collection<ProjectSourceVersion.BuilderImpl> secondarySourceVersions) {
            this.secondarySourceVersions = ProjectSecondarySourceVersionsCopier.copyFromBuilder(secondarySourceVersions);
        }

        public final BuildArtifacts.Builder getArtifacts() {
            return artifacts != null ? artifacts.toBuilder() : null;
        }

        @Override
        public final Builder artifacts(BuildArtifacts artifacts) {
            this.artifacts = artifacts;
            return this;
        }

        public final void setArtifacts(BuildArtifacts.BuilderImpl artifacts) {
            this.artifacts = artifacts != null ? artifacts.build() : null;
        }

        public final List<BuildArtifacts.Builder> getSecondaryArtifacts() {
            List<BuildArtifacts.Builder> result = BuildArtifactsListCopier.copyToBuilder(this.secondaryArtifacts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder secondaryArtifacts(Collection<BuildArtifacts> secondaryArtifacts) {
            this.secondaryArtifacts = BuildArtifactsListCopier.copy(secondaryArtifacts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondaryArtifacts(BuildArtifacts... secondaryArtifacts) {
            secondaryArtifacts(Arrays.asList(secondaryArtifacts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secondaryArtifacts(Consumer<BuildArtifacts.Builder>... secondaryArtifacts) {
            secondaryArtifacts(Stream.of(secondaryArtifacts).map(c -> BuildArtifacts.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSecondaryArtifacts(Collection<BuildArtifacts.BuilderImpl> secondaryArtifacts) {
            this.secondaryArtifacts = BuildArtifactsListCopier.copyFromBuilder(secondaryArtifacts);
        }

        public final ProjectCache.Builder getCache() {
            return cache != null ? cache.toBuilder() : null;
        }

        @Override
        public final Builder cache(ProjectCache cache) {
            this.cache = cache;
            return this;
        }

        public final void setCache(ProjectCache.BuilderImpl cache) {
            this.cache = cache != null ? cache.build() : null;
        }

        public final ProjectEnvironment.Builder getEnvironment() {
            return environment != null ? environment.toBuilder() : null;
        }

        @Override
        public final Builder environment(ProjectEnvironment environment) {
            this.environment = environment;
            return this;
        }

        public final void setEnvironment(ProjectEnvironment.BuilderImpl environment) {
            this.environment = environment != null ? environment.build() : null;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        public final LogsConfig.Builder getLogConfig() {
            return logConfig != null ? logConfig.toBuilder() : null;
        }

        @Override
        public final Builder logConfig(LogsConfig logConfig) {
            this.logConfig = logConfig;
            return this;
        }

        public final void setLogConfig(LogsConfig.BuilderImpl logConfig) {
            this.logConfig = logConfig != null ? logConfig.build() : null;
        }

        public final Integer getBuildTimeoutInMinutes() {
            return buildTimeoutInMinutes;
        }

        @Override
        public final Builder buildTimeoutInMinutes(Integer buildTimeoutInMinutes) {
            this.buildTimeoutInMinutes = buildTimeoutInMinutes;
            return this;
        }

        public final void setBuildTimeoutInMinutes(Integer buildTimeoutInMinutes) {
            this.buildTimeoutInMinutes = buildTimeoutInMinutes;
        }

        public final Integer getQueuedTimeoutInMinutes() {
            return queuedTimeoutInMinutes;
        }

        @Override
        public final Builder queuedTimeoutInMinutes(Integer queuedTimeoutInMinutes) {
            this.queuedTimeoutInMinutes = queuedTimeoutInMinutes;
            return this;
        }

        public final void setQueuedTimeoutInMinutes(Integer queuedTimeoutInMinutes) {
            this.queuedTimeoutInMinutes = queuedTimeoutInMinutes;
        }

        public final Boolean getComplete() {
            return complete;
        }

        @Override
        public final Builder complete(Boolean complete) {
            this.complete = complete;
            return this;
        }

        public final void setComplete(Boolean complete) {
            this.complete = complete;
        }

        public final String getInitiator() {
            return initiator;
        }

        @Override
        public final Builder initiator(String initiator) {
            this.initiator = initiator;
            return this;
        }

        public final void setInitiator(String initiator) {
            this.initiator = initiator;
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        public final String getEncryptionKey() {
            return encryptionKey;
        }

        @Override
        public final Builder encryptionKey(String encryptionKey) {
            this.encryptionKey = encryptionKey;
            return this;
        }

        public final void setEncryptionKey(String encryptionKey) {
            this.encryptionKey = encryptionKey;
        }

        public final Long getBuildBatchNumber() {
            return buildBatchNumber;
        }

        @Override
        public final Builder buildBatchNumber(Long buildBatchNumber) {
            this.buildBatchNumber = buildBatchNumber;
            return this;
        }

        public final void setBuildBatchNumber(Long buildBatchNumber) {
            this.buildBatchNumber = buildBatchNumber;
        }

        public final List<ProjectFileSystemLocation.Builder> getFileSystemLocations() {
            List<ProjectFileSystemLocation.Builder> result = ProjectFileSystemLocationsCopier
                    .copyToBuilder(this.fileSystemLocations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder fileSystemLocations(Collection<ProjectFileSystemLocation> fileSystemLocations) {
            this.fileSystemLocations = ProjectFileSystemLocationsCopier.copy(fileSystemLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fileSystemLocations(ProjectFileSystemLocation... fileSystemLocations) {
            fileSystemLocations(Arrays.asList(fileSystemLocations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fileSystemLocations(Consumer<ProjectFileSystemLocation.Builder>... fileSystemLocations) {
            fileSystemLocations(Stream.of(fileSystemLocations)
                    .map(c -> ProjectFileSystemLocation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setFileSystemLocations(Collection<ProjectFileSystemLocation.BuilderImpl> fileSystemLocations) {
            this.fileSystemLocations = ProjectFileSystemLocationsCopier.copyFromBuilder(fileSystemLocations);
        }

        public final ProjectBuildBatchConfig.Builder getBuildBatchConfig() {
            return buildBatchConfig != null ? buildBatchConfig.toBuilder() : null;
        }

        @Override
        public final Builder buildBatchConfig(ProjectBuildBatchConfig buildBatchConfig) {
            this.buildBatchConfig = buildBatchConfig;
            return this;
        }

        public final void setBuildBatchConfig(ProjectBuildBatchConfig.BuilderImpl buildBatchConfig) {
            this.buildBatchConfig = buildBatchConfig != null ? buildBatchConfig.build() : null;
        }

        public final List<BuildGroup.Builder> getBuildGroups() {
            List<BuildGroup.Builder> result = BuildGroupsCopier.copyToBuilder(this.buildGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder buildGroups(Collection<BuildGroup> buildGroups) {
            this.buildGroups = BuildGroupsCopier.copy(buildGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder buildGroups(BuildGroup... buildGroups) {
            buildGroups(Arrays.asList(buildGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder buildGroups(Consumer<BuildGroup.Builder>... buildGroups) {
            buildGroups(Stream.of(buildGroups).map(c -> BuildGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setBuildGroups(Collection<BuildGroup.BuilderImpl> buildGroups) {
            this.buildGroups = BuildGroupsCopier.copyFromBuilder(buildGroups);
        }

        public final Boolean getDebugSessionEnabled() {
            return debugSessionEnabled;
        }

        @Override
        public final Builder debugSessionEnabled(Boolean debugSessionEnabled) {
            this.debugSessionEnabled = debugSessionEnabled;
            return this;
        }

        public final void setDebugSessionEnabled(Boolean debugSessionEnabled) {
            this.debugSessionEnabled = debugSessionEnabled;
        }

        @Override
        public BuildBatch build() {
            return new BuildBatch(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
