/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a Docker image that is managed by CodeBuild.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EnvironmentImage implements SdkPojo, Serializable,
        ToCopyableBuilder<EnvironmentImage.Builder, EnvironmentImage> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(EnvironmentImage::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(EnvironmentImage::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("versions")
            .getter(getter(EnvironmentImage::versions))
            .setter(setter(Builder::versions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            VERSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final List<String> versions;

    private EnvironmentImage(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.versions = builder.versions;
    }

    /**
     * <p>
     * The name of the Docker image.
     * </p>
     * 
     * @return The name of the Docker image.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the Docker image.
     * </p>
     * 
     * @return The description of the Docker image.
     */
    public final String description() {
        return description;
    }

    /**
     * Returns true if the Versions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasVersions() {
        return versions != null && !(versions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of environment image versions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVersions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of environment image versions.
     */
    public final List<String> versions() {
        return versions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasVersions() ? versions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EnvironmentImage)) {
            return false;
        }
        EnvironmentImage other = (EnvironmentImage) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && hasVersions() == other.hasVersions() && Objects.equals(versions(), other.versions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EnvironmentImage").add("Name", name()).add("Description", description())
                .add("Versions", hasVersions() ? versions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "versions":
            return Optional.ofNullable(clazz.cast(versions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EnvironmentImage, T> g) {
        return obj -> g.apply((EnvironmentImage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EnvironmentImage> {
        /**
         * <p>
         * The name of the Docker image.
         * </p>
         * 
         * @param name
         *        The name of the Docker image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the Docker image.
         * </p>
         * 
         * @param description
         *        The description of the Docker image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of environment image versions.
         * </p>
         * 
         * @param versions
         *        A list of environment image versions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versions(Collection<String> versions);

        /**
         * <p>
         * A list of environment image versions.
         * </p>
         * 
         * @param versions
         *        A list of environment image versions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versions(String... versions);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private List<String> versions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EnvironmentImage model) {
            name(model.name);
            description(model.description);
            versions(model.versions);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getVersions() {
            if (versions instanceof SdkAutoConstructList) {
                return null;
            }
            return versions;
        }

        @Override
        public final Builder versions(Collection<String> versions) {
            this.versions = ImageVersionsCopier.copy(versions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder versions(String... versions) {
            versions(Arrays.asList(versions));
            return this;
        }

        public final void setVersions(Collection<String> versions) {
            this.versions = ImageVersionsCopier.copy(versions);
        }

        @Override
        public EnvironmentImage build() {
            return new EnvironmentImage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
