/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the build environment of the build project.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectEnvironment implements SdkPojo, Serializable,
        ToCopyableBuilder<ProjectEnvironment.Builder, ProjectEnvironment> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ProjectEnvironment::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(ProjectEnvironment::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<String> COMPUTE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("computeType").getter(getter(ProjectEnvironment::computeTypeAsString))
            .setter(setter(Builder::computeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeType").build()).build();

    private static final SdkField<ProjectFleet> FLEET_FIELD = SdkField.<ProjectFleet> builder(MarshallingType.SDK_POJO)
            .memberName("fleet").getter(getter(ProjectEnvironment::fleet)).setter(setter(Builder::fleet))
            .constructor(ProjectFleet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fleet").build()).build();

    private static final SdkField<List<EnvironmentVariable>> ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<List<EnvironmentVariable>> builder(MarshallingType.LIST)
            .memberName("environmentVariables")
            .getter(getter(ProjectEnvironment::environmentVariables))
            .setter(setter(Builder::environmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentVariables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EnvironmentVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(EnvironmentVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PRIVILEGED_MODE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("privilegedMode").getter(getter(ProjectEnvironment::privilegedMode))
            .setter(setter(Builder::privilegedMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privilegedMode").build()).build();

    private static final SdkField<String> CERTIFICATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificate").getter(getter(ProjectEnvironment::certificate)).setter(setter(Builder::certificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificate").build()).build();

    private static final SdkField<RegistryCredential> REGISTRY_CREDENTIAL_FIELD = SdkField
            .<RegistryCredential> builder(MarshallingType.SDK_POJO).memberName("registryCredential")
            .getter(getter(ProjectEnvironment::registryCredential)).setter(setter(Builder::registryCredential))
            .constructor(RegistryCredential::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registryCredential").build())
            .build();

    private static final SdkField<String> IMAGE_PULL_CREDENTIALS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imagePullCredentialsType").getter(getter(ProjectEnvironment::imagePullCredentialsTypeAsString))
            .setter(setter(Builder::imagePullCredentialsType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imagePullCredentialsType").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, IMAGE_FIELD,
            COMPUTE_TYPE_FIELD, FLEET_FIELD, ENVIRONMENT_VARIABLES_FIELD, PRIVILEGED_MODE_FIELD, CERTIFICATE_FIELD,
            REGISTRY_CREDENTIAL_FIELD, IMAGE_PULL_CREDENTIALS_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String image;

    private final String computeType;

    private final ProjectFleet fleet;

    private final List<EnvironmentVariable> environmentVariables;

    private final Boolean privilegedMode;

    private final String certificate;

    private final RegistryCredential registryCredential;

    private final String imagePullCredentialsType;

    private ProjectEnvironment(BuilderImpl builder) {
        this.type = builder.type;
        this.image = builder.image;
        this.computeType = builder.computeType;
        this.fleet = builder.fleet;
        this.environmentVariables = builder.environmentVariables;
        this.privilegedMode = builder.privilegedMode;
        this.certificate = builder.certificate;
        this.registryCredential = builder.registryCredential;
        this.imagePullCredentialsType = builder.imagePullCredentialsType;
    }

    /**
     * <p>
     * The type of build environment to use for related builds.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US East
     * (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific (Sydney), and
     * EU (Frankfurt).
     * </p>
     * </li>
     * <li>
     * <p>
     * The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia), US East
     * (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific (Tokyo), Asia
     * Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China (Beijing), and China (Ningxia).
     * </p>
     * </li>
     * <li>
     * <p>
     * The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N. Virginia), US East
     * (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific (Tokyo), Asia
     * Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney) , China (Beijing), and China (Ningxia).
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are available
     * only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Mumbai), Asia Pacific
     * (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU (Ireland), and South America (São
     * Paulo).
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code> are available
     * only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU (Ireland).
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you're using compute fleets during project creation, <code>type</code> will be ignored.
     * </p>
     * </note>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build environment
     * compute types</a> in the <i>CodeBuild user guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EnvironmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of build environment to use for related builds.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US
     *         East (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific
     *         (Sydney), and EU (Frankfurt).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia), US
     *         East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific
     *         (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China (Beijing), and
     *         China (Ningxia).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N. Virginia),
     *         US East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia
     *         Pacific (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney) , China (Beijing),
     *         and China (Ningxia).
     *         </p>
     *         </li>
     *         </ul>
     *         <ul>
     *         <li>
     *         <p>
     *         The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are
     *         available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Mumbai),
     *         Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU (Ireland), and
     *         South America (São Paulo).
     *         </p>
     *         </li>
     *         </ul>
     *         <ul>
     *         <li>
     *         <p>
     *         The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code> are
     *         available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU (Ireland).
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you're using compute fleets during project creation, <code>type</code> will be ignored.
     *         </p>
     *         </note>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
     *         environment compute types</a> in the <i>CodeBuild user guide</i>.
     * @see EnvironmentType
     */
    public final EnvironmentType type() {
        return EnvironmentType.fromValue(type);
    }

    /**
     * <p>
     * The type of build environment to use for related builds.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US East
     * (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific (Sydney), and
     * EU (Frankfurt).
     * </p>
     * </li>
     * <li>
     * <p>
     * The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia), US East
     * (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific (Tokyo), Asia
     * Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China (Beijing), and China (Ningxia).
     * </p>
     * </li>
     * <li>
     * <p>
     * The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N. Virginia), US East
     * (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific (Tokyo), Asia
     * Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney) , China (Beijing), and China (Ningxia).
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are available
     * only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Mumbai), Asia Pacific
     * (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU (Ireland), and South America (São
     * Paulo).
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code> are available
     * only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU (Ireland).
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you're using compute fleets during project creation, <code>type</code> will be ignored.
     * </p>
     * </note>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build environment
     * compute types</a> in the <i>CodeBuild user guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link EnvironmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of build environment to use for related builds.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US
     *         East (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific
     *         (Sydney), and EU (Frankfurt).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia), US
     *         East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific
     *         (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China (Beijing), and
     *         China (Ningxia).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N. Virginia),
     *         US East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia
     *         Pacific (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney) , China (Beijing),
     *         and China (Ningxia).
     *         </p>
     *         </li>
     *         </ul>
     *         <ul>
     *         <li>
     *         <p>
     *         The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are
     *         available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Mumbai),
     *         Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU (Ireland), and
     *         South America (São Paulo).
     *         </p>
     *         </li>
     *         </ul>
     *         <ul>
     *         <li>
     *         <p>
     *         The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code> are
     *         available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU (Ireland).
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you're using compute fleets during project creation, <code>type</code> will be ignored.
     *         </p>
     *         </note>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
     *         environment compute types</a> in the <i>CodeBuild user guide</i>.
     * @see EnvironmentType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The image tag or image digest that identifies the Docker image to use for this build project. Use the following
     * formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For an image tag: <code>&lt;registry&gt;/&lt;repository&gt;:&lt;tag&gt;</code>. For example, in the Docker
     * repository that CodeBuild uses to manage its Docker images, this would be <code>aws/codebuild/standard:4.0</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * For an image digest: <code>&lt;registry&gt;/&lt;repository&gt;@&lt;digest&gt;</code>. For example, to specify an
     * image with the digest "sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf," use
     * <code>&lt;registry&gt;/&lt;repository&gt;@sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf</code>
     * .
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html">Docker images provided
     * by CodeBuild</a> in the <i>CodeBuild user guide</i>.
     * </p>
     * 
     * @return The image tag or image digest that identifies the Docker image to use for this build project. Use the
     *         following formats:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For an image tag: <code>&lt;registry&gt;/&lt;repository&gt;:&lt;tag&gt;</code>. For example, in the
     *         Docker repository that CodeBuild uses to manage its Docker images, this would be
     *         <code>aws/codebuild/standard:4.0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For an image digest: <code>&lt;registry&gt;/&lt;repository&gt;@&lt;digest&gt;</code>. For example, to
     *         specify an image with the digest
     *         "sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf," use
     *         <code>&lt;registry&gt;/&lt;repository&gt;@sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf</code>
     *         .
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html">Docker images
     *         provided by CodeBuild</a> in the <i>CodeBuild user guide</i>.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * Information about the compute resources the build project uses. Available values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your environment
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your environment
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for builds.
     * This compute type supports Docker images up to 100 GB uncompressed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you use <code>BUILD_GENERAL1_SMALL</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1 NVIDIA A10G
     * Tensor Core GPU for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on ARM-based
     * processors for builds.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you use <code>BUILD_GENERAL1_LARGE</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and 4 NVIDIA
     * Tesla V100 GPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on ARM-based
     * processors for builds.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
     * </p>
     * </note>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build Environment
     * Compute Types</a> in the <i>CodeBuild User Guide.</i>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeType} will
     * return {@link ComputeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeTypeAsString}.
     * </p>
     * 
     * @return Information about the compute resources the build project uses. Available values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your
     *         environment type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your
     *         environment type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for
     *         builds. This compute type supports Docker images up to 100 GB uncompressed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you use <code>BUILD_GENERAL1_SMALL</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1
     *         NVIDIA A10G Tensor Core GPU for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on ARM-based
     *         processors for builds.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you use <code>BUILD_GENERAL1_LARGE</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and 4
     *         NVIDIA Tesla V100 GPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on ARM-based
     *         processors for builds.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
     *         </p>
     *         </note>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
     *         Environment Compute Types</a> in the <i>CodeBuild User Guide.</i>
     * @see ComputeType
     */
    public final ComputeType computeType() {
        return ComputeType.fromValue(computeType);
    }

    /**
     * <p>
     * Information about the compute resources the build project uses. Available values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your environment
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your environment
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for builds.
     * This compute type supports Docker images up to 100 GB uncompressed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
     * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you use <code>BUILD_GENERAL1_SMALL</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1 NVIDIA A10G
     * Tensor Core GPU for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on ARM-based
     * processors for builds.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you use <code>BUILD_GENERAL1_LARGE</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and 4 NVIDIA
     * Tesla V100 GPUs for builds.
     * </p>
     * </li>
     * <li>
     * <p>
     * For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on ARM-based
     * processors for builds.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
     * </p>
     * </note>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build Environment
     * Compute Types</a> in the <i>CodeBuild User Guide.</i>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeType} will
     * return {@link ComputeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeTypeAsString}.
     * </p>
     * 
     * @return Information about the compute resources the build project uses. Available values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your
     *         environment type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your
     *         environment type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for
     *         builds. This compute type supports Docker images up to 100 GB uncompressed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
     *         <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you use <code>BUILD_GENERAL1_SMALL</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1
     *         NVIDIA A10G Tensor Core GPU for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on ARM-based
     *         processors for builds.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you use <code>BUILD_GENERAL1_LARGE</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and 4
     *         NVIDIA Tesla V100 GPUs for builds.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on ARM-based
     *         processors for builds.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
     *         </p>
     *         </note>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
     *         Environment Compute Types</a> in the <i>CodeBuild User Guide.</i>
     * @see ComputeType
     */
    public final String computeTypeAsString() {
        return computeType;
    }

    /**
     * <p>
     * A ProjectFleet object to use for this build project.
     * </p>
     * 
     * @return A ProjectFleet object to use for this build project.
     */
    public final ProjectFleet fleet() {
        return fleet;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentVariables property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentVariables() {
        return environmentVariables != null && !(environmentVariables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of environment variables to make available to builds for this build project.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentVariables} method.
     * </p>
     * 
     * @return A set of environment variables to make available to builds for this build project.
     */
    public final List<EnvironmentVariable> environmentVariables() {
        return environmentVariables;
    }

    /**
     * <p>
     * Enables running the Docker daemon inside a Docker container. Set to true only if the build project is used to
     * build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails. The default
     * setting is <code>false</code>.
     * </p>
     * <p>
     * You can initialize the Docker daemon during the install phase of your build by adding one of the following sets
     * of commands to the install phase of your buildspec file:
     * </p>
     * <p>
     * If the operating system's base image is Ubuntu Linux:
     * </p>
     * <p>
     * <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
     * </p>
     * <p>
     * <code>- timeout 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
     * </p>
     * <p>
     * If the operating system's base image is Alpine Linux and the previous command does not work, add the
     * <code>-t</code> argument to <code>timeout</code>:
     * </p>
     * <p>
     * <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
     * </p>
     * <p>
     * <code>- timeout -t 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
     * </p>
     * 
     * @return Enables running the Docker daemon inside a Docker container. Set to true only if the build project is
     *         used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.
     *         The default setting is <code>false</code>.</p>
     *         <p>
     *         You can initialize the Docker daemon during the install phase of your build by adding one of the
     *         following sets of commands to the install phase of your buildspec file:
     *         </p>
     *         <p>
     *         If the operating system's base image is Ubuntu Linux:
     *         </p>
     *         <p>
     *         <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
     *         </p>
     *         <p>
     *         <code>- timeout 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
     *         </p>
     *         <p>
     *         If the operating system's base image is Alpine Linux and the previous command does not work, add the
     *         <code>-t</code> argument to <code>timeout</code>:
     *         </p>
     *         <p>
     *         <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
     *         </p>
     *         <p>
     *         <code>- timeout -t 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
     */
    public final Boolean privilegedMode() {
        return privilegedMode;
    }

    /**
     * <p>
     * The ARN of the Amazon S3 bucket, path prefix, and object key that contains the PEM-encoded certificate for the
     * build project. For more information, see <a href=
     * "https://docs.aws.amazon.com/codebuild/latest/userguide/create-project-cli.html#cli.environment.certificate"
     * >certificate</a> in the <i>CodeBuild User Guide</i>.
     * </p>
     * 
     * @return The ARN of the Amazon S3 bucket, path prefix, and object key that contains the PEM-encoded certificate
     *         for the build project. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codebuild/latest/userguide/create-project-cli.html#cli.environment.certificate"
     *         >certificate</a> in the <i>CodeBuild User Guide</i>.
     */
    public final String certificate() {
        return certificate;
    }

    /**
     * <p>
     * The credentials for access to a private registry.
     * </p>
     * 
     * @return The credentials for access to a private registry.
     */
    public final RegistryCredential registryCredential() {
        return registryCredential;
    }

    /**
     * <p>
     * The type of credentials CodeBuild uses to pull images in your build. There are two valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you modify your ECR
     * repository policy to trust CodeBuild service principal.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you use an
     * CodeBuild curated image, you must use CODEBUILD credentials.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #imagePullCredentialsType} will return {@link ImagePullCredentialsType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #imagePullCredentialsTypeAsString}.
     * </p>
     * 
     * @return The type of credentials CodeBuild uses to pull images in your build. There are two valid values: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you modify
     *         your ECR repository policy to trust CodeBuild service principal.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you
     *         use an CodeBuild curated image, you must use CODEBUILD credentials.
     * @see ImagePullCredentialsType
     */
    public final ImagePullCredentialsType imagePullCredentialsType() {
        return ImagePullCredentialsType.fromValue(imagePullCredentialsType);
    }

    /**
     * <p>
     * The type of credentials CodeBuild uses to pull images in your build. There are two valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you modify your ECR
     * repository policy to trust CodeBuild service principal.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you use an
     * CodeBuild curated image, you must use CODEBUILD credentials.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #imagePullCredentialsType} will return {@link ImagePullCredentialsType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #imagePullCredentialsTypeAsString}.
     * </p>
     * 
     * @return The type of credentials CodeBuild uses to pull images in your build. There are two valid values: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you modify
     *         your ECR repository policy to trust CodeBuild service principal.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you
     *         use an CodeBuild curated image, you must use CODEBUILD credentials.
     * @see ImagePullCredentialsType
     */
    public final String imagePullCredentialsTypeAsString() {
        return imagePullCredentialsType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(computeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(fleet());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentVariables() ? environmentVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(privilegedMode());
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        hashCode = 31 * hashCode + Objects.hashCode(registryCredential());
        hashCode = 31 * hashCode + Objects.hashCode(imagePullCredentialsTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectEnvironment)) {
            return false;
        }
        ProjectEnvironment other = (ProjectEnvironment) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(image(), other.image())
                && Objects.equals(computeTypeAsString(), other.computeTypeAsString()) && Objects.equals(fleet(), other.fleet())
                && hasEnvironmentVariables() == other.hasEnvironmentVariables()
                && Objects.equals(environmentVariables(), other.environmentVariables())
                && Objects.equals(privilegedMode(), other.privilegedMode()) && Objects.equals(certificate(), other.certificate())
                && Objects.equals(registryCredential(), other.registryCredential())
                && Objects.equals(imagePullCredentialsTypeAsString(), other.imagePullCredentialsTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProjectEnvironment").add("Type", typeAsString()).add("Image", image())
                .add("ComputeType", computeTypeAsString()).add("Fleet", fleet())
                .add("EnvironmentVariables", hasEnvironmentVariables() ? environmentVariables() : null)
                .add("PrivilegedMode", privilegedMode()).add("Certificate", certificate())
                .add("RegistryCredential", registryCredential())
                .add("ImagePullCredentialsType", imagePullCredentialsTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "computeType":
            return Optional.ofNullable(clazz.cast(computeTypeAsString()));
        case "fleet":
            return Optional.ofNullable(clazz.cast(fleet()));
        case "environmentVariables":
            return Optional.ofNullable(clazz.cast(environmentVariables()));
        case "privilegedMode":
            return Optional.ofNullable(clazz.cast(privilegedMode()));
        case "certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        case "registryCredential":
            return Optional.ofNullable(clazz.cast(registryCredential()));
        case "imagePullCredentialsType":
            return Optional.ofNullable(clazz.cast(imagePullCredentialsTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectEnvironment, T> g) {
        return obj -> g.apply((ProjectEnvironment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectEnvironment> {
        /**
         * <p>
         * The type of build environment to use for related builds.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US East
         * (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific (Sydney),
         * and EU (Frankfurt).
         * </p>
         * </li>
         * <li>
         * <p>
         * The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia), US East
         * (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific (Tokyo),
         * Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China (Beijing), and China (Ningxia).
         * </p>
         * </li>
         * <li>
         * <p>
         * The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N. Virginia), US
         * East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific
         * (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney) , China (Beijing), and China
         * (Ningxia).
         * </p>
         * </li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are available
         * only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Mumbai), Asia Pacific
         * (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU (Ireland), and South America
         * (São Paulo).
         * </p>
         * </li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code> are
         * available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU (Ireland).
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you're using compute fleets during project creation, <code>type</code> will be ignored.
         * </p>
         * </note>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         * environment compute types</a> in the <i>CodeBuild user guide</i>.
         * </p>
         * 
         * @param type
         *        The type of build environment to use for related builds.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US
         *        East (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific
         *        (Sydney), and EU (Frankfurt).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia),
         *        US East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia
         *        Pacific (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China
         *        (Beijing), and China (Ningxia).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N.
         *        Virginia), US East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU
         *        (Frankfurt), Asia Pacific (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific
         *        (Sydney) , China (Beijing), and China (Ningxia).
         *        </p>
         *        </li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are
         *        available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific
         *        (Mumbai), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU
         *        (Ireland), and South America (São Paulo).
         *        </p>
         *        </li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code>
         *        are available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU
         *        (Ireland).
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you're using compute fleets during project creation, <code>type</code> will be ignored.
         *        </p>
         *        </note>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         *        environment compute types</a> in the <i>CodeBuild user guide</i>.
         * @see EnvironmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of build environment to use for related builds.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US East
         * (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific (Sydney),
         * and EU (Frankfurt).
         * </p>
         * </li>
         * <li>
         * <p>
         * The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia), US East
         * (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific (Tokyo),
         * Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China (Beijing), and China (Ningxia).
         * </p>
         * </li>
         * <li>
         * <p>
         * The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N. Virginia), US
         * East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia Pacific
         * (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney) , China (Beijing), and China
         * (Ningxia).
         * </p>
         * </li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are available
         * only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific (Mumbai), Asia Pacific
         * (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU (Ireland), and South America
         * (São Paulo).
         * </p>
         * </li>
         * </ul>
         * <ul>
         * <li>
         * <p>
         * The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code> are
         * available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU (Ireland).
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you're using compute fleets during project creation, <code>type</code> will be ignored.
         * </p>
         * </note>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         * environment compute types</a> in the <i>CodeBuild user guide</i>.
         * </p>
         * 
         * @param type
         *        The type of build environment to use for related builds.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The environment type <code>ARM_CONTAINER</code> is available only in regions US East (N. Virginia), US
         *        East (Ohio), US West (Oregon), EU (Ireland), Asia Pacific (Mumbai), Asia Pacific (Tokyo), Asia Pacific
         *        (Sydney), and EU (Frankfurt).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The environment type <code>LINUX_CONTAINER</code> is available only in regions US East (N. Virginia),
         *        US East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU (Frankfurt), Asia
         *        Pacific (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific (Sydney), China
         *        (Beijing), and China (Ningxia).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The environment type <code>LINUX_GPU_CONTAINER</code> is available only in regions US East (N.
         *        Virginia), US East (Ohio), US West (Oregon), Canada (Central), EU (Ireland), EU (London), EU
         *        (Frankfurt), Asia Pacific (Tokyo), Asia Pacific (Seoul), Asia Pacific (Singapore), Asia Pacific
         *        (Sydney) , China (Beijing), and China (Ningxia).
         *        </p>
         *        </li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        The environment types <code>ARM_LAMBDA_CONTAINER</code> and <code>LINUX_LAMBDA_CONTAINER</code> are
         *        available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), Asia Pacific
         *        (Mumbai), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), EU (Frankfurt), EU
         *        (Ireland), and South America (São Paulo).
         *        </p>
         *        </li>
         *        </ul>
         *        <ul>
         *        <li>
         *        <p>
         *        The environment types <code>WINDOWS_CONTAINER</code> and <code>WINDOWS_SERVER_2019_CONTAINER</code>
         *        are available only in regions US East (N. Virginia), US East (Ohio), US West (Oregon), and EU
         *        (Ireland).
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you're using compute fleets during project creation, <code>type</code> will be ignored.
         *        </p>
         *        </note>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         *        environment compute types</a> in the <i>CodeBuild user guide</i>.
         * @see EnvironmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EnvironmentType
         */
        Builder type(EnvironmentType type);

        /**
         * <p>
         * The image tag or image digest that identifies the Docker image to use for this build project. Use the
         * following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For an image tag: <code>&lt;registry&gt;/&lt;repository&gt;:&lt;tag&gt;</code>. For example, in the Docker
         * repository that CodeBuild uses to manage its Docker images, this would be
         * <code>aws/codebuild/standard:4.0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For an image digest: <code>&lt;registry&gt;/&lt;repository&gt;@&lt;digest&gt;</code>. For example, to specify
         * an image with the digest "sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf," use
         * <code>&lt;registry&gt;/&lt;repository&gt;@sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf</code>
         * .
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html">Docker images
         * provided by CodeBuild</a> in the <i>CodeBuild user guide</i>.
         * </p>
         * 
         * @param image
         *        The image tag or image digest that identifies the Docker image to use for this build project. Use the
         *        following formats:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For an image tag: <code>&lt;registry&gt;/&lt;repository&gt;:&lt;tag&gt;</code>. For example, in the
         *        Docker repository that CodeBuild uses to manage its Docker images, this would be
         *        <code>aws/codebuild/standard:4.0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For an image digest: <code>&lt;registry&gt;/&lt;repository&gt;@&lt;digest&gt;</code>. For example, to
         *        specify an image with the digest
         *        "sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf," use
         *        <code>&lt;registry&gt;/&lt;repository&gt;@sha256:cbbf2f9a99b47fc460d422812b6a5adff7dfee951d8fa2e4a98caa0382cfbdbf</code>
         *        .
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html">Docker
         *        images provided by CodeBuild</a> in the <i>CodeBuild user guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * Information about the compute resources the build project uses. Available values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your
         * environment type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your
         * environment type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for builds.
         * This compute type supports Docker images up to 100 GB uncompressed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you use <code>BUILD_GENERAL1_SMALL</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1 NVIDIA
         * A10G Tensor Core GPU for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on ARM-based
         * processors for builds.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you use <code>BUILD_GENERAL1_LARGE</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and 4
         * NVIDIA Tesla V100 GPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on ARM-based
         * processors for builds.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
         * </p>
         * </note>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         * Environment Compute Types</a> in the <i>CodeBuild User Guide.</i>
         * </p>
         * 
         * @param computeType
         *        Information about the compute resources the build project uses. Available values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your
         *        environment type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your
         *        environment type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for
         *        builds. This compute type supports Docker images up to 100 GB uncompressed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you use <code>BUILD_GENERAL1_SMALL</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for
         *        builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1
         *        NVIDIA A10G Tensor Core GPU for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on
         *        ARM-based processors for builds.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you use <code>BUILD_GENERAL1_LARGE</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for
         *        builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and
         *        4 NVIDIA Tesla V100 GPUs for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on
         *        ARM-based processors for builds.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
         *        </p>
         *        </note>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         *        Environment Compute Types</a> in the <i>CodeBuild User Guide.</i>
         * @see ComputeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeType
         */
        Builder computeType(String computeType);

        /**
         * <p>
         * Information about the compute resources the build project uses. Available values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your
         * environment type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your
         * environment type.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for builds.
         * This compute type supports Docker images up to 100 GB uncompressed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
         * <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you use <code>BUILD_GENERAL1_SMALL</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1 NVIDIA
         * A10G Tensor Core GPU for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on ARM-based
         * processors for builds.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you use <code>BUILD_GENERAL1_LARGE</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and 4
         * NVIDIA Tesla V100 GPUs for builds.
         * </p>
         * </li>
         * <li>
         * <p>
         * For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on ARM-based
         * processors for builds.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
         * </p>
         * </note>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         * Environment Compute Types</a> in the <i>CodeBuild User Guide.</i>
         * </p>
         * 
         * @param computeType
         *        Information about the compute resources the build project uses. Available values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_SMALL</code>: Use up to 3 GB memory and 2 vCPUs for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_MEDIUM</code>: Use up to 7 GB memory and 4 vCPUs for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_LARGE</code>: Use up to 16 GB memory and 8 vCPUs for builds, depending on your
         *        environment type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_XLARGE</code>: Use up to 70 GB memory and 36 vCPUs for builds, depending on your
         *        environment type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_GENERAL1_2XLARGE</code>: Use up to 145 GB memory, 72 vCPUs, and 824 GB of SSD storage for
         *        builds. This compute type supports Docker images up to 100 GB uncompressed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_1GB</code>: Use up to 1 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_2GB</code>: Use up to 2 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_4GB</code>: Use up to 4 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_8GB</code>: Use up to 8 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BUILD_LAMBDA_10GB</code>: Use up to 10 GB memory for builds. Only available for environment type
         *        <code>LINUX_LAMBDA_CONTAINER</code> and <code>ARM_LAMBDA_CONTAINER</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you use <code>BUILD_GENERAL1_SMALL</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_CONTAINER</code>, you can use up to 3 GB memory and 2 vCPUs for
         *        builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 16 GB memory, 4 vCPUs, and 1
         *        NVIDIA A10G Tensor Core GPU for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>ARM_CONTAINER</code>, you can use up to 4 GB memory and 2 vCPUs on
         *        ARM-based processors for builds.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you use <code>BUILD_GENERAL1_LARGE</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_CONTAINER</code>, you can use up to 15 GB memory and 8 vCPUs for
         *        builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>LINUX_GPU_CONTAINER</code>, you can use up to 255 GB memory, 32 vCPUs, and
         *        4 NVIDIA Tesla V100 GPUs for builds.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For environment type <code>ARM_CONTAINER</code>, you can use up to 16 GB memory and 8 vCPUs on
         *        ARM-based processors for builds.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you're using compute fleets during project creation, <code>computeType</code> will be ignored.
         *        </p>
         *        </note>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html">Build
         *        Environment Compute Types</a> in the <i>CodeBuild User Guide.</i>
         * @see ComputeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeType
         */
        Builder computeType(ComputeType computeType);

        /**
         * <p>
         * A ProjectFleet object to use for this build project.
         * </p>
         * 
         * @param fleet
         *        A ProjectFleet object to use for this build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleet(ProjectFleet fleet);

        /**
         * <p>
         * A ProjectFleet object to use for this build project.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProjectFleet.Builder} avoiding the need
         * to create one manually via {@link ProjectFleet#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProjectFleet.Builder#build()} is called immediately and its
         * result is passed to {@link #fleet(ProjectFleet)}.
         * 
         * @param fleet
         *        a consumer that will call methods on {@link ProjectFleet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fleet(ProjectFleet)
         */
        default Builder fleet(Consumer<ProjectFleet.Builder> fleet) {
            return fleet(ProjectFleet.builder().applyMutation(fleet).build());
        }

        /**
         * <p>
         * A set of environment variables to make available to builds for this build project.
         * </p>
         * 
         * @param environmentVariables
         *        A set of environment variables to make available to builds for this build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentVariables(Collection<EnvironmentVariable> environmentVariables);

        /**
         * <p>
         * A set of environment variables to make available to builds for this build project.
         * </p>
         * 
         * @param environmentVariables
         *        A set of environment variables to make available to builds for this build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentVariables(EnvironmentVariable... environmentVariables);

        /**
         * <p>
         * A set of environment variables to make available to builds for this build project.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codebuild.model.EnvironmentVariable.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.codebuild.model.EnvironmentVariable#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codebuild.model.EnvironmentVariable.Builder#build()} is called
         * immediately and its result is passed to {@link #environmentVariables(List<EnvironmentVariable>)}.
         * 
         * @param environmentVariables
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codebuild.model.EnvironmentVariable.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environmentVariables(java.util.Collection<EnvironmentVariable>)
         */
        Builder environmentVariables(Consumer<EnvironmentVariable.Builder>... environmentVariables);

        /**
         * <p>
         * Enables running the Docker daemon inside a Docker container. Set to true only if the build project is used to
         * build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails. The default
         * setting is <code>false</code>.
         * </p>
         * <p>
         * You can initialize the Docker daemon during the install phase of your build by adding one of the following
         * sets of commands to the install phase of your buildspec file:
         * </p>
         * <p>
         * If the operating system's base image is Ubuntu Linux:
         * </p>
         * <p>
         * <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
         * </p>
         * <p>
         * <code>- timeout 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
         * </p>
         * <p>
         * If the operating system's base image is Alpine Linux and the previous command does not work, add the
         * <code>-t</code> argument to <code>timeout</code>:
         * </p>
         * <p>
         * <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
         * </p>
         * <p>
         * <code>- timeout -t 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
         * </p>
         * 
         * @param privilegedMode
         *        Enables running the Docker daemon inside a Docker container. Set to true only if the build project is
         *        used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon
         *        fails. The default setting is <code>false</code>.</p>
         *        <p>
         *        You can initialize the Docker daemon during the install phase of your build by adding one of the
         *        following sets of commands to the install phase of your buildspec file:
         *        </p>
         *        <p>
         *        If the operating system's base image is Ubuntu Linux:
         *        </p>
         *        <p>
         *        <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
         *        </p>
         *        <p>
         *        <code>- timeout 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
         *        </p>
         *        <p>
         *        If the operating system's base image is Alpine Linux and the previous command does not work, add the
         *        <code>-t</code> argument to <code>timeout</code>:
         *        </p>
         *        <p>
         *        <code>- nohup /usr/local/bin/dockerd --host=unix:///var/run/docker.sock --host=tcp://0.0.0.0:2375 --storage-driver=overlay&amp;</code>
         *        </p>
         *        <p>
         *        <code>- timeout -t 15 sh -c "until docker info; do echo .; sleep 1; done"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privilegedMode(Boolean privilegedMode);

        /**
         * <p>
         * The ARN of the Amazon S3 bucket, path prefix, and object key that contains the PEM-encoded certificate for
         * the build project. For more information, see <a href=
         * "https://docs.aws.amazon.com/codebuild/latest/userguide/create-project-cli.html#cli.environment.certificate"
         * >certificate</a> in the <i>CodeBuild User Guide</i>.
         * </p>
         * 
         * @param certificate
         *        The ARN of the Amazon S3 bucket, path prefix, and object key that contains the PEM-encoded certificate
         *        for the build project. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codebuild/latest/userguide/create-project-cli.html#cli.environment.certificate"
         *        >certificate</a> in the <i>CodeBuild User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(String certificate);

        /**
         * <p>
         * The credentials for access to a private registry.
         * </p>
         * 
         * @param registryCredential
         *        The credentials for access to a private registry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registryCredential(RegistryCredential registryCredential);

        /**
         * <p>
         * The credentials for access to a private registry.
         * </p>
         * This is a convenience method that creates an instance of the {@link RegistryCredential.Builder} avoiding the
         * need to create one manually via {@link RegistryCredential#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RegistryCredential.Builder#build()} is called immediately and its
         * result is passed to {@link #registryCredential(RegistryCredential)}.
         * 
         * @param registryCredential
         *        a consumer that will call methods on {@link RegistryCredential.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #registryCredential(RegistryCredential)
         */
        default Builder registryCredential(Consumer<RegistryCredential.Builder> registryCredential) {
            return registryCredential(RegistryCredential.builder().applyMutation(registryCredential).build());
        }

        /**
         * <p>
         * The type of credentials CodeBuild uses to pull images in your build. There are two valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you modify your
         * ECR repository policy to trust CodeBuild service principal.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you use
         * an CodeBuild curated image, you must use CODEBUILD credentials.
         * </p>
         * 
         * @param imagePullCredentialsType
         *        The type of credentials CodeBuild uses to pull images in your build. There are two valid values: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you
         *        modify your ECR repository policy to trust CodeBuild service principal.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When
         *        you use an CodeBuild curated image, you must use CODEBUILD credentials.
         * @see ImagePullCredentialsType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImagePullCredentialsType
         */
        Builder imagePullCredentialsType(String imagePullCredentialsType);

        /**
         * <p>
         * The type of credentials CodeBuild uses to pull images in your build. There are two valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you modify your
         * ECR repository policy to trust CodeBuild service principal.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When you use
         * an CodeBuild curated image, you must use CODEBUILD credentials.
         * </p>
         * 
         * @param imagePullCredentialsType
         *        The type of credentials CodeBuild uses to pull images in your build. There are two valid values: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CODEBUILD</code> specifies that CodeBuild uses its own credentials. This requires that you
         *        modify your ECR repository policy to trust CodeBuild service principal.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SERVICE_ROLE</code> specifies that CodeBuild uses your build project's service role.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When you use a cross-account or private registry image, you must use SERVICE_ROLE credentials. When
         *        you use an CodeBuild curated image, you must use CODEBUILD credentials.
         * @see ImagePullCredentialsType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImagePullCredentialsType
         */
        Builder imagePullCredentialsType(ImagePullCredentialsType imagePullCredentialsType);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String image;

        private String computeType;

        private ProjectFleet fleet;

        private List<EnvironmentVariable> environmentVariables = DefaultSdkAutoConstructList.getInstance();

        private Boolean privilegedMode;

        private String certificate;

        private RegistryCredential registryCredential;

        private String imagePullCredentialsType;

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectEnvironment model) {
            type(model.type);
            image(model.image);
            computeType(model.computeType);
            fleet(model.fleet);
            environmentVariables(model.environmentVariables);
            privilegedMode(model.privilegedMode);
            certificate(model.certificate);
            registryCredential(model.registryCredential);
            imagePullCredentialsType(model.imagePullCredentialsType);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(EnvironmentType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final String getComputeType() {
            return computeType;
        }

        public final void setComputeType(String computeType) {
            this.computeType = computeType;
        }

        @Override
        public final Builder computeType(String computeType) {
            this.computeType = computeType;
            return this;
        }

        @Override
        public final Builder computeType(ComputeType computeType) {
            this.computeType(computeType == null ? null : computeType.toString());
            return this;
        }

        public final ProjectFleet.Builder getFleet() {
            return fleet != null ? fleet.toBuilder() : null;
        }

        public final void setFleet(ProjectFleet.BuilderImpl fleet) {
            this.fleet = fleet != null ? fleet.build() : null;
        }

        @Override
        public final Builder fleet(ProjectFleet fleet) {
            this.fleet = fleet;
            return this;
        }

        public final List<EnvironmentVariable.Builder> getEnvironmentVariables() {
            List<EnvironmentVariable.Builder> result = EnvironmentVariablesCopier.copyToBuilder(this.environmentVariables);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironmentVariables(Collection<EnvironmentVariable.BuilderImpl> environmentVariables) {
            this.environmentVariables = EnvironmentVariablesCopier.copyFromBuilder(environmentVariables);
        }

        @Override
        public final Builder environmentVariables(Collection<EnvironmentVariable> environmentVariables) {
            this.environmentVariables = EnvironmentVariablesCopier.copy(environmentVariables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentVariables(EnvironmentVariable... environmentVariables) {
            environmentVariables(Arrays.asList(environmentVariables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentVariables(Consumer<EnvironmentVariable.Builder>... environmentVariables) {
            environmentVariables(Stream.of(environmentVariables).map(c -> EnvironmentVariable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getPrivilegedMode() {
            return privilegedMode;
        }

        public final void setPrivilegedMode(Boolean privilegedMode) {
            this.privilegedMode = privilegedMode;
        }

        @Override
        public final Builder privilegedMode(Boolean privilegedMode) {
            this.privilegedMode = privilegedMode;
            return this;
        }

        public final String getCertificate() {
            return certificate;
        }

        public final void setCertificate(String certificate) {
            this.certificate = certificate;
        }

        @Override
        public final Builder certificate(String certificate) {
            this.certificate = certificate;
            return this;
        }

        public final RegistryCredential.Builder getRegistryCredential() {
            return registryCredential != null ? registryCredential.toBuilder() : null;
        }

        public final void setRegistryCredential(RegistryCredential.BuilderImpl registryCredential) {
            this.registryCredential = registryCredential != null ? registryCredential.build() : null;
        }

        @Override
        public final Builder registryCredential(RegistryCredential registryCredential) {
            this.registryCredential = registryCredential;
            return this;
        }

        public final String getImagePullCredentialsType() {
            return imagePullCredentialsType;
        }

        public final void setImagePullCredentialsType(String imagePullCredentialsType) {
            this.imagePullCredentialsType = imagePullCredentialsType;
        }

        @Override
        public final Builder imagePullCredentialsType(String imagePullCredentialsType) {
            this.imagePullCredentialsType = imagePullCredentialsType;
            return this;
        }

        @Override
        public final Builder imagePullCredentialsType(ImagePullCredentialsType imagePullCredentialsType) {
            this.imagePullCredentialsType(imagePullCredentialsType == null ? null : imagePullCredentialsType.toString());
            return this;
        }

        @Override
        public ProjectEnvironment build() {
            return new ProjectEnvironment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
