/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the build input source code for the build project.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectSource implements SdkPojo, Serializable, ToCopyableBuilder<ProjectSource.Builder, ProjectSource> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ProjectSource::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("location").getter(getter(ProjectSource::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<Integer> GIT_CLONE_DEPTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("gitCloneDepth").getter(getter(ProjectSource::gitCloneDepth)).setter(setter(Builder::gitCloneDepth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gitCloneDepth").build()).build();

    private static final SdkField<GitSubmodulesConfig> GIT_SUBMODULES_CONFIG_FIELD = SdkField
            .<GitSubmodulesConfig> builder(MarshallingType.SDK_POJO).memberName("gitSubmodulesConfig")
            .getter(getter(ProjectSource::gitSubmodulesConfig)).setter(setter(Builder::gitSubmodulesConfig))
            .constructor(GitSubmodulesConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gitSubmodulesConfig").build())
            .build();

    private static final SdkField<String> BUILDSPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("buildspec").getter(getter(ProjectSource::buildspec)).setter(setter(Builder::buildspec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildspec").build()).build();

    private static final SdkField<SourceAuth> AUTH_FIELD = SdkField.<SourceAuth> builder(MarshallingType.SDK_POJO)
            .memberName("auth").getter(getter(ProjectSource::auth)).setter(setter(Builder::auth))
            .constructor(SourceAuth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("auth").build()).build();

    private static final SdkField<Boolean> REPORT_BUILD_STATUS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("reportBuildStatus").getter(getter(ProjectSource::reportBuildStatus))
            .setter(setter(Builder::reportBuildStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reportBuildStatus").build()).build();

    private static final SdkField<BuildStatusConfig> BUILD_STATUS_CONFIG_FIELD = SdkField
            .<BuildStatusConfig> builder(MarshallingType.SDK_POJO).memberName("buildStatusConfig")
            .getter(getter(ProjectSource::buildStatusConfig)).setter(setter(Builder::buildStatusConfig))
            .constructor(BuildStatusConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buildStatusConfig").build()).build();

    private static final SdkField<Boolean> INSECURE_SSL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("insecureSsl").getter(getter(ProjectSource::insecureSsl)).setter(setter(Builder::insecureSsl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("insecureSsl").build()).build();

    private static final SdkField<String> SOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceIdentifier").getter(getter(ProjectSource::sourceIdentifier))
            .setter(setter(Builder::sourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceIdentifier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, LOCATION_FIELD,
            GIT_CLONE_DEPTH_FIELD, GIT_SUBMODULES_CONFIG_FIELD, BUILDSPEC_FIELD, AUTH_FIELD, REPORT_BUILD_STATUS_FIELD,
            BUILD_STATUS_CONFIG_FIELD, INSECURE_SSL_FIELD, SOURCE_IDENTIFIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String location;

    private final Integer gitCloneDepth;

    private final GitSubmodulesConfig gitSubmodulesConfig;

    private final String buildspec;

    private final SourceAuth auth;

    private final Boolean reportBuildStatus;

    private final BuildStatusConfig buildStatusConfig;

    private final Boolean insecureSsl;

    private final String sourceIdentifier;

    private ProjectSource(BuilderImpl builder) {
        this.type = builder.type;
        this.location = builder.location;
        this.gitCloneDepth = builder.gitCloneDepth;
        this.gitSubmodulesConfig = builder.gitSubmodulesConfig;
        this.buildspec = builder.buildspec;
        this.auth = builder.auth;
        this.reportBuildStatus = builder.reportBuildStatus;
        this.buildStatusConfig = builder.buildStatusConfig;
        this.insecureSsl = builder.insecureSsl;
        this.sourceIdentifier = builder.sourceIdentifier;
    }

    /**
     * <p>
     * The type of repository that contains the source code to be built. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
     * CodePipeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB</code>: The source code is in a GitHub repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITLAB</code>: The source code is in a GitLab repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_SOURCE</code>: The project does not have input source code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The source code is in an Amazon S3 bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of repository that contains the source code to be built. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
     *         CodePipeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB</code>: The source code is in a GitHub repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITLAB</code>: The source code is in a GitLab repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_SOURCE</code>: The project does not have input source code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The source code is in an Amazon S3 bucket.
     *         </p>
     *         </li>
     * @see SourceType
     */
    public final SourceType type() {
        return SourceType.fromValue(type);
    }

    /**
     * <p>
     * The type of repository that contains the source code to be built. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
     * CodePipeline.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB</code>: The source code is in a GitHub repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITLAB</code>: The source code is in a GitLab repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NO_SOURCE</code>: The project does not have input source code.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The source code is in an Amazon S3 bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of repository that contains the source code to be built. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
     *         CodePipeline.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB</code>: The source code is in a GitHub repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITLAB</code>: The source code is in a GitLab repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NO_SOURCE</code>: The project does not have input source code.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The source code is in an Amazon S3 bucket.
     *         </p>
     *         </li>
     * @see SourceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Information about the location of the source code to be built. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For source code settings that are specified in the source action of a pipeline in CodePipeline,
     * <code>location</code> should not be specified. If it is specified, CodePipeline ignores it. This is because
     * CodePipeline uses the settings in a pipeline's source action instead of this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in an CodeCommit repository, the HTTPS clone URL to the repository that contains the source code
     * and the buildspec file (for example,
     * <code>https://git-codecommit.&lt;region-ID&gt;.amazonaws.com/v1/repos/&lt;repo-name&gt;</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in an Amazon S3 input bucket, one of the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The path to the ZIP file that contains the source code (for example,
     * <code>&lt;bucket-name&gt;/&lt;path&gt;/&lt;object-name&gt;.zip</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The path to the folder that contains the source code (for example,
     * <code>&lt;bucket-name&gt;/&lt;path-to-source-code&gt;/&lt;folder&gt;/</code>).
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source and the
     * buildspec file. You must connect your Amazon Web Services account to your GitHub account. Use the CodeBuild
     * console to start creating a build project. When you use the console to connect (or reconnect) with GitHub, on the
     * GitHub <b>Authorize application</b> page, for <b>Organization access</b>, choose <b>Request access</b> next to
     * each repository you want to allow CodeBuild to have access to, and then choose <b>Authorize application</b>.
     * (After you have connected to your GitHub account, you do not need to finish creating the build project. You can
     * leave the CodeBuild console.) To instruct CodeBuild to use this connection, in the <code>source</code> object,
     * set the <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in an GitLab or self-managed GitLab repository, the HTTPS clone URL to the repository that
     * contains the source and the buildspec file. You must connect your Amazon Web Services account to your GitLab
     * account. Use the CodeBuild console to start creating a build project. When you use the console to connect (or
     * reconnect) with GitLab, on the Connections <b>Authorize application</b> page, choose <b>Authorize</b>. Then on
     * the CodeConnections <b>Create GitLab connection</b> page, choose <b>Connect to GitLab</b>. (After you have
     * connected to your GitLab account, you do not need to finish creating the build project. You can leave the
     * CodeBuild console.) To instruct CodeBuild to override the default connection and use this connection instead, set
     * the <code>auth</code> object's <code>type</code> value to <code>CODECONNECTIONS</code> in the <code>source</code>
     * object.
     * </p>
     * </li>
     * <li>
     * <p>
     * For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source and the
     * buildspec file. You must connect your Amazon Web Services account to your Bitbucket account. Use the CodeBuild
     * console to start creating a build project. When you use the console to connect (or reconnect) with Bitbucket, on
     * the Bitbucket <b>Confirm access to your account</b> page, choose <b>Grant access</b>. (After you have connected
     * to your Bitbucket account, you do not need to finish creating the build project. You can leave the CodeBuild
     * console.) To instruct CodeBuild to use this connection, in the <code>source</code> object, set the
     * <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify <code>CODEPIPELINE</code> for the <code>Type</code> property, don't specify this property. For all
     * of the other types, you must specify <code>Location</code>.
     * </p>
     * 
     * @return Information about the location of the source code to be built. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For source code settings that are specified in the source action of a pipeline in CodePipeline,
     *         <code>location</code> should not be specified. If it is specified, CodePipeline ignores it. This is
     *         because CodePipeline uses the settings in a pipeline's source action instead of this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in an CodeCommit repository, the HTTPS clone URL to the repository that contains the
     *         source code and the buildspec file (for example,
     *         <code>https://git-codecommit.&lt;region-ID&gt;.amazonaws.com/v1/repos/&lt;repo-name&gt;</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in an Amazon S3 input bucket, one of the following.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The path to the ZIP file that contains the source code (for example,
     *         <code>&lt;bucket-name&gt;/&lt;path&gt;/&lt;object-name&gt;.zip</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The path to the folder that contains the source code (for example,
     *         <code>&lt;bucket-name&gt;/&lt;path-to-source-code&gt;/&lt;folder&gt;/</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source
     *         and the buildspec file. You must connect your Amazon Web Services account to your GitHub account. Use the
     *         CodeBuild console to start creating a build project. When you use the console to connect (or reconnect)
     *         with GitHub, on the GitHub <b>Authorize application</b> page, for <b>Organization access</b>, choose
     *         <b>Request access</b> next to each repository you want to allow CodeBuild to have access to, and then
     *         choose <b>Authorize application</b>. (After you have connected to your GitHub account, you do not need to
     *         finish creating the build project. You can leave the CodeBuild console.) To instruct CodeBuild to use
     *         this connection, in the <code>source</code> object, set the <code>auth</code> object's <code>type</code>
     *         value to <code>OAUTH</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in an GitLab or self-managed GitLab repository, the HTTPS clone URL to the repository
     *         that contains the source and the buildspec file. You must connect your Amazon Web Services account to
     *         your GitLab account. Use the CodeBuild console to start creating a build project. When you use the
     *         console to connect (or reconnect) with GitLab, on the Connections <b>Authorize application</b> page,
     *         choose <b>Authorize</b>. Then on the CodeConnections <b>Create GitLab connection</b> page, choose
     *         <b>Connect to GitLab</b>. (After you have connected to your GitLab account, you do not need to finish
     *         creating the build project. You can leave the CodeBuild console.) To instruct CodeBuild to override the
     *         default connection and use this connection instead, set the <code>auth</code> object's <code>type</code>
     *         value to <code>CODECONNECTIONS</code> in the <code>source</code> object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source
     *         and the buildspec file. You must connect your Amazon Web Services account to your Bitbucket account. Use
     *         the CodeBuild console to start creating a build project. When you use the console to connect (or
     *         reconnect) with Bitbucket, on the Bitbucket <b>Confirm access to your account</b> page, choose <b>Grant
     *         access</b>. (After you have connected to your Bitbucket account, you do not need to finish creating the
     *         build project. You can leave the CodeBuild console.) To instruct CodeBuild to use this connection, in the
     *         <code>source</code> object, set the <code>auth</code> object's <code>type</code> value to
     *         <code>OAUTH</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify <code>CODEPIPELINE</code> for the <code>Type</code> property, don't specify this property.
     *         For all of the other types, you must specify <code>Location</code>.
     */
    public final String location() {
        return location;
    }

    /**
     * <p>
     * Information about the Git clone depth for the build project.
     * </p>
     * 
     * @return Information about the Git clone depth for the build project.
     */
    public final Integer gitCloneDepth() {
        return gitCloneDepth;
    }

    /**
     * <p>
     * Information about the Git submodules configuration for the build project.
     * </p>
     * 
     * @return Information about the Git submodules configuration for the build project.
     */
    public final GitSubmodulesConfig gitSubmodulesConfig() {
        return gitSubmodulesConfig;
    }

    /**
     * <p>
     * The buildspec file declaration to use for the builds in this build project.
     * </p>
     * <p>
     * If this value is set, it can be either an inline buildspec definition, the path to an alternate buildspec file
     * relative to the value of the built-in <code>CODEBUILD_SRC_DIR</code> environment variable, or the path to an S3
     * bucket. The bucket must be in the same Amazon Web Services Region as the build project. Specify the buildspec
     * file using its ARN (for example, <code>arn:aws:s3:::my-codebuild-sample2/buildspec.yml</code>). If this value is
     * not provided or is set to an empty string, the source code must contain a buildspec file in its root directory.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-name-storage"
     * >Buildspec File Name and Storage Location</a>.
     * </p>
     * 
     * @return The buildspec file declaration to use for the builds in this build project.</p>
     *         <p>
     *         If this value is set, it can be either an inline buildspec definition, the path to an alternate buildspec
     *         file relative to the value of the built-in <code>CODEBUILD_SRC_DIR</code> environment variable, or the
     *         path to an S3 bucket. The bucket must be in the same Amazon Web Services Region as the build project.
     *         Specify the buildspec file using its ARN (for example,
     *         <code>arn:aws:s3:::my-codebuild-sample2/buildspec.yml</code>). If this value is not provided or is set to
     *         an empty string, the source code must contain a buildspec file in its root directory. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-name-storage"
     *         >Buildspec File Name and Storage Location</a>.
     */
    public final String buildspec() {
        return buildspec;
    }

    /**
     * <p>
     * Information about the authorization settings for CodeBuild to access the source code to be built.
     * </p>
     * <p>
     * This information is for the CodeBuild console's use only. Your code should not get or set this information
     * directly.
     * </p>
     * 
     * @return Information about the authorization settings for CodeBuild to access the source code to be built.</p>
     *         <p>
     *         This information is for the CodeBuild console's use only. Your code should not get or set this
     *         information directly.
     */
    public final SourceAuth auth() {
        return auth;
    }

    /**
     * <p>
     * Set to true to report the status of a build's start and finish to your source provider. This option is valid only
     * when your source provider is GitHub, GitHub Enterprise, GitLab, GitLab Self Managed, or Bitbucket. If this is set
     * and you use a different source provider, an <code>invalidInputException</code> is thrown.
     * </p>
     * <p>
     * To be able to report the build status to the source provider, the user associated with the source provider must
     * have write access to the repo. If the user does not have write access, the build status cannot be updated. For
     * more information, see <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/access-tokens.html">Source
     * provider access</a> in the <i>CodeBuild User Guide</i>.
     * </p>
     * <p>
     * The status of a build triggered by a webhook is always reported to your source provider.
     * </p>
     * <p>
     * If your project's builds are triggered by a webhook, you must push a new commit to the repo for a change to this
     * property to take effect.
     * </p>
     * 
     * @return Set to true to report the status of a build's start and finish to your source provider. This option is
     *         valid only when your source provider is GitHub, GitHub Enterprise, GitLab, GitLab Self Managed, or
     *         Bitbucket. If this is set and you use a different source provider, an <code>invalidInputException</code>
     *         is thrown. </p>
     *         <p>
     *         To be able to report the build status to the source provider, the user associated with the source
     *         provider must have write access to the repo. If the user does not have write access, the build status
     *         cannot be updated. For more information, see <a
     *         href="https://docs.aws.amazon.com/codebuild/latest/userguide/access-tokens.html">Source provider
     *         access</a> in the <i>CodeBuild User Guide</i>.
     *         </p>
     *         <p>
     *         The status of a build triggered by a webhook is always reported to your source provider.
     *         </p>
     *         <p>
     *         If your project's builds are triggered by a webhook, you must push a new commit to the repo for a change
     *         to this property to take effect.
     */
    public final Boolean reportBuildStatus() {
        return reportBuildStatus;
    }

    /**
     * <p>
     * Contains information that defines how the build project reports the build status to the source provider. This
     * option is only used when the source provider is <code>GITHUB</code>, <code>GITHUB_ENTERPRISE</code>, or
     * <code>BITBUCKET</code>.
     * </p>
     * 
     * @return Contains information that defines how the build project reports the build status to the source provider.
     *         This option is only used when the source provider is <code>GITHUB</code>, <code>GITHUB_ENTERPRISE</code>,
     *         or <code>BITBUCKET</code>.
     */
    public final BuildStatusConfig buildStatusConfig() {
        return buildStatusConfig;
    }

    /**
     * <p>
     * Enable this flag to ignore SSL warnings while connecting to the project source code.
     * </p>
     * 
     * @return Enable this flag to ignore SSL warnings while connecting to the project source code.
     */
    public final Boolean insecureSsl() {
        return insecureSsl;
    }

    /**
     * <p>
     * An identifier for this project source. The identifier can only contain alphanumeric characters and underscores,
     * and must be less than 128 characters in length.
     * </p>
     * 
     * @return An identifier for this project source. The identifier can only contain alphanumeric characters and
     *         underscores, and must be less than 128 characters in length.
     */
    public final String sourceIdentifier() {
        return sourceIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(gitCloneDepth());
        hashCode = 31 * hashCode + Objects.hashCode(gitSubmodulesConfig());
        hashCode = 31 * hashCode + Objects.hashCode(buildspec());
        hashCode = 31 * hashCode + Objects.hashCode(auth());
        hashCode = 31 * hashCode + Objects.hashCode(reportBuildStatus());
        hashCode = 31 * hashCode + Objects.hashCode(buildStatusConfig());
        hashCode = 31 * hashCode + Objects.hashCode(insecureSsl());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIdentifier());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectSource)) {
            return false;
        }
        ProjectSource other = (ProjectSource) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(location(), other.location())
                && Objects.equals(gitCloneDepth(), other.gitCloneDepth())
                && Objects.equals(gitSubmodulesConfig(), other.gitSubmodulesConfig())
                && Objects.equals(buildspec(), other.buildspec()) && Objects.equals(auth(), other.auth())
                && Objects.equals(reportBuildStatus(), other.reportBuildStatus())
                && Objects.equals(buildStatusConfig(), other.buildStatusConfig())
                && Objects.equals(insecureSsl(), other.insecureSsl())
                && Objects.equals(sourceIdentifier(), other.sourceIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProjectSource").add("Type", typeAsString()).add("Location", location())
                .add("GitCloneDepth", gitCloneDepth()).add("GitSubmodulesConfig", gitSubmodulesConfig())
                .add("Buildspec", buildspec()).add("Auth", auth()).add("ReportBuildStatus", reportBuildStatus())
                .add("BuildStatusConfig", buildStatusConfig()).add("InsecureSsl", insecureSsl())
                .add("SourceIdentifier", sourceIdentifier()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "gitCloneDepth":
            return Optional.ofNullable(clazz.cast(gitCloneDepth()));
        case "gitSubmodulesConfig":
            return Optional.ofNullable(clazz.cast(gitSubmodulesConfig()));
        case "buildspec":
            return Optional.ofNullable(clazz.cast(buildspec()));
        case "auth":
            return Optional.ofNullable(clazz.cast(auth()));
        case "reportBuildStatus":
            return Optional.ofNullable(clazz.cast(reportBuildStatus()));
        case "buildStatusConfig":
            return Optional.ofNullable(clazz.cast(buildStatusConfig()));
        case "insecureSsl":
            return Optional.ofNullable(clazz.cast(insecureSsl()));
        case "sourceIdentifier":
            return Optional.ofNullable(clazz.cast(sourceIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectSource, T> g) {
        return obj -> g.apply((ProjectSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectSource> {
        /**
         * <p>
         * The type of repository that contains the source code to be built. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
         * CodePipeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB</code>: The source code is in a GitHub repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITLAB</code>: The source code is in a GitLab repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_SOURCE</code>: The project does not have input source code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The source code is in an Amazon S3 bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of repository that contains the source code to be built. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline
         *        in CodePipeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB</code>: The source code is in a GitHub repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITLAB</code>: The source code is in a GitLab repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_SOURCE</code>: The project does not have input source code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The source code is in an Amazon S3 bucket.
         *        </p>
         *        </li>
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of repository that contains the source code to be built. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline in
         * CodePipeline.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB</code>: The source code is in a GitHub repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITLAB</code>: The source code is in a GitLab repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NO_SOURCE</code>: The project does not have input source code.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The source code is in an Amazon S3 bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of repository that contains the source code to be built. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BITBUCKET</code>: The source code is in a Bitbucket repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODECOMMIT</code>: The source code is in an CodeCommit repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CODEPIPELINE</code>: The source code settings are specified in the source action of a pipeline
         *        in CodePipeline.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB</code>: The source code is in a GitHub repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITHUB_ENTERPRISE</code>: The source code is in a GitHub Enterprise Server repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITLAB</code>: The source code is in a GitLab repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GITLAB_SELF_MANAGED</code>: The source code is in a self-managed GitLab repository.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NO_SOURCE</code>: The project does not have input source code.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The source code is in an Amazon S3 bucket.
         *        </p>
         *        </li>
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder type(SourceType type);

        /**
         * <p>
         * Information about the location of the source code to be built. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For source code settings that are specified in the source action of a pipeline in CodePipeline,
         * <code>location</code> should not be specified. If it is specified, CodePipeline ignores it. This is because
         * CodePipeline uses the settings in a pipeline's source action instead of this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in an CodeCommit repository, the HTTPS clone URL to the repository that contains the source
         * code and the buildspec file (for example,
         * <code>https://git-codecommit.&lt;region-ID&gt;.amazonaws.com/v1/repos/&lt;repo-name&gt;</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in an Amazon S3 input bucket, one of the following.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The path to the ZIP file that contains the source code (for example,
         * <code>&lt;bucket-name&gt;/&lt;path&gt;/&lt;object-name&gt;.zip</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The path to the folder that contains the source code (for example,
         * <code>&lt;bucket-name&gt;/&lt;path-to-source-code&gt;/&lt;folder&gt;/</code>).
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source and
         * the buildspec file. You must connect your Amazon Web Services account to your GitHub account. Use the
         * CodeBuild console to start creating a build project. When you use the console to connect (or reconnect) with
         * GitHub, on the GitHub <b>Authorize application</b> page, for <b>Organization access</b>, choose <b>Request
         * access</b> next to each repository you want to allow CodeBuild to have access to, and then choose
         * <b>Authorize application</b>. (After you have connected to your GitHub account, you do not need to finish
         * creating the build project. You can leave the CodeBuild console.) To instruct CodeBuild to use this
         * connection, in the <code>source</code> object, set the <code>auth</code> object's <code>type</code> value to
         * <code>OAUTH</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in an GitLab or self-managed GitLab repository, the HTTPS clone URL to the repository that
         * contains the source and the buildspec file. You must connect your Amazon Web Services account to your GitLab
         * account. Use the CodeBuild console to start creating a build project. When you use the console to connect (or
         * reconnect) with GitLab, on the Connections <b>Authorize application</b> page, choose <b>Authorize</b>. Then
         * on the CodeConnections <b>Create GitLab connection</b> page, choose <b>Connect to GitLab</b>. (After you have
         * connected to your GitLab account, you do not need to finish creating the build project. You can leave the
         * CodeBuild console.) To instruct CodeBuild to override the default connection and use this connection instead,
         * set the <code>auth</code> object's <code>type</code> value to <code>CODECONNECTIONS</code> in the
         * <code>source</code> object.
         * </p>
         * </li>
         * <li>
         * <p>
         * For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the source and
         * the buildspec file. You must connect your Amazon Web Services account to your Bitbucket account. Use the
         * CodeBuild console to start creating a build project. When you use the console to connect (or reconnect) with
         * Bitbucket, on the Bitbucket <b>Confirm access to your account</b> page, choose <b>Grant access</b>. (After
         * you have connected to your Bitbucket account, you do not need to finish creating the build project. You can
         * leave the CodeBuild console.) To instruct CodeBuild to use this connection, in the <code>source</code>
         * object, set the <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify <code>CODEPIPELINE</code> for the <code>Type</code> property, don't specify this property. For
         * all of the other types, you must specify <code>Location</code>.
         * </p>
         * 
         * @param location
         *        Information about the location of the source code to be built. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For source code settings that are specified in the source action of a pipeline in CodePipeline,
         *        <code>location</code> should not be specified. If it is specified, CodePipeline ignores it. This is
         *        because CodePipeline uses the settings in a pipeline's source action instead of this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in an CodeCommit repository, the HTTPS clone URL to the repository that contains the
         *        source code and the buildspec file (for example,
         *        <code>https://git-codecommit.&lt;region-ID&gt;.amazonaws.com/v1/repos/&lt;repo-name&gt;</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in an Amazon S3 input bucket, one of the following.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The path to the ZIP file that contains the source code (for example,
         *        <code>&lt;bucket-name&gt;/&lt;path&gt;/&lt;object-name&gt;.zip</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The path to the folder that contains the source code (for example,
         *        <code>&lt;bucket-name&gt;/&lt;path-to-source-code&gt;/&lt;folder&gt;/</code>).
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in a GitHub repository, the HTTPS clone URL to the repository that contains the source
         *        and the buildspec file. You must connect your Amazon Web Services account to your GitHub account. Use
         *        the CodeBuild console to start creating a build project. When you use the console to connect (or
         *        reconnect) with GitHub, on the GitHub <b>Authorize application</b> page, for <b>Organization
         *        access</b>, choose <b>Request access</b> next to each repository you want to allow CodeBuild to have
         *        access to, and then choose <b>Authorize application</b>. (After you have connected to your GitHub
         *        account, you do not need to finish creating the build project. You can leave the CodeBuild console.)
         *        To instruct CodeBuild to use this connection, in the <code>source</code> object, set the
         *        <code>auth</code> object's <code>type</code> value to <code>OAUTH</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in an GitLab or self-managed GitLab repository, the HTTPS clone URL to the repository
         *        that contains the source and the buildspec file. You must connect your Amazon Web Services account to
         *        your GitLab account. Use the CodeBuild console to start creating a build project. When you use the
         *        console to connect (or reconnect) with GitLab, on the Connections <b>Authorize application</b> page,
         *        choose <b>Authorize</b>. Then on the CodeConnections <b>Create GitLab connection</b> page, choose
         *        <b>Connect to GitLab</b>. (After you have connected to your GitLab account, you do not need to finish
         *        creating the build project. You can leave the CodeBuild console.) To instruct CodeBuild to override
         *        the default connection and use this connection instead, set the <code>auth</code> object's
         *        <code>type</code> value to <code>CODECONNECTIONS</code> in the <code>source</code> object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For source code in a Bitbucket repository, the HTTPS clone URL to the repository that contains the
         *        source and the buildspec file. You must connect your Amazon Web Services account to your Bitbucket
         *        account. Use the CodeBuild console to start creating a build project. When you use the console to
         *        connect (or reconnect) with Bitbucket, on the Bitbucket <b>Confirm access to your account</b> page,
         *        choose <b>Grant access</b>. (After you have connected to your Bitbucket account, you do not need to
         *        finish creating the build project. You can leave the CodeBuild console.) To instruct CodeBuild to use
         *        this connection, in the <code>source</code> object, set the <code>auth</code> object's
         *        <code>type</code> value to <code>OAUTH</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify <code>CODEPIPELINE</code> for the <code>Type</code> property, don't specify this
         *        property. For all of the other types, you must specify <code>Location</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * Information about the Git clone depth for the build project.
         * </p>
         * 
         * @param gitCloneDepth
         *        Information about the Git clone depth for the build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gitCloneDepth(Integer gitCloneDepth);

        /**
         * <p>
         * Information about the Git submodules configuration for the build project.
         * </p>
         * 
         * @param gitSubmodulesConfig
         *        Information about the Git submodules configuration for the build project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gitSubmodulesConfig(GitSubmodulesConfig gitSubmodulesConfig);

        /**
         * <p>
         * Information about the Git submodules configuration for the build project.
         * </p>
         * This is a convenience method that creates an instance of the {@link GitSubmodulesConfig.Builder} avoiding the
         * need to create one manually via {@link GitSubmodulesConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GitSubmodulesConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #gitSubmodulesConfig(GitSubmodulesConfig)}.
         * 
         * @param gitSubmodulesConfig
         *        a consumer that will call methods on {@link GitSubmodulesConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gitSubmodulesConfig(GitSubmodulesConfig)
         */
        default Builder gitSubmodulesConfig(Consumer<GitSubmodulesConfig.Builder> gitSubmodulesConfig) {
            return gitSubmodulesConfig(GitSubmodulesConfig.builder().applyMutation(gitSubmodulesConfig).build());
        }

        /**
         * <p>
         * The buildspec file declaration to use for the builds in this build project.
         * </p>
         * <p>
         * If this value is set, it can be either an inline buildspec definition, the path to an alternate buildspec
         * file relative to the value of the built-in <code>CODEBUILD_SRC_DIR</code> environment variable, or the path
         * to an S3 bucket. The bucket must be in the same Amazon Web Services Region as the build project. Specify the
         * buildspec file using its ARN (for example, <code>arn:aws:s3:::my-codebuild-sample2/buildspec.yml</code>). If
         * this value is not provided or is set to an empty string, the source code must contain a buildspec file in its
         * root directory. For more information, see <a href=
         * "https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-name-storage"
         * >Buildspec File Name and Storage Location</a>.
         * </p>
         * 
         * @param buildspec
         *        The buildspec file declaration to use for the builds in this build project.</p>
         *        <p>
         *        If this value is set, it can be either an inline buildspec definition, the path to an alternate
         *        buildspec file relative to the value of the built-in <code>CODEBUILD_SRC_DIR</code> environment
         *        variable, or the path to an S3 bucket. The bucket must be in the same Amazon Web Services Region as
         *        the build project. Specify the buildspec file using its ARN (for example,
         *        <code>arn:aws:s3:::my-codebuild-sample2/buildspec.yml</code>). If this value is not provided or is set
         *        to an empty string, the source code must contain a buildspec file in its root directory. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-name-storage"
         *        >Buildspec File Name and Storage Location</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildspec(String buildspec);

        /**
         * <p>
         * Information about the authorization settings for CodeBuild to access the source code to be built.
         * </p>
         * <p>
         * This information is for the CodeBuild console's use only. Your code should not get or set this information
         * directly.
         * </p>
         * 
         * @param auth
         *        Information about the authorization settings for CodeBuild to access the source code to be built.</p>
         *        <p>
         *        This information is for the CodeBuild console's use only. Your code should not get or set this
         *        information directly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder auth(SourceAuth auth);

        /**
         * <p>
         * Information about the authorization settings for CodeBuild to access the source code to be built.
         * </p>
         * <p>
         * This information is for the CodeBuild console's use only. Your code should not get or set this information
         * directly.
         * </p>
         * This is a convenience method that creates an instance of the {@link SourceAuth.Builder} avoiding the need to
         * create one manually via {@link SourceAuth#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SourceAuth.Builder#build()} is called immediately and its result
         * is passed to {@link #auth(SourceAuth)}.
         * 
         * @param auth
         *        a consumer that will call methods on {@link SourceAuth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #auth(SourceAuth)
         */
        default Builder auth(Consumer<SourceAuth.Builder> auth) {
            return auth(SourceAuth.builder().applyMutation(auth).build());
        }

        /**
         * <p>
         * Set to true to report the status of a build's start and finish to your source provider. This option is valid
         * only when your source provider is GitHub, GitHub Enterprise, GitLab, GitLab Self Managed, or Bitbucket. If
         * this is set and you use a different source provider, an <code>invalidInputException</code> is thrown.
         * </p>
         * <p>
         * To be able to report the build status to the source provider, the user associated with the source provider
         * must have write access to the repo. If the user does not have write access, the build status cannot be
         * updated. For more information, see <a
         * href="https://docs.aws.amazon.com/codebuild/latest/userguide/access-tokens.html">Source provider access</a>
         * in the <i>CodeBuild User Guide</i>.
         * </p>
         * <p>
         * The status of a build triggered by a webhook is always reported to your source provider.
         * </p>
         * <p>
         * If your project's builds are triggered by a webhook, you must push a new commit to the repo for a change to
         * this property to take effect.
         * </p>
         * 
         * @param reportBuildStatus
         *        Set to true to report the status of a build's start and finish to your source provider. This option is
         *        valid only when your source provider is GitHub, GitHub Enterprise, GitLab, GitLab Self Managed, or
         *        Bitbucket. If this is set and you use a different source provider, an
         *        <code>invalidInputException</code> is thrown. </p>
         *        <p>
         *        To be able to report the build status to the source provider, the user associated with the source
         *        provider must have write access to the repo. If the user does not have write access, the build status
         *        cannot be updated. For more information, see <a
         *        href="https://docs.aws.amazon.com/codebuild/latest/userguide/access-tokens.html">Source provider
         *        access</a> in the <i>CodeBuild User Guide</i>.
         *        </p>
         *        <p>
         *        The status of a build triggered by a webhook is always reported to your source provider.
         *        </p>
         *        <p>
         *        If your project's builds are triggered by a webhook, you must push a new commit to the repo for a
         *        change to this property to take effect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportBuildStatus(Boolean reportBuildStatus);

        /**
         * <p>
         * Contains information that defines how the build project reports the build status to the source provider. This
         * option is only used when the source provider is <code>GITHUB</code>, <code>GITHUB_ENTERPRISE</code>, or
         * <code>BITBUCKET</code>.
         * </p>
         * 
         * @param buildStatusConfig
         *        Contains information that defines how the build project reports the build status to the source
         *        provider. This option is only used when the source provider is <code>GITHUB</code>,
         *        <code>GITHUB_ENTERPRISE</code>, or <code>BITBUCKET</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildStatusConfig(BuildStatusConfig buildStatusConfig);

        /**
         * <p>
         * Contains information that defines how the build project reports the build status to the source provider. This
         * option is only used when the source provider is <code>GITHUB</code>, <code>GITHUB_ENTERPRISE</code>, or
         * <code>BITBUCKET</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link BuildStatusConfig.Builder} avoiding the
         * need to create one manually via {@link BuildStatusConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BuildStatusConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #buildStatusConfig(BuildStatusConfig)}.
         * 
         * @param buildStatusConfig
         *        a consumer that will call methods on {@link BuildStatusConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #buildStatusConfig(BuildStatusConfig)
         */
        default Builder buildStatusConfig(Consumer<BuildStatusConfig.Builder> buildStatusConfig) {
            return buildStatusConfig(BuildStatusConfig.builder().applyMutation(buildStatusConfig).build());
        }

        /**
         * <p>
         * Enable this flag to ignore SSL warnings while connecting to the project source code.
         * </p>
         * 
         * @param insecureSsl
         *        Enable this flag to ignore SSL warnings while connecting to the project source code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insecureSsl(Boolean insecureSsl);

        /**
         * <p>
         * An identifier for this project source. The identifier can only contain alphanumeric characters and
         * underscores, and must be less than 128 characters in length.
         * </p>
         * 
         * @param sourceIdentifier
         *        An identifier for this project source. The identifier can only contain alphanumeric characters and
         *        underscores, and must be less than 128 characters in length.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String location;

        private Integer gitCloneDepth;

        private GitSubmodulesConfig gitSubmodulesConfig;

        private String buildspec;

        private SourceAuth auth;

        private Boolean reportBuildStatus;

        private BuildStatusConfig buildStatusConfig;

        private Boolean insecureSsl;

        private String sourceIdentifier;

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectSource model) {
            type(model.type);
            location(model.location);
            gitCloneDepth(model.gitCloneDepth);
            gitSubmodulesConfig(model.gitSubmodulesConfig);
            buildspec(model.buildspec);
            auth(model.auth);
            reportBuildStatus(model.reportBuildStatus);
            buildStatusConfig(model.buildStatusConfig);
            insecureSsl(model.insecureSsl);
            sourceIdentifier(model.sourceIdentifier);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SourceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final Integer getGitCloneDepth() {
            return gitCloneDepth;
        }

        public final void setGitCloneDepth(Integer gitCloneDepth) {
            this.gitCloneDepth = gitCloneDepth;
        }

        @Override
        public final Builder gitCloneDepth(Integer gitCloneDepth) {
            this.gitCloneDepth = gitCloneDepth;
            return this;
        }

        public final GitSubmodulesConfig.Builder getGitSubmodulesConfig() {
            return gitSubmodulesConfig != null ? gitSubmodulesConfig.toBuilder() : null;
        }

        public final void setGitSubmodulesConfig(GitSubmodulesConfig.BuilderImpl gitSubmodulesConfig) {
            this.gitSubmodulesConfig = gitSubmodulesConfig != null ? gitSubmodulesConfig.build() : null;
        }

        @Override
        public final Builder gitSubmodulesConfig(GitSubmodulesConfig gitSubmodulesConfig) {
            this.gitSubmodulesConfig = gitSubmodulesConfig;
            return this;
        }

        public final String getBuildspec() {
            return buildspec;
        }

        public final void setBuildspec(String buildspec) {
            this.buildspec = buildspec;
        }

        @Override
        public final Builder buildspec(String buildspec) {
            this.buildspec = buildspec;
            return this;
        }

        public final SourceAuth.Builder getAuth() {
            return auth != null ? auth.toBuilder() : null;
        }

        public final void setAuth(SourceAuth.BuilderImpl auth) {
            this.auth = auth != null ? auth.build() : null;
        }

        @Override
        public final Builder auth(SourceAuth auth) {
            this.auth = auth;
            return this;
        }

        public final Boolean getReportBuildStatus() {
            return reportBuildStatus;
        }

        public final void setReportBuildStatus(Boolean reportBuildStatus) {
            this.reportBuildStatus = reportBuildStatus;
        }

        @Override
        public final Builder reportBuildStatus(Boolean reportBuildStatus) {
            this.reportBuildStatus = reportBuildStatus;
            return this;
        }

        public final BuildStatusConfig.Builder getBuildStatusConfig() {
            return buildStatusConfig != null ? buildStatusConfig.toBuilder() : null;
        }

        public final void setBuildStatusConfig(BuildStatusConfig.BuilderImpl buildStatusConfig) {
            this.buildStatusConfig = buildStatusConfig != null ? buildStatusConfig.build() : null;
        }

        @Override
        public final Builder buildStatusConfig(BuildStatusConfig buildStatusConfig) {
            this.buildStatusConfig = buildStatusConfig;
            return this;
        }

        public final Boolean getInsecureSsl() {
            return insecureSsl;
        }

        public final void setInsecureSsl(Boolean insecureSsl) {
            this.insecureSsl = insecureSsl;
        }

        @Override
        public final Builder insecureSsl(Boolean insecureSsl) {
            this.insecureSsl = insecureSsl;
            return this;
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        @Override
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        @Override
        public ProjectSource build() {
            return new ProjectSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
