/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A filter used to determine which webhooks trigger a build.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebhookFilter implements SdkPojo, Serializable, ToCopyableBuilder<WebhookFilter.Builder, WebhookFilter> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(WebhookFilter::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("pattern")
            .getter(getter(WebhookFilter::pattern)).setter(setter(Builder::pattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pattern").build()).build();

    private static final SdkField<Boolean> EXCLUDE_MATCHED_PATTERN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("excludeMatchedPattern").getter(getter(WebhookFilter::excludeMatchedPattern))
            .setter(setter(Builder::excludeMatchedPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("excludeMatchedPattern").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, PATTERN_FIELD,
            EXCLUDE_MATCHED_PATTERN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String pattern;

    private final Boolean excludeMatchedPattern;

    private WebhookFilter(BuilderImpl builder) {
        this.type = builder.type;
        this.pattern = builder.pattern;
        this.excludeMatchedPattern = builder.excludeMatchedPattern;
    }

    /**
     * <p>
     * The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
     * <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
     * <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and <code>WORKFLOW_NAME</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * EVENT
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event types:
     * <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
     * <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>,
     * <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The <code>EVENT</code>
     * patterns are specified as a comma-separated string. For example,
     * <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created, and pull
     * request updated events.
     * </p>
     * <note>
     * <p>
     * Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and GitHub
     * Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * ACTOR_ACCOUNT_ID
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the regular
     * expression <code>pattern</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * HEAD_REF
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when the head reference matches the regular expression <code>pattern</code>. For
     * example, <code>refs/heads/branch-name</code> and <code>refs/tags/tag-name</code>.
     * </p>
     * <note>
     * <p>
     * Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push, and
     * Bitbucket pull request events.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * BASE_REF
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when the base reference matches the regular expression <code>pattern</code>. For
     * example, <code>refs/heads/branch-name</code>.
     * </p>
     * <note>
     * <p>
     * Works with pull request events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * FILE_PATH
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the path of a changed file matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
     * events, but does not work with GitHub Enterprise pull request events.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * COMMIT_MESSAGE
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the head commit message matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
     * events, but does not work with GitHub Enterprise pull request events.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * TAG_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the tag name of the release matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * RELEASE_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * REPOSITORY_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the repository name matches the regular expression pattern.
     * </p>
     * <note>
     * <p>
     * Works with GitHub global or organization webhooks only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * WORKFLOW_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link WebhookFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
     *         <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
     *         <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and
     *         <code>WORKFLOW_NAME</code>. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         EVENT
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event types:
     *         <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
     *         <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>,
     *         <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The
     *         <code>EVENT</code> patterns are specified as a comma-separated string. For example,
     *         <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created, and
     *         pull request updated events.
     *         </p>
     *         <note>
     *         <p>
     *         Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and GitHub
     *         Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         ACTOR_ACCOUNT_ID
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the
     *         regular expression <code>pattern</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         HEAD_REF
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when the head reference matches the regular expression
     *         <code>pattern</code>. For example, <code>refs/heads/branch-name</code> and
     *         <code>refs/tags/tag-name</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push,
     *         and Bitbucket pull request events.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         BASE_REF
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when the base reference matches the regular expression
     *         <code>pattern</code>. For example, <code>refs/heads/branch-name</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with pull request events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         FILE_PATH
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the path of a changed file matches the regular expression
     *         <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise
     *         push events, but does not work with GitHub Enterprise pull request events.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         COMMIT_MESSAGE
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the head commit message matches the regular expression
     *         <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise
     *         push events, but does not work with GitHub Enterprise pull request events.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         TAG_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the tag name of the release matches the regular expression
     *         <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         RELEASE_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         REPOSITORY_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the repository name matches the regular expression pattern.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub global or organization webhooks only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         WORKFLOW_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     * @see WebhookFilterType
     */
    public final WebhookFilterType type() {
        return WebhookFilterType.fromValue(type);
    }

    /**
     * <p>
     * The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
     * <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
     * <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and <code>WORKFLOW_NAME</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * EVENT
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event types:
     * <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
     * <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>,
     * <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The <code>EVENT</code>
     * patterns are specified as a comma-separated string. For example,
     * <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created, and pull
     * request updated events.
     * </p>
     * <note>
     * <p>
     * Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and GitHub
     * Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * ACTOR_ACCOUNT_ID
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the regular
     * expression <code>pattern</code>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * HEAD_REF
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when the head reference matches the regular expression <code>pattern</code>. For
     * example, <code>refs/heads/branch-name</code> and <code>refs/tags/tag-name</code>.
     * </p>
     * <note>
     * <p>
     * Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push, and
     * Bitbucket pull request events.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * BASE_REF
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook event triggers a build when the base reference matches the regular expression <code>pattern</code>. For
     * example, <code>refs/heads/branch-name</code>.
     * </p>
     * <note>
     * <p>
     * Works with pull request events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * FILE_PATH
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the path of a changed file matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
     * events, but does not work with GitHub Enterprise pull request events.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * COMMIT_MESSAGE
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the head commit message matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
     * events, but does not work with GitHub Enterprise pull request events.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * TAG_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the tag name of the release matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * RELEASE_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * REPOSITORY_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the repository name matches the regular expression pattern.
     * </p>
     * <note>
     * <p>
     * Works with GitHub global or organization webhooks only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * WORKFLOW_NAME
     * </p>
     * <ul>
     * <li>
     * <p>
     * A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
     * </p>
     * <note>
     * <p>
     * Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
     * </p>
     * </note></li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link WebhookFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
     *         <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
     *         <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and
     *         <code>WORKFLOW_NAME</code>. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         EVENT
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event types:
     *         <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
     *         <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>,
     *         <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The
     *         <code>EVENT</code> patterns are specified as a comma-separated string. For example,
     *         <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created, and
     *         pull request updated events.
     *         </p>
     *         <note>
     *         <p>
     *         Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and GitHub
     *         Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         ACTOR_ACCOUNT_ID
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the
     *         regular expression <code>pattern</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         HEAD_REF
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when the head reference matches the regular expression
     *         <code>pattern</code>. For example, <code>refs/heads/branch-name</code> and
     *         <code>refs/tags/tag-name</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push,
     *         and Bitbucket pull request events.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         BASE_REF
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook event triggers a build when the base reference matches the regular expression
     *         <code>pattern</code>. For example, <code>refs/heads/branch-name</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with pull request events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         FILE_PATH
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the path of a changed file matches the regular expression
     *         <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise
     *         push events, but does not work with GitHub Enterprise pull request events.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         COMMIT_MESSAGE
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the head commit message matches the regular expression
     *         <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise
     *         push events, but does not work with GitHub Enterprise pull request events.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         TAG_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the tag name of the release matches the regular expression
     *         <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         RELEASE_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         REPOSITORY_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the repository name matches the regular expression pattern.
     *         </p>
     *         <note>
     *         <p>
     *         Works with GitHub global or organization webhooks only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         WORKFLOW_NAME
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     * @see WebhookFilterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * For a <code>WebHookFilter</code> that uses <code>EVENT</code> type, a comma-separated string that specifies one
     * or more events. For example, the webhook filter <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code>
     * allows all push, pull request created, and pull request updated events to trigger a build.
     * </p>
     * <p>
     * For a <code>WebHookFilter</code> that uses any of the other filter types, a regular expression pattern. For
     * example, a <code>WebHookFilter</code> that uses <code>HEAD_REF</code> for its <code>type</code> and the pattern
     * <code>^refs/heads/</code> triggers a build when the head reference is a branch with a reference name
     * <code>refs/heads/branch-name</code>.
     * </p>
     * 
     * @return For a <code>WebHookFilter</code> that uses <code>EVENT</code> type, a comma-separated string that
     *         specifies one or more events. For example, the webhook filter
     *         <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> allows all push, pull request created, and
     *         pull request updated events to trigger a build. </p>
     *         <p>
     *         For a <code>WebHookFilter</code> that uses any of the other filter types, a regular expression pattern.
     *         For example, a <code>WebHookFilter</code> that uses <code>HEAD_REF</code> for its <code>type</code> and
     *         the pattern <code>^refs/heads/</code> triggers a build when the head reference is a branch with a
     *         reference name <code>refs/heads/branch-name</code>.
     */
    public final String pattern() {
        return pattern;
    }

    /**
     * <p>
     * Used to indicate that the <code>pattern</code> determines which webhook events do not trigger a build. If true,
     * then a webhook event that does not match the <code>pattern</code> triggers a build. If false, then a webhook
     * event that matches the <code>pattern</code> triggers a build.
     * </p>
     * 
     * @return Used to indicate that the <code>pattern</code> determines which webhook events do not trigger a build. If
     *         true, then a webhook event that does not match the <code>pattern</code> triggers a build. If false, then
     *         a webhook event that matches the <code>pattern</code> triggers a build.
     */
    public final Boolean excludeMatchedPattern() {
        return excludeMatchedPattern;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(pattern());
        hashCode = 31 * hashCode + Objects.hashCode(excludeMatchedPattern());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebhookFilter)) {
            return false;
        }
        WebhookFilter other = (WebhookFilter) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(pattern(), other.pattern())
                && Objects.equals(excludeMatchedPattern(), other.excludeMatchedPattern());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WebhookFilter").add("Type", typeAsString()).add("Pattern", pattern())
                .add("ExcludeMatchedPattern", excludeMatchedPattern()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "pattern":
            return Optional.ofNullable(clazz.cast(pattern()));
        case "excludeMatchedPattern":
            return Optional.ofNullable(clazz.cast(excludeMatchedPattern()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WebhookFilter, T> g) {
        return obj -> g.apply((WebhookFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebhookFilter> {
        /**
         * <p>
         * The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
         * <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
         * <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and <code>WORKFLOW_NAME</code>
         * .
         * </p>
         * <ul>
         * <li>
         * <p>
         * EVENT
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event types:
         * <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
         * <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>,
         * <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The <code>EVENT</code>
         * patterns are specified as a comma-separated string. For example,
         * <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created, and
         * pull request updated events.
         * </p>
         * <note>
         * <p>
         * Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and GitHub
         * Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * ACTOR_ACCOUNT_ID
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the
         * regular expression <code>pattern</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * HEAD_REF
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when the head reference matches the regular expression <code>pattern</code>.
         * For example, <code>refs/heads/branch-name</code> and <code>refs/tags/tag-name</code>.
         * </p>
         * <note>
         * <p>
         * Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push, and
         * Bitbucket pull request events.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * BASE_REF
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when the base reference matches the regular expression <code>pattern</code>.
         * For example, <code>refs/heads/branch-name</code>.
         * </p>
         * <note>
         * <p>
         * Works with pull request events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * FILE_PATH
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the path of a changed file matches the regular expression
         * <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
         * events, but does not work with GitHub Enterprise pull request events.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * COMMIT_MESSAGE
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the head commit message matches the regular expression <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
         * events, but does not work with GitHub Enterprise pull request events.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * TAG_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the tag name of the release matches the regular expression
         * <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * RELEASE_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * REPOSITORY_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the repository name matches the regular expression pattern.
         * </p>
         * <note>
         * <p>
         * Works with GitHub global or organization webhooks only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * WORKFLOW_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
         *        <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
         *        <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and
         *        <code>WORKFLOW_NAME</code>. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        EVENT
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event
         *        types: <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
         *        <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>, <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The
         *        <code>EVENT</code> patterns are specified as a comma-separated string. For example,
         *        <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created,
         *        and pull request updated events.
         *        </p>
         *        <note>
         *        <p>
         *        Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and
         *        GitHub Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub
         *        only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        ACTOR_ACCOUNT_ID
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the
         *        regular expression <code>pattern</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        HEAD_REF
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when the head reference matches the regular expression
         *        <code>pattern</code>. For example, <code>refs/heads/branch-name</code> and
         *        <code>refs/tags/tag-name</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket
         *        push, and Bitbucket pull request events.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        BASE_REF
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when the base reference matches the regular expression
         *        <code>pattern</code>. For example, <code>refs/heads/branch-name</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with pull request events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        FILE_PATH
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the path of a changed file matches the regular expression
         *        <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub
         *        Enterprise push events, but does not work with GitHub Enterprise pull request events.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        COMMIT_MESSAGE
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the head commit message matches the regular expression
         *        <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub
         *        Enterprise push events, but does not work with GitHub Enterprise pull request events.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        TAG_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the tag name of the release matches the regular expression
         *        <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        RELEASE_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        REPOSITORY_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the repository name matches the regular expression pattern.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub global or organization webhooks only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        WORKFLOW_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         * @see WebhookFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebhookFilterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
         * <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
         * <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and <code>WORKFLOW_NAME</code>
         * .
         * </p>
         * <ul>
         * <li>
         * <p>
         * EVENT
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event types:
         * <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
         * <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>,
         * <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The <code>EVENT</code>
         * patterns are specified as a comma-separated string. For example,
         * <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created, and
         * pull request updated events.
         * </p>
         * <note>
         * <p>
         * Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and GitHub
         * Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * ACTOR_ACCOUNT_ID
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the
         * regular expression <code>pattern</code>.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * HEAD_REF
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when the head reference matches the regular expression <code>pattern</code>.
         * For example, <code>refs/heads/branch-name</code> and <code>refs/tags/tag-name</code>.
         * </p>
         * <note>
         * <p>
         * Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket push, and
         * Bitbucket pull request events.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * BASE_REF
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook event triggers a build when the base reference matches the regular expression <code>pattern</code>.
         * For example, <code>refs/heads/branch-name</code>.
         * </p>
         * <note>
         * <p>
         * Works with pull request events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * FILE_PATH
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the path of a changed file matches the regular expression
         * <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
         * events, but does not work with GitHub Enterprise pull request events.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * COMMIT_MESSAGE
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the head commit message matches the regular expression <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub Enterprise push
         * events, but does not work with GitHub Enterprise pull request events.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * TAG_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the tag name of the release matches the regular expression
         * <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * RELEASE_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * REPOSITORY_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the repository name matches the regular expression pattern.
         * </p>
         * <note>
         * <p>
         * Works with GitHub global or organization webhooks only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * WORKFLOW_NAME
         * </p>
         * <ul>
         * <li>
         * <p>
         * A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
         * </p>
         * <note>
         * <p>
         * Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
         * </p>
         * </note></li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of webhook filter. There are nine webhook filter types: <code>EVENT</code>,
         *        <code>ACTOR_ACCOUNT_ID</code>, <code>HEAD_REF</code>, <code>BASE_REF</code>, <code>FILE_PATH</code>,
         *        <code>COMMIT_MESSAGE</code>, <code>TAG_NAME</code>, <code>RELEASE_NAME</code>, and
         *        <code>WORKFLOW_NAME</code>. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        EVENT
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when the provided <code>pattern</code> matches one of nine event
         *        types: <code>PUSH</code>, <code>PULL_REQUEST_CREATED</code>, <code>PULL_REQUEST_UPDATED</code>,
         *        <code>PULL_REQUEST_CLOSED</code>, <code>PULL_REQUEST_REOPENED</code>, <code>PULL_REQUEST_MERGED</code>, <code>RELEASED</code>, <code>PRERELEASED</code>, and <code>WORKFLOW_JOB_QUEUED</code>. The
         *        <code>EVENT</code> patterns are specified as a comma-separated string. For example,
         *        <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> filters all push, pull request created,
         *        and pull request updated events.
         *        </p>
         *        <note>
         *        <p>
         *        Types <code>PULL_REQUEST_REOPENED</code> and <code>WORKFLOW_JOB_QUEUED</code> work with GitHub and
         *        GitHub Enterprise only. Types <code>RELEASED</code> and <code>PRERELEASED</code> work with GitHub
         *        only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        ACTOR_ACCOUNT_ID
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when a GitHub, GitHub Enterprise, or Bitbucket account ID matches the
         *        regular expression <code>pattern</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        HEAD_REF
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when the head reference matches the regular expression
         *        <code>pattern</code>. For example, <code>refs/heads/branch-name</code> and
         *        <code>refs/tags/tag-name</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub and GitHub Enterprise push, GitHub and GitHub Enterprise pull request, Bitbucket
         *        push, and Bitbucket pull request events.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        BASE_REF
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook event triggers a build when the base reference matches the regular expression
         *        <code>pattern</code>. For example, <code>refs/heads/branch-name</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with pull request events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        FILE_PATH
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the path of a changed file matches the regular expression
         *        <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub
         *        Enterprise push events, but does not work with GitHub Enterprise pull request events.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        COMMIT_MESSAGE
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the head commit message matches the regular expression
         *        <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub and Bitbucket events push and pull requests events. Also works with GitHub
         *        Enterprise push events, but does not work with GitHub Enterprise pull request events.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        TAG_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the tag name of the release matches the regular expression
         *        <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        RELEASE_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the release name matches the regular expression <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with <code>RELEASED</code> and <code>PRERELEASED</code> events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        REPOSITORY_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the repository name matches the regular expression pattern.
         *        </p>
         *        <note>
         *        <p>
         *        Works with GitHub global or organization webhooks only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        WORKFLOW_NAME
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A webhook triggers a build when the workflow name matches the regular expression <code>pattern</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Works with <code>WORKFLOW_JOB_QUEUED</code> events only.
         *        </p>
         *        </note></li>
         *        </ul>
         *        </li>
         * @see WebhookFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebhookFilterType
         */
        Builder type(WebhookFilterType type);

        /**
         * <p>
         * For a <code>WebHookFilter</code> that uses <code>EVENT</code> type, a comma-separated string that specifies
         * one or more events. For example, the webhook filter
         * <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> allows all push, pull request created, and pull
         * request updated events to trigger a build.
         * </p>
         * <p>
         * For a <code>WebHookFilter</code> that uses any of the other filter types, a regular expression pattern. For
         * example, a <code>WebHookFilter</code> that uses <code>HEAD_REF</code> for its <code>type</code> and the
         * pattern <code>^refs/heads/</code> triggers a build when the head reference is a branch with a reference name
         * <code>refs/heads/branch-name</code>.
         * </p>
         * 
         * @param pattern
         *        For a <code>WebHookFilter</code> that uses <code>EVENT</code> type, a comma-separated string that
         *        specifies one or more events. For example, the webhook filter
         *        <code>PUSH, PULL_REQUEST_CREATED, PULL_REQUEST_UPDATED</code> allows all push, pull request created,
         *        and pull request updated events to trigger a build. </p>
         *        <p>
         *        For a <code>WebHookFilter</code> that uses any of the other filter types, a regular expression
         *        pattern. For example, a <code>WebHookFilter</code> that uses <code>HEAD_REF</code> for its
         *        <code>type</code> and the pattern <code>^refs/heads/</code> triggers a build when the head reference
         *        is a branch with a reference name <code>refs/heads/branch-name</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pattern(String pattern);

        /**
         * <p>
         * Used to indicate that the <code>pattern</code> determines which webhook events do not trigger a build. If
         * true, then a webhook event that does not match the <code>pattern</code> triggers a build. If false, then a
         * webhook event that matches the <code>pattern</code> triggers a build.
         * </p>
         * 
         * @param excludeMatchedPattern
         *        Used to indicate that the <code>pattern</code> determines which webhook events do not trigger a build.
         *        If true, then a webhook event that does not match the <code>pattern</code> triggers a build. If false,
         *        then a webhook event that matches the <code>pattern</code> triggers a build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeMatchedPattern(Boolean excludeMatchedPattern);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String pattern;

        private Boolean excludeMatchedPattern;

        private BuilderImpl() {
        }

        private BuilderImpl(WebhookFilter model) {
            type(model.type);
            pattern(model.pattern);
            excludeMatchedPattern(model.excludeMatchedPattern);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(WebhookFilterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getPattern() {
            return pattern;
        }

        public final void setPattern(String pattern) {
            this.pattern = pattern;
        }

        @Override
        public final Builder pattern(String pattern) {
            this.pattern = pattern;
            return this;
        }

        public final Boolean getExcludeMatchedPattern() {
            return excludeMatchedPattern;
        }

        public final void setExcludeMatchedPattern(Boolean excludeMatchedPattern) {
            this.excludeMatchedPattern = excludeMatchedPattern;
        }

        @Override
        public final Builder excludeMatchedPattern(Boolean excludeMatchedPattern) {
            this.excludeMatchedPattern = excludeMatchedPattern;
            return this;
        }

        @Override
        public WebhookFilter build() {
            return new WebhookFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
