/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the build output artifacts for the build project.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectArtifacts implements SdkPojo, Serializable,
        ToCopyableBuilder<ProjectArtifacts.Builder, ProjectArtifacts> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ProjectArtifacts::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("location").getter(getter(ProjectArtifacts::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("path")
            .getter(getter(ProjectArtifacts::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("path").build()).build();

    private static final SdkField<String> NAMESPACE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespaceType").getter(getter(ProjectArtifacts::namespaceTypeAsString))
            .setter(setter(Builder::namespaceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespaceType").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ProjectArtifacts::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> PACKAGING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("packaging").getter(getter(ProjectArtifacts::packagingAsString)).setter(setter(Builder::packaging))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("packaging").build()).build();

    private static final SdkField<Boolean> OVERRIDE_ARTIFACT_NAME_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("overrideArtifactName").getter(getter(ProjectArtifacts::overrideArtifactName))
            .setter(setter(Builder::overrideArtifactName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("overrideArtifactName").build())
            .build();

    private static final SdkField<Boolean> ENCRYPTION_DISABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("encryptionDisabled").getter(getter(ProjectArtifacts::encryptionDisabled))
            .setter(setter(Builder::encryptionDisabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionDisabled").build())
            .build();

    private static final SdkField<String> ARTIFACT_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("artifactIdentifier").getter(getter(ProjectArtifacts::artifactIdentifier))
            .setter(setter(Builder::artifactIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifactIdentifier").build())
            .build();

    private static final SdkField<String> BUCKET_OWNER_ACCESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketOwnerAccess").getter(getter(ProjectArtifacts::bucketOwnerAccessAsString))
            .setter(setter(Builder::bucketOwnerAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketOwnerAccess").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, LOCATION_FIELD,
            PATH_FIELD, NAMESPACE_TYPE_FIELD, NAME_FIELD, PACKAGING_FIELD, OVERRIDE_ARTIFACT_NAME_FIELD,
            ENCRYPTION_DISABLED_FIELD, ARTIFACT_IDENTIFIER_FIELD, BUCKET_OWNER_ACCESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String location;

    private final String path;

    private final String namespaceType;

    private final String name;

    private final String packaging;

    private final Boolean overrideArtifactName;

    private final Boolean encryptionDisabled;

    private final String artifactIdentifier;

    private final String bucketOwnerAccess;

    private ProjectArtifacts(BuilderImpl builder) {
        this.type = builder.type;
        this.location = builder.location;
        this.path = builder.path;
        this.namespaceType = builder.namespaceType;
        this.name = builder.name;
        this.packaging = builder.packaging;
        this.overrideArtifactName = builder.overrideArtifactName;
        this.encryptionDisabled = builder.encryptionDisabled;
        this.artifactIdentifier = builder.artifactIdentifier;
        this.bucketOwnerAccess = builder.bucketOwnerAccess;
    }

    /**
     * <p>
     * The type of build output artifact. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
     * </p>
     * <note>
     * <p>
     * The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The build project stores build output in Amazon S3.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ArtifactsType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of build output artifact. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
     *         </p>
     *         <note>
     *         <p>
     *         The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The build project stores build output in Amazon S3.
     *         </p>
     *         </li>
     * @see ArtifactsType
     */
    public final ArtifactsType type() {
        return ArtifactsType.fromValue(type);
    }

    /**
     * <p>
     * The type of build output artifact. Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
     * </p>
     * <note>
     * <p>
     * The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code>: The build project stores build output in Amazon S3.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ArtifactsType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of build output artifact. Valid values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
     *         </p>
     *         <note>
     *         <p>
     *         The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code>: The build project stores build output in Amazon S3.
     *         </p>
     *         </li>
     * @see ArtifactsType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Information about the build output artifact location:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output locations instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, this is the name of the output bucket.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Information about the build output artifact location:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output locations instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, this is the name of the output bucket.
     *         </p>
     *         </li>
     */
    public final String location() {
        return location;
    }

    /**
     * <p>
     * Along with <code>namespaceType</code> and <code>name</code>, the pattern that CodeBuild uses to name and store
     * the output artifact:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output names instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, this is the path to the output artifact. If <code>path</code> is
     * not specified, <code>path</code> is not used.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     * <code>NONE</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is stored in
     * the output bucket at <code>MyArtifacts/MyArtifact.zip</code>.
     * </p>
     * 
     * @return Along with <code>namespaceType</code> and <code>name</code>, the pattern that CodeBuild uses to name and
     *         store the output artifact:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output names instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, this is the path to the output artifact. If
     *         <code>path</code> is not specified, <code>path</code> is not used.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set
     *         to <code>NONE</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is
     *         stored in the output bucket at <code>MyArtifacts/MyArtifact.zip</code>.
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the name and
     * location to store the output artifact:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output names instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is not
     * specified.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     * <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is stored
     * in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #namespaceType}
     * will return {@link ArtifactNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #namespaceTypeAsString}.
     * </p>
     * 
     * @return Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the name
     *         and location to store the output artifact:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output names instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is not
     *         specified.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set
     *         to <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output
     *         artifact is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
     * @see ArtifactNamespace
     */
    public final ArtifactNamespace namespaceType() {
        return ArtifactNamespace.fromValue(namespaceType);
    }

    /**
     * <p>
     * Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the name and
     * location to store the output artifact:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output names instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is not
     * specified.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     * <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is stored
     * in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #namespaceType}
     * will return {@link ArtifactNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #namespaceTypeAsString}.
     * </p>
     * 
     * @return Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the name
     *         and location to store the output artifact:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output names instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is not
     *         specified.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set
     *         to <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output
     *         artifact is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
     * @see ArtifactNamespace
     */
    public final String namespaceTypeAsString() {
        return namespaceType;
    }

    /**
     * <p>
     * Along with <code>path</code> and <code>namespaceType</code>, the pattern that CodeBuild uses to name and store
     * the output artifact:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output names instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, this is the name of the output artifact object. If you set the
     * name to be a forward slash ("/"), the artifact is stored in the root of the output bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     * <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, then the output artifact is
     * stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>path</code> is empty, <code>namespaceType</code> is set to <code>NONE</code>, and <code>name</code> is
     * set to "<code>/</code>", the output artifact is stored in the root of the output bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     * <code>BUILD_ID</code>, and <code>name</code> is set to "<code>/</code>", the output artifact is stored in
     * <code>MyArtifacts/&lt;build-ID&gt;</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Along with <code>path</code> and <code>namespaceType</code>, the pattern that CodeBuild uses to name and
     *         store the output artifact:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output names instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, this is the name of the output artifact object. If you
     *         set the name to be a forward slash ("/"), the artifact is stored in the root of the output bucket.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     *         <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, then the output
     *         artifact is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>path</code> is empty, <code>namespaceType</code> is set to <code>NONE</code>, and
     *         <code>name</code> is set to "<code>/</code>", the output artifact is stored in the root of the output
     *         bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
     *         <code>BUILD_ID</code>, and <code>name</code> is set to "<code>/</code>", the output artifact is stored in
     *         <code>MyArtifacts/&lt;build-ID&gt;</code>.
     *         </p>
     *         </li>
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of build output artifact to create:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output artifacts instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output. This is the
     * default if <code>packaging</code> is not specified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #packaging} will
     * return {@link ArtifactPackaging#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #packagingAsString}.
     * </p>
     * 
     * @return The type of build output artifact to create:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output artifacts instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output. This
     *         is the default if <code>packaging</code> is not specified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see ArtifactPackaging
     */
    public final ArtifactPackaging packaging() {
        return ArtifactPackaging.fromValue(packaging);
    }

    /**
     * <p>
     * The type of build output artifact to create:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This is
     * because CodePipeline manages its build output artifacts instead of CodeBuild.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no build
     * output is produced.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>type</code> is set to <code>S3</code>, valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output. This is the
     * default if <code>packaging</code> is not specified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #packaging} will
     * return {@link ArtifactPackaging#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #packagingAsString}.
     * </p>
     * 
     * @return The type of build output artifact to create:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified.
     *         This is because CodePipeline manages its build output artifacts instead of CodeBuild.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
     *         build output is produced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>type</code> is set to <code>S3</code>, valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output. This
     *         is the default if <code>packaging</code> is not specified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see ArtifactPackaging
     */
    public final String packagingAsString() {
        return packaging;
    }

    /**
     * <p>
     * If this flag is set, a name specified in the buildspec file overrides the artifact name. The name specified in a
     * buildspec file is calculated at build time and uses the Shell Command Language. For example, you can append a
     * date and time to your artifact name so that it is always unique.
     * </p>
     * 
     * @return If this flag is set, a name specified in the buildspec file overrides the artifact name. The name
     *         specified in a buildspec file is calculated at build time and uses the Shell Command Language. For
     *         example, you can append a date and time to your artifact name so that it is always unique.
     */
    public final Boolean overrideArtifactName() {
        return overrideArtifactName;
    }

    /**
     * <p>
     * Set to true if you do not want your output artifacts encrypted. This option is valid only if your artifacts type
     * is Amazon S3. If this is set with another artifacts type, an invalidInputException is thrown.
     * </p>
     * 
     * @return Set to true if you do not want your output artifacts encrypted. This option is valid only if your
     *         artifacts type is Amazon S3. If this is set with another artifacts type, an invalidInputException is
     *         thrown.
     */
    public final Boolean encryptionDisabled() {
        return encryptionDisabled;
    }

    /**
     * <p>
     * An identifier for this artifact definition.
     * </p>
     * 
     * @return An identifier for this artifact definition.
     */
    public final String artifactIdentifier() {
        return artifactIdentifier;
    }

    /**
     * Returns the value of the BucketOwnerAccess property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bucketOwnerAccess}
     * will return {@link BucketOwnerAccess#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #bucketOwnerAccessAsString}.
     * </p>
     * 
     * @return The value of the BucketOwnerAccess property for this object.
     * @see BucketOwnerAccess
     */
    public final BucketOwnerAccess bucketOwnerAccess() {
        return BucketOwnerAccess.fromValue(bucketOwnerAccess);
    }

    /**
     * Returns the value of the BucketOwnerAccess property for this object.
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bucketOwnerAccess}
     * will return {@link BucketOwnerAccess#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #bucketOwnerAccessAsString}.
     * </p>
     * 
     * @return The value of the BucketOwnerAccess property for this object.
     * @see BucketOwnerAccess
     */
    public final String bucketOwnerAccessAsString() {
        return bucketOwnerAccess;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(namespaceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(packagingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(overrideArtifactName());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionDisabled());
        hashCode = 31 * hashCode + Objects.hashCode(artifactIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(bucketOwnerAccessAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectArtifacts)) {
            return false;
        }
        ProjectArtifacts other = (ProjectArtifacts) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(location(), other.location())
                && Objects.equals(path(), other.path()) && Objects.equals(namespaceTypeAsString(), other.namespaceTypeAsString())
                && Objects.equals(name(), other.name()) && Objects.equals(packagingAsString(), other.packagingAsString())
                && Objects.equals(overrideArtifactName(), other.overrideArtifactName())
                && Objects.equals(encryptionDisabled(), other.encryptionDisabled())
                && Objects.equals(artifactIdentifier(), other.artifactIdentifier())
                && Objects.equals(bucketOwnerAccessAsString(), other.bucketOwnerAccessAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProjectArtifacts").add("Type", typeAsString()).add("Location", location()).add("Path", path())
                .add("NamespaceType", namespaceTypeAsString()).add("Name", name()).add("Packaging", packagingAsString())
                .add("OverrideArtifactName", overrideArtifactName()).add("EncryptionDisabled", encryptionDisabled())
                .add("ArtifactIdentifier", artifactIdentifier()).add("BucketOwnerAccess", bucketOwnerAccessAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "path":
            return Optional.ofNullable(clazz.cast(path()));
        case "namespaceType":
            return Optional.ofNullable(clazz.cast(namespaceTypeAsString()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "packaging":
            return Optional.ofNullable(clazz.cast(packagingAsString()));
        case "overrideArtifactName":
            return Optional.ofNullable(clazz.cast(overrideArtifactName()));
        case "encryptionDisabled":
            return Optional.ofNullable(clazz.cast(encryptionDisabled()));
        case "artifactIdentifier":
            return Optional.ofNullable(clazz.cast(artifactIdentifier()));
        case "bucketOwnerAccess":
            return Optional.ofNullable(clazz.cast(bucketOwnerAccessAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("location", LOCATION_FIELD);
        map.put("path", PATH_FIELD);
        map.put("namespaceType", NAMESPACE_TYPE_FIELD);
        map.put("name", NAME_FIELD);
        map.put("packaging", PACKAGING_FIELD);
        map.put("overrideArtifactName", OVERRIDE_ARTIFACT_NAME_FIELD);
        map.put("encryptionDisabled", ENCRYPTION_DISABLED_FIELD);
        map.put("artifactIdentifier", ARTIFACT_IDENTIFIER_FIELD);
        map.put("bucketOwnerAccess", BUCKET_OWNER_ACCESS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ProjectArtifacts, T> g) {
        return obj -> g.apply((ProjectArtifacts) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectArtifacts> {
        /**
         * <p>
         * The type of build output artifact. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
         * </p>
         * <note>
         * <p>
         * The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The build project stores build output in Amazon S3.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of build output artifact. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
         *        </p>
         *        <note>
         *        <p>
         *        The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The build project stores build output in Amazon S3.
         *        </p>
         *        </li>
         * @see ArtifactsType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactsType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of build output artifact. Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
         * </p>
         * <note>
         * <p>
         * The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code>: The build project stores build output in Amazon S3.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of build output artifact. Valid values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CODEPIPELINE</code>: The build project has build output generated through CodePipeline.
         *        </p>
         *        <note>
         *        <p>
         *        The <code>CODEPIPELINE</code> type is not supported for <code>secondaryArtifacts</code>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NO_ARTIFACTS</code>: The build project does not produce any build output.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code>: The build project stores build output in Amazon S3.
         *        </p>
         *        </li>
         * @see ArtifactsType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactsType
         */
        Builder type(ArtifactsType type);

        /**
         * <p>
         * Information about the build output artifact location:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output locations instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, this is the name of the output bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param location
         *        Information about the build output artifact location:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output locations instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, this is the name of the output bucket.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * Along with <code>namespaceType</code> and <code>name</code>, the pattern that CodeBuild uses to name and
         * store the output artifact:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output names instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, this is the path to the output artifact. If <code>path</code>
         * is not specified, <code>path</code> is not used.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         * <code>NONE</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is stored
         * in the output bucket at <code>MyArtifacts/MyArtifact.zip</code>.
         * </p>
         * 
         * @param path
         *        Along with <code>namespaceType</code> and <code>name</code>, the pattern that CodeBuild uses to name
         *        and store the output artifact:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output names instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, this is the path to the output artifact. If
         *        <code>path</code> is not specified, <code>path</code> is not used.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is
         *        set to <code>NONE</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output
         *        artifact is stored in the output bucket at <code>MyArtifacts/MyArtifact.zip</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the name and
         * location to store the output artifact:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output names instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is not
         * specified.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         * <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is
         * stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
         * </p>
         * 
         * @param namespaceType
         *        Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the
         *        name and location to store the output artifact:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output names instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is
         *        not specified.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is
         *        set to <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output
         *        artifact is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
         * @see ArtifactNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactNamespace
         */
        Builder namespaceType(String namespaceType);

        /**
         * <p>
         * Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the name and
         * location to store the output artifact:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output names instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is not
         * specified.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         * <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output artifact is
         * stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
         * </p>
         * 
         * @param namespaceType
         *        Along with <code>path</code> and <code>name</code>, the pattern that CodeBuild uses to determine the
         *        name and location to store the output artifact:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output names instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BUILD_ID</code>: Include the build ID in the location of the build output artifact.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code>: Do not include the build ID. This is the default if <code>namespaceType</code> is
         *        not specified.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, if <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is
         *        set to <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, the output
         *        artifact is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
         * @see ArtifactNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactNamespace
         */
        Builder namespaceType(ArtifactNamespace namespaceType);

        /**
         * <p>
         * Along with <code>path</code> and <code>namespaceType</code>, the pattern that CodeBuild uses to name and
         * store the output artifact:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output names instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, this is the name of the output artifact object. If you set
         * the name to be a forward slash ("/"), the artifact is stored in the root of the output bucket.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         * <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, then the output artifact
         * is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>path</code> is empty, <code>namespaceType</code> is set to <code>NONE</code>, and <code>name</code>
         * is set to "<code>/</code>", the output artifact is stored in the root of the output bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         * <code>BUILD_ID</code>, and <code>name</code> is set to "<code>/</code>", the output artifact is stored in
         * <code>MyArtifacts/&lt;build-ID&gt;</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        Along with <code>path</code> and <code>namespaceType</code>, the pattern that CodeBuild uses to name
         *        and store the output artifact:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output names instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, this is the name of the output artifact object. If you
         *        set the name to be a forward slash ("/"), the artifact is stored in the root of the output bucket.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         *        <code>BUILD_ID</code>, and <code>name</code> is set to <code>MyArtifact.zip</code>, then the output
         *        artifact is stored in <code>MyArtifacts/&lt;build-ID&gt;/MyArtifact.zip</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>path</code> is empty, <code>namespaceType</code> is set to <code>NONE</code>, and
         *        <code>name</code> is set to "<code>/</code>", the output artifact is stored in the root of the output
         *        bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>path</code> is set to <code>MyArtifacts</code>, <code>namespaceType</code> is set to
         *        <code>BUILD_ID</code>, and <code>name</code> is set to "<code>/</code>", the output artifact is stored
         *        in <code>MyArtifacts/&lt;build-ID&gt;</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of build output artifact to create:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output artifacts instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output. This is
         * the default if <code>packaging</code> is not specified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param packaging
         *        The type of build output artifact to create:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output artifacts instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output.
         *        This is the default if <code>packaging</code> is not specified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see ArtifactPackaging
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactPackaging
         */
        Builder packaging(String packaging);

        /**
         * <p>
         * The type of build output artifact to create:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if specified. This
         * is because CodePipeline manages its build output artifacts instead of CodeBuild.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because no
         * build output is produced.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>type</code> is set to <code>S3</code>, valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output. This is
         * the default if <code>packaging</code> is not specified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param packaging
         *        The type of build output artifact to create:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>CODEPIPELINE</code>, CodePipeline ignores this value if
         *        specified. This is because CodePipeline manages its build output artifacts instead of CodeBuild.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>NO_ARTIFACTS</code>, this value is ignored if specified, because
         *        no build output is produced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>type</code> is set to <code>S3</code>, valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code>: CodeBuild creates in the output bucket a folder that contains the build output.
         *        This is the default if <code>packaging</code> is not specified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ZIP</code>: CodeBuild creates in the output bucket a ZIP file that contains the build output.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see ArtifactPackaging
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactPackaging
         */
        Builder packaging(ArtifactPackaging packaging);

        /**
         * <p>
         * If this flag is set, a name specified in the buildspec file overrides the artifact name. The name specified
         * in a buildspec file is calculated at build time and uses the Shell Command Language. For example, you can
         * append a date and time to your artifact name so that it is always unique.
         * </p>
         * 
         * @param overrideArtifactName
         *        If this flag is set, a name specified in the buildspec file overrides the artifact name. The name
         *        specified in a buildspec file is calculated at build time and uses the Shell Command Language. For
         *        example, you can append a date and time to your artifact name so that it is always unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrideArtifactName(Boolean overrideArtifactName);

        /**
         * <p>
         * Set to true if you do not want your output artifacts encrypted. This option is valid only if your artifacts
         * type is Amazon S3. If this is set with another artifacts type, an invalidInputException is thrown.
         * </p>
         * 
         * @param encryptionDisabled
         *        Set to true if you do not want your output artifacts encrypted. This option is valid only if your
         *        artifacts type is Amazon S3. If this is set with another artifacts type, an invalidInputException is
         *        thrown.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionDisabled(Boolean encryptionDisabled);

        /**
         * <p>
         * An identifier for this artifact definition.
         * </p>
         * 
         * @param artifactIdentifier
         *        An identifier for this artifact definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactIdentifier(String artifactIdentifier);

        /**
         * Sets the value of the BucketOwnerAccess property for this object.
         *
         * @param bucketOwnerAccess
         *        The new value for the BucketOwnerAccess property for this object.
         * @see BucketOwnerAccess
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BucketOwnerAccess
         */
        Builder bucketOwnerAccess(String bucketOwnerAccess);

        /**
         * Sets the value of the BucketOwnerAccess property for this object.
         *
         * @param bucketOwnerAccess
         *        The new value for the BucketOwnerAccess property for this object.
         * @see BucketOwnerAccess
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BucketOwnerAccess
         */
        Builder bucketOwnerAccess(BucketOwnerAccess bucketOwnerAccess);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String location;

        private String path;

        private String namespaceType;

        private String name;

        private String packaging;

        private Boolean overrideArtifactName;

        private Boolean encryptionDisabled;

        private String artifactIdentifier;

        private String bucketOwnerAccess;

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectArtifacts model) {
            type(model.type);
            location(model.location);
            path(model.path);
            namespaceType(model.namespaceType);
            name(model.name);
            packaging(model.packaging);
            overrideArtifactName(model.overrideArtifactName);
            encryptionDisabled(model.encryptionDisabled);
            artifactIdentifier(model.artifactIdentifier);
            bucketOwnerAccess(model.bucketOwnerAccess);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ArtifactsType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final String getNamespaceType() {
            return namespaceType;
        }

        public final void setNamespaceType(String namespaceType) {
            this.namespaceType = namespaceType;
        }

        @Override
        public final Builder namespaceType(String namespaceType) {
            this.namespaceType = namespaceType;
            return this;
        }

        @Override
        public final Builder namespaceType(ArtifactNamespace namespaceType) {
            this.namespaceType(namespaceType == null ? null : namespaceType.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getPackaging() {
            return packaging;
        }

        public final void setPackaging(String packaging) {
            this.packaging = packaging;
        }

        @Override
        public final Builder packaging(String packaging) {
            this.packaging = packaging;
            return this;
        }

        @Override
        public final Builder packaging(ArtifactPackaging packaging) {
            this.packaging(packaging == null ? null : packaging.toString());
            return this;
        }

        public final Boolean getOverrideArtifactName() {
            return overrideArtifactName;
        }

        public final void setOverrideArtifactName(Boolean overrideArtifactName) {
            this.overrideArtifactName = overrideArtifactName;
        }

        @Override
        public final Builder overrideArtifactName(Boolean overrideArtifactName) {
            this.overrideArtifactName = overrideArtifactName;
            return this;
        }

        public final Boolean getEncryptionDisabled() {
            return encryptionDisabled;
        }

        public final void setEncryptionDisabled(Boolean encryptionDisabled) {
            this.encryptionDisabled = encryptionDisabled;
        }

        @Override
        public final Builder encryptionDisabled(Boolean encryptionDisabled) {
            this.encryptionDisabled = encryptionDisabled;
            return this;
        }

        public final String getArtifactIdentifier() {
            return artifactIdentifier;
        }

        public final void setArtifactIdentifier(String artifactIdentifier) {
            this.artifactIdentifier = artifactIdentifier;
        }

        @Override
        public final Builder artifactIdentifier(String artifactIdentifier) {
            this.artifactIdentifier = artifactIdentifier;
            return this;
        }

        public final String getBucketOwnerAccess() {
            return bucketOwnerAccess;
        }

        public final void setBucketOwnerAccess(String bucketOwnerAccess) {
            this.bucketOwnerAccess = bucketOwnerAccess;
        }

        @Override
        public final Builder bucketOwnerAccess(String bucketOwnerAccess) {
            this.bucketOwnerAccess = bucketOwnerAccess;
            return this;
        }

        @Override
        public final Builder bucketOwnerAccess(BucketOwnerAccess bucketOwnerAccess) {
            this.bucketOwnerAccess(bucketOwnerAccess == null ? null : bucketOwnerAccess.toString());
            return this;
        }

        @Override
        public ProjectArtifacts build() {
            return new ProjectArtifacts(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
