/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codebuild.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A PullRequestBuildPolicy object that defines comment-based approval requirements for triggering builds on pull
 * requests. This policy helps control when automated builds are executed based on contributor permissions and approval
 * workflows.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PullRequestBuildPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<PullRequestBuildPolicy.Builder, PullRequestBuildPolicy> {
    private static final SdkField<String> REQUIRES_COMMENT_APPROVAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("requiresCommentApproval").getter(getter(PullRequestBuildPolicy::requiresCommentApprovalAsString))
            .setter(setter(Builder::requiresCommentApproval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requiresCommentApproval").build())
            .build();

    private static final SdkField<List<String>> APPROVER_ROLES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("approverRoles")
            .getter(getter(PullRequestBuildPolicy::approverRolesAsStrings))
            .setter(setter(Builder::approverRolesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("approverRoles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            REQUIRES_COMMENT_APPROVAL_FIELD, APPROVER_ROLES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String requiresCommentApproval;

    private final List<String> approverRoles;

    private PullRequestBuildPolicy(BuilderImpl builder) {
        this.requiresCommentApproval = builder.requiresCommentApproval;
        this.approverRoles = builder.approverRoles;
    }

    /**
     * <p>
     * Specifies when comment-based approval is required before triggering a build on pull requests. This setting
     * determines whether builds run automatically or require explicit approval through comments.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless contributor is
     * one of the approver roles)
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval (unless
     * contributor is one of the approver roles)
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #requiresCommentApproval} will return {@link PullRequestBuildCommentApproval#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #requiresCommentApprovalAsString}.
     * </p>
     * 
     * @return Specifies when comment-based approval is required before triggering a build on pull requests. This
     *         setting determines whether builds run automatically or require explicit approval through comments.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless
     *         contributor is one of the approver roles)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval (unless
     *         contributor is one of the approver roles)
     *         </p>
     *         </li>
     * @see PullRequestBuildCommentApproval
     */
    public final PullRequestBuildCommentApproval requiresCommentApproval() {
        return PullRequestBuildCommentApproval.fromValue(requiresCommentApproval);
    }

    /**
     * <p>
     * Specifies when comment-based approval is required before triggering a build on pull requests. This setting
     * determines whether builds run automatically or require explicit approval through comments.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless contributor is
     * one of the approver roles)
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval (unless
     * contributor is one of the approver roles)
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #requiresCommentApproval} will return {@link PullRequestBuildCommentApproval#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #requiresCommentApprovalAsString}.
     * </p>
     * 
     * @return Specifies when comment-based approval is required before triggering a build on pull requests. This
     *         setting determines whether builds run automatically or require explicit approval through comments.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless
     *         contributor is one of the approver roles)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval (unless
     *         contributor is one of the approver roles)
     *         </p>
     *         </li>
     * @see PullRequestBuildCommentApproval
     */
    public final String requiresCommentApprovalAsString() {
        return requiresCommentApproval;
    }

    /**
     * <p>
     * List of repository roles that have approval privileges for pull request builds when comment approval is required.
     * Only users with these roles can provide valid comment approvals. If a pull request contributor is one of these
     * roles, their pull request builds will trigger automatically. This field is only applicable when
     * <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApproverRoles} method.
     * </p>
     * 
     * @return List of repository roles that have approval privileges for pull request builds when comment approval is
     *         required. Only users with these roles can provide valid comment approvals. If a pull request contributor
     *         is one of these roles, their pull request builds will trigger automatically. This field is only
     *         applicable when <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
     */
    public final List<PullRequestBuildApproverRole> approverRoles() {
        return PullRequestBuildApproverRolesCopier.copyStringToEnum(approverRoles);
    }

    /**
     * For responses, this returns true if the service returned a value for the ApproverRoles property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasApproverRoles() {
        return approverRoles != null && !(approverRoles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of repository roles that have approval privileges for pull request builds when comment approval is required.
     * Only users with these roles can provide valid comment approvals. If a pull request contributor is one of these
     * roles, their pull request builds will trigger automatically. This field is only applicable when
     * <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApproverRoles} method.
     * </p>
     * 
     * @return List of repository roles that have approval privileges for pull request builds when comment approval is
     *         required. Only users with these roles can provide valid comment approvals. If a pull request contributor
     *         is one of these roles, their pull request builds will trigger automatically. This field is only
     *         applicable when <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
     */
    public final List<String> approverRolesAsStrings() {
        return approverRoles;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(requiresCommentApprovalAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasApproverRoles() ? approverRolesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PullRequestBuildPolicy)) {
            return false;
        }
        PullRequestBuildPolicy other = (PullRequestBuildPolicy) obj;
        return Objects.equals(requiresCommentApprovalAsString(), other.requiresCommentApprovalAsString())
                && hasApproverRoles() == other.hasApproverRoles()
                && Objects.equals(approverRolesAsStrings(), other.approverRolesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PullRequestBuildPolicy").add("RequiresCommentApproval", requiresCommentApprovalAsString())
                .add("ApproverRoles", hasApproverRoles() ? approverRolesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "requiresCommentApproval":
            return Optional.ofNullable(clazz.cast(requiresCommentApprovalAsString()));
        case "approverRoles":
            return Optional.ofNullable(clazz.cast(approverRolesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("requiresCommentApproval", REQUIRES_COMMENT_APPROVAL_FIELD);
        map.put("approverRoles", APPROVER_ROLES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PullRequestBuildPolicy, T> g) {
        return obj -> g.apply((PullRequestBuildPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PullRequestBuildPolicy> {
        /**
         * <p>
         * Specifies when comment-based approval is required before triggering a build on pull requests. This setting
         * determines whether builds run automatically or require explicit approval through comments.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless
         * contributor is one of the approver roles)
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval (unless
         * contributor is one of the approver roles)
         * </p>
         * </li>
         * </ul>
         * 
         * @param requiresCommentApproval
         *        Specifies when comment-based approval is required before triggering a build on pull requests. This
         *        setting determines whether builds run automatically or require explicit approval through comments.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless
         *        contributor is one of the approver roles)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval
         *        (unless contributor is one of the approver roles)
         *        </p>
         *        </li>
         * @see PullRequestBuildCommentApproval
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PullRequestBuildCommentApproval
         */
        Builder requiresCommentApproval(String requiresCommentApproval);

        /**
         * <p>
         * Specifies when comment-based approval is required before triggering a build on pull requests. This setting
         * determines whether builds run automatically or require explicit approval through comments.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless
         * contributor is one of the approver roles)
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval (unless
         * contributor is one of the approver roles)
         * </p>
         * </li>
         * </ul>
         * 
         * @param requiresCommentApproval
         *        Specifies when comment-based approval is required before triggering a build on pull requests. This
         *        setting determines whether builds run automatically or require explicit approval through comments.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>DISABLED</i>: Builds trigger automatically without requiring comment approval
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>ALL_PULL_REQUESTS</i>: All pull requests require comment approval before builds execute (unless
         *        contributor is one of the approver roles)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>FORK_PULL_REQUESTS</i>: Only pull requests from forked repositories require comment approval
         *        (unless contributor is one of the approver roles)
         *        </p>
         *        </li>
         * @see PullRequestBuildCommentApproval
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PullRequestBuildCommentApproval
         */
        Builder requiresCommentApproval(PullRequestBuildCommentApproval requiresCommentApproval);

        /**
         * <p>
         * List of repository roles that have approval privileges for pull request builds when comment approval is
         * required. Only users with these roles can provide valid comment approvals. If a pull request contributor is
         * one of these roles, their pull request builds will trigger automatically. This field is only applicable when
         * <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * </p>
         * 
         * @param approverRoles
         *        List of repository roles that have approval privileges for pull request builds when comment approval
         *        is required. Only users with these roles can provide valid comment approvals. If a pull request
         *        contributor is one of these roles, their pull request builds will trigger automatically. This field is
         *        only applicable when <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approverRolesWithStrings(Collection<String> approverRoles);

        /**
         * <p>
         * List of repository roles that have approval privileges for pull request builds when comment approval is
         * required. Only users with these roles can provide valid comment approvals. If a pull request contributor is
         * one of these roles, their pull request builds will trigger automatically. This field is only applicable when
         * <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * </p>
         * 
         * @param approverRoles
         *        List of repository roles that have approval privileges for pull request builds when comment approval
         *        is required. Only users with these roles can provide valid comment approvals. If a pull request
         *        contributor is one of these roles, their pull request builds will trigger automatically. This field is
         *        only applicable when <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approverRolesWithStrings(String... approverRoles);

        /**
         * <p>
         * List of repository roles that have approval privileges for pull request builds when comment approval is
         * required. Only users with these roles can provide valid comment approvals. If a pull request contributor is
         * one of these roles, their pull request builds will trigger automatically. This field is only applicable when
         * <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * </p>
         * 
         * @param approverRoles
         *        List of repository roles that have approval privileges for pull request builds when comment approval
         *        is required. Only users with these roles can provide valid comment approvals. If a pull request
         *        contributor is one of these roles, their pull request builds will trigger automatically. This field is
         *        only applicable when <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approverRoles(Collection<PullRequestBuildApproverRole> approverRoles);

        /**
         * <p>
         * List of repository roles that have approval privileges for pull request builds when comment approval is
         * required. Only users with these roles can provide valid comment approvals. If a pull request contributor is
         * one of these roles, their pull request builds will trigger automatically. This field is only applicable when
         * <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * </p>
         * 
         * @param approverRoles
         *        List of repository roles that have approval privileges for pull request builds when comment approval
         *        is required. Only users with these roles can provide valid comment approvals. If a pull request
         *        contributor is one of these roles, their pull request builds will trigger automatically. This field is
         *        only applicable when <code>requiresCommentApproval</code> is not <i>DISABLED</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approverRoles(PullRequestBuildApproverRole... approverRoles);
    }

    static final class BuilderImpl implements Builder {
        private String requiresCommentApproval;

        private List<String> approverRoles = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PullRequestBuildPolicy model) {
            requiresCommentApproval(model.requiresCommentApproval);
            approverRolesWithStrings(model.approverRoles);
        }

        public final String getRequiresCommentApproval() {
            return requiresCommentApproval;
        }

        public final void setRequiresCommentApproval(String requiresCommentApproval) {
            this.requiresCommentApproval = requiresCommentApproval;
        }

        @Override
        public final Builder requiresCommentApproval(String requiresCommentApproval) {
            this.requiresCommentApproval = requiresCommentApproval;
            return this;
        }

        @Override
        public final Builder requiresCommentApproval(PullRequestBuildCommentApproval requiresCommentApproval) {
            this.requiresCommentApproval(requiresCommentApproval == null ? null : requiresCommentApproval.toString());
            return this;
        }

        public final Collection<String> getApproverRoles() {
            if (approverRoles instanceof SdkAutoConstructList) {
                return null;
            }
            return approverRoles;
        }

        public final void setApproverRoles(Collection<String> approverRoles) {
            this.approverRoles = PullRequestBuildApproverRolesCopier.copy(approverRoles);
        }

        @Override
        public final Builder approverRolesWithStrings(Collection<String> approverRoles) {
            this.approverRoles = PullRequestBuildApproverRolesCopier.copy(approverRoles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder approverRolesWithStrings(String... approverRoles) {
            approverRolesWithStrings(Arrays.asList(approverRoles));
            return this;
        }

        @Override
        public final Builder approverRoles(Collection<PullRequestBuildApproverRole> approverRoles) {
            this.approverRoles = PullRequestBuildApproverRolesCopier.copyEnumToString(approverRoles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder approverRoles(PullRequestBuildApproverRole... approverRoles) {
            approverRoles(Arrays.asList(approverRoles));
            return this;
        }

        @Override
        public PullRequestBuildPolicy build() {
            return new PullRequestBuildPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
