/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecatalyst.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a workflow run.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkflowRunSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkflowRunSummary.Builder, WorkflowRunSummary> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(WorkflowRunSummary::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> WORKFLOW_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("workflowId").getter(getter(WorkflowRunSummary::workflowId)).setter(setter(Builder::workflowId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workflowId").build()).build();

    private static final SdkField<String> WORKFLOW_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("workflowName").getter(getter(WorkflowRunSummary::workflowName)).setter(setter(Builder::workflowName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("workflowName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(WorkflowRunSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<WorkflowRunStatusReason>> STATUS_REASONS_FIELD = SdkField
            .<List<WorkflowRunStatusReason>> builder(MarshallingType.LIST)
            .memberName("statusReasons")
            .getter(getter(WorkflowRunSummary::statusReasons))
            .setter(setter(Builder::statusReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<WorkflowRunStatusReason> builder(MarshallingType.SDK_POJO)
                                            .constructor(WorkflowRunStatusReason::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime")
            .getter(getter(WorkflowRunSummary::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime")
            .getter(getter(WorkflowRunSummary::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdatedTime")
            .getter(getter(WorkflowRunSummary::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, WORKFLOW_ID_FIELD,
            WORKFLOW_NAME_FIELD, STATUS_FIELD, STATUS_REASONS_FIELD, START_TIME_FIELD, END_TIME_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String workflowId;

    private final String workflowName;

    private final String status;

    private final List<WorkflowRunStatusReason> statusReasons;

    private final Instant startTime;

    private final Instant endTime;

    private final Instant lastUpdatedTime;

    private WorkflowRunSummary(BuilderImpl builder) {
        this.id = builder.id;
        this.workflowId = builder.workflowId;
        this.workflowName = builder.workflowName;
        this.status = builder.status;
        this.statusReasons = builder.statusReasons;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The system-generated unique ID of the workflow run.
     * </p>
     * 
     * @return The system-generated unique ID of the workflow run.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The system-generated unique ID of the workflow.
     * </p>
     * 
     * @return The system-generated unique ID of the workflow.
     */
    public final String workflowId() {
        return workflowId;
    }

    /**
     * <p>
     * The name of the workflow.
     * </p>
     * 
     * @return The name of the workflow.
     */
    public final String workflowName() {
        return workflowName;
    }

    /**
     * <p>
     * The status of the workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkflowRunStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the workflow run.
     * @see WorkflowRunStatus
     */
    public final WorkflowRunStatus status() {
        return WorkflowRunStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the workflow run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkflowRunStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the workflow run.
     * @see WorkflowRunStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the StatusReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStatusReasons() {
        return statusReasons != null && !(statusReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reasons for the workflow run status.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatusReasons} method.
     * </p>
     * 
     * @return The reasons for the workflow run status.
     */
    public final List<WorkflowRunStatusReason> statusReasons() {
        return statusReasons;
    }

    /**
     * <p>
     * The date and time the workflow run began, in coordinated universal time (UTC) timestamp format as specified in <a
     * href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>.
     * </p>
     * 
     * @return The date and time the workflow run began, in coordinated universal time (UTC) timestamp format as
     *         specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time the workflow run ended, in coordinated universal time (UTC) timestamp format as specified in <a
     * href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
     * </p>
     * 
     * @return The date and time the workflow run ended, in coordinated universal time (UTC) timestamp format as
     *         specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The date and time the workflow was last updated, in coordinated universal time (UTC) timestamp format as
     * specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
     * </p>
     * 
     * @return The date and time the workflow was last updated, in coordinated universal time (UTC) timestamp format as
     *         specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(workflowId());
        hashCode = 31 * hashCode + Objects.hashCode(workflowName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatusReasons() ? statusReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkflowRunSummary)) {
            return false;
        }
        WorkflowRunSummary other = (WorkflowRunSummary) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(workflowId(), other.workflowId())
                && Objects.equals(workflowName(), other.workflowName())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasStatusReasons() == other.hasStatusReasons()
                && Objects.equals(statusReasons(), other.statusReasons()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkflowRunSummary").add("Id", id()).add("WorkflowId", workflowId())
                .add("WorkflowName", workflowName()).add("Status", statusAsString())
                .add("StatusReasons", hasStatusReasons() ? statusReasons() : null).add("StartTime", startTime())
                .add("EndTime", endTime()).add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "workflowId":
            return Optional.ofNullable(clazz.cast(workflowId()));
        case "workflowName":
            return Optional.ofNullable(clazz.cast(workflowName()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReasons":
            return Optional.ofNullable(clazz.cast(statusReasons()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "lastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("id", ID_FIELD);
        map.put("workflowId", WORKFLOW_ID_FIELD);
        map.put("workflowName", WORKFLOW_NAME_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReasons", STATUS_REASONS_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("endTime", END_TIME_FIELD);
        map.put("lastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<WorkflowRunSummary, T> g) {
        return obj -> g.apply((WorkflowRunSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkflowRunSummary> {
        /**
         * <p>
         * The system-generated unique ID of the workflow run.
         * </p>
         * 
         * @param id
         *        The system-generated unique ID of the workflow run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The system-generated unique ID of the workflow.
         * </p>
         * 
         * @param workflowId
         *        The system-generated unique ID of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowId(String workflowId);

        /**
         * <p>
         * The name of the workflow.
         * </p>
         * 
         * @param workflowName
         *        The name of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workflowName(String workflowName);

        /**
         * <p>
         * The status of the workflow run.
         * </p>
         * 
         * @param status
         *        The status of the workflow run.
         * @see WorkflowRunStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkflowRunStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the workflow run.
         * </p>
         * 
         * @param status
         *        The status of the workflow run.
         * @see WorkflowRunStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkflowRunStatus
         */
        Builder status(WorkflowRunStatus status);

        /**
         * <p>
         * The reasons for the workflow run status.
         * </p>
         * 
         * @param statusReasons
         *        The reasons for the workflow run status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasons(Collection<WorkflowRunStatusReason> statusReasons);

        /**
         * <p>
         * The reasons for the workflow run status.
         * </p>
         * 
         * @param statusReasons
         *        The reasons for the workflow run status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasons(WorkflowRunStatusReason... statusReasons);

        /**
         * <p>
         * The reasons for the workflow run status.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codecatalyst.model.WorkflowRunStatusReason.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.codecatalyst.model.WorkflowRunStatusReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codecatalyst.model.WorkflowRunStatusReason.Builder#build()} is called
         * immediately and its result is passed to {@link #statusReasons(List<WorkflowRunStatusReason>)}.
         * 
         * @param statusReasons
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codecatalyst.model.WorkflowRunStatusReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusReasons(java.util.Collection<WorkflowRunStatusReason>)
         */
        Builder statusReasons(Consumer<WorkflowRunStatusReason.Builder>... statusReasons);

        /**
         * <p>
         * The date and time the workflow run began, in coordinated universal time (UTC) timestamp format as specified
         * in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>.
         * </p>
         * 
         * @param startTime
         *        The date and time the workflow run began, in coordinated universal time (UTC) timestamp format as
         *        specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time the workflow run ended, in coordinated universal time (UTC) timestamp format as specified
         * in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
         * </p>
         * 
         * @param endTime
         *        The date and time the workflow run ended, in coordinated universal time (UTC) timestamp format as
         *        specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The date and time the workflow was last updated, in coordinated universal time (UTC) timestamp format as
         * specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
         * </p>
         * 
         * @param lastUpdatedTime
         *        The date and time the workflow was last updated, in coordinated universal time (UTC) timestamp format
         *        as specified in <a href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String workflowId;

        private String workflowName;

        private String status;

        private List<WorkflowRunStatusReason> statusReasons = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(WorkflowRunSummary model) {
            id(model.id);
            workflowId(model.workflowId);
            workflowName(model.workflowName);
            status(model.status);
            statusReasons(model.statusReasons);
            startTime(model.startTime);
            endTime(model.endTime);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getWorkflowId() {
            return workflowId;
        }

        public final void setWorkflowId(String workflowId) {
            this.workflowId = workflowId;
        }

        @Override
        public final Builder workflowId(String workflowId) {
            this.workflowId = workflowId;
            return this;
        }

        public final String getWorkflowName() {
            return workflowName;
        }

        public final void setWorkflowName(String workflowName) {
            this.workflowName = workflowName;
        }

        @Override
        public final Builder workflowName(String workflowName) {
            this.workflowName = workflowName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(WorkflowRunStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<WorkflowRunStatusReason.Builder> getStatusReasons() {
            List<WorkflowRunStatusReason.Builder> result = WorkflowRunStatusReasonsCopier.copyToBuilder(this.statusReasons);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStatusReasons(Collection<WorkflowRunStatusReason.BuilderImpl> statusReasons) {
            this.statusReasons = WorkflowRunStatusReasonsCopier.copyFromBuilder(statusReasons);
        }

        @Override
        public final Builder statusReasons(Collection<WorkflowRunStatusReason> statusReasons) {
            this.statusReasons = WorkflowRunStatusReasonsCopier.copy(statusReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusReasons(WorkflowRunStatusReason... statusReasons) {
            statusReasons(Arrays.asList(statusReasons));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusReasons(Consumer<WorkflowRunStatusReason.Builder>... statusReasons) {
            statusReasons(Stream.of(statusReasons).map(c -> WorkflowRunStatusReason.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        @Override
        public WorkflowRunSummary build() {
            return new WorkflowRunSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
