/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecatalyst.endpoints.internal;

import java.net.URI;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.codecatalyst.endpoints.CodeCatalystEndpointParams;
import software.amazon.awssdk.services.codecatalyst.endpoints.CodeCatalystEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultCodeCatalystEndpointProvider implements CodeCatalystEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(CodeCatalystEndpointParams params) {
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        try {
            Region region = params.region();
            String regionId = region == null ? null : region.id();
            RuleResult result = endpointRule0(params, regionId);
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(CodeCatalystEndpointParams params, String region) {
        if (params.endpoint() != null) {
            return RuleResult.endpoint(Endpoint.builder().url(URI.create(params.endpoint())).build());
        }
        RuleResult result = endpointRule2(params, region);
        if (result.isResolved()) {
            return result;
        }
        return endpointRule7(params, region);
    }

    private static RuleResult endpointRule2(CodeCatalystEndpointParams params, String region) {
        if (region == null) {
            RulePartition partitionResult = RulesFunctions.awsPartition("us-west-2");
            if (partitionResult != null) {
                RuleResult result = endpointRule3(params, partitionResult);
                if (result.isResolved()) {
                    return result;
                }
                return RuleResult.endpoint(Endpoint.builder()
                        .url(URI.create("https://codecatalyst.global." + partitionResult.dualStackDnsSuffix())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule3(CodeCatalystEndpointParams params, RulePartition partitionResult) {
        if (params.useFips()) {
            if (!partitionResult.supportsFIPS()) {
                return RuleResult.error("Partition does not support FIPS.");
            }
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://codecatalyst-fips.global." + partitionResult.dualStackDnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule7(CodeCatalystEndpointParams params, String region) {
        if (region != null) {
            RulePartition partitionResult = RulesFunctions.awsPartition(region);
            if (partitionResult != null) {
                RuleResult result = endpointRule8(params, partitionResult);
                if (result.isResolved()) {
                    return result;
                }
                return RuleResult.endpoint(Endpoint.builder()
                        .url(URI.create("https://codecatalyst.global." + partitionResult.dualStackDnsSuffix())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule8(CodeCatalystEndpointParams params, RulePartition partitionResult) {
        if (params.useFips()) {
            if (!partitionResult.supportsFIPS()) {
                return RuleResult.error("Partition does not support FIPS.");
            }
            return RuleResult.endpoint(Endpoint.builder()
                    .url(URI.create("https://codecatalyst-fips.global." + partitionResult.dualStackDnsSuffix())).build());
        }
        return RuleResult.carryOn();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }
}
