/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information about a pull request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PullRequest implements SdkPojo, Serializable, ToCopyableBuilder<PullRequest.Builder, PullRequest> {
    private static final SdkField<String> PULL_REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pullRequestId").getter(getter(PullRequest::pullRequestId)).setter(setter(Builder::pullRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pullRequestId").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(PullRequest::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(PullRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Instant> LAST_ACTIVITY_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastActivityDate").getter(getter(PullRequest::lastActivityDate))
            .setter(setter(Builder::lastActivityDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastActivityDate").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationDate").getter(getter(PullRequest::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationDate").build()).build();

    private static final SdkField<String> PULL_REQUEST_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pullRequestStatus").getter(getter(PullRequest::pullRequestStatusAsString))
            .setter(setter(Builder::pullRequestStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pullRequestStatus").build()).build();

    private static final SdkField<String> AUTHOR_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authorArn").getter(getter(PullRequest::authorArn)).setter(setter(Builder::authorArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorArn").build()).build();

    private static final SdkField<List<PullRequestTarget>> PULL_REQUEST_TARGETS_FIELD = SdkField
            .<List<PullRequestTarget>> builder(MarshallingType.LIST)
            .memberName("pullRequestTargets")
            .getter(getter(PullRequest::pullRequestTargets))
            .setter(setter(Builder::pullRequestTargets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pullRequestTargets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PullRequestTarget> builder(MarshallingType.SDK_POJO)
                                            .constructor(PullRequestTarget::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clientRequestToken").getter(getter(PullRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientRequestToken").build())
            .build();

    private static final SdkField<String> REVISION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("revisionId").getter(getter(PullRequest::revisionId)).setter(setter(Builder::revisionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revisionId").build()).build();

    private static final SdkField<List<ApprovalRule>> APPROVAL_RULES_FIELD = SdkField
            .<List<ApprovalRule>> builder(MarshallingType.LIST)
            .memberName("approvalRules")
            .getter(getter(PullRequest::approvalRules))
            .setter(setter(Builder::approvalRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("approvalRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ApprovalRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(ApprovalRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PULL_REQUEST_ID_FIELD,
            TITLE_FIELD, DESCRIPTION_FIELD, LAST_ACTIVITY_DATE_FIELD, CREATION_DATE_FIELD, PULL_REQUEST_STATUS_FIELD,
            AUTHOR_ARN_FIELD, PULL_REQUEST_TARGETS_FIELD, CLIENT_REQUEST_TOKEN_FIELD, REVISION_ID_FIELD, APPROVAL_RULES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String pullRequestId;

    private final String title;

    private final String description;

    private final Instant lastActivityDate;

    private final Instant creationDate;

    private final String pullRequestStatus;

    private final String authorArn;

    private final List<PullRequestTarget> pullRequestTargets;

    private final String clientRequestToken;

    private final String revisionId;

    private final List<ApprovalRule> approvalRules;

    private PullRequest(BuilderImpl builder) {
        this.pullRequestId = builder.pullRequestId;
        this.title = builder.title;
        this.description = builder.description;
        this.lastActivityDate = builder.lastActivityDate;
        this.creationDate = builder.creationDate;
        this.pullRequestStatus = builder.pullRequestStatus;
        this.authorArn = builder.authorArn;
        this.pullRequestTargets = builder.pullRequestTargets;
        this.clientRequestToken = builder.clientRequestToken;
        this.revisionId = builder.revisionId;
        this.approvalRules = builder.approvalRules;
    }

    /**
     * <p>
     * The system-generated ID of the pull request.
     * </p>
     * 
     * @return The system-generated ID of the pull request.
     */
    public final String pullRequestId() {
        return pullRequestId;
    }

    /**
     * <p>
     * The user-defined title of the pull request. This title is displayed in the list of pull requests to other
     * repository users.
     * </p>
     * 
     * @return The user-defined title of the pull request. This title is displayed in the list of pull requests to other
     *         repository users.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The user-defined description of the pull request. This description can be used to clarify what should be reviewed
     * and other details of the request.
     * </p>
     * 
     * @return The user-defined description of the pull request. This description can be used to clarify what should be
     *         reviewed and other details of the request.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The day and time of the last user or system activity on the pull request, in timestamp format.
     * </p>
     * 
     * @return The day and time of the last user or system activity on the pull request, in timestamp format.
     */
    public final Instant lastActivityDate() {
        return lastActivityDate;
    }

    /**
     * <p>
     * The date and time the pull request was originally created, in timestamp format.
     * </p>
     * 
     * @return The date and time the pull request was originally created, in timestamp format.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The status of the pull request. Pull request status can only change from <code>OPEN</code> to <code>CLOSED</code>
     * .
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pullRequestStatus}
     * will return {@link PullRequestStatusEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #pullRequestStatusAsString}.
     * </p>
     * 
     * @return The status of the pull request. Pull request status can only change from <code>OPEN</code> to
     *         <code>CLOSED</code>.
     * @see PullRequestStatusEnum
     */
    public final PullRequestStatusEnum pullRequestStatus() {
        return PullRequestStatusEnum.fromValue(pullRequestStatus);
    }

    /**
     * <p>
     * The status of the pull request. Pull request status can only change from <code>OPEN</code> to <code>CLOSED</code>
     * .
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #pullRequestStatus}
     * will return {@link PullRequestStatusEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #pullRequestStatusAsString}.
     * </p>
     * 
     * @return The status of the pull request. Pull request status can only change from <code>OPEN</code> to
     *         <code>CLOSED</code>.
     * @see PullRequestStatusEnum
     */
    public final String pullRequestStatusAsString() {
        return pullRequestStatus;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the user who created the pull request.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the user who created the pull request.
     */
    public final String authorArn() {
        return authorArn;
    }

    /**
     * Returns true if the PullRequestTargets property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasPullRequestTargets() {
        return pullRequestTargets != null && !(pullRequestTargets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The targets of the pull request, including the source branch and destination branch for the pull request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPullRequestTargets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The targets of the pull request, including the source branch and destination branch for the pull request.
     */
    public final List<PullRequestTarget> pullRequestTargets() {
        return pullRequestTargets;
    }

    /**
     * <p>
     * A unique, client-generated idempotency token that, when provided in a request, ensures the request cannot be
     * repeated with a changed parameter. If a request is received with the same parameters and a token is included, the
     * request returns information about the initial request that used that token.
     * </p>
     * 
     * @return A unique, client-generated idempotency token that, when provided in a request, ensures the request cannot
     *         be repeated with a changed parameter. If a request is received with the same parameters and a token is
     *         included, the request returns information about the initial request that used that token.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The system-generated revision ID for the pull request.
     * </p>
     * 
     * @return The system-generated revision ID for the pull request.
     */
    public final String revisionId() {
        return revisionId;
    }

    /**
     * Returns true if the ApprovalRules property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasApprovalRules() {
        return approvalRules != null && !(approvalRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The approval rules applied to the pull request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasApprovalRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The approval rules applied to the pull request.
     */
    public final List<ApprovalRule> approvalRules() {
        return approvalRules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(pullRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(lastActivityDate());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(pullRequestStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authorArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasPullRequestTargets() ? pullRequestTargets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(revisionId());
        hashCode = 31 * hashCode + Objects.hashCode(hasApprovalRules() ? approvalRules() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PullRequest)) {
            return false;
        }
        PullRequest other = (PullRequest) obj;
        return Objects.equals(pullRequestId(), other.pullRequestId()) && Objects.equals(title(), other.title())
                && Objects.equals(description(), other.description())
                && Objects.equals(lastActivityDate(), other.lastActivityDate())
                && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(pullRequestStatusAsString(), other.pullRequestStatusAsString())
                && Objects.equals(authorArn(), other.authorArn()) && hasPullRequestTargets() == other.hasPullRequestTargets()
                && Objects.equals(pullRequestTargets(), other.pullRequestTargets())
                && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(revisionId(), other.revisionId()) && hasApprovalRules() == other.hasApprovalRules()
                && Objects.equals(approvalRules(), other.approvalRules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PullRequest").add("PullRequestId", pullRequestId()).add("Title", title())
                .add("Description", description()).add("LastActivityDate", lastActivityDate())
                .add("CreationDate", creationDate()).add("PullRequestStatus", pullRequestStatusAsString())
                .add("AuthorArn", authorArn()).add("PullRequestTargets", hasPullRequestTargets() ? pullRequestTargets() : null)
                .add("ClientRequestToken", clientRequestToken()).add("RevisionId", revisionId())
                .add("ApprovalRules", hasApprovalRules() ? approvalRules() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "pullRequestId":
            return Optional.ofNullable(clazz.cast(pullRequestId()));
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "lastActivityDate":
            return Optional.ofNullable(clazz.cast(lastActivityDate()));
        case "creationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "pullRequestStatus":
            return Optional.ofNullable(clazz.cast(pullRequestStatusAsString()));
        case "authorArn":
            return Optional.ofNullable(clazz.cast(authorArn()));
        case "pullRequestTargets":
            return Optional.ofNullable(clazz.cast(pullRequestTargets()));
        case "clientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "revisionId":
            return Optional.ofNullable(clazz.cast(revisionId()));
        case "approvalRules":
            return Optional.ofNullable(clazz.cast(approvalRules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PullRequest, T> g) {
        return obj -> g.apply((PullRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PullRequest> {
        /**
         * <p>
         * The system-generated ID of the pull request.
         * </p>
         * 
         * @param pullRequestId
         *        The system-generated ID of the pull request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pullRequestId(String pullRequestId);

        /**
         * <p>
         * The user-defined title of the pull request. This title is displayed in the list of pull requests to other
         * repository users.
         * </p>
         * 
         * @param title
         *        The user-defined title of the pull request. This title is displayed in the list of pull requests to
         *        other repository users.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The user-defined description of the pull request. This description can be used to clarify what should be
         * reviewed and other details of the request.
         * </p>
         * 
         * @param description
         *        The user-defined description of the pull request. This description can be used to clarify what should
         *        be reviewed and other details of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The day and time of the last user or system activity on the pull request, in timestamp format.
         * </p>
         * 
         * @param lastActivityDate
         *        The day and time of the last user or system activity on the pull request, in timestamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastActivityDate(Instant lastActivityDate);

        /**
         * <p>
         * The date and time the pull request was originally created, in timestamp format.
         * </p>
         * 
         * @param creationDate
         *        The date and time the pull request was originally created, in timestamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The status of the pull request. Pull request status can only change from <code>OPEN</code> to
         * <code>CLOSED</code>.
         * </p>
         * 
         * @param pullRequestStatus
         *        The status of the pull request. Pull request status can only change from <code>OPEN</code> to
         *        <code>CLOSED</code>.
         * @see PullRequestStatusEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PullRequestStatusEnum
         */
        Builder pullRequestStatus(String pullRequestStatus);

        /**
         * <p>
         * The status of the pull request. Pull request status can only change from <code>OPEN</code> to
         * <code>CLOSED</code>.
         * </p>
         * 
         * @param pullRequestStatus
         *        The status of the pull request. Pull request status can only change from <code>OPEN</code> to
         *        <code>CLOSED</code>.
         * @see PullRequestStatusEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PullRequestStatusEnum
         */
        Builder pullRequestStatus(PullRequestStatusEnum pullRequestStatus);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the user who created the pull request.
         * </p>
         * 
         * @param authorArn
         *        The Amazon Resource Name (ARN) of the user who created the pull request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorArn(String authorArn);

        /**
         * <p>
         * The targets of the pull request, including the source branch and destination branch for the pull request.
         * </p>
         * 
         * @param pullRequestTargets
         *        The targets of the pull request, including the source branch and destination branch for the pull
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pullRequestTargets(Collection<PullRequestTarget> pullRequestTargets);

        /**
         * <p>
         * The targets of the pull request, including the source branch and destination branch for the pull request.
         * </p>
         * 
         * @param pullRequestTargets
         *        The targets of the pull request, including the source branch and destination branch for the pull
         *        request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pullRequestTargets(PullRequestTarget... pullRequestTargets);

        /**
         * <p>
         * The targets of the pull request, including the source branch and destination branch for the pull request.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PullRequestTarget>.Builder} avoiding the
         * need to create one manually via {@link List<PullRequestTarget>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PullRequestTarget>.Builder#build()} is called immediately
         * and its result is passed to {@link #pullRequestTargets(List<PullRequestTarget>)}.
         * 
         * @param pullRequestTargets
         *        a consumer that will call methods on {@link List<PullRequestTarget>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pullRequestTargets(List<PullRequestTarget>)
         */
        Builder pullRequestTargets(Consumer<PullRequestTarget.Builder>... pullRequestTargets);

        /**
         * <p>
         * A unique, client-generated idempotency token that, when provided in a request, ensures the request cannot be
         * repeated with a changed parameter. If a request is received with the same parameters and a token is included,
         * the request returns information about the initial request that used that token.
         * </p>
         * 
         * @param clientRequestToken
         *        A unique, client-generated idempotency token that, when provided in a request, ensures the request
         *        cannot be repeated with a changed parameter. If a request is received with the same parameters and a
         *        token is included, the request returns information about the initial request that used that token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The system-generated revision ID for the pull request.
         * </p>
         * 
         * @param revisionId
         *        The system-generated revision ID for the pull request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionId(String revisionId);

        /**
         * <p>
         * The approval rules applied to the pull request.
         * </p>
         * 
         * @param approvalRules
         *        The approval rules applied to the pull request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvalRules(Collection<ApprovalRule> approvalRules);

        /**
         * <p>
         * The approval rules applied to the pull request.
         * </p>
         * 
         * @param approvalRules
         *        The approval rules applied to the pull request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder approvalRules(ApprovalRule... approvalRules);

        /**
         * <p>
         * The approval rules applied to the pull request.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ApprovalRule>.Builder} avoiding the need to
         * create one manually via {@link List<ApprovalRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ApprovalRule>.Builder#build()} is called immediately and its
         * result is passed to {@link #approvalRules(List<ApprovalRule>)}.
         * 
         * @param approvalRules
         *        a consumer that will call methods on {@link List<ApprovalRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #approvalRules(List<ApprovalRule>)
         */
        Builder approvalRules(Consumer<ApprovalRule.Builder>... approvalRules);
    }

    static final class BuilderImpl implements Builder {
        private String pullRequestId;

        private String title;

        private String description;

        private Instant lastActivityDate;

        private Instant creationDate;

        private String pullRequestStatus;

        private String authorArn;

        private List<PullRequestTarget> pullRequestTargets = DefaultSdkAutoConstructList.getInstance();

        private String clientRequestToken;

        private String revisionId;

        private List<ApprovalRule> approvalRules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PullRequest model) {
            pullRequestId(model.pullRequestId);
            title(model.title);
            description(model.description);
            lastActivityDate(model.lastActivityDate);
            creationDate(model.creationDate);
            pullRequestStatus(model.pullRequestStatus);
            authorArn(model.authorArn);
            pullRequestTargets(model.pullRequestTargets);
            clientRequestToken(model.clientRequestToken);
            revisionId(model.revisionId);
            approvalRules(model.approvalRules);
        }

        public final String getPullRequestId() {
            return pullRequestId;
        }

        @Override
        public final Builder pullRequestId(String pullRequestId) {
            this.pullRequestId = pullRequestId;
            return this;
        }

        public final void setPullRequestId(String pullRequestId) {
            this.pullRequestId = pullRequestId;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Instant getLastActivityDate() {
            return lastActivityDate;
        }

        @Override
        public final Builder lastActivityDate(Instant lastActivityDate) {
            this.lastActivityDate = lastActivityDate;
            return this;
        }

        public final void setLastActivityDate(Instant lastActivityDate) {
            this.lastActivityDate = lastActivityDate;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final String getPullRequestStatus() {
            return pullRequestStatus;
        }

        @Override
        public final Builder pullRequestStatus(String pullRequestStatus) {
            this.pullRequestStatus = pullRequestStatus;
            return this;
        }

        @Override
        public final Builder pullRequestStatus(PullRequestStatusEnum pullRequestStatus) {
            this.pullRequestStatus(pullRequestStatus == null ? null : pullRequestStatus.toString());
            return this;
        }

        public final void setPullRequestStatus(String pullRequestStatus) {
            this.pullRequestStatus = pullRequestStatus;
        }

        public final String getAuthorArn() {
            return authorArn;
        }

        @Override
        public final Builder authorArn(String authorArn) {
            this.authorArn = authorArn;
            return this;
        }

        public final void setAuthorArn(String authorArn) {
            this.authorArn = authorArn;
        }

        public final Collection<PullRequestTarget.Builder> getPullRequestTargets() {
            if (pullRequestTargets instanceof SdkAutoConstructList) {
                return null;
            }
            return pullRequestTargets != null ? pullRequestTargets.stream().map(PullRequestTarget::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder pullRequestTargets(Collection<PullRequestTarget> pullRequestTargets) {
            this.pullRequestTargets = PullRequestTargetListCopier.copy(pullRequestTargets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder pullRequestTargets(PullRequestTarget... pullRequestTargets) {
            pullRequestTargets(Arrays.asList(pullRequestTargets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder pullRequestTargets(Consumer<PullRequestTarget.Builder>... pullRequestTargets) {
            pullRequestTargets(Stream.of(pullRequestTargets).map(c -> PullRequestTarget.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPullRequestTargets(Collection<PullRequestTarget.BuilderImpl> pullRequestTargets) {
            this.pullRequestTargets = PullRequestTargetListCopier.copyFromBuilder(pullRequestTargets);
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final String getRevisionId() {
            return revisionId;
        }

        @Override
        public final Builder revisionId(String revisionId) {
            this.revisionId = revisionId;
            return this;
        }

        public final void setRevisionId(String revisionId) {
            this.revisionId = revisionId;
        }

        public final Collection<ApprovalRule.Builder> getApprovalRules() {
            if (approvalRules instanceof SdkAutoConstructList) {
                return null;
            }
            return approvalRules != null ? approvalRules.stream().map(ApprovalRule::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder approvalRules(Collection<ApprovalRule> approvalRules) {
            this.approvalRules = ApprovalRulesListCopier.copy(approvalRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder approvalRules(ApprovalRule... approvalRules) {
            approvalRules(Arrays.asList(approvalRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder approvalRules(Consumer<ApprovalRule.Builder>... approvalRules) {
            approvalRules(Stream.of(approvalRules).map(c -> ApprovalRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setApprovalRules(Collection<ApprovalRule.BuilderImpl> approvalRules) {
            this.approvalRules = ApprovalRulesListCopier.copyFromBuilder(approvalRules);
        }

        @Override
        public PullRequest build() {
            return new PullRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
