/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreatePullRequestRequest extends CodeCommitRequest implements
        ToCopyableBuilder<CreatePullRequestRequest.Builder, CreatePullRequestRequest> {
    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(CreatePullRequestRequest::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(CreatePullRequestRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<Target>> TARGETS_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("targets")
            .getter(getter(CreatePullRequestRequest::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("clientRequestToken")
            .getter(getter(CreatePullRequestRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TITLE_FIELD,
            DESCRIPTION_FIELD, TARGETS_FIELD, CLIENT_REQUEST_TOKEN_FIELD));

    private final String title;

    private final String description;

    private final List<Target> targets;

    private final String clientRequestToken;

    private CreatePullRequestRequest(BuilderImpl builder) {
        super(builder);
        this.title = builder.title;
        this.description = builder.description;
        this.targets = builder.targets;
        this.clientRequestToken = builder.clientRequestToken;
    }

    /**
     * <p>
     * The title of the pull request. This title is used to identify the pull request to other users in the repository.
     * </p>
     * 
     * @return The title of the pull request. This title is used to identify the pull request to other users in the
     *         repository.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * A description of the pull request.
     * </p>
     * 
     * @return A description of the pull request.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The targets for the pull request, including the source of the code to be reviewed (the source branch) and the
     * destination where the creator of the pull request intends the code to be merged after the pull request is closed
     * (the destination branch).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return The targets for the pull request, including the source of the code to be reviewed (the source branch) and
     *         the destination where the creator of the pull request intends the code to be merged after the pull
     *         request is closed (the destination branch).
     */
    public final List<Target> targets() {
        return targets;
    }

    /**
     * <p>
     * A unique, client-generated idempotency token that, when provided in a request, ensures the request cannot be
     * repeated with a changed parameter. If a request is received with the same parameters and a token is included, the
     * request returns information about the initial request that used that token.
     * </p>
     * <note>
     * <p>
     * The AWS SDKs prepopulate client request tokens. If you are using an AWS SDK, an idempotency token is created for
     * you.
     * </p>
     * </note>
     * 
     * @return A unique, client-generated idempotency token that, when provided in a request, ensures the request cannot
     *         be repeated with a changed parameter. If a request is received with the same parameters and a token is
     *         included, the request returns information about the initial request that used that token.</p> <note>
     *         <p>
     *         The AWS SDKs prepopulate client request tokens. If you are using an AWS SDK, an idempotency token is
     *         created for you.
     *         </p>
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreatePullRequestRequest)) {
            return false;
        }
        CreatePullRequestRequest other = (CreatePullRequestRequest) obj;
        return Objects.equals(title(), other.title()) && Objects.equals(description(), other.description())
                && hasTargets() == other.hasTargets() && Objects.equals(targets(), other.targets())
                && Objects.equals(clientRequestToken(), other.clientRequestToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreatePullRequestRequest").add("Title", title()).add("Description", description())
                .add("Targets", hasTargets() ? targets() : null).add("ClientRequestToken", clientRequestToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "clientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreatePullRequestRequest, T> g) {
        return obj -> g.apply((CreatePullRequestRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CodeCommitRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreatePullRequestRequest> {
        /**
         * <p>
         * The title of the pull request. This title is used to identify the pull request to other users in the
         * repository.
         * </p>
         * 
         * @param title
         *        The title of the pull request. This title is used to identify the pull request to other users in the
         *        repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * A description of the pull request.
         * </p>
         * 
         * @param description
         *        A description of the pull request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The targets for the pull request, including the source of the code to be reviewed (the source branch) and the
         * destination where the creator of the pull request intends the code to be merged after the pull request is
         * closed (the destination branch).
         * </p>
         * 
         * @param targets
         *        The targets for the pull request, including the source of the code to be reviewed (the source branch)
         *        and the destination where the creator of the pull request intends the code to be merged after the pull
         *        request is closed (the destination branch).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<Target> targets);

        /**
         * <p>
         * The targets for the pull request, including the source of the code to be reviewed (the source branch) and the
         * destination where the creator of the pull request intends the code to be merged after the pull request is
         * closed (the destination branch).
         * </p>
         * 
         * @param targets
         *        The targets for the pull request, including the source of the code to be reviewed (the source branch)
         *        and the destination where the creator of the pull request intends the code to be merged after the pull
         *        request is closed (the destination branch).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Target... targets);

        /**
         * <p>
         * The targets for the pull request, including the source of the code to be reviewed (the source branch) and the
         * destination where the creator of the pull request intends the code to be merged after the pull request is
         * closed (the destination branch).
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codecommit.model.Target.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.codecommit.model.Target#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codecommit.model.Target.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(List<Target>)}.
         * 
         * @param targets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codecommit.model.Target.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(java.util.Collection<Target>)
         */
        Builder targets(Consumer<Target.Builder>... targets);

        /**
         * <p>
         * A unique, client-generated idempotency token that, when provided in a request, ensures the request cannot be
         * repeated with a changed parameter. If a request is received with the same parameters and a token is included,
         * the request returns information about the initial request that used that token.
         * </p>
         * <note>
         * <p>
         * The AWS SDKs prepopulate client request tokens. If you are using an AWS SDK, an idempotency token is created
         * for you.
         * </p>
         * </note>
         * 
         * @param clientRequestToken
         *        A unique, client-generated idempotency token that, when provided in a request, ensures the request
         *        cannot be repeated with a changed parameter. If a request is received with the same parameters and a
         *        token is included, the request returns information about the initial request that used that token.</p>
         *        <note>
         *        <p>
         *        The AWS SDKs prepopulate client request tokens. If you are using an AWS SDK, an idempotency token is
         *        created for you.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeCommitRequest.BuilderImpl implements Builder {
        private String title;

        private String description;

        private List<Target> targets = DefaultSdkAutoConstructList.getInstance();

        private String clientRequestToken;

        private BuilderImpl() {
        }

        private BuilderImpl(CreatePullRequestRequest model) {
            super(model);
            title(model.title);
            description(model.description);
            targets(model.targets);
            clientRequestToken(model.clientRequestToken);
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<Target.Builder> getTargets() {
            List<Target.Builder> result = TargetListCopier.copyToBuilder(this.targets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargets(Collection<Target.BuilderImpl> targets) {
            this.targets = TargetListCopier.copyFromBuilder(targets);
        }

        @Override
        public final Builder targets(Collection<Target> targets) {
            this.targets = TargetListCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Target... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(Consumer<Target.Builder>... targets) {
            targets(Stream.of(targets).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreatePullRequestRequest build() {
            return new CreatePullRequestRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
