/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codecommit.internal.CodeCommitServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.codecommit.model.ActorDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleContentRequiredException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleNameAlreadyExistsException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleNameRequiredException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleTemplateContentRequiredException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleTemplateDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleTemplateInUseException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleTemplateNameAlreadyExistsException;
import software.amazon.awssdk.services.codecommit.model.ApprovalRuleTemplateNameRequiredException;
import software.amazon.awssdk.services.codecommit.model.ApprovalStateRequiredException;
import software.amazon.awssdk.services.codecommit.model.AssociateApprovalRuleTemplateWithRepositoryRequest;
import software.amazon.awssdk.services.codecommit.model.AssociateApprovalRuleTemplateWithRepositoryResponse;
import software.amazon.awssdk.services.codecommit.model.AuthorDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.BatchAssociateApprovalRuleTemplateWithRepositoriesRequest;
import software.amazon.awssdk.services.codecommit.model.BatchAssociateApprovalRuleTemplateWithRepositoriesResponse;
import software.amazon.awssdk.services.codecommit.model.BatchDescribeMergeConflictsRequest;
import software.amazon.awssdk.services.codecommit.model.BatchDescribeMergeConflictsResponse;
import software.amazon.awssdk.services.codecommit.model.BatchDisassociateApprovalRuleTemplateFromRepositoriesRequest;
import software.amazon.awssdk.services.codecommit.model.BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse;
import software.amazon.awssdk.services.codecommit.model.BatchGetCommitsRequest;
import software.amazon.awssdk.services.codecommit.model.BatchGetCommitsResponse;
import software.amazon.awssdk.services.codecommit.model.BatchGetRepositoriesRequest;
import software.amazon.awssdk.services.codecommit.model.BatchGetRepositoriesResponse;
import software.amazon.awssdk.services.codecommit.model.BeforeCommitIdAndAfterCommitIdAreSameException;
import software.amazon.awssdk.services.codecommit.model.BlobIdDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.BlobIdRequiredException;
import software.amazon.awssdk.services.codecommit.model.BranchDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.BranchNameExistsException;
import software.amazon.awssdk.services.codecommit.model.BranchNameIsTagNameException;
import software.amazon.awssdk.services.codecommit.model.BranchNameRequiredException;
import software.amazon.awssdk.services.codecommit.model.CannotDeleteApprovalRuleFromTemplateException;
import software.amazon.awssdk.services.codecommit.model.CannotModifyApprovalRuleFromTemplateException;
import software.amazon.awssdk.services.codecommit.model.ClientRequestTokenRequiredException;
import software.amazon.awssdk.services.codecommit.model.CodeCommitException;
import software.amazon.awssdk.services.codecommit.model.CommentContentRequiredException;
import software.amazon.awssdk.services.codecommit.model.CommentContentSizeLimitExceededException;
import software.amazon.awssdk.services.codecommit.model.CommentDeletedException;
import software.amazon.awssdk.services.codecommit.model.CommentDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.CommentIdRequiredException;
import software.amazon.awssdk.services.codecommit.model.CommentNotCreatedByCallerException;
import software.amazon.awssdk.services.codecommit.model.CommitDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.CommitIdDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.CommitIdRequiredException;
import software.amazon.awssdk.services.codecommit.model.CommitIdsLimitExceededException;
import software.amazon.awssdk.services.codecommit.model.CommitIdsListRequiredException;
import software.amazon.awssdk.services.codecommit.model.CommitMessageLengthExceededException;
import software.amazon.awssdk.services.codecommit.model.CommitRequiredException;
import software.amazon.awssdk.services.codecommit.model.ConcurrentReferenceUpdateException;
import software.amazon.awssdk.services.codecommit.model.CreateApprovalRuleTemplateRequest;
import software.amazon.awssdk.services.codecommit.model.CreateApprovalRuleTemplateResponse;
import software.amazon.awssdk.services.codecommit.model.CreateBranchRequest;
import software.amazon.awssdk.services.codecommit.model.CreateBranchResponse;
import software.amazon.awssdk.services.codecommit.model.CreateCommitRequest;
import software.amazon.awssdk.services.codecommit.model.CreateCommitResponse;
import software.amazon.awssdk.services.codecommit.model.CreatePullRequestApprovalRuleRequest;
import software.amazon.awssdk.services.codecommit.model.CreatePullRequestApprovalRuleResponse;
import software.amazon.awssdk.services.codecommit.model.CreatePullRequestRequest;
import software.amazon.awssdk.services.codecommit.model.CreatePullRequestResponse;
import software.amazon.awssdk.services.codecommit.model.CreateRepositoryRequest;
import software.amazon.awssdk.services.codecommit.model.CreateRepositoryResponse;
import software.amazon.awssdk.services.codecommit.model.CreateUnreferencedMergeCommitRequest;
import software.amazon.awssdk.services.codecommit.model.CreateUnreferencedMergeCommitResponse;
import software.amazon.awssdk.services.codecommit.model.DefaultBranchCannotBeDeletedException;
import software.amazon.awssdk.services.codecommit.model.DeleteApprovalRuleTemplateRequest;
import software.amazon.awssdk.services.codecommit.model.DeleteApprovalRuleTemplateResponse;
import software.amazon.awssdk.services.codecommit.model.DeleteBranchRequest;
import software.amazon.awssdk.services.codecommit.model.DeleteBranchResponse;
import software.amazon.awssdk.services.codecommit.model.DeleteCommentContentRequest;
import software.amazon.awssdk.services.codecommit.model.DeleteCommentContentResponse;
import software.amazon.awssdk.services.codecommit.model.DeleteFileRequest;
import software.amazon.awssdk.services.codecommit.model.DeleteFileResponse;
import software.amazon.awssdk.services.codecommit.model.DeletePullRequestApprovalRuleRequest;
import software.amazon.awssdk.services.codecommit.model.DeletePullRequestApprovalRuleResponse;
import software.amazon.awssdk.services.codecommit.model.DeleteRepositoryRequest;
import software.amazon.awssdk.services.codecommit.model.DeleteRepositoryResponse;
import software.amazon.awssdk.services.codecommit.model.DescribeMergeConflictsRequest;
import software.amazon.awssdk.services.codecommit.model.DescribeMergeConflictsResponse;
import software.amazon.awssdk.services.codecommit.model.DescribePullRequestEventsRequest;
import software.amazon.awssdk.services.codecommit.model.DescribePullRequestEventsResponse;
import software.amazon.awssdk.services.codecommit.model.DirectoryNameConflictsWithFileNameException;
import software.amazon.awssdk.services.codecommit.model.DisassociateApprovalRuleTemplateFromRepositoryRequest;
import software.amazon.awssdk.services.codecommit.model.DisassociateApprovalRuleTemplateFromRepositoryResponse;
import software.amazon.awssdk.services.codecommit.model.EncryptionIntegrityChecksFailedException;
import software.amazon.awssdk.services.codecommit.model.EncryptionKeyAccessDeniedException;
import software.amazon.awssdk.services.codecommit.model.EncryptionKeyDisabledException;
import software.amazon.awssdk.services.codecommit.model.EncryptionKeyNotFoundException;
import software.amazon.awssdk.services.codecommit.model.EncryptionKeyUnavailableException;
import software.amazon.awssdk.services.codecommit.model.EvaluatePullRequestApprovalRulesRequest;
import software.amazon.awssdk.services.codecommit.model.EvaluatePullRequestApprovalRulesResponse;
import software.amazon.awssdk.services.codecommit.model.FileContentAndSourceFileSpecifiedException;
import software.amazon.awssdk.services.codecommit.model.FileContentRequiredException;
import software.amazon.awssdk.services.codecommit.model.FileContentSizeLimitExceededException;
import software.amazon.awssdk.services.codecommit.model.FileDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.FileEntryRequiredException;
import software.amazon.awssdk.services.codecommit.model.FileModeRequiredException;
import software.amazon.awssdk.services.codecommit.model.FileNameConflictsWithDirectoryNameException;
import software.amazon.awssdk.services.codecommit.model.FilePathConflictsWithSubmodulePathException;
import software.amazon.awssdk.services.codecommit.model.FileTooLargeException;
import software.amazon.awssdk.services.codecommit.model.FolderContentSizeLimitExceededException;
import software.amazon.awssdk.services.codecommit.model.FolderDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.GetApprovalRuleTemplateRequest;
import software.amazon.awssdk.services.codecommit.model.GetApprovalRuleTemplateResponse;
import software.amazon.awssdk.services.codecommit.model.GetBlobRequest;
import software.amazon.awssdk.services.codecommit.model.GetBlobResponse;
import software.amazon.awssdk.services.codecommit.model.GetBranchRequest;
import software.amazon.awssdk.services.codecommit.model.GetBranchResponse;
import software.amazon.awssdk.services.codecommit.model.GetCommentReactionsRequest;
import software.amazon.awssdk.services.codecommit.model.GetCommentReactionsResponse;
import software.amazon.awssdk.services.codecommit.model.GetCommentRequest;
import software.amazon.awssdk.services.codecommit.model.GetCommentResponse;
import software.amazon.awssdk.services.codecommit.model.GetCommentsForComparedCommitRequest;
import software.amazon.awssdk.services.codecommit.model.GetCommentsForComparedCommitResponse;
import software.amazon.awssdk.services.codecommit.model.GetCommentsForPullRequestRequest;
import software.amazon.awssdk.services.codecommit.model.GetCommentsForPullRequestResponse;
import software.amazon.awssdk.services.codecommit.model.GetCommitRequest;
import software.amazon.awssdk.services.codecommit.model.GetCommitResponse;
import software.amazon.awssdk.services.codecommit.model.GetDifferencesRequest;
import software.amazon.awssdk.services.codecommit.model.GetDifferencesResponse;
import software.amazon.awssdk.services.codecommit.model.GetFileRequest;
import software.amazon.awssdk.services.codecommit.model.GetFileResponse;
import software.amazon.awssdk.services.codecommit.model.GetFolderRequest;
import software.amazon.awssdk.services.codecommit.model.GetFolderResponse;
import software.amazon.awssdk.services.codecommit.model.GetMergeCommitRequest;
import software.amazon.awssdk.services.codecommit.model.GetMergeCommitResponse;
import software.amazon.awssdk.services.codecommit.model.GetMergeConflictsRequest;
import software.amazon.awssdk.services.codecommit.model.GetMergeConflictsResponse;
import software.amazon.awssdk.services.codecommit.model.GetMergeOptionsRequest;
import software.amazon.awssdk.services.codecommit.model.GetMergeOptionsResponse;
import software.amazon.awssdk.services.codecommit.model.GetPullRequestApprovalStatesRequest;
import software.amazon.awssdk.services.codecommit.model.GetPullRequestApprovalStatesResponse;
import software.amazon.awssdk.services.codecommit.model.GetPullRequestOverrideStateRequest;
import software.amazon.awssdk.services.codecommit.model.GetPullRequestOverrideStateResponse;
import software.amazon.awssdk.services.codecommit.model.GetPullRequestRequest;
import software.amazon.awssdk.services.codecommit.model.GetPullRequestResponse;
import software.amazon.awssdk.services.codecommit.model.GetRepositoryRequest;
import software.amazon.awssdk.services.codecommit.model.GetRepositoryResponse;
import software.amazon.awssdk.services.codecommit.model.GetRepositoryTriggersRequest;
import software.amazon.awssdk.services.codecommit.model.GetRepositoryTriggersResponse;
import software.amazon.awssdk.services.codecommit.model.IdempotencyParameterMismatchException;
import software.amazon.awssdk.services.codecommit.model.InvalidActorArnException;
import software.amazon.awssdk.services.codecommit.model.InvalidApprovalRuleContentException;
import software.amazon.awssdk.services.codecommit.model.InvalidApprovalRuleNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidApprovalRuleTemplateContentException;
import software.amazon.awssdk.services.codecommit.model.InvalidApprovalRuleTemplateDescriptionException;
import software.amazon.awssdk.services.codecommit.model.InvalidApprovalRuleTemplateNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidApprovalStateException;
import software.amazon.awssdk.services.codecommit.model.InvalidAuthorArnException;
import software.amazon.awssdk.services.codecommit.model.InvalidBlobIdException;
import software.amazon.awssdk.services.codecommit.model.InvalidBranchNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidClientRequestTokenException;
import software.amazon.awssdk.services.codecommit.model.InvalidCommentIdException;
import software.amazon.awssdk.services.codecommit.model.InvalidCommitException;
import software.amazon.awssdk.services.codecommit.model.InvalidCommitIdException;
import software.amazon.awssdk.services.codecommit.model.InvalidConflictDetailLevelException;
import software.amazon.awssdk.services.codecommit.model.InvalidConflictResolutionException;
import software.amazon.awssdk.services.codecommit.model.InvalidConflictResolutionStrategyException;
import software.amazon.awssdk.services.codecommit.model.InvalidContinuationTokenException;
import software.amazon.awssdk.services.codecommit.model.InvalidDeletionParameterException;
import software.amazon.awssdk.services.codecommit.model.InvalidDescriptionException;
import software.amazon.awssdk.services.codecommit.model.InvalidDestinationCommitSpecifierException;
import software.amazon.awssdk.services.codecommit.model.InvalidEmailException;
import software.amazon.awssdk.services.codecommit.model.InvalidFileLocationException;
import software.amazon.awssdk.services.codecommit.model.InvalidFileModeException;
import software.amazon.awssdk.services.codecommit.model.InvalidFilePositionException;
import software.amazon.awssdk.services.codecommit.model.InvalidMaxConflictFilesException;
import software.amazon.awssdk.services.codecommit.model.InvalidMaxMergeHunksException;
import software.amazon.awssdk.services.codecommit.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.codecommit.model.InvalidMergeOptionException;
import software.amazon.awssdk.services.codecommit.model.InvalidOrderException;
import software.amazon.awssdk.services.codecommit.model.InvalidOverrideStatusException;
import software.amazon.awssdk.services.codecommit.model.InvalidParentCommitIdException;
import software.amazon.awssdk.services.codecommit.model.InvalidPathException;
import software.amazon.awssdk.services.codecommit.model.InvalidPullRequestEventTypeException;
import software.amazon.awssdk.services.codecommit.model.InvalidPullRequestIdException;
import software.amazon.awssdk.services.codecommit.model.InvalidPullRequestStatusException;
import software.amazon.awssdk.services.codecommit.model.InvalidPullRequestStatusUpdateException;
import software.amazon.awssdk.services.codecommit.model.InvalidReactionUserArnException;
import software.amazon.awssdk.services.codecommit.model.InvalidReactionValueException;
import software.amazon.awssdk.services.codecommit.model.InvalidReferenceNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidRelativeFileVersionEnumException;
import software.amazon.awssdk.services.codecommit.model.InvalidReplacementContentException;
import software.amazon.awssdk.services.codecommit.model.InvalidReplacementTypeException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryDescriptionException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryTriggerBranchNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryTriggerCustomDataException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryTriggerDestinationArnException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryTriggerEventsException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryTriggerNameException;
import software.amazon.awssdk.services.codecommit.model.InvalidRepositoryTriggerRegionException;
import software.amazon.awssdk.services.codecommit.model.InvalidResourceArnException;
import software.amazon.awssdk.services.codecommit.model.InvalidRevisionIdException;
import software.amazon.awssdk.services.codecommit.model.InvalidRuleContentSha256Exception;
import software.amazon.awssdk.services.codecommit.model.InvalidSortByException;
import software.amazon.awssdk.services.codecommit.model.InvalidSourceCommitSpecifierException;
import software.amazon.awssdk.services.codecommit.model.InvalidSystemTagUsageException;
import software.amazon.awssdk.services.codecommit.model.InvalidTagKeysListException;
import software.amazon.awssdk.services.codecommit.model.InvalidTagsMapException;
import software.amazon.awssdk.services.codecommit.model.InvalidTargetBranchException;
import software.amazon.awssdk.services.codecommit.model.InvalidTargetException;
import software.amazon.awssdk.services.codecommit.model.InvalidTargetsException;
import software.amazon.awssdk.services.codecommit.model.InvalidTitleException;
import software.amazon.awssdk.services.codecommit.model.ListApprovalRuleTemplatesRequest;
import software.amazon.awssdk.services.codecommit.model.ListApprovalRuleTemplatesResponse;
import software.amazon.awssdk.services.codecommit.model.ListAssociatedApprovalRuleTemplatesForRepositoryRequest;
import software.amazon.awssdk.services.codecommit.model.ListAssociatedApprovalRuleTemplatesForRepositoryResponse;
import software.amazon.awssdk.services.codecommit.model.ListBranchesRequest;
import software.amazon.awssdk.services.codecommit.model.ListBranchesResponse;
import software.amazon.awssdk.services.codecommit.model.ListFileCommitHistoryRequest;
import software.amazon.awssdk.services.codecommit.model.ListFileCommitHistoryResponse;
import software.amazon.awssdk.services.codecommit.model.ListPullRequestsRequest;
import software.amazon.awssdk.services.codecommit.model.ListPullRequestsResponse;
import software.amazon.awssdk.services.codecommit.model.ListRepositoriesForApprovalRuleTemplateRequest;
import software.amazon.awssdk.services.codecommit.model.ListRepositoriesForApprovalRuleTemplateResponse;
import software.amazon.awssdk.services.codecommit.model.ListRepositoriesRequest;
import software.amazon.awssdk.services.codecommit.model.ListRepositoriesResponse;
import software.amazon.awssdk.services.codecommit.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codecommit.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codecommit.model.ManualMergeRequiredException;
import software.amazon.awssdk.services.codecommit.model.MaximumBranchesExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumConflictResolutionEntriesExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumFileContentToLoadExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumFileEntriesExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumItemsToCompareExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumNumberOfApprovalsExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumOpenPullRequestsExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumRepositoryNamesExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumRepositoryTriggersExceededException;
import software.amazon.awssdk.services.codecommit.model.MaximumRuleTemplatesAssociatedWithRepositoryException;
import software.amazon.awssdk.services.codecommit.model.MergeBranchesByFastForwardRequest;
import software.amazon.awssdk.services.codecommit.model.MergeBranchesByFastForwardResponse;
import software.amazon.awssdk.services.codecommit.model.MergeBranchesBySquashRequest;
import software.amazon.awssdk.services.codecommit.model.MergeBranchesBySquashResponse;
import software.amazon.awssdk.services.codecommit.model.MergeBranchesByThreeWayRequest;
import software.amazon.awssdk.services.codecommit.model.MergeBranchesByThreeWayResponse;
import software.amazon.awssdk.services.codecommit.model.MergeOptionRequiredException;
import software.amazon.awssdk.services.codecommit.model.MergePullRequestByFastForwardRequest;
import software.amazon.awssdk.services.codecommit.model.MergePullRequestByFastForwardResponse;
import software.amazon.awssdk.services.codecommit.model.MergePullRequestBySquashRequest;
import software.amazon.awssdk.services.codecommit.model.MergePullRequestBySquashResponse;
import software.amazon.awssdk.services.codecommit.model.MergePullRequestByThreeWayRequest;
import software.amazon.awssdk.services.codecommit.model.MergePullRequestByThreeWayResponse;
import software.amazon.awssdk.services.codecommit.model.MultipleConflictResolutionEntriesException;
import software.amazon.awssdk.services.codecommit.model.MultipleRepositoriesInPullRequestException;
import software.amazon.awssdk.services.codecommit.model.NameLengthExceededException;
import software.amazon.awssdk.services.codecommit.model.NoChangeException;
import software.amazon.awssdk.services.codecommit.model.NumberOfRuleTemplatesExceededException;
import software.amazon.awssdk.services.codecommit.model.NumberOfRulesExceededException;
import software.amazon.awssdk.services.codecommit.model.OverrideAlreadySetException;
import software.amazon.awssdk.services.codecommit.model.OverridePullRequestApprovalRulesRequest;
import software.amazon.awssdk.services.codecommit.model.OverridePullRequestApprovalRulesResponse;
import software.amazon.awssdk.services.codecommit.model.OverrideStatusRequiredException;
import software.amazon.awssdk.services.codecommit.model.ParentCommitDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.ParentCommitIdOutdatedException;
import software.amazon.awssdk.services.codecommit.model.ParentCommitIdRequiredException;
import software.amazon.awssdk.services.codecommit.model.PathDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.PathRequiredException;
import software.amazon.awssdk.services.codecommit.model.PostCommentForComparedCommitRequest;
import software.amazon.awssdk.services.codecommit.model.PostCommentForComparedCommitResponse;
import software.amazon.awssdk.services.codecommit.model.PostCommentForPullRequestRequest;
import software.amazon.awssdk.services.codecommit.model.PostCommentForPullRequestResponse;
import software.amazon.awssdk.services.codecommit.model.PostCommentReplyRequest;
import software.amazon.awssdk.services.codecommit.model.PostCommentReplyResponse;
import software.amazon.awssdk.services.codecommit.model.PullRequestAlreadyClosedException;
import software.amazon.awssdk.services.codecommit.model.PullRequestApprovalRulesNotSatisfiedException;
import software.amazon.awssdk.services.codecommit.model.PullRequestCannotBeApprovedByAuthorException;
import software.amazon.awssdk.services.codecommit.model.PullRequestDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.PullRequestIdRequiredException;
import software.amazon.awssdk.services.codecommit.model.PullRequestStatusRequiredException;
import software.amazon.awssdk.services.codecommit.model.PutCommentReactionRequest;
import software.amazon.awssdk.services.codecommit.model.PutCommentReactionResponse;
import software.amazon.awssdk.services.codecommit.model.PutFileEntryConflictException;
import software.amazon.awssdk.services.codecommit.model.PutFileRequest;
import software.amazon.awssdk.services.codecommit.model.PutFileResponse;
import software.amazon.awssdk.services.codecommit.model.PutRepositoryTriggersRequest;
import software.amazon.awssdk.services.codecommit.model.PutRepositoryTriggersResponse;
import software.amazon.awssdk.services.codecommit.model.ReactionLimitExceededException;
import software.amazon.awssdk.services.codecommit.model.ReactionValueRequiredException;
import software.amazon.awssdk.services.codecommit.model.ReferenceDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.ReferenceNameRequiredException;
import software.amazon.awssdk.services.codecommit.model.ReferenceTypeNotSupportedException;
import software.amazon.awssdk.services.codecommit.model.ReplacementContentRequiredException;
import software.amazon.awssdk.services.codecommit.model.ReplacementTypeRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryDoesNotExistException;
import software.amazon.awssdk.services.codecommit.model.RepositoryLimitExceededException;
import software.amazon.awssdk.services.codecommit.model.RepositoryNameExistsException;
import software.amazon.awssdk.services.codecommit.model.RepositoryNameRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryNamesRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryNotAssociatedWithPullRequestException;
import software.amazon.awssdk.services.codecommit.model.RepositoryTriggerBranchNameListRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryTriggerDestinationArnRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryTriggerEventsListRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryTriggerNameRequiredException;
import software.amazon.awssdk.services.codecommit.model.RepositoryTriggersListRequiredException;
import software.amazon.awssdk.services.codecommit.model.ResourceArnRequiredException;
import software.amazon.awssdk.services.codecommit.model.RestrictedSourceFileException;
import software.amazon.awssdk.services.codecommit.model.RevisionIdRequiredException;
import software.amazon.awssdk.services.codecommit.model.RevisionNotCurrentException;
import software.amazon.awssdk.services.codecommit.model.SameFileContentException;
import software.amazon.awssdk.services.codecommit.model.SamePathRequestException;
import software.amazon.awssdk.services.codecommit.model.SourceAndDestinationAreSameException;
import software.amazon.awssdk.services.codecommit.model.SourceFileOrContentRequiredException;
import software.amazon.awssdk.services.codecommit.model.TagKeysListRequiredException;
import software.amazon.awssdk.services.codecommit.model.TagPolicyException;
import software.amazon.awssdk.services.codecommit.model.TagResourceRequest;
import software.amazon.awssdk.services.codecommit.model.TagResourceResponse;
import software.amazon.awssdk.services.codecommit.model.TagsMapRequiredException;
import software.amazon.awssdk.services.codecommit.model.TargetRequiredException;
import software.amazon.awssdk.services.codecommit.model.TargetsRequiredException;
import software.amazon.awssdk.services.codecommit.model.TestRepositoryTriggersRequest;
import software.amazon.awssdk.services.codecommit.model.TestRepositoryTriggersResponse;
import software.amazon.awssdk.services.codecommit.model.TipOfSourceReferenceIsDifferentException;
import software.amazon.awssdk.services.codecommit.model.TipsDivergenceExceededException;
import software.amazon.awssdk.services.codecommit.model.TitleRequiredException;
import software.amazon.awssdk.services.codecommit.model.TooManyTagsException;
import software.amazon.awssdk.services.codecommit.model.UntagResourceRequest;
import software.amazon.awssdk.services.codecommit.model.UntagResourceResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateApprovalRuleTemplateContentRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateApprovalRuleTemplateContentResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateApprovalRuleTemplateDescriptionRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateApprovalRuleTemplateDescriptionResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateApprovalRuleTemplateNameRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateApprovalRuleTemplateNameResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateCommentRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateCommentResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateDefaultBranchRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateDefaultBranchResponse;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestApprovalRuleContentRequest;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestApprovalRuleContentResponse;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestApprovalStateRequest;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestApprovalStateResponse;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestDescriptionRequest;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestDescriptionResponse;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestStatusRequest;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestStatusResponse;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestTitleRequest;
import software.amazon.awssdk.services.codecommit.model.UpdatePullRequestTitleResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateRepositoryDescriptionRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateRepositoryDescriptionResponse;
import software.amazon.awssdk.services.codecommit.model.UpdateRepositoryNameRequest;
import software.amazon.awssdk.services.codecommit.model.UpdateRepositoryNameResponse;
import software.amazon.awssdk.services.codecommit.transform.AssociateApprovalRuleTemplateWithRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.BatchAssociateApprovalRuleTemplateWithRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.BatchDescribeMergeConflictsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.BatchDisassociateApprovalRuleTemplateFromRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.BatchGetCommitsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.BatchGetRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreateApprovalRuleTemplateRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreateBranchRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreateCommitRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreatePullRequestApprovalRuleRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreatePullRequestRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.CreateUnreferencedMergeCommitRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DeleteApprovalRuleTemplateRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DeleteBranchRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DeleteCommentContentRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DeleteFileRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DeletePullRequestApprovalRuleRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DeleteRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DescribeMergeConflictsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DescribePullRequestEventsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.DisassociateApprovalRuleTemplateFromRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.EvaluatePullRequestApprovalRulesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetApprovalRuleTemplateRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetBlobRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetBranchRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetCommentReactionsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetCommentRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetCommentsForComparedCommitRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetCommentsForPullRequestRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetCommitRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetDifferencesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetFileRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetFolderRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetMergeCommitRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetMergeConflictsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetMergeOptionsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetPullRequestApprovalStatesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetPullRequestOverrideStateRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetPullRequestRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.GetRepositoryTriggersRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListApprovalRuleTemplatesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListAssociatedApprovalRuleTemplatesForRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListBranchesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListFileCommitHistoryRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListPullRequestsRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListRepositoriesForApprovalRuleTemplateRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListRepositoriesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.MergeBranchesByFastForwardRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.MergeBranchesBySquashRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.MergeBranchesByThreeWayRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.MergePullRequestByFastForwardRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.MergePullRequestBySquashRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.MergePullRequestByThreeWayRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.OverridePullRequestApprovalRulesRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.PostCommentForComparedCommitRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.PostCommentForPullRequestRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.PostCommentReplyRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.PutCommentReactionRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.PutFileRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.PutRepositoryTriggersRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.TestRepositoryTriggersRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateApprovalRuleTemplateContentRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateApprovalRuleTemplateDescriptionRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateApprovalRuleTemplateNameRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateCommentRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateDefaultBranchRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdatePullRequestApprovalRuleContentRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdatePullRequestApprovalStateRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdatePullRequestDescriptionRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdatePullRequestStatusRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdatePullRequestTitleRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateRepositoryDescriptionRequestMarshaller;
import software.amazon.awssdk.services.codecommit.transform.UpdateRepositoryNameRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodeCommitAsyncClient}.
 *
 * @see CodeCommitAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeCommitAsyncClient implements CodeCommitAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodeCommitAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CodeCommitServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCodeCommitAsyncClient(CodeCommitServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an association between an approval rule template and a specified repository. Then, the next time a pull
     * request is created in the repository where the destination reference (if specified) matches the destination
     * reference (branch) for the pull request, an approval rule that matches the template conditions is automatically
     * created for that pull request. If no destination references are specified in the template, an approval rule that
     * matches the template contents is created for all pull requests in that repository.
     * </p>
     *
     * @param associateApprovalRuleTemplateWithRepositoryRequest
     * @return A Java Future containing the result of the AssociateApprovalRuleTemplateWithRepository operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>MaximumRuleTemplatesAssociatedWithRepositoryException The maximum number of approval rule templates
     *         for a repository has been exceeded. You cannot associate more than 25 approval rule templates with a
     *         repository.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.AssociateApprovalRuleTemplateWithRepository
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/AssociateApprovalRuleTemplateWithRepository"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateApprovalRuleTemplateWithRepositoryResponse> associateApprovalRuleTemplateWithRepository(
            AssociateApprovalRuleTemplateWithRepositoryRequest associateApprovalRuleTemplateWithRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                associateApprovalRuleTemplateWithRepositoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateApprovalRuleTemplateWithRepositoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateApprovalRuleTemplateWithRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateApprovalRuleTemplateWithRepositoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateApprovalRuleTemplateWithRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateApprovalRuleTemplateWithRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateApprovalRuleTemplateWithRepositoryRequest, AssociateApprovalRuleTemplateWithRepositoryResponse>()
                            .withOperationName("AssociateApprovalRuleTemplateWithRepository")
                            .withMarshaller(new AssociateApprovalRuleTemplateWithRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateApprovalRuleTemplateWithRepositoryRequest));
            CompletableFuture<AssociateApprovalRuleTemplateWithRepositoryResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an association between an approval rule template and one or more specified repositories.
     * </p>
     *
     * @param batchAssociateApprovalRuleTemplateWithRepositoriesRequest
     * @return A Java Future containing the result of the BatchAssociateApprovalRuleTemplateWithRepositories operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>RepositoryNamesRequiredException At least one repository name object is required, but was not
     *         specified.</li>
     *         <li>MaximumRepositoryNamesExceededException The maximum number of allowed repository names was exceeded.
     *         Currently, this number is 100.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.BatchAssociateApprovalRuleTemplateWithRepositories
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/BatchAssociateApprovalRuleTemplateWithRepositories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchAssociateApprovalRuleTemplateWithRepositoriesResponse> batchAssociateApprovalRuleTemplateWithRepositories(
            BatchAssociateApprovalRuleTemplateWithRepositoriesRequest batchAssociateApprovalRuleTemplateWithRepositoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchAssociateApprovalRuleTemplateWithRepositoriesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchAssociateApprovalRuleTemplateWithRepositoriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchAssociateApprovalRuleTemplateWithRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchAssociateApprovalRuleTemplateWithRepositoriesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchAssociateApprovalRuleTemplateWithRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchAssociateApprovalRuleTemplateWithRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchAssociateApprovalRuleTemplateWithRepositoriesRequest, BatchAssociateApprovalRuleTemplateWithRepositoriesResponse>()
                            .withOperationName("BatchAssociateApprovalRuleTemplateWithRepositories")
                            .withMarshaller(
                                    new BatchAssociateApprovalRuleTemplateWithRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchAssociateApprovalRuleTemplateWithRepositoriesRequest));
            CompletableFuture<BatchAssociateApprovalRuleTemplateWithRepositoriesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about one or more merge conflicts in the attempted merge of two commit specifiers using the
     * squash or three-way merge strategy.
     * </p>
     *
     * @param batchDescribeMergeConflictsRequest
     * @return A Java Future containing the result of the BatchDescribeMergeConflicts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>MergeOptionRequiredException A merge option or stategy is required, and none was provided.</li>
     *         <li>InvalidMergeOptionException The specified merge option is not valid for this operation. Not all merge
     *         strategies are supported for all operations.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>InvalidMaxConflictFilesException The specified value for the number of conflict files to return is
     *         not valid.</li>
     *         <li>InvalidMaxMergeHunksException The specified value for the number of merge hunks to return is not
     *         valid.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.BatchDescribeMergeConflicts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/BatchDescribeMergeConflicts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDescribeMergeConflictsResponse> batchDescribeMergeConflicts(
            BatchDescribeMergeConflictsRequest batchDescribeMergeConflictsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDescribeMergeConflictsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDescribeMergeConflictsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDescribeMergeConflicts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDescribeMergeConflictsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDescribeMergeConflictsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDescribeMergeConflictsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDescribeMergeConflictsRequest, BatchDescribeMergeConflictsResponse>()
                            .withOperationName("BatchDescribeMergeConflicts")
                            .withMarshaller(new BatchDescribeMergeConflictsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDescribeMergeConflictsRequest));
            CompletableFuture<BatchDescribeMergeConflictsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association between an approval rule template and one or more specified repositories.
     * </p>
     *
     * @param batchDisassociateApprovalRuleTemplateFromRepositoriesRequest
     * @return A Java Future containing the result of the BatchDisassociateApprovalRuleTemplateFromRepositories
     *         operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>RepositoryNamesRequiredException At least one repository name object is required, but was not
     *         specified.</li>
     *         <li>MaximumRepositoryNamesExceededException The maximum number of allowed repository names was exceeded.
     *         Currently, this number is 100.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.BatchDisassociateApprovalRuleTemplateFromRepositories
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/BatchDisassociateApprovalRuleTemplateFromRepositories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse> batchDisassociateApprovalRuleTemplateFromRepositories(
            BatchDisassociateApprovalRuleTemplateFromRepositoriesRequest batchDisassociateApprovalRuleTemplateFromRepositoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchDisassociateApprovalRuleTemplateFromRepositoriesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchDisassociateApprovalRuleTemplateFromRepositoriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME,
                    "BatchDisassociateApprovalRuleTemplateFromRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata,
                            BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDisassociateApprovalRuleTemplateFromRepositoriesRequest, BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse>()
                            .withOperationName("BatchDisassociateApprovalRuleTemplateFromRepositories")
                            .withMarshaller(
                                    new BatchDisassociateApprovalRuleTemplateFromRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchDisassociateApprovalRuleTemplateFromRepositoriesRequest));
            CompletableFuture<BatchDisassociateApprovalRuleTemplateFromRepositoriesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the contents of one or more commits in a repository.
     * </p>
     *
     * @param batchGetCommitsRequest
     * @return A Java Future containing the result of the BatchGetCommits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CommitIdsListRequiredException A list of commit IDs is required, but was either not specified or the
     *         list was empty.</li>
     *         <li>CommitIdsLimitExceededException The maximum number of allowed commit IDs in a batch request is 100.
     *         Verify that your batch requests contains no more than 100 commit IDs, and then try again.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.BatchGetCommits
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/BatchGetCommits" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetCommitsResponse> batchGetCommits(BatchGetCommitsRequest batchGetCommitsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetCommitsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetCommitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetCommits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetCommitsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetCommitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetCommitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetCommitsRequest, BatchGetCommitsResponse>()
                            .withOperationName("BatchGetCommits")
                            .withMarshaller(new BatchGetCommitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetCommitsRequest));
            CompletableFuture<BatchGetCommitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about one or more repositories.
     * </p>
     * <note>
     * <p>
     * The description field for a repository accepts all HTML characters and all valid Unicode characters. Applications
     * that do not HTML-encode the description and display it in a webpage can expose users to potentially malicious
     * code. Make sure that you HTML-encode the description field in any application that uses this API to display the
     * repository description on a webpage.
     * </p>
     * </note>
     *
     * @param batchGetRepositoriesRequest
     *        Represents the input of a batch get repositories operation.
     * @return A Java Future containing the result of the BatchGetRepositories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNamesRequiredException At least one repository name object is required, but was not
     *         specified.</li>
     *         <li>MaximumRepositoryNamesExceededException The maximum number of allowed repository names was exceeded.
     *         Currently, this number is 100.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.BatchGetRepositories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/BatchGetRepositories"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetRepositoriesResponse> batchGetRepositories(
            BatchGetRepositoriesRequest batchGetRepositoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetRepositoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetRepositoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetRepositoriesRequest, BatchGetRepositoriesResponse>()
                            .withOperationName("BatchGetRepositories")
                            .withMarshaller(new BatchGetRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetRepositoriesRequest));
            CompletableFuture<BatchGetRepositoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a template for approval rules that can then be associated with one or more repositories in your Amazon
     * Web Services account. When you associate a template with a repository, CodeCommit creates an approval rule that
     * matches the conditions of the template for all pull requests that meet the conditions of the template. For more
     * information, see <a>AssociateApprovalRuleTemplateWithRepository</a>.
     * </p>
     *
     * @param createApprovalRuleTemplateRequest
     * @return A Java Future containing the result of the CreateApprovalRuleTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateNameAlreadyExistsException You cannot create an approval rule template with that
     *         name because a template with that name already exists in this Amazon Web Services Region for your Amazon
     *         Web Services account. Approval rule template names must be unique.</li>
     *         <li>ApprovalRuleTemplateContentRequiredException The content for the approval rule template is empty. You
     *         must provide some content for an approval rule template. The content cannot be null.</li>
     *         <li>InvalidApprovalRuleTemplateContentException The content of the approval rule template is not valid.</li>
     *         <li>InvalidApprovalRuleTemplateDescriptionException The description for the approval rule template is not
     *         valid because it exceeds the maximum characters allowed for a description. For more information about
     *         limits in CodeCommit, see <a
     *         href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a> in the
     *         <i>CodeCommit User Guide</i>.</li>
     *         <li>NumberOfRuleTemplatesExceededException The maximum number of approval rule templates has been
     *         exceeded for this Amazon Web Services Region.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreateApprovalRuleTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreateApprovalRuleTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApprovalRuleTemplateResponse> createApprovalRuleTemplate(
            CreateApprovalRuleTemplateRequest createApprovalRuleTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApprovalRuleTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApprovalRuleTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApprovalRuleTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApprovalRuleTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApprovalRuleTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApprovalRuleTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApprovalRuleTemplateRequest, CreateApprovalRuleTemplateResponse>()
                            .withOperationName("CreateApprovalRuleTemplate")
                            .withMarshaller(new CreateApprovalRuleTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createApprovalRuleTemplateRequest));
            CompletableFuture<CreateApprovalRuleTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a branch in a repository and points the branch to a commit.
     * </p>
     * <note>
     * <p>
     * Calling the create branch operation does not set a repository's default branch. To do this, call the update
     * default branch operation.
     * </p>
     * </note>
     *
     * @param createBranchRequest
     *        Represents the input of a create branch operation.
     * @return A Java Future containing the result of the CreateBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>BranchNameExistsException Cannot create the branch with the specified name because the commit
     *         conflicts with an existing branch with the same name. Branch names must be unique.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>CommitIdRequiredException A commit ID was not specified.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreateBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBranchResponse> createBranch(CreateBranchRequest createBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBranchRequest, CreateBranchResponse>()
                            .withOperationName("CreateBranch").withMarshaller(new CreateBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBranchRequest));
            CompletableFuture<CreateBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a commit for a repository on the tip of a specified branch.
     * </p>
     *
     * @param createCommitRequest
     * @return A Java Future containing the result of the CreateCommit operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>ParentCommitIdRequiredException A parent commit ID is required. To view the full commit ID of a
     *         branch in a repository, use <a>GetBranch</a> or a Git command (for example, git pull or git log).</li>
     *         <li>InvalidParentCommitIdException The parent commit ID is not valid. The commit ID cannot be empty, and
     *         must match the head commit ID for the branch of the repository where you want to add or update a file.</li>
     *         <li>ParentCommitDoesNotExistException The parent commit ID is not valid because it does not exist. The
     *         specified parent commit ID does not exist in the specified branch of the repository.</li>
     *         <li>ParentCommitIdOutdatedException The file could not be added because the provided parent commit ID is
     *         not the current tip of the specified branch. To view the full commit ID of the current head of the
     *         branch, use <a>GetBranch</a>.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>BranchNameIsTagNameException The specified branch name is not valid because it is a tag name. Enter
     *         the name of a branch in the repository. For a list of valid branch names, use <a>ListBranches</a>.</li>
     *         <li>FileEntryRequiredException The commit cannot be created because no files have been specified as
     *         added, updated, or changed (PutFile or DeleteFile) for the commit.</li>
     *         <li>MaximumFileEntriesExceededException The number of specified files to change as part of this commit
     *         exceeds the maximum number of files that can be changed in a single commit. Consider using a Git client
     *         for these changes.</li>
     *         <li>PutFileEntryConflictException The commit cannot be created because one or more files specified in the
     *         commit reference both a file and a folder.</li>
     *         <li>SourceFileOrContentRequiredException The commit cannot be created because no source files or file
     *         content have been specified for the commit.</li>
     *         <li>FileContentAndSourceFileSpecifiedException The commit cannot be created because both a source file
     *         and file content have been specified for the same file. You cannot provide both. Either specify a source
     *         file or provide the file content directly.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>SamePathRequestException The commit cannot be created because one or more changes in this commit
     *         duplicate actions in the same file path. For example, you cannot make the same delete request to the same
     *         file in the same file path twice, or make a delete request and a move request to the same file as part of
     *         the same commit.</li>
     *         <li>FileDoesNotExistException The specified file does not exist. Verify that you have used the correct
     *         file name, full path, and extension.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>InvalidDeletionParameterException The specified deletion parameter is not valid.</li>
     *         <li>RestrictedSourceFileException The commit cannot be created because one of the changes specifies
     *         copying or moving a .gitkeep file.</li>
     *         <li>FileModeRequiredException The commit cannot be created because no file mode has been specified. A
     *         file mode is required to update mode permissions for a file.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>NoChangeException The commit cannot be created because no changes will be made to the repository as a
     *         result of this commit. A commit must contain at least one change.</li>
     *         <li>FileNameConflictsWithDirectoryNameException A file cannot be added to the repository because the
     *         specified file name has the same name as a directory in this repository. Either provide another name for
     *         the file, or add the file in a directory that does not match the file name.</li>
     *         <li>DirectoryNameConflictsWithFileNameException A file cannot be added to the repository because the
     *         specified path name has the same name as a file that already exists in this repository. Either provide a
     *         different name for the file, or specify a different path for the file.</li>
     *         <li>FilePathConflictsWithSubmodulePathException The commit cannot be created because a specified file
     *         path points to a submodule. Verify that the destination files have valid file paths that do not point to
     *         a submodule.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreateCommit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreateCommit" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCommitResponse> createCommit(CreateCommitRequest createCommitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCommitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCommitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCommit");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCommitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateCommitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCommitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCommitRequest, CreateCommitResponse>()
                            .withOperationName("CreateCommit").withMarshaller(new CreateCommitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCommitRequest));
            CompletableFuture<CreateCommitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a pull request in the specified repository.
     * </p>
     *
     * @param createPullRequestRequest
     * @return A Java Future containing the result of the CreatePullRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>ClientRequestTokenRequiredException A client request token is required. A client request token is an
     *         unique, client-generated idempotency token that, when provided in a request, ensures the request cannot
     *         be repeated with a changed parameter. If a request is received with the same parameters and a token is
     *         included, the request returns information about the initial request that used that token.</li>
     *         <li>InvalidClientRequestTokenException The client request token is not valid.</li>
     *         <li>IdempotencyParameterMismatchException The client request token is not valid. Either the token is not
     *         in a valid format, or the token has been used in a previous request and cannot be reused.</li>
     *         <li>ReferenceNameRequiredException A reference name is required, but none was provided.</li>
     *         <li>InvalidReferenceNameException The specified reference name format is not valid. Reference names must
     *         conform to the Git references format (for example, refs/heads/main). For more information, see <a
     *         href="https://git-scm.com/book/en/v2/Git-Internals-Git-References">Git Internals - Git References</a> or
     *         consult your Git documentation.</li>
     *         <li>ReferenceDoesNotExistException The specified reference does not exist. You must provide a full commit
     *         ID.</li>
     *         <li>ReferenceTypeNotSupportedException The specified reference is not a supported type.</li>
     *         <li>TitleRequiredException A pull request title is required. It cannot be empty or null.</li>
     *         <li>InvalidTitleException The title of the pull request is not valid. Pull request titles cannot exceed
     *         100 characters in length.</li>
     *         <li>InvalidDescriptionException The pull request description is not valid. Descriptions cannot be more
     *         than 1,000 characters.</li>
     *         <li>TargetsRequiredException An array of target objects is required. It cannot be empty or null.</li>
     *         <li>InvalidTargetsException The targets for the pull request is not valid or not in a valid format.
     *         Targets are a list of target objects. Each target object must contain the full values for the repository
     *         name, source branch, and destination branch for a pull request.</li>
     *         <li>TargetRequiredException A pull request target is required. It cannot be empty or null. A pull request
     *         target must contain the full values for the repository name, source branch, and destination branch for
     *         the pull request.</li>
     *         <li>InvalidTargetException The target for the pull request is not valid. A target must contain the full
     *         values for the repository name, source branch, and destination branch for the pull request.</li>
     *         <li>MultipleRepositoriesInPullRequestException You cannot include more than one repository in a pull
     *         request. Make sure you have specified only one repository name in your request, and then try again.</li>
     *         <li>MaximumOpenPullRequestsExceededException You cannot create the pull request because the repository
     *         has too many open pull requests. The maximum number of open pull requests for a repository is 1,000.
     *         Close one or more open pull requests, and then try again.</li>
     *         <li>SourceAndDestinationAreSameException The source branch and destination branch for the pull request
     *         are the same. You must specify different branches for the source and destination.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreatePullRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreatePullRequest" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePullRequestResponse> createPullRequest(CreatePullRequestRequest createPullRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPullRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPullRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePullRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePullRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePullRequestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePullRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePullRequestRequest, CreatePullRequestResponse>()
                            .withOperationName("CreatePullRequest")
                            .withMarshaller(new CreatePullRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPullRequestRequest));
            CompletableFuture<CreatePullRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an approval rule for a pull request.
     * </p>
     *
     * @param createPullRequestApprovalRuleRequest
     * @return A Java Future containing the result of the CreatePullRequestApprovalRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleNameRequiredException An approval rule name is required, but was not specified.</li>
     *         <li>InvalidApprovalRuleNameException The name for the approval rule is not valid.</li>
     *         <li>ApprovalRuleNameAlreadyExistsException An approval rule with that name already exists. Approval rule
     *         names must be unique within the scope of a pull request.</li>
     *         <li>ApprovalRuleContentRequiredException The content for the approval rule is empty. You must provide
     *         some content for an approval rule. The content cannot be null.</li>
     *         <li>InvalidApprovalRuleContentException The content for the approval rule is not valid.</li>
     *         <li>NumberOfRulesExceededException The approval rule cannot be added. The pull request has the maximum
     *         number of approval rules associated with it.</li>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreatePullRequestApprovalRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreatePullRequestApprovalRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePullRequestApprovalRuleResponse> createPullRequestApprovalRule(
            CreatePullRequestApprovalRuleRequest createPullRequestApprovalRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPullRequestApprovalRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createPullRequestApprovalRuleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePullRequestApprovalRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePullRequestApprovalRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePullRequestApprovalRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePullRequestApprovalRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePullRequestApprovalRuleRequest, CreatePullRequestApprovalRuleResponse>()
                            .withOperationName("CreatePullRequestApprovalRule")
                            .withMarshaller(new CreatePullRequestApprovalRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPullRequestApprovalRuleRequest));
            CompletableFuture<CreatePullRequestApprovalRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new, empty repository.
     * </p>
     *
     * @param createRepositoryRequest
     *        Represents the input of a create repository operation.
     * @return A Java Future containing the result of the CreateRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameExistsException The specified repository name already exists.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>InvalidRepositoryDescriptionException The specified repository description is not valid.</li>
     *         <li>RepositoryLimitExceededException A repository resource limit was exceeded.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>InvalidTagsMapException The map of tags is not valid.</li>
     *         <li>TooManyTagsException The maximum number of tags for an CodeCommit resource has been exceeded.</li>
     *         <li>InvalidSystemTagUsageException The specified tag is not valid. Key names cannot be prefixed with
     *         aws:.</li>
     *         <li>TagPolicyException The tag policy is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreateRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreateRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRepositoryResponse> createRepository(CreateRepositoryRequest createRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryRequest, CreateRepositoryResponse>()
                            .withOperationName("CreateRepository")
                            .withMarshaller(new CreateRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRepositoryRequest));
            CompletableFuture<CreateRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an unreferenced commit that represents the result of merging two branches using a specified merge
     * strategy. This can help you determine the outcome of a potential merge. This API cannot be used with the
     * fast-forward merge strategy because that strategy does not create a merge commit.
     * </p>
     * <note>
     * <p>
     * This unreferenced merge commit can only be accessed using the GetCommit API or through git commands such as git
     * fetch. To retrieve this commit, you must specify its commit ID or otherwise reference it.
     * </p>
     * </note>
     *
     * @param createUnreferencedMergeCommitRequest
     * @return A Java Future containing the result of the CreateUnreferencedMergeCommit operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>MergeOptionRequiredException A merge option or stategy is required, and none was provided.</li>
     *         <li>InvalidMergeOptionException The specified merge option is not valid for this operation. Not all merge
     *         strategies are supported for all operations.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>InvalidConflictResolutionException The specified conflict resolution list is not valid.</li>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>MaximumConflictResolutionEntriesExceededException The number of allowed conflict resolution entries
     *         was exceeded.</li>
     *         <li>MultipleConflictResolutionEntriesException More than one conflict resolution entries exists for the
     *         conflict. A conflict can have only one conflict resolution entry.</li>
     *         <li>ReplacementTypeRequiredException A replacement type is required.</li>
     *         <li>InvalidReplacementTypeException Automerge was specified for resolving the conflict, but the specified
     *         replacement type is not valid.</li>
     *         <li>ReplacementContentRequiredException USE_NEW_CONTENT was specified, but no replacement content has
     *         been provided.</li>
     *         <li>InvalidReplacementContentException Automerge was specified for resolving the conflict, but the
     *         replacement type is not valid or content is missing.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>FileModeRequiredException The commit cannot be created because no file mode has been specified. A
     *         file mode is required to update mode permissions for a file.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.CreateUnreferencedMergeCommit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/CreateUnreferencedMergeCommit"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUnreferencedMergeCommitResponse> createUnreferencedMergeCommit(
            CreateUnreferencedMergeCommitRequest createUnreferencedMergeCommitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createUnreferencedMergeCommitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createUnreferencedMergeCommitRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUnreferencedMergeCommit");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUnreferencedMergeCommitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateUnreferencedMergeCommitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUnreferencedMergeCommitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUnreferencedMergeCommitRequest, CreateUnreferencedMergeCommitResponse>()
                            .withOperationName("CreateUnreferencedMergeCommit")
                            .withMarshaller(new CreateUnreferencedMergeCommitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createUnreferencedMergeCommitRequest));
            CompletableFuture<CreateUnreferencedMergeCommitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified approval rule template. Deleting a template does not remove approval rules on pull requests
     * already created with the template.
     * </p>
     *
     * @param deleteApprovalRuleTemplateRequest
     * @return A Java Future containing the result of the DeleteApprovalRuleTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateInUseException The approval rule template is associated with one or more
     *         repositories. You cannot delete a template that is associated with a repository. Remove all associations,
     *         and then try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DeleteApprovalRuleTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DeleteApprovalRuleTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApprovalRuleTemplateResponse> deleteApprovalRuleTemplate(
            DeleteApprovalRuleTemplateRequest deleteApprovalRuleTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteApprovalRuleTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApprovalRuleTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApprovalRuleTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApprovalRuleTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApprovalRuleTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApprovalRuleTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApprovalRuleTemplateRequest, DeleteApprovalRuleTemplateResponse>()
                            .withOperationName("DeleteApprovalRuleTemplate")
                            .withMarshaller(new DeleteApprovalRuleTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApprovalRuleTemplateRequest));
            CompletableFuture<DeleteApprovalRuleTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a branch from a repository, unless that branch is the default branch for the repository.
     * </p>
     *
     * @param deleteBranchRequest
     *        Represents the input of a delete branch operation.
     * @return A Java Future containing the result of the DeleteBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>DefaultBranchCannotBeDeletedException The specified branch is the default branch for the repository,
     *         and cannot be deleted. To delete this branch, you must first set another branch as the default branch.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DeleteBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DeleteBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBranchResponse> deleteBranch(DeleteBranchRequest deleteBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBranchRequest, DeleteBranchResponse>()
                            .withOperationName("DeleteBranch").withMarshaller(new DeleteBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBranchRequest));
            CompletableFuture<DeleteBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the content of a comment made on a change, file, or commit in a repository.
     * </p>
     *
     * @param deleteCommentContentRequest
     * @return A Java Future containing the result of the DeleteCommentContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CommentDoesNotExistException No comment exists with the provided ID. Verify that you have used the
     *         correct ID, and then try again.</li>
     *         <li>CommentIdRequiredException The comment ID is missing or null. A comment ID is required.</li>
     *         <li>InvalidCommentIdException The comment ID is not in a valid format. Make sure that you have provided
     *         the full comment ID.</li>
     *         <li>CommentDeletedException This comment has already been deleted. You cannot edit or delete a deleted
     *         comment.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DeleteCommentContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DeleteCommentContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCommentContentResponse> deleteCommentContent(
            DeleteCommentContentRequest deleteCommentContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCommentContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCommentContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCommentContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCommentContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCommentContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCommentContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCommentContentRequest, DeleteCommentContentResponse>()
                            .withOperationName("DeleteCommentContent")
                            .withMarshaller(new DeleteCommentContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCommentContentRequest));
            CompletableFuture<DeleteCommentContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified file from a specified branch. A commit is created on the branch that contains the revision.
     * The file still exists in the commits earlier to the commit that contains the deletion.
     * </p>
     *
     * @param deleteFileRequest
     * @return A Java Future containing the result of the DeleteFile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>ParentCommitIdRequiredException A parent commit ID is required. To view the full commit ID of a
     *         branch in a repository, use <a>GetBranch</a> or a Git command (for example, git pull or git log).</li>
     *         <li>InvalidParentCommitIdException The parent commit ID is not valid. The commit ID cannot be empty, and
     *         must match the head commit ID for the branch of the repository where you want to add or update a file.</li>
     *         <li>ParentCommitDoesNotExistException The parent commit ID is not valid because it does not exist. The
     *         specified parent commit ID does not exist in the specified branch of the repository.</li>
     *         <li>ParentCommitIdOutdatedException The file could not be added because the provided parent commit ID is
     *         not the current tip of the specified branch. To view the full commit ID of the current head of the
     *         branch, use <a>GetBranch</a>.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FileDoesNotExistException The specified file does not exist. Verify that you have used the correct
     *         file name, full path, and extension.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>BranchNameIsTagNameException The specified branch name is not valid because it is a tag name. Enter
     *         the name of a branch in the repository. For a list of valid branch names, use <a>ListBranches</a>.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DeleteFile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DeleteFile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFileResponse> deleteFile(DeleteFileRequest deleteFileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFileRequest, DeleteFileResponse>().withOperationName("DeleteFile")
                            .withMarshaller(new DeleteFileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFileRequest));
            CompletableFuture<DeleteFileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an approval rule from a specified pull request. Approval rules can be deleted from a pull request only if
     * the pull request is open, and if the approval rule was created specifically for a pull request and not generated
     * from an approval rule template associated with the repository where the pull request was created. You cannot
     * delete an approval rule from a merged or closed pull request.
     * </p>
     *
     * @param deletePullRequestApprovalRuleRequest
     * @return A Java Future containing the result of the DeletePullRequestApprovalRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>ApprovalRuleNameRequiredException An approval rule name is required, but was not specified.</li>
     *         <li>InvalidApprovalRuleNameException The name for the approval rule is not valid.</li>
     *         <li>CannotDeleteApprovalRuleFromTemplateException The approval rule cannot be deleted from the pull
     *         request because it was created by an approval rule template and applied to the pull request
     *         automatically.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DeletePullRequestApprovalRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DeletePullRequestApprovalRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePullRequestApprovalRuleResponse> deletePullRequestApprovalRule(
            DeletePullRequestApprovalRuleRequest deletePullRequestApprovalRuleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePullRequestApprovalRuleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deletePullRequestApprovalRuleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePullRequestApprovalRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePullRequestApprovalRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePullRequestApprovalRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePullRequestApprovalRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePullRequestApprovalRuleRequest, DeletePullRequestApprovalRuleResponse>()
                            .withOperationName("DeletePullRequestApprovalRule")
                            .withMarshaller(new DeletePullRequestApprovalRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePullRequestApprovalRuleRequest));
            CompletableFuture<DeletePullRequestApprovalRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a repository. If a specified repository was already deleted, a null repository ID is returned.
     * </p>
     * <important>
     * <p>
     * Deleting a repository also deletes all associated objects and metadata. After a repository is deleted, all future
     * push calls to the deleted repository fail.
     * </p>
     * </important>
     *
     * @param deleteRepositoryRequest
     *        Represents the input of a delete repository operation.
     * @return A Java Future containing the result of the DeleteRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DeleteRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DeleteRepository" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryResponse> deleteRepository(DeleteRepositoryRequest deleteRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryRequest, DeleteRepositoryResponse>()
                            .withOperationName("DeleteRepository")
                            .withMarshaller(new DeleteRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryRequest));
            CompletableFuture<DeleteRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about one or more merge conflicts in the attempted merge of two commit specifiers using the
     * squash or three-way merge strategy. If the merge option for the attempted merge is specified as
     * FAST_FORWARD_MERGE, an exception is thrown.
     * </p>
     *
     * @param describeMergeConflictsRequest
     * @return A Java Future containing the result of the DescribeMergeConflicts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>MergeOptionRequiredException A merge option or stategy is required, and none was provided.</li>
     *         <li>InvalidMergeOptionException The specified merge option is not valid for this operation. Not all merge
     *         strategies are supported for all operations.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FileDoesNotExistException The specified file does not exist. Verify that you have used the correct
     *         file name, full path, and extension.</li>
     *         <li>InvalidMaxMergeHunksException The specified value for the number of merge hunks to return is not
     *         valid.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DescribeMergeConflicts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DescribeMergeConflicts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMergeConflictsResponse> describeMergeConflicts(
            DescribeMergeConflictsRequest describeMergeConflictsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMergeConflictsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMergeConflictsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMergeConflicts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMergeConflictsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMergeConflictsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMergeConflictsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMergeConflictsRequest, DescribeMergeConflictsResponse>()
                            .withOperationName("DescribeMergeConflicts")
                            .withMarshaller(new DescribeMergeConflictsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMergeConflictsRequest));
            CompletableFuture<DescribeMergeConflictsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about one or more pull request events.
     * </p>
     *
     * @param describePullRequestEventsRequest
     * @return A Java Future containing the result of the DescribePullRequestEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidPullRequestEventTypeException The pull request event type is not valid.</li>
     *         <li>InvalidActorArnException The Amazon Resource Name (ARN) is not valid. Make sure that you have
     *         provided the full ARN for the user who initiated the change for the pull request, and then try again.</li>
     *         <li>ActorDoesNotExistException The specified Amazon Resource Name (ARN) does not exist in the Amazon Web
     *         Services account.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DescribePullRequestEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DescribePullRequestEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePullRequestEventsResponse> describePullRequestEvents(
            DescribePullRequestEventsRequest describePullRequestEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePullRequestEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePullRequestEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePullRequestEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePullRequestEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePullRequestEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePullRequestEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePullRequestEventsRequest, DescribePullRequestEventsResponse>()
                            .withOperationName("DescribePullRequestEvents")
                            .withMarshaller(new DescribePullRequestEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePullRequestEventsRequest));
            CompletableFuture<DescribePullRequestEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association between a template and a repository so that approval rules based on the template are not
     * automatically created when pull requests are created in the specified repository. This does not delete any
     * approval rules previously created for pull requests through the template association.
     * </p>
     *
     * @param disassociateApprovalRuleTemplateFromRepositoryRequest
     * @return A Java Future containing the result of the DisassociateApprovalRuleTemplateFromRepository operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.DisassociateApprovalRuleTemplateFromRepository
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/DisassociateApprovalRuleTemplateFromRepository"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateApprovalRuleTemplateFromRepositoryResponse> disassociateApprovalRuleTemplateFromRepository(
            DisassociateApprovalRuleTemplateFromRepositoryRequest disassociateApprovalRuleTemplateFromRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disassociateApprovalRuleTemplateFromRepositoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateApprovalRuleTemplateFromRepositoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateApprovalRuleTemplateFromRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateApprovalRuleTemplateFromRepositoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisassociateApprovalRuleTemplateFromRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateApprovalRuleTemplateFromRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateApprovalRuleTemplateFromRepositoryRequest, DisassociateApprovalRuleTemplateFromRepositoryResponse>()
                            .withOperationName("DisassociateApprovalRuleTemplateFromRepository")
                            .withMarshaller(new DisassociateApprovalRuleTemplateFromRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateApprovalRuleTemplateFromRepositoryRequest));
            CompletableFuture<DisassociateApprovalRuleTemplateFromRepositoryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Evaluates whether a pull request has met all the conditions specified in its associated approval rules.
     * </p>
     *
     * @param evaluatePullRequestApprovalRulesRequest
     * @return A Java Future containing the result of the EvaluatePullRequestApprovalRules operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidRevisionIdException The revision ID is not valid. Use GetPullRequest to determine the value.</li>
     *         <li>RevisionIdRequiredException A revision ID is required, but was not provided.</li>
     *         <li>RevisionNotCurrentException The revision ID provided in the request does not match the current
     *         revision ID. Use GetPullRequest to retrieve the current revision ID.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.EvaluatePullRequestApprovalRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/EvaluatePullRequestApprovalRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EvaluatePullRequestApprovalRulesResponse> evaluatePullRequestApprovalRules(
            EvaluatePullRequestApprovalRulesRequest evaluatePullRequestApprovalRulesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(evaluatePullRequestApprovalRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                evaluatePullRequestApprovalRulesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluatePullRequestApprovalRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EvaluatePullRequestApprovalRulesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EvaluatePullRequestApprovalRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EvaluatePullRequestApprovalRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EvaluatePullRequestApprovalRulesRequest, EvaluatePullRequestApprovalRulesResponse>()
                            .withOperationName("EvaluatePullRequestApprovalRules")
                            .withMarshaller(new EvaluatePullRequestApprovalRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(evaluatePullRequestApprovalRulesRequest));
            CompletableFuture<EvaluatePullRequestApprovalRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specified approval rule template.
     * </p>
     *
     * @param getApprovalRuleTemplateRequest
     * @return A Java Future containing the result of the GetApprovalRuleTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetApprovalRuleTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetApprovalRuleTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApprovalRuleTemplateResponse> getApprovalRuleTemplate(
            GetApprovalRuleTemplateRequest getApprovalRuleTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApprovalRuleTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApprovalRuleTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApprovalRuleTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApprovalRuleTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApprovalRuleTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApprovalRuleTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApprovalRuleTemplateRequest, GetApprovalRuleTemplateResponse>()
                            .withOperationName("GetApprovalRuleTemplate")
                            .withMarshaller(new GetApprovalRuleTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getApprovalRuleTemplateRequest));
            CompletableFuture<GetApprovalRuleTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the base-64 encoded content of an individual blob in a repository.
     * </p>
     *
     * @param getBlobRequest
     *        Represents the input of a get blob operation.
     * @return A Java Future containing the result of the GetBlob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>BlobIdRequiredException A blob ID is required, but was not specified.</li>
     *         <li>InvalidBlobIdException The specified blob is not valid.</li>
     *         <li>BlobIdDoesNotExistException The specified blob does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>FileTooLargeException The specified file exceeds the file size limit for CodeCommit. For more
     *         information about limits in CodeCommit, see <a
     *         href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a> in the
     *         <i>CodeCommit User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetBlob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetBlob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBlobResponse> getBlob(GetBlobRequest getBlobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBlobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBlobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBlobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBlobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBlobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBlobRequest, GetBlobResponse>().withOperationName("GetBlob")
                            .withMarshaller(new GetBlobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBlobRequest));
            CompletableFuture<GetBlobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a repository branch, including its name and the last commit ID.
     * </p>
     *
     * @param getBranchRequest
     *        Represents the input of a get branch operation.
     * @return A Java Future containing the result of the GetBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBranchResponse> getBranch(GetBranchRequest getBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBranchRequest, GetBranchResponse>().withOperationName("GetBranch")
                            .withMarshaller(new GetBranchRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBranchRequest));
            CompletableFuture<GetBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the content of a comment made on a change, file, or commit in a repository.
     * </p>
     * <note>
     * <p>
     * Reaction counts might include numbers from user identities who were deleted after the reaction was made. For a
     * count of reactions from active identities, use GetCommentReactions.
     * </p>
     * </note>
     *
     * @param getCommentRequest
     * @return A Java Future containing the result of the GetComment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CommentDoesNotExistException No comment exists with the provided ID. Verify that you have used the
     *         correct ID, and then try again.</li>
     *         <li>CommentDeletedException This comment has already been deleted. You cannot edit or delete a deleted
     *         comment.</li>
     *         <li>CommentIdRequiredException The comment ID is missing or null. A comment ID is required.</li>
     *         <li>InvalidCommentIdException The comment ID is not in a valid format. Make sure that you have provided
     *         the full comment ID.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetComment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetComment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCommentResponse> getComment(GetCommentRequest getCommentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCommentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCommentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCommentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCommentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCommentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCommentRequest, GetCommentResponse>().withOperationName("GetComment")
                            .withMarshaller(new GetCommentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCommentRequest));
            CompletableFuture<GetCommentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about reactions to a specified comment ID. Reactions from users who have been deleted will
     * not be included in the count.
     * </p>
     *
     * @param getCommentReactionsRequest
     * @return A Java Future containing the result of the GetCommentReactions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CommentDoesNotExistException No comment exists with the provided ID. Verify that you have used the
     *         correct ID, and then try again.</li>
     *         <li>CommentIdRequiredException The comment ID is missing or null. A comment ID is required.</li>
     *         <li>InvalidCommentIdException The comment ID is not in a valid format. Make sure that you have provided
     *         the full comment ID.</li>
     *         <li>InvalidReactionUserArnException The Amazon Resource Name (ARN) of the user or identity is not valid.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>CommentDeletedException This comment has already been deleted. You cannot edit or delete a deleted
     *         comment.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetCommentReactions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetCommentReactions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCommentReactionsResponse> getCommentReactions(
            GetCommentReactionsRequest getCommentReactionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCommentReactionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCommentReactionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCommentReactions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCommentReactionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCommentReactionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCommentReactionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCommentReactionsRequest, GetCommentReactionsResponse>()
                            .withOperationName("GetCommentReactions")
                            .withMarshaller(new GetCommentReactionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCommentReactionsRequest));
            CompletableFuture<GetCommentReactionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about comments made on the comparison between two commits.
     * </p>
     * <note>
     * <p>
     * Reaction counts might include numbers from user identities who were deleted after the reaction was made. For a
     * count of reactions from active identities, use GetCommentReactions.
     * </p>
     * </note>
     *
     * @param getCommentsForComparedCommitRequest
     * @return A Java Future containing the result of the GetCommentsForComparedCommit operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>CommitIdRequiredException A commit ID was not specified.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetCommentsForComparedCommit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetCommentsForComparedCommit"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCommentsForComparedCommitResponse> getCommentsForComparedCommit(
            GetCommentsForComparedCommitRequest getCommentsForComparedCommitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCommentsForComparedCommitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCommentsForComparedCommitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCommentsForComparedCommit");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCommentsForComparedCommitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCommentsForComparedCommitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCommentsForComparedCommitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCommentsForComparedCommitRequest, GetCommentsForComparedCommitResponse>()
                            .withOperationName("GetCommentsForComparedCommit")
                            .withMarshaller(new GetCommentsForComparedCommitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCommentsForComparedCommitRequest));
            CompletableFuture<GetCommentsForComparedCommitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns comments made on a pull request.
     * </p>
     * <note>
     * <p>
     * Reaction counts might include numbers from user identities who were deleted after the reaction was made. For a
     * count of reactions from active identities, use GetCommentReactions.
     * </p>
     * </note>
     *
     * @param getCommentsForPullRequestRequest
     * @return A Java Future containing the result of the GetCommentsForPullRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>CommitIdRequiredException A commit ID was not specified.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>RepositoryNotAssociatedWithPullRequestException The repository does not contain any pull requests
     *         with that pull request ID. Use GetPullRequest to verify the correct repository name for the pull request
     *         ID.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetCommentsForPullRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetCommentsForPullRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCommentsForPullRequestResponse> getCommentsForPullRequest(
            GetCommentsForPullRequestRequest getCommentsForPullRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCommentsForPullRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCommentsForPullRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCommentsForPullRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCommentsForPullRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCommentsForPullRequestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCommentsForPullRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCommentsForPullRequestRequest, GetCommentsForPullRequestResponse>()
                            .withOperationName("GetCommentsForPullRequest")
                            .withMarshaller(new GetCommentsForPullRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCommentsForPullRequestRequest));
            CompletableFuture<GetCommentsForPullRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a commit, including commit message and committer information.
     * </p>
     *
     * @param getCommitRequest
     *        Represents the input of a get commit operation.
     * @return A Java Future containing the result of the GetCommit operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>CommitIdRequiredException A commit ID was not specified.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>CommitIdDoesNotExistException The specified commit ID does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetCommit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetCommit" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCommitResponse> getCommit(GetCommitRequest getCommitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCommitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCommitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCommit");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCommitResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCommitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCommitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCommitRequest, GetCommitResponse>().withOperationName("GetCommit")
                            .withMarshaller(new GetCommitRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCommitRequest));
            CompletableFuture<GetCommitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the differences in a valid commit specifier (such as a branch, tag, HEAD, commit ID, or
     * other fully qualified reference). Results can be limited to a specified path.
     * </p>
     *
     * @param getDifferencesRequest
     * @return A Java Future containing the result of the GetDifferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>PathDoesNotExistException The specified path does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetDifferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetDifferences" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDifferencesResponse> getDifferences(GetDifferencesRequest getDifferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDifferencesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDifferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDifferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDifferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDifferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDifferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDifferencesRequest, GetDifferencesResponse>()
                            .withOperationName("GetDifferences")
                            .withMarshaller(new GetDifferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDifferencesRequest));
            CompletableFuture<GetDifferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the base-64 encoded contents of a specified file and its metadata.
     * </p>
     *
     * @param getFileRequest
     * @return A Java Future containing the result of the GetFile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FileDoesNotExistException The specified file does not exist. Verify that you have used the correct
     *         file name, full path, and extension.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>FileTooLargeException The specified file exceeds the file size limit for CodeCommit. For more
     *         information about limits in CodeCommit, see <a
     *         href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a> in the
     *         <i>CodeCommit User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetFile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetFile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFileResponse> getFile(GetFileRequest getFileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFileRequest, GetFileResponse>().withOperationName("GetFile")
                            .withMarshaller(new GetFileRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFileRequest));
            CompletableFuture<GetFileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the contents of a specified folder in a repository.
     * </p>
     *
     * @param getFolderRequest
     * @return A Java Future containing the result of the GetFolder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FolderDoesNotExistException The specified folder does not exist. Either the folder name is not
     *         correct, or you did not enter the full path to the folder.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetFolder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetFolder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFolderResponse> getFolder(GetFolderRequest getFolderRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFolderRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFolderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFolder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFolderResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFolderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFolderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFolderRequest, GetFolderResponse>().withOperationName("GetFolder")
                            .withMarshaller(new GetFolderRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFolderRequest));
            CompletableFuture<GetFolderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specified merge commit.
     * </p>
     *
     * @param getMergeCommitRequest
     * @return A Java Future containing the result of the GetMergeCommit operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetMergeCommit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetMergeCommit" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMergeCommitResponse> getMergeCommit(GetMergeCommitRequest getMergeCommitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMergeCommitRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMergeCommitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMergeCommit");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMergeCommitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMergeCommitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMergeCommitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMergeCommitRequest, GetMergeCommitResponse>()
                            .withOperationName("GetMergeCommit")
                            .withMarshaller(new GetMergeCommitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMergeCommitRequest));
            CompletableFuture<GetMergeCommitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about merge conflicts between the before and after commit IDs for a pull request in a
     * repository.
     * </p>
     *
     * @param getMergeConflictsRequest
     * @return A Java Future containing the result of the GetMergeConflicts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>MergeOptionRequiredException A merge option or stategy is required, and none was provided.</li>
     *         <li>InvalidMergeOptionException The specified merge option is not valid for this operation. Not all merge
     *         strategies are supported for all operations.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>InvalidMaxConflictFilesException The specified value for the number of conflict files to return is
     *         not valid.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidDestinationCommitSpecifierException The destination commit specifier is not valid. You must
     *         provide a valid branch name, tag, or full commit ID.</li>
     *         <li>InvalidSourceCommitSpecifierException The source commit specifier is not valid. You must provide a
     *         valid branch name, tag, or full commit ID.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetMergeConflicts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetMergeConflicts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMergeConflictsResponse> getMergeConflicts(GetMergeConflictsRequest getMergeConflictsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMergeConflictsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMergeConflictsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMergeConflicts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMergeConflictsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMergeConflictsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMergeConflictsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMergeConflictsRequest, GetMergeConflictsResponse>()
                            .withOperationName("GetMergeConflicts")
                            .withMarshaller(new GetMergeConflictsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMergeConflictsRequest));
            CompletableFuture<GetMergeConflictsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the merge options available for merging two specified branches. For details about why a
     * merge option is not available, use GetMergeConflicts or DescribeMergeConflicts.
     * </p>
     *
     * @param getMergeOptionsRequest
     * @return A Java Future containing the result of the GetMergeOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetMergeOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetMergeOptions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMergeOptionsResponse> getMergeOptions(GetMergeOptionsRequest getMergeOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMergeOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMergeOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMergeOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMergeOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMergeOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMergeOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMergeOptionsRequest, GetMergeOptionsResponse>()
                            .withOperationName("GetMergeOptions")
                            .withMarshaller(new GetMergeOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMergeOptionsRequest));
            CompletableFuture<GetMergeOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a pull request in a specified repository.
     * </p>
     *
     * @param getPullRequestRequest
     * @return A Java Future containing the result of the GetPullRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetPullRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetPullRequest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPullRequestResponse> getPullRequest(GetPullRequestRequest getPullRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPullRequestRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPullRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPullRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPullRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPullRequestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPullRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPullRequestRequest, GetPullRequestResponse>()
                            .withOperationName("GetPullRequest")
                            .withMarshaller(new GetPullRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPullRequestRequest));
            CompletableFuture<GetPullRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the approval states for a specified pull request. Approval states only apply to pull
     * requests that have one or more approval rules applied to them.
     * </p>
     *
     * @param getPullRequestApprovalStatesRequest
     * @return A Java Future containing the result of the GetPullRequestApprovalStates operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidRevisionIdException The revision ID is not valid. Use GetPullRequest to determine the value.</li>
     *         <li>RevisionIdRequiredException A revision ID is required, but was not provided.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetPullRequestApprovalStates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetPullRequestApprovalStates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPullRequestApprovalStatesResponse> getPullRequestApprovalStates(
            GetPullRequestApprovalStatesRequest getPullRequestApprovalStatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPullRequestApprovalStatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPullRequestApprovalStatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPullRequestApprovalStates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPullRequestApprovalStatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPullRequestApprovalStatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPullRequestApprovalStatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPullRequestApprovalStatesRequest, GetPullRequestApprovalStatesResponse>()
                            .withOperationName("GetPullRequestApprovalStates")
                            .withMarshaller(new GetPullRequestApprovalStatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPullRequestApprovalStatesRequest));
            CompletableFuture<GetPullRequestApprovalStatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about whether approval rules have been set aside (overridden) for a pull request, and if so,
     * the Amazon Resource Name (ARN) of the user or identity that overrode the rules and their requirements for the
     * pull request.
     * </p>
     *
     * @param getPullRequestOverrideStateRequest
     * @return A Java Future containing the result of the GetPullRequestOverrideState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidRevisionIdException The revision ID is not valid. Use GetPullRequest to determine the value.</li>
     *         <li>RevisionIdRequiredException A revision ID is required, but was not provided.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetPullRequestOverrideState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetPullRequestOverrideState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPullRequestOverrideStateResponse> getPullRequestOverrideState(
            GetPullRequestOverrideStateRequest getPullRequestOverrideStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPullRequestOverrideStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPullRequestOverrideStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPullRequestOverrideState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPullRequestOverrideStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPullRequestOverrideStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPullRequestOverrideStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPullRequestOverrideStateRequest, GetPullRequestOverrideStateResponse>()
                            .withOperationName("GetPullRequestOverrideState")
                            .withMarshaller(new GetPullRequestOverrideStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPullRequestOverrideStateRequest));
            CompletableFuture<GetPullRequestOverrideStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a repository.
     * </p>
     * <note>
     * <p>
     * The description field for a repository accepts all HTML characters and all valid Unicode characters. Applications
     * that do not HTML-encode the description and display it in a webpage can expose users to potentially malicious
     * code. Make sure that you HTML-encode the description field in any application that uses this API to display the
     * repository description on a webpage.
     * </p>
     * </note>
     *
     * @param getRepositoryRequest
     *        Represents the input of a get repository operation.
     * @return A Java Future containing the result of the GetRepository operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetRepository
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetRepository" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryResponse> getRepository(GetRepositoryRequest getRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryRequest, GetRepositoryResponse>()
                            .withOperationName("GetRepository")
                            .withMarshaller(new GetRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositoryRequest));
            CompletableFuture<GetRepositoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about triggers configured for a repository.
     * </p>
     *
     * @param getRepositoryTriggersRequest
     *        Represents the input of a get repository triggers operation.
     * @return A Java Future containing the result of the GetRepositoryTriggers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.GetRepositoryTriggers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/GetRepositoryTriggers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryTriggersResponse> getRepositoryTriggers(
            GetRepositoryTriggersRequest getRepositoryTriggersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryTriggersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryTriggersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryTriggers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryTriggersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryTriggersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRepositoryTriggersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryTriggersRequest, GetRepositoryTriggersResponse>()
                            .withOperationName("GetRepositoryTriggers")
                            .withMarshaller(new GetRepositoryTriggersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositoryTriggersRequest));
            CompletableFuture<GetRepositoryTriggersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all approval rule templates in the specified Amazon Web Services Region in your Amazon Web Services
     * account. If an Amazon Web Services Region is not specified, the Amazon Web Services Region where you are signed
     * in is used.
     * </p>
     *
     * @param listApprovalRuleTemplatesRequest
     * @return A Java Future containing the result of the ListApprovalRuleTemplates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListApprovalRuleTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListApprovalRuleTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApprovalRuleTemplatesResponse> listApprovalRuleTemplates(
            ListApprovalRuleTemplatesRequest listApprovalRuleTemplatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApprovalRuleTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApprovalRuleTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApprovalRuleTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApprovalRuleTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApprovalRuleTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApprovalRuleTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApprovalRuleTemplatesRequest, ListApprovalRuleTemplatesResponse>()
                            .withOperationName("ListApprovalRuleTemplates")
                            .withMarshaller(new ListApprovalRuleTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listApprovalRuleTemplatesRequest));
            CompletableFuture<ListApprovalRuleTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all approval rule templates that are associated with a specified repository.
     * </p>
     *
     * @param listAssociatedApprovalRuleTemplatesForRepositoryRequest
     * @return A Java Future containing the result of the ListAssociatedApprovalRuleTemplatesForRepository operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListAssociatedApprovalRuleTemplatesForRepository
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListAssociatedApprovalRuleTemplatesForRepository"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociatedApprovalRuleTemplatesForRepositoryResponse> listAssociatedApprovalRuleTemplatesForRepository(
            ListAssociatedApprovalRuleTemplatesForRepositoryRequest listAssociatedApprovalRuleTemplatesForRepositoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listAssociatedApprovalRuleTemplatesForRepositoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAssociatedApprovalRuleTemplatesForRepositoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociatedApprovalRuleTemplatesForRepository");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociatedApprovalRuleTemplatesForRepositoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListAssociatedApprovalRuleTemplatesForRepositoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssociatedApprovalRuleTemplatesForRepositoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociatedApprovalRuleTemplatesForRepositoryRequest, ListAssociatedApprovalRuleTemplatesForRepositoryResponse>()
                            .withOperationName("ListAssociatedApprovalRuleTemplatesForRepository")
                            .withMarshaller(
                                    new ListAssociatedApprovalRuleTemplatesForRepositoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociatedApprovalRuleTemplatesForRepositoryRequest));
            CompletableFuture<ListAssociatedApprovalRuleTemplatesForRepositoryResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about one or more branches in a repository.
     * </p>
     *
     * @param listBranchesRequest
     *        Represents the input of a list branches operation.
     * @return A Java Future containing the result of the ListBranches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListBranches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListBranches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBranchesResponse> listBranches(ListBranchesRequest listBranchesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBranchesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBranchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBranches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBranchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBranchesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBranchesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBranchesRequest, ListBranchesResponse>()
                            .withOperationName("ListBranches").withMarshaller(new ListBranchesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBranchesRequest));
            CompletableFuture<ListBranchesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of commits and changes to a specified file.
     * </p>
     *
     * @param listFileCommitHistoryRequest
     * @return A Java Future containing the result of the ListFileCommitHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListFileCommitHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListFileCommitHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFileCommitHistoryResponse> listFileCommitHistory(
            ListFileCommitHistoryRequest listFileCommitHistoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFileCommitHistoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFileCommitHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFileCommitHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFileCommitHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListFileCommitHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFileCommitHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFileCommitHistoryRequest, ListFileCommitHistoryResponse>()
                            .withOperationName("ListFileCommitHistory")
                            .withMarshaller(new ListFileCommitHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFileCommitHistoryRequest));
            CompletableFuture<ListFileCommitHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of pull requests for a specified repository. The return list can be refined by pull request status
     * or pull request author ARN.
     * </p>
     *
     * @param listPullRequestsRequest
     * @return A Java Future containing the result of the ListPullRequests operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidPullRequestStatusException The pull request status is not valid. The only valid values are
     *         <code>OPEN</code> and <code>CLOSED</code>.</li>
     *         <li>InvalidAuthorArnException The Amazon Resource Name (ARN) is not valid. Make sure that you have
     *         provided the full ARN for the author of the pull request, and then try again.</li>
     *         <li>AuthorDoesNotExistException The specified Amazon Resource Name (ARN) does not exist in the Amazon Web
     *         Services account.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListPullRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListPullRequests" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPullRequestsResponse> listPullRequests(ListPullRequestsRequest listPullRequestsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPullRequestsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPullRequestsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPullRequests");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPullRequestsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPullRequestsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPullRequestsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPullRequestsRequest, ListPullRequestsResponse>()
                            .withOperationName("ListPullRequests")
                            .withMarshaller(new ListPullRequestsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPullRequestsRequest));
            CompletableFuture<ListPullRequestsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about one or more repositories.
     * </p>
     *
     * @param listRepositoriesRequest
     *        Represents the input of a list repositories operation.
     * @return A Java Future containing the result of the ListRepositories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidSortByException The specified sort by value is not valid.</li>
     *         <li>InvalidOrderException The specified sort order is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListRepositories
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListRepositories" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRepositoriesResponse> listRepositories(ListRepositoriesRequest listRepositoriesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRepositoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRepositoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRepositoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRepositoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRepositoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRepositoriesRequest, ListRepositoriesResponse>()
                            .withOperationName("ListRepositories")
                            .withMarshaller(new ListRepositoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRepositoriesRequest));
            CompletableFuture<ListRepositoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all repositories associated with the specified approval rule template.
     * </p>
     *
     * @param listRepositoriesForApprovalRuleTemplateRequest
     * @return A Java Future containing the result of the ListRepositoriesForApprovalRuleTemplate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>InvalidMaxResultsException The specified number of maximum results is not valid.</li>
     *         <li>InvalidContinuationTokenException The specified continuation token is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListRepositoriesForApprovalRuleTemplate
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListRepositoriesForApprovalRuleTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRepositoriesForApprovalRuleTemplateResponse> listRepositoriesForApprovalRuleTemplate(
            ListRepositoriesForApprovalRuleTemplateRequest listRepositoriesForApprovalRuleTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRepositoriesForApprovalRuleTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRepositoriesForApprovalRuleTemplateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositoriesForApprovalRuleTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRepositoriesForApprovalRuleTemplateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListRepositoriesForApprovalRuleTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRepositoriesForApprovalRuleTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRepositoriesForApprovalRuleTemplateRequest, ListRepositoriesForApprovalRuleTemplateResponse>()
                            .withOperationName("ListRepositoriesForApprovalRuleTemplate")
                            .withMarshaller(new ListRepositoriesForApprovalRuleTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRepositoriesForApprovalRuleTemplateRequest));
            CompletableFuture<ListRepositoriesForApprovalRuleTemplateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about Amazon Web Servicestags for a specified Amazon Resource Name (ARN) in CodeCommit. For a
     * list of valid resources in CodeCommit, see <a href=
     * "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     * >CodeCommit Resources and Operations</a> in the<i> CodeCommit User Guide</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>ResourceArnRequiredException A valid Amazon Resource Name (ARN) for an CodeCommit resource is
     *         required. For a list of valid resources in CodeCommit, see <a href=
     *         "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     *         >CodeCommit Resources and Operations</a> in the CodeCommit User Guide.</li>
     *         <li>InvalidResourceArnException The value for the resource ARN is not valid. For more information about
     *         resources in CodeCommit, see <a href=
     *         "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     *         >CodeCommit Resources and Operations</a> in the CodeCommit User Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Merges two branches using the fast-forward merge strategy.
     * </p>
     *
     * @param mergeBranchesByFastForwardRequest
     * @return A Java Future containing the result of the MergeBranchesByFastForward operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidTargetBranchException The specified target branch is not valid.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>BranchNameIsTagNameException The specified branch name is not valid because it is a tag name. Enter
     *         the name of a branch in the repository. For a list of valid branch names, use <a>ListBranches</a>.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.MergeBranchesByFastForward
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/MergeBranchesByFastForward"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergeBranchesByFastForwardResponse> mergeBranchesByFastForward(
            MergeBranchesByFastForwardRequest mergeBranchesByFastForwardRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergeBranchesByFastForwardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeBranchesByFastForwardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeBranchesByFastForward");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergeBranchesByFastForwardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergeBranchesByFastForwardResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergeBranchesByFastForwardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergeBranchesByFastForwardRequest, MergeBranchesByFastForwardResponse>()
                            .withOperationName("MergeBranchesByFastForward")
                            .withMarshaller(new MergeBranchesByFastForwardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(mergeBranchesByFastForwardRequest));
            CompletableFuture<MergeBranchesByFastForwardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Merges two branches using the squash merge strategy.
     * </p>
     *
     * @param mergeBranchesBySquashRequest
     * @return A Java Future containing the result of the MergeBranchesBySquash operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidTargetBranchException The specified target branch is not valid.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>BranchNameIsTagNameException The specified branch name is not valid because it is a tag name. Enter
     *         the name of a branch in the repository. For a list of valid branch names, use <a>ListBranches</a>.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>InvalidConflictResolutionException The specified conflict resolution list is not valid.</li>
     *         <li>MaximumConflictResolutionEntriesExceededException The number of allowed conflict resolution entries
     *         was exceeded.</li>
     *         <li>MultipleConflictResolutionEntriesException More than one conflict resolution entries exists for the
     *         conflict. A conflict can have only one conflict resolution entry.</li>
     *         <li>ReplacementTypeRequiredException A replacement type is required.</li>
     *         <li>InvalidReplacementTypeException Automerge was specified for resolving the conflict, but the specified
     *         replacement type is not valid.</li>
     *         <li>ReplacementContentRequiredException USE_NEW_CONTENT was specified, but no replacement content has
     *         been provided.</li>
     *         <li>InvalidReplacementContentException Automerge was specified for resolving the conflict, but the
     *         replacement type is not valid or content is missing.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>FileModeRequiredException The commit cannot be created because no file mode has been specified. A
     *         file mode is required to update mode permissions for a file.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.MergeBranchesBySquash
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/MergeBranchesBySquash"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergeBranchesBySquashResponse> mergeBranchesBySquash(
            MergeBranchesBySquashRequest mergeBranchesBySquashRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergeBranchesBySquashRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeBranchesBySquashRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeBranchesBySquash");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergeBranchesBySquashResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergeBranchesBySquashResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergeBranchesBySquashResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergeBranchesBySquashRequest, MergeBranchesBySquashResponse>()
                            .withOperationName("MergeBranchesBySquash")
                            .withMarshaller(new MergeBranchesBySquashRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(mergeBranchesBySquashRequest));
            CompletableFuture<MergeBranchesBySquashResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Merges two specified branches using the three-way merge strategy.
     * </p>
     *
     * @param mergeBranchesByThreeWayRequest
     * @return A Java Future containing the result of the MergeBranchesByThreeWay operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>CommitRequiredException A commit was not specified.</li>
     *         <li>InvalidCommitException The specified commit is not valid.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidTargetBranchException The specified target branch is not valid.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>BranchNameIsTagNameException The specified branch name is not valid because it is a tag name. Enter
     *         the name of a branch in the repository. For a list of valid branch names, use <a>ListBranches</a>.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>InvalidConflictResolutionException The specified conflict resolution list is not valid.</li>
     *         <li>MaximumConflictResolutionEntriesExceededException The number of allowed conflict resolution entries
     *         was exceeded.</li>
     *         <li>MultipleConflictResolutionEntriesException More than one conflict resolution entries exists for the
     *         conflict. A conflict can have only one conflict resolution entry.</li>
     *         <li>ReplacementTypeRequiredException A replacement type is required.</li>
     *         <li>InvalidReplacementTypeException Automerge was specified for resolving the conflict, but the specified
     *         replacement type is not valid.</li>
     *         <li>ReplacementContentRequiredException USE_NEW_CONTENT was specified, but no replacement content has
     *         been provided.</li>
     *         <li>InvalidReplacementContentException Automerge was specified for resolving the conflict, but the
     *         replacement type is not valid or content is missing.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>FileModeRequiredException The commit cannot be created because no file mode has been specified. A
     *         file mode is required to update mode permissions for a file.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.MergeBranchesByThreeWay
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/MergeBranchesByThreeWay"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergeBranchesByThreeWayResponse> mergeBranchesByThreeWay(
            MergeBranchesByThreeWayRequest mergeBranchesByThreeWayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergeBranchesByThreeWayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeBranchesByThreeWayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeBranchesByThreeWay");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergeBranchesByThreeWayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergeBranchesByThreeWayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergeBranchesByThreeWayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergeBranchesByThreeWayRequest, MergeBranchesByThreeWayResponse>()
                            .withOperationName("MergeBranchesByThreeWay")
                            .withMarshaller(new MergeBranchesByThreeWayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(mergeBranchesByThreeWayRequest));
            CompletableFuture<MergeBranchesByThreeWayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attempts to merge the source commit of a pull request into the specified destination branch for that pull request
     * at the specified commit using the fast-forward merge strategy. If the merge is successful, it closes the pull
     * request.
     * </p>
     *
     * @param mergePullRequestByFastForwardRequest
     * @return A Java Future containing the result of the MergePullRequestByFastForward operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>TipOfSourceReferenceIsDifferentException The tip of the source branch in the destination repository
     *         does not match the tip of the source branch specified in your request. The pull request might have been
     *         updated. Make sure that you have the latest changes.</li>
     *         <li>ReferenceDoesNotExistException The specified reference does not exist. You must provide a full commit
     *         ID.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>RepositoryNotAssociatedWithPullRequestException The repository does not contain any pull requests
     *         with that pull request ID. Use GetPullRequest to verify the correct repository name for the pull request
     *         ID.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>PullRequestApprovalRulesNotSatisfiedException The pull request cannot be merged because one or more
     *         approval rules applied to the pull request have conditions that have not been met.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.MergePullRequestByFastForward
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/MergePullRequestByFastForward"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergePullRequestByFastForwardResponse> mergePullRequestByFastForward(
            MergePullRequestByFastForwardRequest mergePullRequestByFastForwardRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergePullRequestByFastForwardRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                mergePullRequestByFastForwardRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergePullRequestByFastForward");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergePullRequestByFastForwardResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergePullRequestByFastForwardResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergePullRequestByFastForwardResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergePullRequestByFastForwardRequest, MergePullRequestByFastForwardResponse>()
                            .withOperationName("MergePullRequestByFastForward")
                            .withMarshaller(new MergePullRequestByFastForwardRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(mergePullRequestByFastForwardRequest));
            CompletableFuture<MergePullRequestByFastForwardResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attempts to merge the source commit of a pull request into the specified destination branch for that pull request
     * at the specified commit using the squash merge strategy. If the merge is successful, it closes the pull request.
     * </p>
     *
     * @param mergePullRequestBySquashRequest
     * @return A Java Future containing the result of the MergePullRequestBySquash operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>TipOfSourceReferenceIsDifferentException The tip of the source branch in the destination repository
     *         does not match the tip of the source branch specified in your request. The pull request might have been
     *         updated. Make sure that you have the latest changes.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>InvalidConflictResolutionException The specified conflict resolution list is not valid.</li>
     *         <li>ReplacementTypeRequiredException A replacement type is required.</li>
     *         <li>InvalidReplacementTypeException Automerge was specified for resolving the conflict, but the specified
     *         replacement type is not valid.</li>
     *         <li>MultipleConflictResolutionEntriesException More than one conflict resolution entries exists for the
     *         conflict. A conflict can have only one conflict resolution entry.</li>
     *         <li>ReplacementContentRequiredException USE_NEW_CONTENT was specified, but no replacement content has
     *         been provided.</li>
     *         <li>MaximumConflictResolutionEntriesExceededException The number of allowed conflict resolution entries
     *         was exceeded.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>InvalidReplacementContentException Automerge was specified for resolving the conflict, but the
     *         replacement type is not valid or content is missing.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>RepositoryNotAssociatedWithPullRequestException The repository does not contain any pull requests
     *         with that pull request ID. Use GetPullRequest to verify the correct repository name for the pull request
     *         ID.</li>
     *         <li>PullRequestApprovalRulesNotSatisfiedException The pull request cannot be merged because one or more
     *         approval rules applied to the pull request have conditions that have not been met.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.MergePullRequestBySquash
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/MergePullRequestBySquash"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergePullRequestBySquashResponse> mergePullRequestBySquash(
            MergePullRequestBySquashRequest mergePullRequestBySquashRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergePullRequestBySquashRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergePullRequestBySquashRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergePullRequestBySquash");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergePullRequestBySquashResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergePullRequestBySquashResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergePullRequestBySquashResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergePullRequestBySquashRequest, MergePullRequestBySquashResponse>()
                            .withOperationName("MergePullRequestBySquash")
                            .withMarshaller(new MergePullRequestBySquashRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(mergePullRequestBySquashRequest));
            CompletableFuture<MergePullRequestBySquashResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attempts to merge the source commit of a pull request into the specified destination branch for that pull request
     * at the specified commit using the three-way merge strategy. If the merge is successful, it closes the pull
     * request.
     * </p>
     *
     * @param mergePullRequestByThreeWayRequest
     * @return A Java Future containing the result of the MergePullRequestByThreeWay operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>ManualMergeRequiredException The pull request cannot be merged automatically into the destination
     *         branch. You must manually merge the branches and resolve any conflicts.</li>
     *         <li>TipOfSourceReferenceIsDifferentException The tip of the source branch in the destination repository
     *         does not match the tip of the source branch specified in your request. The pull request might have been
     *         updated. Make sure that you have the latest changes.</li>
     *         <li>TipsDivergenceExceededException The divergence between the tips of the provided commit specifiers is
     *         too great to determine whether there might be any merge conflicts. Locally compare the specifiers using
     *         <code>git diff</code> or a diff tool.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>InvalidConflictDetailLevelException The specified conflict detail level is not valid.</li>
     *         <li>InvalidConflictResolutionStrategyException The specified conflict resolution strategy is not valid.</li>
     *         <li>InvalidConflictResolutionException The specified conflict resolution list is not valid.</li>
     *         <li>ReplacementTypeRequiredException A replacement type is required.</li>
     *         <li>InvalidReplacementTypeException Automerge was specified for resolving the conflict, but the specified
     *         replacement type is not valid.</li>
     *         <li>MultipleConflictResolutionEntriesException More than one conflict resolution entries exists for the
     *         conflict. A conflict can have only one conflict resolution entry.</li>
     *         <li>ReplacementContentRequiredException USE_NEW_CONTENT was specified, but no replacement content has
     *         been provided.</li>
     *         <li>MaximumConflictResolutionEntriesExceededException The number of allowed conflict resolution entries
     *         was exceeded.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>InvalidReplacementContentException Automerge was specified for resolving the conflict, but the
     *         replacement type is not valid or content is missing.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>MaximumFileContentToLoadExceededException The number of files to load exceeds the allowed limit.</li>
     *         <li>MaximumItemsToCompareExceededException The number of items to compare between the source or
     *         destination branches and the merge base has exceeded the maximum allowed.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>RepositoryNotAssociatedWithPullRequestException The repository does not contain any pull requests
     *         with that pull request ID. Use GetPullRequest to verify the correct repository name for the pull request
     *         ID.</li>
     *         <li>ConcurrentReferenceUpdateException The merge cannot be completed because the target branch has been
     *         modified. Another user might have modified the target branch while the merge was in progress. Wait a few
     *         minutes, and then try again.</li>
     *         <li>PullRequestApprovalRulesNotSatisfiedException The pull request cannot be merged because one or more
     *         approval rules applied to the pull request have conditions that have not been met.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.MergePullRequestByThreeWay
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/MergePullRequestByThreeWay"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergePullRequestByThreeWayResponse> mergePullRequestByThreeWay(
            MergePullRequestByThreeWayRequest mergePullRequestByThreeWayRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergePullRequestByThreeWayRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergePullRequestByThreeWayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergePullRequestByThreeWay");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergePullRequestByThreeWayResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergePullRequestByThreeWayResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergePullRequestByThreeWayResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergePullRequestByThreeWayRequest, MergePullRequestByThreeWayResponse>()
                            .withOperationName("MergePullRequestByThreeWay")
                            .withMarshaller(new MergePullRequestByThreeWayRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(mergePullRequestByThreeWayRequest));
            CompletableFuture<MergePullRequestByThreeWayResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets aside (overrides) all approval rule requirements for a specified pull request.
     * </p>
     *
     * @param overridePullRequestApprovalRulesRequest
     * @return A Java Future containing the result of the OverridePullRequestApprovalRules operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidRevisionIdException The revision ID is not valid. Use GetPullRequest to determine the value.</li>
     *         <li>RevisionIdRequiredException A revision ID is required, but was not provided.</li>
     *         <li>InvalidOverrideStatusException The override status is not valid. Valid statuses are OVERRIDE and
     *         REVOKE.</li>
     *         <li>OverrideStatusRequiredException An override status is required, but no value was provided. Valid
     *         values include OVERRIDE and REVOKE.</li>
     *         <li>OverrideAlreadySetException The pull request has already had its approval rules set to override.</li>
     *         <li>RevisionNotCurrentException The revision ID provided in the request does not match the current
     *         revision ID. Use GetPullRequest to retrieve the current revision ID.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.OverridePullRequestApprovalRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/OverridePullRequestApprovalRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<OverridePullRequestApprovalRulesResponse> overridePullRequestApprovalRules(
            OverridePullRequestApprovalRulesRequest overridePullRequestApprovalRulesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(overridePullRequestApprovalRulesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                overridePullRequestApprovalRulesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OverridePullRequestApprovalRules");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<OverridePullRequestApprovalRulesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, OverridePullRequestApprovalRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<OverridePullRequestApprovalRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<OverridePullRequestApprovalRulesRequest, OverridePullRequestApprovalRulesResponse>()
                            .withOperationName("OverridePullRequestApprovalRules")
                            .withMarshaller(new OverridePullRequestApprovalRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(overridePullRequestApprovalRulesRequest));
            CompletableFuture<OverridePullRequestApprovalRulesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Posts a comment on the comparison between two commits.
     * </p>
     *
     * @param postCommentForComparedCommitRequest
     * @return A Java Future containing the result of the PostCommentForComparedCommit operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>ClientRequestTokenRequiredException A client request token is required. A client request token is an
     *         unique, client-generated idempotency token that, when provided in a request, ensures the request cannot
     *         be repeated with a changed parameter. If a request is received with the same parameters and a token is
     *         included, the request returns information about the initial request that used that token.</li>
     *         <li>InvalidClientRequestTokenException The client request token is not valid.</li>
     *         <li>IdempotencyParameterMismatchException The client request token is not valid. Either the token is not
     *         in a valid format, or the token has been used in a previous request and cannot be reused.</li>
     *         <li>CommentContentRequiredException The comment is empty. You must provide some content for a comment.
     *         The content cannot be null.</li>
     *         <li>CommentContentSizeLimitExceededException The comment is too large. Comments are limited to 10,240
     *         characters.</li>
     *         <li>InvalidFileLocationException The location of the file is not valid. Make sure that you include the
     *         file name and extension.</li>
     *         <li>InvalidRelativeFileVersionEnumException Either the enum is not in a valid format, or the specified
     *         file version enum is not valid in respect to the current file version.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidFilePositionException The position is not valid. Make sure that the line number exists in the
     *         version of the file you want to comment on.</li>
     *         <li>CommitIdRequiredException A commit ID was not specified.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>BeforeCommitIdAndAfterCommitIdAreSameException The before commit ID and the after commit ID are the
     *         same, which is not valid. The before commit ID and the after commit ID must be different commit IDs.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>PathDoesNotExistException The specified path does not exist.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.PostCommentForComparedCommit
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/PostCommentForComparedCommit"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PostCommentForComparedCommitResponse> postCommentForComparedCommit(
            PostCommentForComparedCommitRequest postCommentForComparedCommitRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postCommentForComparedCommitRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postCommentForComparedCommitRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostCommentForComparedCommit");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PostCommentForComparedCommitResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PostCommentForComparedCommitResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PostCommentForComparedCommitResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PostCommentForComparedCommitRequest, PostCommentForComparedCommitResponse>()
                            .withOperationName("PostCommentForComparedCommit")
                            .withMarshaller(new PostCommentForComparedCommitRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(postCommentForComparedCommitRequest));
            CompletableFuture<PostCommentForComparedCommitResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Posts a comment on a pull request.
     * </p>
     *
     * @param postCommentForPullRequestRequest
     * @return A Java Future containing the result of the PostCommentForPullRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>RepositoryNotAssociatedWithPullRequestException The repository does not contain any pull requests
     *         with that pull request ID. Use GetPullRequest to verify the correct repository name for the pull request
     *         ID.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>ClientRequestTokenRequiredException A client request token is required. A client request token is an
     *         unique, client-generated idempotency token that, when provided in a request, ensures the request cannot
     *         be repeated with a changed parameter. If a request is received with the same parameters and a token is
     *         included, the request returns information about the initial request that used that token.</li>
     *         <li>InvalidClientRequestTokenException The client request token is not valid.</li>
     *         <li>IdempotencyParameterMismatchException The client request token is not valid. Either the token is not
     *         in a valid format, or the token has been used in a previous request and cannot be reused.</li>
     *         <li>CommentContentRequiredException The comment is empty. You must provide some content for a comment.
     *         The content cannot be null.</li>
     *         <li>CommentContentSizeLimitExceededException The comment is too large. Comments are limited to 10,240
     *         characters.</li>
     *         <li>InvalidFileLocationException The location of the file is not valid. Make sure that you include the
     *         file name and extension.</li>
     *         <li>InvalidRelativeFileVersionEnumException Either the enum is not in a valid format, or the specified
     *         file version enum is not valid in respect to the current file version.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidFilePositionException The position is not valid. Make sure that the line number exists in the
     *         version of the file you want to comment on.</li>
     *         <li>CommitIdRequiredException A commit ID was not specified.</li>
     *         <li>InvalidCommitIdException The specified commit ID is not valid.</li>
     *         <li>BeforeCommitIdAndAfterCommitIdAreSameException The before commit ID and the after commit ID are the
     *         same, which is not valid. The before commit ID and the after commit ID must be different commit IDs.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>CommitDoesNotExistException The specified commit does not exist or no commit was specified, and the
     *         specified repository has no default branch.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>PathDoesNotExistException The specified path does not exist.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.PostCommentForPullRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/PostCommentForPullRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PostCommentForPullRequestResponse> postCommentForPullRequest(
            PostCommentForPullRequestRequest postCommentForPullRequestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postCommentForPullRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postCommentForPullRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostCommentForPullRequest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PostCommentForPullRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PostCommentForPullRequestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PostCommentForPullRequestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PostCommentForPullRequestRequest, PostCommentForPullRequestResponse>()
                            .withOperationName("PostCommentForPullRequest")
                            .withMarshaller(new PostCommentForPullRequestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(postCommentForPullRequestRequest));
            CompletableFuture<PostCommentForPullRequestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Posts a comment in reply to an existing comment on a comparison between commits or a pull request.
     * </p>
     *
     * @param postCommentReplyRequest
     * @return A Java Future containing the result of the PostCommentReply operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientRequestTokenRequiredException A client request token is required. A client request token is an
     *         unique, client-generated idempotency token that, when provided in a request, ensures the request cannot
     *         be repeated with a changed parameter. If a request is received with the same parameters and a token is
     *         included, the request returns information about the initial request that used that token.</li>
     *         <li>InvalidClientRequestTokenException The client request token is not valid.</li>
     *         <li>IdempotencyParameterMismatchException The client request token is not valid. Either the token is not
     *         in a valid format, or the token has been used in a previous request and cannot be reused.</li>
     *         <li>CommentContentRequiredException The comment is empty. You must provide some content for a comment.
     *         The content cannot be null.</li>
     *         <li>CommentContentSizeLimitExceededException The comment is too large. Comments are limited to 10,240
     *         characters.</li>
     *         <li>CommentDoesNotExistException No comment exists with the provided ID. Verify that you have used the
     *         correct ID, and then try again.</li>
     *         <li>CommentIdRequiredException The comment ID is missing or null. A comment ID is required.</li>
     *         <li>InvalidCommentIdException The comment ID is not in a valid format. Make sure that you have provided
     *         the full comment ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.PostCommentReply
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/PostCommentReply" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PostCommentReplyResponse> postCommentReply(PostCommentReplyRequest postCommentReplyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postCommentReplyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postCommentReplyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostCommentReply");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PostCommentReplyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PostCommentReplyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PostCommentReplyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PostCommentReplyRequest, PostCommentReplyResponse>()
                            .withOperationName("PostCommentReply")
                            .withMarshaller(new PostCommentReplyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(postCommentReplyRequest));
            CompletableFuture<PostCommentReplyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates a reaction to a specified comment for the user whose identity is used to make the request. You
     * can only add or update a reaction for yourself. You cannot add, modify, or delete a reaction for another user.
     * </p>
     *
     * @param putCommentReactionRequest
     * @return A Java Future containing the result of the PutCommentReaction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CommentDoesNotExistException No comment exists with the provided ID. Verify that you have used the
     *         correct ID, and then try again.</li>
     *         <li>CommentIdRequiredException The comment ID is missing or null. A comment ID is required.</li>
     *         <li>InvalidCommentIdException The comment ID is not in a valid format. Make sure that you have provided
     *         the full comment ID.</li>
     *         <li>InvalidReactionValueException The value of the reaction is not valid. For more information, see the
     *         <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/welcome.html">CodeCommit User Guide</a>.
     *         </li>
     *         <li>ReactionValueRequiredException A reaction value is required.</li>
     *         <li>ReactionLimitExceededException The number of reactions has been exceeded. Reactions are limited to
     *         one reaction per user for each individual comment ID.</li>
     *         <li>CommentDeletedException This comment has already been deleted. You cannot edit or delete a deleted
     *         comment.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.PutCommentReaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/PutCommentReaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutCommentReactionResponse> putCommentReaction(PutCommentReactionRequest putCommentReactionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putCommentReactionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCommentReactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCommentReaction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutCommentReactionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutCommentReactionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutCommentReactionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutCommentReactionRequest, PutCommentReactionResponse>()
                            .withOperationName("PutCommentReaction")
                            .withMarshaller(new PutCommentReactionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putCommentReactionRequest));
            CompletableFuture<PutCommentReactionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates a file in a branch in an CodeCommit repository, and generates a commit for the addition in the
     * specified branch.
     * </p>
     *
     * @param putFileRequest
     * @return A Java Future containing the result of the PutFile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>ParentCommitIdRequiredException A parent commit ID is required. To view the full commit ID of a
     *         branch in a repository, use <a>GetBranch</a> or a Git command (for example, git pull or git log).</li>
     *         <li>InvalidParentCommitIdException The parent commit ID is not valid. The commit ID cannot be empty, and
     *         must match the head commit ID for the branch of the repository where you want to add or update a file.</li>
     *         <li>ParentCommitDoesNotExistException The parent commit ID is not valid because it does not exist. The
     *         specified parent commit ID does not exist in the specified branch of the repository.</li>
     *         <li>ParentCommitIdOutdatedException The file could not be added because the provided parent commit ID is
     *         not the current tip of the specified branch. To view the full commit ID of the current head of the
     *         branch, use <a>GetBranch</a>.</li>
     *         <li>FileContentRequiredException The file cannot be added because it is empty. Empty files cannot be
     *         added to the repository with this API.</li>
     *         <li>FileContentSizeLimitExceededException The file cannot be added because it is too large. The maximum
     *         file size is 6 MB, and the combined file content change size is 7 MB. Consider making these changes using
     *         a Git client.</li>
     *         <li>FolderContentSizeLimitExceededException The commit cannot be created because at least one of the
     *         overall changes in the commit results in a folder whose contents exceed the limit of 6 MB. Either reduce
     *         the number and size of your changes, or split the changes across multiple folders.</li>
     *         <li>PathRequiredException The folderPath for a location cannot be null.</li>
     *         <li>InvalidPathException The specified path is not valid.</li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>BranchNameIsTagNameException The specified branch name is not valid because it is a tag name. Enter
     *         the name of a branch in the repository. For a list of valid branch names, use <a>ListBranches</a>.</li>
     *         <li>InvalidFileModeException The specified file mode permission is not valid. For a list of valid file
     *         mode permissions, see <a>PutFile</a>.</li>
     *         <li>NameLengthExceededException The user name is not valid because it has exceeded the character limit
     *         for author names.</li>
     *         <li>InvalidEmailException The specified email address either contains one or more characters that are not
     *         allowed, or it exceeds the maximum number of characters allowed for an email address.</li>
     *         <li>CommitMessageLengthExceededException The commit message is too long. Provide a shorter string.</li>
     *         <li>InvalidDeletionParameterException The specified deletion parameter is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SameFileContentException The file was not added or updated because the content of the file is exactly
     *         the same as the content of that file in the repository and branch that you specified.</li>
     *         <li>FileNameConflictsWithDirectoryNameException A file cannot be added to the repository because the
     *         specified file name has the same name as a directory in this repository. Either provide another name for
     *         the file, or add the file in a directory that does not match the file name.</li>
     *         <li>DirectoryNameConflictsWithFileNameException A file cannot be added to the repository because the
     *         specified path name has the same name as a file that already exists in this repository. Either provide a
     *         different name for the file, or specify a different path for the file.</li>
     *         <li>FilePathConflictsWithSubmodulePathException The commit cannot be created because a specified file
     *         path points to a submodule. Verify that the destination files have valid file paths that do not point to
     *         a submodule.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.PutFile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/PutFile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutFileResponse> putFile(PutFileRequest putFileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putFileRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutFileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutFileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutFileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutFileRequest, PutFileResponse>().withOperationName("PutFile")
                            .withMarshaller(new PutFileRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putFileRequest));
            CompletableFuture<PutFileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replaces all triggers for a repository. Used to create or delete triggers.
     * </p>
     *
     * @param putRepositoryTriggersRequest
     *        Represents the input of a put repository triggers operation.
     * @return A Java Future containing the result of the PutRepositoryTriggers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryTriggersListRequiredException The list of triggers for the repository is required, but was
     *         not specified.</li>
     *         <li>MaximumRepositoryTriggersExceededException The number of triggers allowed for the repository was
     *         exceeded.</li>
     *         <li>InvalidRepositoryTriggerNameException The name of the trigger is not valid.</li>
     *         <li>InvalidRepositoryTriggerDestinationArnException The Amazon Resource Name (ARN) for the trigger is not
     *         valid for the specified destination. The most common reason for this error is that the ARN does not meet
     *         the requirements for the service type.</li>
     *         <li>InvalidRepositoryTriggerRegionException The Amazon Web Services Region for the trigger target does
     *         not match the Amazon Web Services Region for the repository. Triggers must be created in the same Amazon
     *         Web Services Region as the target for the trigger.</li>
     *         <li>InvalidRepositoryTriggerCustomDataException The custom data provided for the trigger is not valid.</li>
     *         <li>MaximumBranchesExceededException The number of branches for the trigger was exceeded.</li>
     *         <li>InvalidRepositoryTriggerBranchNameException One or more branch names specified for the trigger is not
     *         valid.</li>
     *         <li>InvalidRepositoryTriggerEventsException One or more events specified for the trigger is not valid.
     *         Check to make sure that all events specified match the requirements for allowed events.</li>
     *         <li>RepositoryTriggerNameRequiredException A name for the trigger is required, but was not specified.</li>
     *         <li>RepositoryTriggerDestinationArnRequiredException A destination ARN for the target service for the
     *         trigger is required, but was not specified.</li>
     *         <li>RepositoryTriggerBranchNameListRequiredException At least one branch name is required, but was not
     *         specified in the trigger configuration.</li>
     *         <li>RepositoryTriggerEventsListRequiredException At least one event for the trigger is required, but was
     *         not specified.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.PutRepositoryTriggers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/PutRepositoryTriggers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRepositoryTriggersResponse> putRepositoryTriggers(
            PutRepositoryTriggersRequest putRepositoryTriggersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRepositoryTriggersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRepositoryTriggersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRepositoryTriggers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRepositoryTriggersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRepositoryTriggersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRepositoryTriggersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRepositoryTriggersRequest, PutRepositoryTriggersResponse>()
                            .withOperationName("PutRepositoryTriggers")
                            .withMarshaller(new PutRepositoryTriggersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRepositoryTriggersRequest));
            CompletableFuture<PutRepositoryTriggersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tags for a resource in CodeCommit. For a list of valid resources in CodeCommit, see <a href=
     * "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     * >CodeCommit Resources and Operations</a> in the <i>CodeCommit User Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>ResourceArnRequiredException A valid Amazon Resource Name (ARN) for an CodeCommit resource is
     *         required. For a list of valid resources in CodeCommit, see <a href=
     *         "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     *         >CodeCommit Resources and Operations</a> in the CodeCommit User Guide.</li>
     *         <li>InvalidResourceArnException The value for the resource ARN is not valid. For more information about
     *         resources in CodeCommit, see <a href=
     *         "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     *         >CodeCommit Resources and Operations</a> in the CodeCommit User Guide.</li>
     *         <li>TagsMapRequiredException A map of tags is required.</li>
     *         <li>InvalidTagsMapException The map of tags is not valid.</li>
     *         <li>TooManyTagsException The maximum number of tags for an CodeCommit resource has been exceeded.</li>
     *         <li>InvalidSystemTagUsageException The specified tag is not valid. Key names cannot be prefixed with
     *         aws:.</li>
     *         <li>TagPolicyException The tag policy is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tests the functionality of repository triggers by sending information to the trigger target. If real data is
     * available in the repository, the test sends data from the last commit. If no data is available, sample data is
     * generated.
     * </p>
     *
     * @param testRepositoryTriggersRequest
     *        Represents the input of a test repository triggers operation.
     * @return A Java Future containing the result of the TestRepositoryTriggers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>RepositoryTriggersListRequiredException The list of triggers for the repository is required, but was
     *         not specified.</li>
     *         <li>MaximumRepositoryTriggersExceededException The number of triggers allowed for the repository was
     *         exceeded.</li>
     *         <li>InvalidRepositoryTriggerNameException The name of the trigger is not valid.</li>
     *         <li>InvalidRepositoryTriggerDestinationArnException The Amazon Resource Name (ARN) for the trigger is not
     *         valid for the specified destination. The most common reason for this error is that the ARN does not meet
     *         the requirements for the service type.</li>
     *         <li>InvalidRepositoryTriggerRegionException The Amazon Web Services Region for the trigger target does
     *         not match the Amazon Web Services Region for the repository. Triggers must be created in the same Amazon
     *         Web Services Region as the target for the trigger.</li>
     *         <li>InvalidRepositoryTriggerCustomDataException The custom data provided for the trigger is not valid.</li>
     *         <li>MaximumBranchesExceededException The number of branches for the trigger was exceeded.</li>
     *         <li>InvalidRepositoryTriggerBranchNameException One or more branch names specified for the trigger is not
     *         valid.</li>
     *         <li>InvalidRepositoryTriggerEventsException One or more events specified for the trigger is not valid.
     *         Check to make sure that all events specified match the requirements for allowed events.</li>
     *         <li>RepositoryTriggerNameRequiredException A name for the trigger is required, but was not specified.</li>
     *         <li>RepositoryTriggerDestinationArnRequiredException A destination ARN for the target service for the
     *         trigger is required, but was not specified.</li>
     *         <li>RepositoryTriggerBranchNameListRequiredException At least one branch name is required, but was not
     *         specified in the trigger configuration.</li>
     *         <li>RepositoryTriggerEventsListRequiredException At least one event for the trigger is required, but was
     *         not specified.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.TestRepositoryTriggers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/TestRepositoryTriggers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TestRepositoryTriggersResponse> testRepositoryTriggers(
            TestRepositoryTriggersRequest testRepositoryTriggersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testRepositoryTriggersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testRepositoryTriggersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestRepositoryTriggers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TestRepositoryTriggersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TestRepositoryTriggersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TestRepositoryTriggersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TestRepositoryTriggersRequest, TestRepositoryTriggersResponse>()
                            .withOperationName("TestRepositoryTriggers")
                            .withMarshaller(new TestRepositoryTriggersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(testRepositoryTriggersRequest));
            CompletableFuture<TestRepositoryTriggersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags for a resource in CodeCommit. For a list of valid resources in CodeCommit, see <a href=
     * "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     * >CodeCommit Resources and Operations</a> in the <i>CodeCommit User Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>ResourceArnRequiredException A valid Amazon Resource Name (ARN) for an CodeCommit resource is
     *         required. For a list of valid resources in CodeCommit, see <a href=
     *         "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     *         >CodeCommit Resources and Operations</a> in the CodeCommit User Guide.</li>
     *         <li>InvalidResourceArnException The value for the resource ARN is not valid. For more information about
     *         resources in CodeCommit, see <a href=
     *         "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats"
     *         >CodeCommit Resources and Operations</a> in the CodeCommit User Guide.</li>
     *         <li>TagKeysListRequiredException A list of tag keys is required. The list cannot be empty or null.</li>
     *         <li>InvalidTagKeysListException The list of tags is not valid.</li>
     *         <li>TooManyTagsException The maximum number of tags for an CodeCommit resource has been exceeded.</li>
     *         <li>InvalidSystemTagUsageException The specified tag is not valid. Key names cannot be prefixed with
     *         aws:.</li>
     *         <li>TagPolicyException The tag policy is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the content of an approval rule template. You can change the number of required approvals, the membership
     * of the approval rule, and whether an approval pool is defined.
     * </p>
     *
     * @param updateApprovalRuleTemplateContentRequest
     * @return A Java Future containing the result of the UpdateApprovalRuleTemplateContent operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>InvalidApprovalRuleTemplateContentException The content of the approval rule template is not valid.</li>
     *         <li>InvalidRuleContentSha256Exception The SHA-256 hash signature for the rule content is not valid.</li>
     *         <li>ApprovalRuleTemplateContentRequiredException The content for the approval rule template is empty. You
     *         must provide some content for an approval rule template. The content cannot be null.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateApprovalRuleTemplateContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateApprovalRuleTemplateContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApprovalRuleTemplateContentResponse> updateApprovalRuleTemplateContent(
            UpdateApprovalRuleTemplateContentRequest updateApprovalRuleTemplateContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApprovalRuleTemplateContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateApprovalRuleTemplateContentRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApprovalRuleTemplateContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApprovalRuleTemplateContentResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateApprovalRuleTemplateContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApprovalRuleTemplateContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApprovalRuleTemplateContentRequest, UpdateApprovalRuleTemplateContentResponse>()
                            .withOperationName("UpdateApprovalRuleTemplateContent")
                            .withMarshaller(new UpdateApprovalRuleTemplateContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApprovalRuleTemplateContentRequest));
            CompletableFuture<UpdateApprovalRuleTemplateContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the description for a specified approval rule template.
     * </p>
     *
     * @param updateApprovalRuleTemplateDescriptionRequest
     * @return A Java Future containing the result of the UpdateApprovalRuleTemplateDescription operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>InvalidApprovalRuleTemplateDescriptionException The description for the approval rule template is not
     *         valid because it exceeds the maximum characters allowed for a description. For more information about
     *         limits in CodeCommit, see <a
     *         href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a> in the
     *         <i>CodeCommit User Guide</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateApprovalRuleTemplateDescription
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateApprovalRuleTemplateDescription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApprovalRuleTemplateDescriptionResponse> updateApprovalRuleTemplateDescription(
            UpdateApprovalRuleTemplateDescriptionRequest updateApprovalRuleTemplateDescriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApprovalRuleTemplateDescriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateApprovalRuleTemplateDescriptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApprovalRuleTemplateDescription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApprovalRuleTemplateDescriptionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateApprovalRuleTemplateDescriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApprovalRuleTemplateDescriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApprovalRuleTemplateDescriptionRequest, UpdateApprovalRuleTemplateDescriptionResponse>()
                            .withOperationName("UpdateApprovalRuleTemplateDescription")
                            .withMarshaller(new UpdateApprovalRuleTemplateDescriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApprovalRuleTemplateDescriptionRequest));
            CompletableFuture<UpdateApprovalRuleTemplateDescriptionResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name of a specified approval rule template.
     * </p>
     *
     * @param updateApprovalRuleTemplateNameRequest
     * @return A Java Future containing the result of the UpdateApprovalRuleTemplateName operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidApprovalRuleTemplateNameException The name of the approval rule template is not valid.
     *         Template names must be between 1 and 100 valid characters in length. For more information about limits in
     *         CodeCommit, see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a>
     *         in the <i>CodeCommit User Guide</i>.</li>
     *         <li>ApprovalRuleTemplateNameRequiredException An approval rule template name is required, but was not
     *         specified.</li>
     *         <li>ApprovalRuleTemplateDoesNotExistException The specified approval rule template does not exist. Verify
     *         that the name is correct and that you are signed in to the Amazon Web Services Region where the template
     *         was created, and then try again.</li>
     *         <li>ApprovalRuleTemplateNameAlreadyExistsException You cannot create an approval rule template with that
     *         name because a template with that name already exists in this Amazon Web Services Region for your Amazon
     *         Web Services account. Approval rule template names must be unique.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateApprovalRuleTemplateName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateApprovalRuleTemplateName"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApprovalRuleTemplateNameResponse> updateApprovalRuleTemplateName(
            UpdateApprovalRuleTemplateNameRequest updateApprovalRuleTemplateNameRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApprovalRuleTemplateNameRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateApprovalRuleTemplateNameRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApprovalRuleTemplateName");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApprovalRuleTemplateNameResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApprovalRuleTemplateNameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApprovalRuleTemplateNameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApprovalRuleTemplateNameRequest, UpdateApprovalRuleTemplateNameResponse>()
                            .withOperationName("UpdateApprovalRuleTemplateName")
                            .withMarshaller(new UpdateApprovalRuleTemplateNameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApprovalRuleTemplateNameRequest));
            CompletableFuture<UpdateApprovalRuleTemplateNameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replaces the contents of a comment.
     * </p>
     *
     * @param updateCommentRequest
     * @return A Java Future containing the result of the UpdateComment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CommentContentRequiredException The comment is empty. You must provide some content for a comment.
     *         The content cannot be null.</li>
     *         <li>CommentContentSizeLimitExceededException The comment is too large. Comments are limited to 10,240
     *         characters.</li>
     *         <li>CommentDoesNotExistException No comment exists with the provided ID. Verify that you have used the
     *         correct ID, and then try again.</li>
     *         <li>CommentIdRequiredException The comment ID is missing or null. A comment ID is required.</li>
     *         <li>InvalidCommentIdException The comment ID is not in a valid format. Make sure that you have provided
     *         the full comment ID.</li>
     *         <li>CommentNotCreatedByCallerException You cannot modify or delete this comment. Only comment authors can
     *         modify or delete their comments.</li>
     *         <li>CommentDeletedException This comment has already been deleted. You cannot edit or delete a deleted
     *         comment.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateComment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateComment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCommentResponse> updateComment(UpdateCommentRequest updateCommentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCommentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCommentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateComment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCommentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateCommentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCommentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCommentRequest, UpdateCommentResponse>()
                            .withOperationName("UpdateComment")
                            .withMarshaller(new UpdateCommentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateCommentRequest));
            CompletableFuture<UpdateCommentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets or changes the default branch name for the specified repository.
     * </p>
     * <note>
     * <p>
     * If you use this operation to change the default branch name to the current default branch name, a success message
     * is returned even though the default branch did not change.
     * </p>
     * </note>
     *
     * @param updateDefaultBranchRequest
     *        Represents the input of an update default branch operation.
     * @return A Java Future containing the result of the UpdateDefaultBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>BranchNameRequiredException A branch name is required, but was not specified.</li>
     *         <li>InvalidBranchNameException The specified reference name is not valid.</li>
     *         <li>BranchDoesNotExistException The specified branch does not exist.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateDefaultBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateDefaultBranch"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDefaultBranchResponse> updateDefaultBranch(
            UpdateDefaultBranchRequest updateDefaultBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDefaultBranchRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDefaultBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDefaultBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDefaultBranchResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDefaultBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDefaultBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDefaultBranchRequest, UpdateDefaultBranchResponse>()
                            .withOperationName("UpdateDefaultBranch")
                            .withMarshaller(new UpdateDefaultBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDefaultBranchRequest));
            CompletableFuture<UpdateDefaultBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the structure of an approval rule created specifically for a pull request. For example, you can change
     * the number of required approvers and the approval pool for approvers.
     * </p>
     *
     * @param updatePullRequestApprovalRuleContentRequest
     * @return A Java Future containing the result of the UpdatePullRequestApprovalRuleContent operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>ApprovalRuleNameRequiredException An approval rule name is required, but was not specified.</li>
     *         <li>InvalidApprovalRuleNameException The name for the approval rule is not valid.</li>
     *         <li>ApprovalRuleDoesNotExistException The specified approval rule does not exist.</li>
     *         <li>InvalidRuleContentSha256Exception The SHA-256 hash signature for the rule content is not valid.</li>
     *         <li>ApprovalRuleContentRequiredException The content for the approval rule is empty. You must provide
     *         some content for an approval rule. The content cannot be null.</li>
     *         <li>InvalidApprovalRuleContentException The content for the approval rule is not valid.</li>
     *         <li>CannotModifyApprovalRuleFromTemplateException The approval rule cannot be modified for the pull
     *         request because it was created by an approval rule template and applied to the pull request
     *         automatically.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdatePullRequestApprovalRuleContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdatePullRequestApprovalRuleContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePullRequestApprovalRuleContentResponse> updatePullRequestApprovalRuleContent(
            UpdatePullRequestApprovalRuleContentRequest updatePullRequestApprovalRuleContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullRequestApprovalRuleContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updatePullRequestApprovalRuleContentRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullRequestApprovalRuleContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePullRequestApprovalRuleContentResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdatePullRequestApprovalRuleContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePullRequestApprovalRuleContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePullRequestApprovalRuleContentRequest, UpdatePullRequestApprovalRuleContentResponse>()
                            .withOperationName("UpdatePullRequestApprovalRuleContent")
                            .withMarshaller(new UpdatePullRequestApprovalRuleContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePullRequestApprovalRuleContentRequest));
            CompletableFuture<UpdatePullRequestApprovalRuleContentResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the state of a user's approval on a pull request. The user is derived from the signed-in account when the
     * request is made.
     * </p>
     *
     * @param updatePullRequestApprovalStateRequest
     * @return A Java Future containing the result of the UpdatePullRequestApprovalState operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidRevisionIdException The revision ID is not valid. Use GetPullRequest to determine the value.</li>
     *         <li>RevisionIdRequiredException A revision ID is required, but was not provided.</li>
     *         <li>InvalidApprovalStateException The state for the approval is not valid. Valid values include APPROVE
     *         and REVOKE.</li>
     *         <li>ApprovalStateRequiredException An approval state is required, but was not specified.</li>
     *         <li>PullRequestCannotBeApprovedByAuthorException The approval cannot be applied because the user
     *         approving the pull request matches the user who created the pull request. You cannot approve a pull
     *         request that you created.</li>
     *         <li>RevisionNotCurrentException The revision ID provided in the request does not match the current
     *         revision ID. Use GetPullRequest to retrieve the current revision ID.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>MaximumNumberOfApprovalsExceededException The number of approvals required for the approval rule
     *         exceeds the maximum number allowed.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdatePullRequestApprovalState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdatePullRequestApprovalState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePullRequestApprovalStateResponse> updatePullRequestApprovalState(
            UpdatePullRequestApprovalStateRequest updatePullRequestApprovalStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullRequestApprovalStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updatePullRequestApprovalStateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullRequestApprovalState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePullRequestApprovalStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePullRequestApprovalStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePullRequestApprovalStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePullRequestApprovalStateRequest, UpdatePullRequestApprovalStateResponse>()
                            .withOperationName("UpdatePullRequestApprovalState")
                            .withMarshaller(new UpdatePullRequestApprovalStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePullRequestApprovalStateRequest));
            CompletableFuture<UpdatePullRequestApprovalStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replaces the contents of the description of a pull request.
     * </p>
     *
     * @param updatePullRequestDescriptionRequest
     * @return A Java Future containing the result of the UpdatePullRequestDescription operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidDescriptionException The pull request description is not valid. Descriptions cannot be more
     *         than 1,000 characters.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdatePullRequestDescription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdatePullRequestDescription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePullRequestDescriptionResponse> updatePullRequestDescription(
            UpdatePullRequestDescriptionRequest updatePullRequestDescriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullRequestDescriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePullRequestDescriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullRequestDescription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePullRequestDescriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePullRequestDescriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePullRequestDescriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePullRequestDescriptionRequest, UpdatePullRequestDescriptionResponse>()
                            .withOperationName("UpdatePullRequestDescription")
                            .withMarshaller(new UpdatePullRequestDescriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePullRequestDescriptionRequest));
            CompletableFuture<UpdatePullRequestDescriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status of a pull request.
     * </p>
     *
     * @param updatePullRequestStatusRequest
     * @return A Java Future containing the result of the UpdatePullRequestStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>InvalidPullRequestStatusUpdateException The pull request status update is not valid. The only valid
     *         update is from <code>OPEN</code> to <code>CLOSED</code>.</li>
     *         <li>InvalidPullRequestStatusException The pull request status is not valid. The only valid values are
     *         <code>OPEN</code> and <code>CLOSED</code>.</li>
     *         <li>PullRequestStatusRequiredException A pull request status is required, but none was provided.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdatePullRequestStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdatePullRequestStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePullRequestStatusResponse> updatePullRequestStatus(
            UpdatePullRequestStatusRequest updatePullRequestStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullRequestStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePullRequestStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullRequestStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePullRequestStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePullRequestStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePullRequestStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePullRequestStatusRequest, UpdatePullRequestStatusResponse>()
                            .withOperationName("UpdatePullRequestStatus")
                            .withMarshaller(new UpdatePullRequestStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePullRequestStatusRequest));
            CompletableFuture<UpdatePullRequestStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replaces the title of a pull request.
     * </p>
     *
     * @param updatePullRequestTitleRequest
     * @return A Java Future containing the result of the UpdatePullRequestTitle operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PullRequestDoesNotExistException The pull request ID could not be found. Make sure that you have
     *         specified the correct repository name and pull request ID, and then try again.</li>
     *         <li>InvalidPullRequestIdException The pull request ID is not valid. Make sure that you have provided the
     *         full ID and that the pull request is in the specified repository, and then try again.</li>
     *         <li>PullRequestIdRequiredException A pull request ID is required, but none was provided.</li>
     *         <li>TitleRequiredException A pull request title is required. It cannot be empty or null.</li>
     *         <li>InvalidTitleException The title of the pull request is not valid. Pull request titles cannot exceed
     *         100 characters in length.</li>
     *         <li>PullRequestAlreadyClosedException The pull request status cannot be updated because it is already
     *         closed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdatePullRequestTitle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdatePullRequestTitle"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePullRequestTitleResponse> updatePullRequestTitle(
            UpdatePullRequestTitleRequest updatePullRequestTitleRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePullRequestTitleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePullRequestTitleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePullRequestTitle");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePullRequestTitleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePullRequestTitleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePullRequestTitleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePullRequestTitleRequest, UpdatePullRequestTitleResponse>()
                            .withOperationName("UpdatePullRequestTitle")
                            .withMarshaller(new UpdatePullRequestTitleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePullRequestTitleRequest));
            CompletableFuture<UpdatePullRequestTitleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets or changes the comment or description for a repository.
     * </p>
     * <note>
     * <p>
     * The description field for a repository accepts all HTML characters and all valid Unicode characters. Applications
     * that do not HTML-encode the description and display it in a webpage can expose users to potentially malicious
     * code. Make sure that you HTML-encode the description field in any application that uses this API to display the
     * repository description on a webpage.
     * </p>
     * </note>
     *
     * @param updateRepositoryDescriptionRequest
     *        Represents the input of an update repository description operation.
     * @return A Java Future containing the result of the UpdateRepositoryDescription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>InvalidRepositoryDescriptionException The specified repository description is not valid.</li>
     *         <li>EncryptionIntegrityChecksFailedException An encryption integrity check failed.</li>
     *         <li>EncryptionKeyAccessDeniedException An encryption key could not be accessed.</li>
     *         <li>EncryptionKeyDisabledException The encryption key is disabled.</li>
     *         <li>EncryptionKeyNotFoundException No encryption key was found.</li>
     *         <li>EncryptionKeyUnavailableException The encryption key is not available.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateRepositoryDescription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateRepositoryDescription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRepositoryDescriptionResponse> updateRepositoryDescription(
            UpdateRepositoryDescriptionRequest updateRepositoryDescriptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRepositoryDescriptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRepositoryDescriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepositoryDescription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRepositoryDescriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRepositoryDescriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRepositoryDescriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRepositoryDescriptionRequest, UpdateRepositoryDescriptionResponse>()
                            .withOperationName("UpdateRepositoryDescription")
                            .withMarshaller(new UpdateRepositoryDescriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRepositoryDescriptionRequest));
            CompletableFuture<UpdateRepositoryDescriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Renames a repository. The repository name must be unique across the calling Amazon Web Services account.
     * Repository names are limited to 100 alphanumeric, dash, and underscore characters, and cannot include certain
     * characters. The suffix .git is prohibited. For more information about the limits on repository names, see <a
     * href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Quotas</a> in the CodeCommit User
     * Guide.
     * </p>
     *
     * @param updateRepositoryNameRequest
     *        Represents the input of an update repository description operation.
     * @return A Java Future containing the result of the UpdateRepositoryName operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RepositoryDoesNotExistException The specified repository does not exist.</li>
     *         <li>RepositoryNameExistsException The specified repository name already exists.</li>
     *         <li>RepositoryNameRequiredException A repository name is required, but was not specified.</li>
     *         <li>InvalidRepositoryNameException A specified repository name is not valid.</p> <note>
     *         <p>
     *         This exception occurs only when a specified repository name is not valid. Other exceptions occur when a
     *         required repository parameter is missing, or when a specified repository does not exist.
     *         </p>
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeCommitException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeCommitAsyncClient.UpdateRepositoryName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codecommit-2015-04-13/UpdateRepositoryName"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRepositoryNameResponse> updateRepositoryName(
            UpdateRepositoryNameRequest updateRepositoryNameRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRepositoryNameRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRepositoryNameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeCommit");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepositoryName");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRepositoryNameResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRepositoryNameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRepositoryNameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRepositoryNameRequest, UpdateRepositoryNameResponse>()
                            .withOperationName("UpdateRepositoryName")
                            .withMarshaller(new UpdateRepositoryNameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRepositoryNameRequest));
            CompletableFuture<UpdateRepositoryNameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CodeCommitServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeCommitException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileContentRequiredException")
                                .exceptionBuilderSupplier(FileContentRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalRuleTemplateDescriptionException")
                                .exceptionBuilderSupplier(InvalidApprovalRuleTemplateDescriptionException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleTemplateDoesNotExistException")
                                .exceptionBuilderSupplier(ApprovalRuleTemplateDoesNotExistException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommentDeletedException")
                                .exceptionBuilderSupplier(CommentDeletedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumItemsToCompareExceededException")
                                .exceptionBuilderSupplier(MaximumItemsToCompareExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidBlobIdException")
                                .exceptionBuilderSupplier(InvalidBlobIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryDescriptionException")
                                .exceptionBuilderSupplier(InvalidRepositoryDescriptionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryTriggerDestinationArnException")
                                .exceptionBuilderSupplier(InvalidRepositoryTriggerDestinationArnException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommentDoesNotExistException")
                                .exceptionBuilderSupplier(CommentDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidConflictDetailLevelException")
                                .exceptionBuilderSupplier(InvalidConflictDetailLevelException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReactionLimitExceededException")
                                .exceptionBuilderSupplier(ReactionLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BeforeCommitIdAndAfterCommitIdAreSameException")
                                .exceptionBuilderSupplier(BeforeCommitIdAndAfterCommitIdAreSameException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CannotModifyApprovalRuleFromTemplateException")
                                .exceptionBuilderSupplier(CannotModifyApprovalRuleFromTemplateException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RevisionIdRequiredException")
                                .exceptionBuilderSupplier(RevisionIdRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullRequestDoesNotExistException")
                                .exceptionBuilderSupplier(PullRequestDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReferenceTypeNotSupportedException")
                                .exceptionBuilderSupplier(ReferenceTypeNotSupportedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilePathConflictsWithSubmodulePathException")
                                .exceptionBuilderSupplier(FilePathConflictsWithSubmodulePathException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryTriggerEventsListRequiredException")
                                .exceptionBuilderSupplier(RepositoryTriggerEventsListRequiredException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileModeRequiredException")
                                .exceptionBuilderSupplier(FileModeRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitDoesNotExistException")
                                .exceptionBuilderSupplier(CommitDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceArnRequiredException")
                                .exceptionBuilderSupplier(ResourceArnRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidReactionUserArnException")
                                .exceptionBuilderSupplier(InvalidReactionUserArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryTriggerEventsException")
                                .exceptionBuilderSupplier(InvalidRepositoryTriggerEventsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNotAssociatedWithPullRequestException")
                                .exceptionBuilderSupplier(RepositoryNotAssociatedWithPullRequestException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFileLocationException")
                                .exceptionBuilderSupplier(InvalidFileLocationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumBranchesExceededException")
                                .exceptionBuilderSupplier(MaximumBranchesExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRelativeFileVersionEnumException")
                                .exceptionBuilderSupplier(InvalidRelativeFileVersionEnumException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidReactionValueException")
                                .exceptionBuilderSupplier(InvalidReactionValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagKeysListRequiredException")
                                .exceptionBuilderSupplier(TagKeysListRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientRequestTokenException")
                                .exceptionBuilderSupplier(InvalidClientRequestTokenException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FolderContentSizeLimitExceededException")
                                .exceptionBuilderSupplier(FolderContentSizeLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNameRequiredException")
                                .exceptionBuilderSupplier(RepositoryNameRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidReferenceNameException")
                                .exceptionBuilderSupplier(InvalidReferenceNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommentContentRequiredException")
                                .exceptionBuilderSupplier(CommentContentRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagKeysListException")
                                .exceptionBuilderSupplier(InvalidTagKeysListException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsMapRequiredException")
                                .exceptionBuilderSupplier(TagsMapRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PathRequiredException")
                                .exceptionBuilderSupplier(PathRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSystemTagUsageException")
                                .exceptionBuilderSupplier(InvalidSystemTagUsageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitIdDoesNotExistException")
                                .exceptionBuilderSupplier(CommitIdDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParentCommitIdException")
                                .exceptionBuilderSupplier(InvalidParentCommitIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReferenceNameRequiredException")
                                .exceptionBuilderSupplier(ReferenceNameRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxConflictFilesException")
                                .exceptionBuilderSupplier(InvalidMaxConflictFilesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumRuleTemplatesAssociatedWithRepositoryException")
                                .exceptionBuilderSupplier(MaximumRuleTemplatesAssociatedWithRepositoryException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleTemplateContentRequiredException")
                                .exceptionBuilderSupplier(ApprovalRuleTemplateContentRequiredException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCommentIdException")
                                .exceptionBuilderSupplier(InvalidCommentIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalStateException")
                                .exceptionBuilderSupplier(InvalidApprovalStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCommitException")
                                .exceptionBuilderSupplier(InvalidCommitException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleTemplateNameRequiredException")
                                .exceptionBuilderSupplier(ApprovalRuleTemplateNameRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoChangeException")
                                .exceptionBuilderSupplier(NoChangeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleDoesNotExistException")
                                .exceptionBuilderSupplier(ApprovalRuleDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumFileContentToLoadExceededException")
                                .exceptionBuilderSupplier(MaximumFileContentToLoadExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MultipleRepositoriesInPullRequestException")
                                .exceptionBuilderSupplier(MultipleRepositoriesInPullRequestException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BlobIdDoesNotExistException")
                                .exceptionBuilderSupplier(BlobIdDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MergeOptionRequiredException")
                                .exceptionBuilderSupplier(MergeOptionRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPullRequestIdException")
                                .exceptionBuilderSupplier(InvalidPullRequestIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleTemplateInUseException")
                                .exceptionBuilderSupplier(ApprovalRuleTemplateInUseException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileDoesNotExistException")
                                .exceptionBuilderSupplier(FileDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNameExistsException")
                                .exceptionBuilderSupplier(RepositoryNameExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PathDoesNotExistException")
                                .exceptionBuilderSupplier(PathDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BranchNameExistsException")
                                .exceptionBuilderSupplier(BranchNameExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EncryptionIntegrityChecksFailedException")
                                .exceptionBuilderSupplier(EncryptionIntegrityChecksFailedException::builder).httpStatusCode(500)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCommitIdException")
                                .exceptionBuilderSupplier(InvalidCommitIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOrderException")
                                .exceptionBuilderSupplier(InvalidOrderException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullRequestAlreadyClosedException")
                                .exceptionBuilderSupplier(PullRequestAlreadyClosedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BranchNameRequiredException")
                                .exceptionBuilderSupplier(BranchNameRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryTriggerNameException")
                                .exceptionBuilderSupplier(InvalidRepositoryTriggerNameException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMergeOptionException")
                                .exceptionBuilderSupplier(InvalidMergeOptionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParentCommitIdOutdatedException")
                                .exceptionBuilderSupplier(ParentCommitIdOutdatedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitIdsListRequiredException")
                                .exceptionBuilderSupplier(CommitIdsListRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidContinuationTokenException")
                                .exceptionBuilderSupplier(InvalidContinuationTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumFileEntriesExceededException")
                                .exceptionBuilderSupplier(MaximumFileEntriesExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullRequestApprovalRulesNotSatisfiedException")
                                .exceptionBuilderSupplier(PullRequestApprovalRulesNotSatisfiedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientRequestTokenRequiredException")
                                .exceptionBuilderSupplier(ClientRequestTokenRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NameLengthExceededException")
                                .exceptionBuilderSupplier(NameLengthExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EncryptionKeyDisabledException")
                                .exceptionBuilderSupplier(EncryptionKeyDisabledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MultipleConflictResolutionEntriesException")
                                .exceptionBuilderSupplier(MultipleConflictResolutionEntriesException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileContentSizeLimitExceededException")
                                .exceptionBuilderSupplier(FileContentSizeLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ManualMergeRequiredException")
                                .exceptionBuilderSupplier(ManualMergeRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalRuleContentException")
                                .exceptionBuilderSupplier(InvalidApprovalRuleContentException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryTriggerBranchNameException")
                                .exceptionBuilderSupplier(InvalidRepositoryTriggerBranchNameException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SameFileContentException")
                                .exceptionBuilderSupplier(SameFileContentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTargetException")
                                .exceptionBuilderSupplier(InvalidTargetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryTriggerRegionException")
                                .exceptionBuilderSupplier(InvalidRepositoryTriggerRegionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommentContentSizeLimitExceededException")
                                .exceptionBuilderSupplier(CommentContentSizeLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParentCommitDoesNotExistException")
                                .exceptionBuilderSupplier(ParentCommitDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileEntryRequiredException")
                                .exceptionBuilderSupplier(FileEntryRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidActorArnException")
                                .exceptionBuilderSupplier(InvalidActorArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitIdsLimitExceededException")
                                .exceptionBuilderSupplier(CommitIdsLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDestinationCommitSpecifierException")
                                .exceptionBuilderSupplier(InvalidDestinationCommitSpecifierException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryTriggerBranchNameListRequiredException")
                                .exceptionBuilderSupplier(RepositoryTriggerBranchNameListRequiredException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalRuleNameException")
                                .exceptionBuilderSupplier(InvalidApprovalRuleNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryNameConflictsWithFileNameException")
                                .exceptionBuilderSupplier(DirectoryNameConflictsWithFileNameException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SourceAndDestinationAreSameException")
                                .exceptionBuilderSupplier(SourceAndDestinationAreSameException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTitleException")
                                .exceptionBuilderSupplier(InvalidTitleException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EncryptionKeyUnavailableException")
                                .exceptionBuilderSupplier(EncryptionKeyUnavailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryNameException")
                                .exceptionBuilderSupplier(InvalidRepositoryNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidBranchNameException")
                                .exceptionBuilderSupplier(InvalidBranchNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TargetsRequiredException")
                                .exceptionBuilderSupplier(TargetsRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryLimitExceededException")
                                .exceptionBuilderSupplier(RepositoryLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BranchNameIsTagNameException")
                                .exceptionBuilderSupplier(BranchNameIsTagNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitIdRequiredException")
                                .exceptionBuilderSupplier(CommitIdRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TitleRequiredException")
                                .exceptionBuilderSupplier(TitleRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTargetsException")
                                .exceptionBuilderSupplier(InvalidTargetsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumRepositoryTriggersExceededException")
                                .exceptionBuilderSupplier(MaximumRepositoryTriggersExceededException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BlobIdRequiredException")
                                .exceptionBuilderSupplier(BlobIdRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidReplacementTypeException")
                                .exceptionBuilderSupplier(InvalidReplacementTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryNamesRequiredException")
                                .exceptionBuilderSupplier(RepositoryNamesRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorDoesNotExistException")
                                .exceptionBuilderSupplier(AuthorDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFilePositionException")
                                .exceptionBuilderSupplier(InvalidFilePositionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileNameConflictsWithDirectoryNameException")
                                .exceptionBuilderSupplier(FileNameConflictsWithDirectoryNameException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceArnException")
                                .exceptionBuilderSupplier(InvalidResourceArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EncryptionKeyAccessDeniedException")
                                .exceptionBuilderSupplier(EncryptionKeyAccessDeniedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullRequestCannotBeApprovedByAuthorException")
                                .exceptionBuilderSupplier(PullRequestCannotBeApprovedByAuthorException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BranchDoesNotExistException")
                                .exceptionBuilderSupplier(BranchDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReactionValueRequiredException")
                                .exceptionBuilderSupplier(ReactionValueRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleTemplateNameAlreadyExistsException")
                                .exceptionBuilderSupplier(ApprovalRuleTemplateNameAlreadyExistsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRepositoryTriggerCustomDataException")
                                .exceptionBuilderSupplier(InvalidRepositoryTriggerCustomDataException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleContentRequiredException")
                                .exceptionBuilderSupplier(ApprovalRuleContentRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumRepositoryNamesExceededException")
                                .exceptionBuilderSupplier(MaximumRepositoryNamesExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryDoesNotExistException")
                                .exceptionBuilderSupplier(RepositoryDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRevisionIdException")
                                .exceptionBuilderSupplier(InvalidRevisionIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumOpenPullRequestsExceededException")
                                .exceptionBuilderSupplier(MaximumOpenPullRequestsExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DefaultBranchCannotBeDeletedException")
                                .exceptionBuilderSupplier(DefaultBranchCannotBeDeletedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EncryptionKeyNotFoundException")
                                .exceptionBuilderSupplier(EncryptionKeyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPathException")
                                .exceptionBuilderSupplier(InvalidPathException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidConflictResolutionException")
                                .exceptionBuilderSupplier(InvalidConflictResolutionException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalRuleTemplateContentException")
                                .exceptionBuilderSupplier(InvalidApprovalRuleTemplateContentException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryTriggerDestinationArnRequiredException")
                                .exceptionBuilderSupplier(RepositoryTriggerDestinationArnRequiredException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidConflictResolutionStrategyException")
                                .exceptionBuilderSupplier(InvalidConflictResolutionStrategyException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidAuthorArnException")
                                .exceptionBuilderSupplier(InvalidAuthorArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTargetBranchException")
                                .exceptionBuilderSupplier(InvalidTargetBranchException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplacementTypeRequiredException")
                                .exceptionBuilderSupplier(ReplacementTypeRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryTriggerNameRequiredException")
                                .exceptionBuilderSupplier(RepositoryTriggerNameRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalStateRequiredException")
                                .exceptionBuilderSupplier(ApprovalStateRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPullRequestStatusException")
                                .exceptionBuilderSupplier(InvalidPullRequestStatusException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OverrideAlreadySetException")
                                .exceptionBuilderSupplier(OverrideAlreadySetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NumberOfRulesExceededException")
                                .exceptionBuilderSupplier(NumberOfRulesExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOverrideStatusException")
                                .exceptionBuilderSupplier(InvalidOverrideStatusException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleNameRequiredException")
                                .exceptionBuilderSupplier(ApprovalRuleNameRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPullRequestStatusUpdateException")
                                .exceptionBuilderSupplier(InvalidPullRequestStatusUpdateException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParentCommitIdRequiredException")
                                .exceptionBuilderSupplier(ParentCommitIdRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPullRequestEventTypeException")
                                .exceptionBuilderSupplier(InvalidPullRequestEventTypeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ActorDoesNotExistException")
                                .exceptionBuilderSupplier(ActorDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotencyParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotencyParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NumberOfRuleTemplatesExceededException")
                                .exceptionBuilderSupplier(NumberOfRuleTemplatesExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDescriptionException")
                                .exceptionBuilderSupplier(InvalidDescriptionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FolderDoesNotExistException")
                                .exceptionBuilderSupplier(FolderDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullRequestIdRequiredException")
                                .exceptionBuilderSupplier(PullRequestIdRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TipOfSourceReferenceIsDifferentException")
                                .exceptionBuilderSupplier(TipOfSourceReferenceIsDifferentException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RepositoryTriggersListRequiredException")
                                .exceptionBuilderSupplier(RepositoryTriggersListRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidReplacementContentException")
                                .exceptionBuilderSupplier(InvalidReplacementContentException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumNumberOfApprovalsExceededException")
                                .exceptionBuilderSupplier(MaximumNumberOfApprovalsExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitMessageLengthExceededException")
                                .exceptionBuilderSupplier(CommitMessageLengthExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxMergeHunksException")
                                .exceptionBuilderSupplier(InvalidMaxMergeHunksException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEmailException")
                                .exceptionBuilderSupplier(InvalidEmailException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDeletionParameterException")
                                .exceptionBuilderSupplier(InvalidDeletionParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRuleContentSha256Exception")
                                .exceptionBuilderSupplier(InvalidRuleContentSha256Exception::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SourceFileOrContentRequiredException")
                                .exceptionBuilderSupplier(SourceFileOrContentRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommentIdRequiredException")
                                .exceptionBuilderSupplier(CommentIdRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommitRequiredException")
                                .exceptionBuilderSupplier(CommitRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RevisionNotCurrentException")
                                .exceptionBuilderSupplier(RevisionNotCurrentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumConflictResolutionEntriesExceededException")
                                .exceptionBuilderSupplier(MaximumConflictResolutionEntriesExceededException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CommentNotCreatedByCallerException")
                                .exceptionBuilderSupplier(CommentNotCreatedByCallerException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TargetRequiredException")
                                .exceptionBuilderSupplier(TargetRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagsMapException")
                                .exceptionBuilderSupplier(InvalidTagsMapException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RestrictedSourceFileException")
                                .exceptionBuilderSupplier(RestrictedSourceFileException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileTooLargeException")
                                .exceptionBuilderSupplier(FileTooLargeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PutFileEntryConflictException")
                                .exceptionBuilderSupplier(PutFileEntryConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FileContentAndSourceFileSpecifiedException")
                                .exceptionBuilderSupplier(FileContentAndSourceFileSpecifiedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PullRequestStatusRequiredException")
                                .exceptionBuilderSupplier(PullRequestStatusRequiredException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentReferenceUpdateException")
                                .exceptionBuilderSupplier(ConcurrentReferenceUpdateException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SamePathRequestException")
                                .exceptionBuilderSupplier(SamePathRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TipsDivergenceExceededException")
                                .exceptionBuilderSupplier(TipsDivergenceExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSortByException")
                                .exceptionBuilderSupplier(InvalidSortByException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CannotDeleteApprovalRuleFromTemplateException")
                                .exceptionBuilderSupplier(CannotDeleteApprovalRuleFromTemplateException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReferenceDoesNotExistException")
                                .exceptionBuilderSupplier(ReferenceDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OverrideStatusRequiredException")
                                .exceptionBuilderSupplier(OverrideStatusRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ApprovalRuleNameAlreadyExistsException")
                                .exceptionBuilderSupplier(ApprovalRuleNameAlreadyExistsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagPolicyException")
                                .exceptionBuilderSupplier(TagPolicyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFileModeException")
                                .exceptionBuilderSupplier(InvalidFileModeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidApprovalRuleTemplateNameException")
                                .exceptionBuilderSupplier(InvalidApprovalRuleTemplateNameException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSourceCommitSpecifierException")
                                .exceptionBuilderSupplier(InvalidSourceCommitSpecifierException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplacementContentRequiredException")
                                .exceptionBuilderSupplier(ReplacementContentRequiredException::builder).httpStatusCode(400)
                                .build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        CodeCommitServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = CodeCommitServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
