/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information about a specific commit.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Commit implements SdkPojo, Serializable, ToCopyableBuilder<Commit.Builder, Commit> {
    private static final SdkField<String> COMMIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("commitId").getter(getter(Commit::commitId)).setter(setter(Builder::commitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("commitId").build()).build();

    private static final SdkField<String> TREE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("treeId")
            .getter(getter(Commit::treeId)).setter(setter(Builder::treeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("treeId").build()).build();

    private static final SdkField<List<String>> PARENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("parents")
            .getter(getter(Commit::parents))
            .setter(setter(Builder::parents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(Commit::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<UserInfo> AUTHOR_FIELD = SdkField.<UserInfo> builder(MarshallingType.SDK_POJO)
            .memberName("author").getter(getter(Commit::author)).setter(setter(Builder::author)).constructor(UserInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("author").build()).build();

    private static final SdkField<UserInfo> COMMITTER_FIELD = SdkField.<UserInfo> builder(MarshallingType.SDK_POJO)
            .memberName("committer").getter(getter(Commit::committer)).setter(setter(Builder::committer))
            .constructor(UserInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("committer").build()).build();

    private static final SdkField<String> ADDITIONAL_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("additionalData").getter(getter(Commit::additionalData)).setter(setter(Builder::additionalData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("additionalData").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMIT_ID_FIELD,
            TREE_ID_FIELD, PARENTS_FIELD, MESSAGE_FIELD, AUTHOR_FIELD, COMMITTER_FIELD, ADDITIONAL_DATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String commitId;

    private final String treeId;

    private final List<String> parents;

    private final String message;

    private final UserInfo author;

    private final UserInfo committer;

    private final String additionalData;

    private Commit(BuilderImpl builder) {
        this.commitId = builder.commitId;
        this.treeId = builder.treeId;
        this.parents = builder.parents;
        this.message = builder.message;
        this.author = builder.author;
        this.committer = builder.committer;
        this.additionalData = builder.additionalData;
    }

    /**
     * <p>
     * The full SHA ID of the specified commit.
     * </p>
     * 
     * @return The full SHA ID of the specified commit.
     */
    public final String commitId() {
        return commitId;
    }

    /**
     * <p>
     * Tree information for the specified commit.
     * </p>
     * 
     * @return Tree information for the specified commit.
     */
    public final String treeId() {
        return treeId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parents property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParents() {
        return parents != null && !(parents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of parent commits for the specified commit. Each parent commit ID is the full commit ID.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParents} method.
     * </p>
     * 
     * @return A list of parent commits for the specified commit. Each parent commit ID is the full commit ID.
     */
    public final List<String> parents() {
        return parents;
    }

    /**
     * <p>
     * The commit message associated with the specified commit.
     * </p>
     * 
     * @return The commit message associated with the specified commit.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * Information about the author of the specified commit. Information includes the date in timestamp format with GMT
     * offset, the name of the author, and the email address for the author, as configured in Git.
     * </p>
     * 
     * @return Information about the author of the specified commit. Information includes the date in timestamp format
     *         with GMT offset, the name of the author, and the email address for the author, as configured in Git.
     */
    public final UserInfo author() {
        return author;
    }

    /**
     * <p>
     * Information about the person who committed the specified commit, also known as the committer. Information
     * includes the date in timestamp format with GMT offset, the name of the committer, and the email address for the
     * committer, as configured in Git.
     * </p>
     * <p>
     * For more information about the difference between an author and a committer in Git, see <a
     * href="http://git-scm.com/book/ch2-3.html">Viewing the Commit History</a> in Pro Git by Scott Chacon and Ben
     * Straub.
     * </p>
     * 
     * @return Information about the person who committed the specified commit, also known as the committer. Information
     *         includes the date in timestamp format with GMT offset, the name of the committer, and the email address
     *         for the committer, as configured in Git.</p>
     *         <p>
     *         For more information about the difference between an author and a committer in Git, see <a
     *         href="http://git-scm.com/book/ch2-3.html">Viewing the Commit History</a> in Pro Git by Scott Chacon and
     *         Ben Straub.
     */
    public final UserInfo committer() {
        return committer;
    }

    /**
     * <p>
     * Any other data associated with the specified commit.
     * </p>
     * 
     * @return Any other data associated with the specified commit.
     */
    public final String additionalData() {
        return additionalData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(commitId());
        hashCode = 31 * hashCode + Objects.hashCode(treeId());
        hashCode = 31 * hashCode + Objects.hashCode(hasParents() ? parents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(author());
        hashCode = 31 * hashCode + Objects.hashCode(committer());
        hashCode = 31 * hashCode + Objects.hashCode(additionalData());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Commit)) {
            return false;
        }
        Commit other = (Commit) obj;
        return Objects.equals(commitId(), other.commitId()) && Objects.equals(treeId(), other.treeId())
                && hasParents() == other.hasParents() && Objects.equals(parents(), other.parents())
                && Objects.equals(message(), other.message()) && Objects.equals(author(), other.author())
                && Objects.equals(committer(), other.committer()) && Objects.equals(additionalData(), other.additionalData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Commit").add("CommitId", commitId()).add("TreeId", treeId())
                .add("Parents", hasParents() ? parents() : null).add("Message", message()).add("Author", author())
                .add("Committer", committer()).add("AdditionalData", additionalData()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "commitId":
            return Optional.ofNullable(clazz.cast(commitId()));
        case "treeId":
            return Optional.ofNullable(clazz.cast(treeId()));
        case "parents":
            return Optional.ofNullable(clazz.cast(parents()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "author":
            return Optional.ofNullable(clazz.cast(author()));
        case "committer":
            return Optional.ofNullable(clazz.cast(committer()));
        case "additionalData":
            return Optional.ofNullable(clazz.cast(additionalData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("commitId", COMMIT_ID_FIELD);
        map.put("treeId", TREE_ID_FIELD);
        map.put("parents", PARENTS_FIELD);
        map.put("message", MESSAGE_FIELD);
        map.put("author", AUTHOR_FIELD);
        map.put("committer", COMMITTER_FIELD);
        map.put("additionalData", ADDITIONAL_DATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Commit, T> g) {
        return obj -> g.apply((Commit) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Commit> {
        /**
         * <p>
         * The full SHA ID of the specified commit.
         * </p>
         * 
         * @param commitId
         *        The full SHA ID of the specified commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commitId(String commitId);

        /**
         * <p>
         * Tree information for the specified commit.
         * </p>
         * 
         * @param treeId
         *        Tree information for the specified commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder treeId(String treeId);

        /**
         * <p>
         * A list of parent commits for the specified commit. Each parent commit ID is the full commit ID.
         * </p>
         * 
         * @param parents
         *        A list of parent commits for the specified commit. Each parent commit ID is the full commit ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parents(Collection<String> parents);

        /**
         * <p>
         * A list of parent commits for the specified commit. Each parent commit ID is the full commit ID.
         * </p>
         * 
         * @param parents
         *        A list of parent commits for the specified commit. Each parent commit ID is the full commit ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parents(String... parents);

        /**
         * <p>
         * The commit message associated with the specified commit.
         * </p>
         * 
         * @param message
         *        The commit message associated with the specified commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * Information about the author of the specified commit. Information includes the date in timestamp format with
         * GMT offset, the name of the author, and the email address for the author, as configured in Git.
         * </p>
         * 
         * @param author
         *        Information about the author of the specified commit. Information includes the date in timestamp
         *        format with GMT offset, the name of the author, and the email address for the author, as configured in
         *        Git.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder author(UserInfo author);

        /**
         * <p>
         * Information about the author of the specified commit. Information includes the date in timestamp format with
         * GMT offset, the name of the author, and the email address for the author, as configured in Git.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserInfo.Builder} avoiding the need to
         * create one manually via {@link UserInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserInfo.Builder#build()} is called immediately and its result is
         * passed to {@link #author(UserInfo)}.
         * 
         * @param author
         *        a consumer that will call methods on {@link UserInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #author(UserInfo)
         */
        default Builder author(Consumer<UserInfo.Builder> author) {
            return author(UserInfo.builder().applyMutation(author).build());
        }

        /**
         * <p>
         * Information about the person who committed the specified commit, also known as the committer. Information
         * includes the date in timestamp format with GMT offset, the name of the committer, and the email address for
         * the committer, as configured in Git.
         * </p>
         * <p>
         * For more information about the difference between an author and a committer in Git, see <a
         * href="http://git-scm.com/book/ch2-3.html">Viewing the Commit History</a> in Pro Git by Scott Chacon and Ben
         * Straub.
         * </p>
         * 
         * @param committer
         *        Information about the person who committed the specified commit, also known as the committer.
         *        Information includes the date in timestamp format with GMT offset, the name of the committer, and the
         *        email address for the committer, as configured in Git.</p>
         *        <p>
         *        For more information about the difference between an author and a committer in Git, see <a
         *        href="http://git-scm.com/book/ch2-3.html">Viewing the Commit History</a> in Pro Git by Scott Chacon
         *        and Ben Straub.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder committer(UserInfo committer);

        /**
         * <p>
         * Information about the person who committed the specified commit, also known as the committer. Information
         * includes the date in timestamp format with GMT offset, the name of the committer, and the email address for
         * the committer, as configured in Git.
         * </p>
         * <p>
         * For more information about the difference between an author and a committer in Git, see <a
         * href="http://git-scm.com/book/ch2-3.html">Viewing the Commit History</a> in Pro Git by Scott Chacon and Ben
         * Straub.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserInfo.Builder} avoiding the need to
         * create one manually via {@link UserInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserInfo.Builder#build()} is called immediately and its result is
         * passed to {@link #committer(UserInfo)}.
         * 
         * @param committer
         *        a consumer that will call methods on {@link UserInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #committer(UserInfo)
         */
        default Builder committer(Consumer<UserInfo.Builder> committer) {
            return committer(UserInfo.builder().applyMutation(committer).build());
        }

        /**
         * <p>
         * Any other data associated with the specified commit.
         * </p>
         * 
         * @param additionalData
         *        Any other data associated with the specified commit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalData(String additionalData);
    }

    static final class BuilderImpl implements Builder {
        private String commitId;

        private String treeId;

        private List<String> parents = DefaultSdkAutoConstructList.getInstance();

        private String message;

        private UserInfo author;

        private UserInfo committer;

        private String additionalData;

        private BuilderImpl() {
        }

        private BuilderImpl(Commit model) {
            commitId(model.commitId);
            treeId(model.treeId);
            parents(model.parents);
            message(model.message);
            author(model.author);
            committer(model.committer);
            additionalData(model.additionalData);
        }

        public final String getCommitId() {
            return commitId;
        }

        public final void setCommitId(String commitId) {
            this.commitId = commitId;
        }

        @Override
        public final Builder commitId(String commitId) {
            this.commitId = commitId;
            return this;
        }

        public final String getTreeId() {
            return treeId;
        }

        public final void setTreeId(String treeId) {
            this.treeId = treeId;
        }

        @Override
        public final Builder treeId(String treeId) {
            this.treeId = treeId;
            return this;
        }

        public final Collection<String> getParents() {
            if (parents instanceof SdkAutoConstructList) {
                return null;
            }
            return parents;
        }

        public final void setParents(Collection<String> parents) {
            this.parents = ParentListCopier.copy(parents);
        }

        @Override
        public final Builder parents(Collection<String> parents) {
            this.parents = ParentListCopier.copy(parents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parents(String... parents) {
            parents(Arrays.asList(parents));
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final UserInfo.Builder getAuthor() {
            return author != null ? author.toBuilder() : null;
        }

        public final void setAuthor(UserInfo.BuilderImpl author) {
            this.author = author != null ? author.build() : null;
        }

        @Override
        public final Builder author(UserInfo author) {
            this.author = author;
            return this;
        }

        public final UserInfo.Builder getCommitter() {
            return committer != null ? committer.toBuilder() : null;
        }

        public final void setCommitter(UserInfo.BuilderImpl committer) {
            this.committer = committer != null ? committer.build() : null;
        }

        @Override
        public final Builder committer(UserInfo committer) {
            this.committer = committer;
            return this;
        }

        public final String getAdditionalData() {
            return additionalData;
        }

        public final void setAdditionalData(String additionalData) {
            this.additionalData = additionalData;
        }

        @Override
        public final Builder additionalData(String additionalData) {
            this.additionalData = additionalData;
            return this;
        }

        @Override
        public Commit build() {
            return new Commit(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
