/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a version of a file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FileVersion implements SdkPojo, Serializable, ToCopyableBuilder<FileVersion.Builder, FileVersion> {
    private static final SdkField<Commit> COMMIT_FIELD = SdkField.<Commit> builder(MarshallingType.SDK_POJO).memberName("commit")
            .getter(getter(FileVersion::commit)).setter(setter(Builder::commit)).constructor(Commit::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("commit").build()).build();

    private static final SdkField<String> BLOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("blobId")
            .getter(getter(FileVersion::blobId)).setter(setter(Builder::blobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("blobId").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("path")
            .getter(getter(FileVersion::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("path").build()).build();

    private static final SdkField<List<String>> REVISION_CHILDREN_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("revisionChildren")
            .getter(getter(FileVersion::revisionChildren))
            .setter(setter(Builder::revisionChildren))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("revisionChildren").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMIT_FIELD, BLOB_ID_FIELD,
            PATH_FIELD, REVISION_CHILDREN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Commit commit;

    private final String blobId;

    private final String path;

    private final List<String> revisionChildren;

    private FileVersion(BuilderImpl builder) {
        this.commit = builder.commit;
        this.blobId = builder.blobId;
        this.path = builder.path;
        this.revisionChildren = builder.revisionChildren;
    }

    /**
     * Returns the value of the Commit property for this object.
     * 
     * @return The value of the Commit property for this object.
     */
    public final Commit commit() {
        return commit;
    }

    /**
     * <p>
     * The blob ID of the object that represents the content of the file in this version.
     * </p>
     * 
     * @return The blob ID of the object that represents the content of the file in this version.
     */
    public final String blobId() {
        return blobId;
    }

    /**
     * <p>
     * The name and path of the file at which this blob is indexed which contains the data for this version of the file.
     * This value will vary between file versions if a file is renamed or if its path changes.
     * </p>
     * 
     * @return The name and path of the file at which this blob is indexed which contains the data for this version of
     *         the file. This value will vary between file versions if a file is renamed or if its path changes.
     */
    public final String path() {
        return path;
    }

    /**
     * For responses, this returns true if the service returned a value for the RevisionChildren property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRevisionChildren() {
        return revisionChildren != null && !(revisionChildren instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of commit IDs that contain more recent versions of this file. If there are no additional versions of the
     * file, this array will be empty.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRevisionChildren} method.
     * </p>
     * 
     * @return An array of commit IDs that contain more recent versions of this file. If there are no additional
     *         versions of the file, this array will be empty.
     */
    public final List<String> revisionChildren() {
        return revisionChildren;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(commit());
        hashCode = 31 * hashCode + Objects.hashCode(blobId());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(hasRevisionChildren() ? revisionChildren() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FileVersion)) {
            return false;
        }
        FileVersion other = (FileVersion) obj;
        return Objects.equals(commit(), other.commit()) && Objects.equals(blobId(), other.blobId())
                && Objects.equals(path(), other.path()) && hasRevisionChildren() == other.hasRevisionChildren()
                && Objects.equals(revisionChildren(), other.revisionChildren());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FileVersion").add("Commit", commit()).add("BlobId", blobId()).add("Path", path())
                .add("RevisionChildren", hasRevisionChildren() ? revisionChildren() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "commit":
            return Optional.ofNullable(clazz.cast(commit()));
        case "blobId":
            return Optional.ofNullable(clazz.cast(blobId()));
        case "path":
            return Optional.ofNullable(clazz.cast(path()));
        case "revisionChildren":
            return Optional.ofNullable(clazz.cast(revisionChildren()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("commit", COMMIT_FIELD);
        map.put("blobId", BLOB_ID_FIELD);
        map.put("path", PATH_FIELD);
        map.put("revisionChildren", REVISION_CHILDREN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FileVersion, T> g) {
        return obj -> g.apply((FileVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FileVersion> {
        /**
         * Sets the value of the Commit property for this object.
         *
         * @param commit
         *        The new value for the Commit property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commit(Commit commit);

        /**
         * Sets the value of the Commit property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Commit.Builder} avoiding the need to
         * create one manually via {@link Commit#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Commit.Builder#build()} is called immediately and its result is
         * passed to {@link #commit(Commit)}.
         * 
         * @param commit
         *        a consumer that will call methods on {@link Commit.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #commit(Commit)
         */
        default Builder commit(Consumer<Commit.Builder> commit) {
            return commit(Commit.builder().applyMutation(commit).build());
        }

        /**
         * <p>
         * The blob ID of the object that represents the content of the file in this version.
         * </p>
         * 
         * @param blobId
         *        The blob ID of the object that represents the content of the file in this version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blobId(String blobId);

        /**
         * <p>
         * The name and path of the file at which this blob is indexed which contains the data for this version of the
         * file. This value will vary between file versions if a file is renamed or if its path changes.
         * </p>
         * 
         * @param path
         *        The name and path of the file at which this blob is indexed which contains the data for this version
         *        of the file. This value will vary between file versions if a file is renamed or if its path changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * An array of commit IDs that contain more recent versions of this file. If there are no additional versions of
         * the file, this array will be empty.
         * </p>
         * 
         * @param revisionChildren
         *        An array of commit IDs that contain more recent versions of this file. If there are no additional
         *        versions of the file, this array will be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionChildren(Collection<String> revisionChildren);

        /**
         * <p>
         * An array of commit IDs that contain more recent versions of this file. If there are no additional versions of
         * the file, this array will be empty.
         * </p>
         * 
         * @param revisionChildren
         *        An array of commit IDs that contain more recent versions of this file. If there are no additional
         *        versions of the file, this array will be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder revisionChildren(String... revisionChildren);
    }

    static final class BuilderImpl implements Builder {
        private Commit commit;

        private String blobId;

        private String path;

        private List<String> revisionChildren = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FileVersion model) {
            commit(model.commit);
            blobId(model.blobId);
            path(model.path);
            revisionChildren(model.revisionChildren);
        }

        public final Commit.Builder getCommit() {
            return commit != null ? commit.toBuilder() : null;
        }

        public final void setCommit(Commit.BuilderImpl commit) {
            this.commit = commit != null ? commit.build() : null;
        }

        @Override
        public final Builder commit(Commit commit) {
            this.commit = commit;
            return this;
        }

        public final String getBlobId() {
            return blobId;
        }

        public final void setBlobId(String blobId) {
            this.blobId = blobId;
        }

        @Override
        public final Builder blobId(String blobId) {
            this.blobId = blobId;
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final Collection<String> getRevisionChildren() {
            if (revisionChildren instanceof SdkAutoConstructList) {
                return null;
            }
            return revisionChildren;
        }

        public final void setRevisionChildren(Collection<String> revisionChildren) {
            this.revisionChildren = RevisionChildrenCopier.copy(revisionChildren);
        }

        @Override
        public final Builder revisionChildren(Collection<String> revisionChildren) {
            this.revisionChildren = RevisionChildrenCopier.copy(revisionChildren);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder revisionChildren(String... revisionChildren) {
            revisionChildren(Arrays.asList(revisionChildren));
            return this;
        }

        @Override
        public FileVersion build() {
            return new FileVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
