/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information about the location of a change or comment in the comparison between two commits or a pull
 * request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Location implements SdkPojo, Serializable, ToCopyableBuilder<Location.Builder, Location> {
    private static final SdkField<String> FILE_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("filePath").getter(getter(Location::filePath)).setter(setter(Builder::filePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filePath").build()).build();

    private static final SdkField<Long> FILE_POSITION_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("filePosition").getter(getter(Location::filePosition)).setter(setter(Builder::filePosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filePosition").build()).build();

    private static final SdkField<String> RELATIVE_FILE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("relativeFileVersion").getter(getter(Location::relativeFileVersionAsString))
            .setter(setter(Builder::relativeFileVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relativeFileVersion").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILE_PATH_FIELD,
            FILE_POSITION_FIELD, RELATIVE_FILE_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String filePath;

    private final Long filePosition;

    private final String relativeFileVersion;

    private Location(BuilderImpl builder) {
        this.filePath = builder.filePath;
        this.filePosition = builder.filePosition;
        this.relativeFileVersion = builder.relativeFileVersion;
    }

    /**
     * <p>
     * The name of the file being compared, including its extension and subdirectory, if any.
     * </p>
     * 
     * @return The name of the file being compared, including its extension and subdirectory, if any.
     */
    public final String filePath() {
        return filePath;
    }

    /**
     * <p>
     * The position of a change in a compared file, in line number format.
     * </p>
     * 
     * @return The position of a change in a compared file, in line number format.
     */
    public final Long filePosition() {
        return filePosition;
    }

    /**
     * <p>
     * In a comparison of commits or a pull request, whether the change is in the before or after of that comparison.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #relativeFileVersion} will return {@link RelativeFileVersionEnum#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #relativeFileVersionAsString}.
     * </p>
     * 
     * @return In a comparison of commits or a pull request, whether the change is in the before or after of that
     *         comparison.
     * @see RelativeFileVersionEnum
     */
    public final RelativeFileVersionEnum relativeFileVersion() {
        return RelativeFileVersionEnum.fromValue(relativeFileVersion);
    }

    /**
     * <p>
     * In a comparison of commits or a pull request, whether the change is in the before or after of that comparison.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #relativeFileVersion} will return {@link RelativeFileVersionEnum#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #relativeFileVersionAsString}.
     * </p>
     * 
     * @return In a comparison of commits or a pull request, whether the change is in the before or after of that
     *         comparison.
     * @see RelativeFileVersionEnum
     */
    public final String relativeFileVersionAsString() {
        return relativeFileVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(filePath());
        hashCode = 31 * hashCode + Objects.hashCode(filePosition());
        hashCode = 31 * hashCode + Objects.hashCode(relativeFileVersionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Location)) {
            return false;
        }
        Location other = (Location) obj;
        return Objects.equals(filePath(), other.filePath()) && Objects.equals(filePosition(), other.filePosition())
                && Objects.equals(relativeFileVersionAsString(), other.relativeFileVersionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Location").add("FilePath", filePath()).add("FilePosition", filePosition())
                .add("RelativeFileVersion", relativeFileVersionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "filePath":
            return Optional.ofNullable(clazz.cast(filePath()));
        case "filePosition":
            return Optional.ofNullable(clazz.cast(filePosition()));
        case "relativeFileVersion":
            return Optional.ofNullable(clazz.cast(relativeFileVersionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("filePath", FILE_PATH_FIELD);
        map.put("filePosition", FILE_POSITION_FIELD);
        map.put("relativeFileVersion", RELATIVE_FILE_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Location, T> g) {
        return obj -> g.apply((Location) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Location> {
        /**
         * <p>
         * The name of the file being compared, including its extension and subdirectory, if any.
         * </p>
         * 
         * @param filePath
         *        The name of the file being compared, including its extension and subdirectory, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePath(String filePath);

        /**
         * <p>
         * The position of a change in a compared file, in line number format.
         * </p>
         * 
         * @param filePosition
         *        The position of a change in a compared file, in line number format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePosition(Long filePosition);

        /**
         * <p>
         * In a comparison of commits or a pull request, whether the change is in the before or after of that
         * comparison.
         * </p>
         * 
         * @param relativeFileVersion
         *        In a comparison of commits or a pull request, whether the change is in the before or after of that
         *        comparison.
         * @see RelativeFileVersionEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RelativeFileVersionEnum
         */
        Builder relativeFileVersion(String relativeFileVersion);

        /**
         * <p>
         * In a comparison of commits or a pull request, whether the change is in the before or after of that
         * comparison.
         * </p>
         * 
         * @param relativeFileVersion
         *        In a comparison of commits or a pull request, whether the change is in the before or after of that
         *        comparison.
         * @see RelativeFileVersionEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RelativeFileVersionEnum
         */
        Builder relativeFileVersion(RelativeFileVersionEnum relativeFileVersion);
    }

    static final class BuilderImpl implements Builder {
        private String filePath;

        private Long filePosition;

        private String relativeFileVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(Location model) {
            filePath(model.filePath);
            filePosition(model.filePosition);
            relativeFileVersion(model.relativeFileVersion);
        }

        public final String getFilePath() {
            return filePath;
        }

        public final void setFilePath(String filePath) {
            this.filePath = filePath;
        }

        @Override
        public final Builder filePath(String filePath) {
            this.filePath = filePath;
            return this;
        }

        public final Long getFilePosition() {
            return filePosition;
        }

        public final void setFilePosition(Long filePosition) {
            this.filePosition = filePosition;
        }

        @Override
        public final Builder filePosition(Long filePosition) {
            this.filePosition = filePosition;
            return this;
        }

        public final String getRelativeFileVersion() {
            return relativeFileVersion;
        }

        public final void setRelativeFileVersion(String relativeFileVersion) {
            this.relativeFileVersion = relativeFileVersion;
        }

        @Override
        public final Builder relativeFileVersion(String relativeFileVersion) {
            this.relativeFileVersion = relativeFileVersion;
            return this;
        }

        @Override
        public final Builder relativeFileVersion(RelativeFileVersionEnum relativeFileVersion) {
            this.relativeFileVersion(relativeFileVersion == null ? null : relativeFileVersion.toString());
            return this;
        }

        @Override
        public Location build() {
            return new Location(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
