/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutFileRequest extends CodeCommitRequest implements ToCopyableBuilder<PutFileRequest.Builder, PutFileRequest> {
    private static final SdkField<String> REPOSITORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::repositoryName)).setter(setter(Builder::repositoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryName").build()).build();

    private static final SdkField<String> BRANCH_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::branchName)).setter(setter(Builder::branchName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("branchName").build()).build();

    private static final SdkField<SdkBytes> FILE_CONTENT_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(PutFileRequest::fileContent)).setter(setter(Builder::fileContent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileContent").build()).build();

    private static final SdkField<String> FILE_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::filePath)).setter(setter(Builder::filePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filePath").build()).build();

    private static final SdkField<String> FILE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::fileModeAsString)).setter(setter(Builder::fileMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileMode").build()).build();

    private static final SdkField<String> PARENT_COMMIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::parentCommitId)).setter(setter(Builder::parentCommitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parentCommitId").build()).build();

    private static final SdkField<String> COMMIT_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::commitMessage)).setter(setter(Builder::commitMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("commitMessage").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> EMAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutFileRequest::email)).setter(setter(Builder::email))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("email").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_NAME_FIELD,
            BRANCH_NAME_FIELD, FILE_CONTENT_FIELD, FILE_PATH_FIELD, FILE_MODE_FIELD, PARENT_COMMIT_ID_FIELD,
            COMMIT_MESSAGE_FIELD, NAME_FIELD, EMAIL_FIELD));

    private final String repositoryName;

    private final String branchName;

    private final SdkBytes fileContent;

    private final String filePath;

    private final String fileMode;

    private final String parentCommitId;

    private final String commitMessage;

    private final String name;

    private final String email;

    private PutFileRequest(BuilderImpl builder) {
        super(builder);
        this.repositoryName = builder.repositoryName;
        this.branchName = builder.branchName;
        this.fileContent = builder.fileContent;
        this.filePath = builder.filePath;
        this.fileMode = builder.fileMode;
        this.parentCommitId = builder.parentCommitId;
        this.commitMessage = builder.commitMessage;
        this.name = builder.name;
        this.email = builder.email;
    }

    /**
     * <p>
     * The name of the repository where you want to add or update the file.
     * </p>
     * 
     * @return The name of the repository where you want to add or update the file.
     */
    public String repositoryName() {
        return repositoryName;
    }

    /**
     * <p>
     * The name of the branch where you want to add or update the file. If this is an empty repository, this branch will
     * be created.
     * </p>
     * 
     * @return The name of the branch where you want to add or update the file. If this is an empty repository, this
     *         branch will be created.
     */
    public String branchName() {
        return branchName;
    }

    /**
     * <p>
     * The content of the file, in binary object format.
     * </p>
     * 
     * @return The content of the file, in binary object format.
     */
    public SdkBytes fileContent() {
        return fileContent;
    }

    /**
     * <p>
     * The name of the file you want to add or update, including the relative path to the file in the repository.
     * </p>
     * <note>
     * <p>
     * If the path does not currently exist in the repository, the path will be created as part of adding the file.
     * </p>
     * </note>
     * 
     * @return The name of the file you want to add or update, including the relative path to the file in the
     *         repository.</p> <note>
     *         <p>
     *         If the path does not currently exist in the repository, the path will be created as part of adding the
     *         file.
     *         </p>
     */
    public String filePath() {
        return filePath;
    }

    /**
     * <p>
     * The file mode permissions of the blob. Valid file mode permissions are listed below.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileMode} will
     * return {@link FileModeTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileModeAsString}.
     * </p>
     * 
     * @return The file mode permissions of the blob. Valid file mode permissions are listed below.
     * @see FileModeTypeEnum
     */
    public FileModeTypeEnum fileMode() {
        return FileModeTypeEnum.fromValue(fileMode);
    }

    /**
     * <p>
     * The file mode permissions of the blob. Valid file mode permissions are listed below.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileMode} will
     * return {@link FileModeTypeEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileModeAsString}.
     * </p>
     * 
     * @return The file mode permissions of the blob. Valid file mode permissions are listed below.
     * @see FileModeTypeEnum
     */
    public String fileModeAsString() {
        return fileMode;
    }

    /**
     * <p>
     * The full commit ID of the head commit in the branch where you want to add or update the file. If this is an empty
     * repository, no commit ID is required. If this is not an empty repository, a commit ID is required.
     * </p>
     * <p>
     * The commit ID must match the ID of the head commit at the time of the operation, or an error will occur, and the
     * file will not be added or updated.
     * </p>
     * 
     * @return The full commit ID of the head commit in the branch where you want to add or update the file. If this is
     *         an empty repository, no commit ID is required. If this is not an empty repository, a commit ID is
     *         required. </p>
     *         <p>
     *         The commit ID must match the ID of the head commit at the time of the operation, or an error will occur,
     *         and the file will not be added or updated.
     */
    public String parentCommitId() {
        return parentCommitId;
    }

    /**
     * <p>
     * A message about why this file was added or updated. While optional, adding a message is strongly encouraged in
     * order to provide a more useful commit history for your repository.
     * </p>
     * 
     * @return A message about why this file was added or updated. While optional, adding a message is strongly
     *         encouraged in order to provide a more useful commit history for your repository.
     */
    public String commitMessage() {
        return commitMessage;
    }

    /**
     * <p>
     * The name of the person adding or updating the file. While optional, adding a name is strongly encouraged in order
     * to provide a more useful commit history for your repository.
     * </p>
     * 
     * @return The name of the person adding or updating the file. While optional, adding a name is strongly encouraged
     *         in order to provide a more useful commit history for your repository.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * An email address for the person adding or updating the file.
     * </p>
     * 
     * @return An email address for the person adding or updating the file.
     */
    public String email() {
        return email;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(repositoryName());
        hashCode = 31 * hashCode + Objects.hashCode(branchName());
        hashCode = 31 * hashCode + Objects.hashCode(fileContent());
        hashCode = 31 * hashCode + Objects.hashCode(filePath());
        hashCode = 31 * hashCode + Objects.hashCode(fileModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(parentCommitId());
        hashCode = 31 * hashCode + Objects.hashCode(commitMessage());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(email());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutFileRequest)) {
            return false;
        }
        PutFileRequest other = (PutFileRequest) obj;
        return Objects.equals(repositoryName(), other.repositoryName()) && Objects.equals(branchName(), other.branchName())
                && Objects.equals(fileContent(), other.fileContent()) && Objects.equals(filePath(), other.filePath())
                && Objects.equals(fileModeAsString(), other.fileModeAsString())
                && Objects.equals(parentCommitId(), other.parentCommitId())
                && Objects.equals(commitMessage(), other.commitMessage()) && Objects.equals(name(), other.name())
                && Objects.equals(email(), other.email());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutFileRequest").add("RepositoryName", repositoryName()).add("BranchName", branchName())
                .add("FileContent", fileContent()).add("FilePath", filePath()).add("FileMode", fileModeAsString())
                .add("ParentCommitId", parentCommitId()).add("CommitMessage", commitMessage()).add("Name", name())
                .add("Email", email()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "repositoryName":
            return Optional.ofNullable(clazz.cast(repositoryName()));
        case "branchName":
            return Optional.ofNullable(clazz.cast(branchName()));
        case "fileContent":
            return Optional.ofNullable(clazz.cast(fileContent()));
        case "filePath":
            return Optional.ofNullable(clazz.cast(filePath()));
        case "fileMode":
            return Optional.ofNullable(clazz.cast(fileModeAsString()));
        case "parentCommitId":
            return Optional.ofNullable(clazz.cast(parentCommitId()));
        case "commitMessage":
            return Optional.ofNullable(clazz.cast(commitMessage()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "email":
            return Optional.ofNullable(clazz.cast(email()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutFileRequest, T> g) {
        return obj -> g.apply((PutFileRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CodeCommitRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutFileRequest> {
        /**
         * <p>
         * The name of the repository where you want to add or update the file.
         * </p>
         * 
         * @param repositoryName
         *        The name of the repository where you want to add or update the file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryName(String repositoryName);

        /**
         * <p>
         * The name of the branch where you want to add or update the file. If this is an empty repository, this branch
         * will be created.
         * </p>
         * 
         * @param branchName
         *        The name of the branch where you want to add or update the file. If this is an empty repository, this
         *        branch will be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder branchName(String branchName);

        /**
         * <p>
         * The content of the file, in binary object format.
         * </p>
         * 
         * @param fileContent
         *        The content of the file, in binary object format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileContent(SdkBytes fileContent);

        /**
         * <p>
         * The name of the file you want to add or update, including the relative path to the file in the repository.
         * </p>
         * <note>
         * <p>
         * If the path does not currently exist in the repository, the path will be created as part of adding the file.
         * </p>
         * </note>
         * 
         * @param filePath
         *        The name of the file you want to add or update, including the relative path to the file in the
         *        repository.</p> <note>
         *        <p>
         *        If the path does not currently exist in the repository, the path will be created as part of adding the
         *        file.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filePath(String filePath);

        /**
         * <p>
         * The file mode permissions of the blob. Valid file mode permissions are listed below.
         * </p>
         * 
         * @param fileMode
         *        The file mode permissions of the blob. Valid file mode permissions are listed below.
         * @see FileModeTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileModeTypeEnum
         */
        Builder fileMode(String fileMode);

        /**
         * <p>
         * The file mode permissions of the blob. Valid file mode permissions are listed below.
         * </p>
         * 
         * @param fileMode
         *        The file mode permissions of the blob. Valid file mode permissions are listed below.
         * @see FileModeTypeEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileModeTypeEnum
         */
        Builder fileMode(FileModeTypeEnum fileMode);

        /**
         * <p>
         * The full commit ID of the head commit in the branch where you want to add or update the file. If this is an
         * empty repository, no commit ID is required. If this is not an empty repository, a commit ID is required.
         * </p>
         * <p>
         * The commit ID must match the ID of the head commit at the time of the operation, or an error will occur, and
         * the file will not be added or updated.
         * </p>
         * 
         * @param parentCommitId
         *        The full commit ID of the head commit in the branch where you want to add or update the file. If this
         *        is an empty repository, no commit ID is required. If this is not an empty repository, a commit ID is
         *        required. </p>
         *        <p>
         *        The commit ID must match the ID of the head commit at the time of the operation, or an error will
         *        occur, and the file will not be added or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentCommitId(String parentCommitId);

        /**
         * <p>
         * A message about why this file was added or updated. While optional, adding a message is strongly encouraged
         * in order to provide a more useful commit history for your repository.
         * </p>
         * 
         * @param commitMessage
         *        A message about why this file was added or updated. While optional, adding a message is strongly
         *        encouraged in order to provide a more useful commit history for your repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commitMessage(String commitMessage);

        /**
         * <p>
         * The name of the person adding or updating the file. While optional, adding a name is strongly encouraged in
         * order to provide a more useful commit history for your repository.
         * </p>
         * 
         * @param name
         *        The name of the person adding or updating the file. While optional, adding a name is strongly
         *        encouraged in order to provide a more useful commit history for your repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An email address for the person adding or updating the file.
         * </p>
         * 
         * @param email
         *        An email address for the person adding or updating the file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder email(String email);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeCommitRequest.BuilderImpl implements Builder {
        private String repositoryName;

        private String branchName;

        private SdkBytes fileContent;

        private String filePath;

        private String fileMode;

        private String parentCommitId;

        private String commitMessage;

        private String name;

        private String email;

        private BuilderImpl() {
        }

        private BuilderImpl(PutFileRequest model) {
            super(model);
            repositoryName(model.repositoryName);
            branchName(model.branchName);
            fileContent(model.fileContent);
            filePath(model.filePath);
            fileMode(model.fileMode);
            parentCommitId(model.parentCommitId);
            commitMessage(model.commitMessage);
            name(model.name);
            email(model.email);
        }

        public final String getRepositoryName() {
            return repositoryName;
        }

        @Override
        public final Builder repositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
            return this;
        }

        public final void setRepositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
        }

        public final String getBranchName() {
            return branchName;
        }

        @Override
        public final Builder branchName(String branchName) {
            this.branchName = branchName;
            return this;
        }

        public final void setBranchName(String branchName) {
            this.branchName = branchName;
        }

        public final ByteBuffer getFileContent() {
            return fileContent == null ? null : fileContent.asByteBuffer();
        }

        @Override
        public final Builder fileContent(SdkBytes fileContent) {
            this.fileContent = StandardMemberCopier.copy(fileContent);
            return this;
        }

        public final void setFileContent(ByteBuffer fileContent) {
            fileContent(fileContent == null ? null : SdkBytes.fromByteBuffer(fileContent));
        }

        public final String getFilePath() {
            return filePath;
        }

        @Override
        public final Builder filePath(String filePath) {
            this.filePath = filePath;
            return this;
        }

        public final void setFilePath(String filePath) {
            this.filePath = filePath;
        }

        public final String getFileModeAsString() {
            return fileMode;
        }

        @Override
        public final Builder fileMode(String fileMode) {
            this.fileMode = fileMode;
            return this;
        }

        @Override
        public final Builder fileMode(FileModeTypeEnum fileMode) {
            this.fileMode(fileMode == null ? null : fileMode.toString());
            return this;
        }

        public final void setFileMode(String fileMode) {
            this.fileMode = fileMode;
        }

        public final String getParentCommitId() {
            return parentCommitId;
        }

        @Override
        public final Builder parentCommitId(String parentCommitId) {
            this.parentCommitId = parentCommitId;
            return this;
        }

        public final void setParentCommitId(String parentCommitId) {
            this.parentCommitId = parentCommitId;
        }

        public final String getCommitMessage() {
            return commitMessage;
        }

        @Override
        public final Builder commitMessage(String commitMessage) {
            this.commitMessage = commitMessage;
            return this;
        }

        public final void setCommitMessage(String commitMessage) {
            this.commitMessage = commitMessage;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getEmail() {
            return email;
        }

        @Override
        public final Builder email(String email) {
            this.email = email;
            return this;
        }

        public final void setEmail(String email) {
            this.email = email;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutFileRequest build() {
            return new PutFileRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
