/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeconnections;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.codeconnections.internal.CodeConnectionsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.codeconnections.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.codeconnections.model.AccessDeniedException;
import software.amazon.awssdk.services.codeconnections.model.CodeConnectionsException;
import software.amazon.awssdk.services.codeconnections.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codeconnections.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.codeconnections.model.ConflictException;
import software.amazon.awssdk.services.codeconnections.model.CreateConnectionRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateConnectionResponse;
import software.amazon.awssdk.services.codeconnections.model.CreateHostRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateHostResponse;
import software.amazon.awssdk.services.codeconnections.model.CreateRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.CreateSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteHostRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteHostResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.model.GetConnectionRequest;
import software.amazon.awssdk.services.codeconnections.model.GetConnectionResponse;
import software.amazon.awssdk.services.codeconnections.model.GetHostRequest;
import software.amazon.awssdk.services.codeconnections.model.GetHostResponse;
import software.amazon.awssdk.services.codeconnections.model.GetRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.GetRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.GetRepositorySyncStatusRequest;
import software.amazon.awssdk.services.codeconnections.model.GetRepositorySyncStatusResponse;
import software.amazon.awssdk.services.codeconnections.model.GetResourceSyncStatusRequest;
import software.amazon.awssdk.services.codeconnections.model.GetResourceSyncStatusResponse;
import software.amazon.awssdk.services.codeconnections.model.GetSyncBlockerSummaryRequest;
import software.amazon.awssdk.services.codeconnections.model.GetSyncBlockerSummaryResponse;
import software.amazon.awssdk.services.codeconnections.model.GetSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.GetSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.model.InternalServerException;
import software.amazon.awssdk.services.codeconnections.model.InvalidInputException;
import software.amazon.awssdk.services.codeconnections.model.LimitExceededException;
import software.amazon.awssdk.services.codeconnections.model.ListConnectionsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListConnectionsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListHostsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListHostsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListRepositoryLinksRequest;
import software.amazon.awssdk.services.codeconnections.model.ListRepositoryLinksResponse;
import software.amazon.awssdk.services.codeconnections.model.ListRepositorySyncDefinitionsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListRepositorySyncDefinitionsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListSyncConfigurationsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListSyncConfigurationsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codeconnections.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codeconnections.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codeconnections.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codeconnections.model.ResourceUnavailableException;
import software.amazon.awssdk.services.codeconnections.model.RetryLatestCommitFailedException;
import software.amazon.awssdk.services.codeconnections.model.SyncBlockerDoesNotExistException;
import software.amazon.awssdk.services.codeconnections.model.SyncConfigurationStillExistsException;
import software.amazon.awssdk.services.codeconnections.model.TagResourceRequest;
import software.amazon.awssdk.services.codeconnections.model.TagResourceResponse;
import software.amazon.awssdk.services.codeconnections.model.ThrottlingException;
import software.amazon.awssdk.services.codeconnections.model.UnsupportedOperationException;
import software.amazon.awssdk.services.codeconnections.model.UnsupportedProviderTypeException;
import software.amazon.awssdk.services.codeconnections.model.UntagResourceRequest;
import software.amazon.awssdk.services.codeconnections.model.UntagResourceResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateHostRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateHostResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateOutOfSyncException;
import software.amazon.awssdk.services.codeconnections.model.UpdateRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncBlockerRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncBlockerResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.transform.CreateConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.CreateHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.CreateRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.CreateSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetRepositorySyncStatusRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetResourceSyncStatusRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetSyncBlockerSummaryRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListConnectionsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListHostsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListRepositoryLinksRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListRepositorySyncDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListSyncConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateSyncBlockerRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodeConnectionsAsyncClient}.
 *
 * @see CodeConnectionsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeConnectionsAsyncClient implements CodeConnectionsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodeConnectionsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeConnectionsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "CodeConnections" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a connection that can then be given to other Amazon Web Services services like CodePipeline so that it
     * can access third-party code repositories. The connection is in pending status until the third-party connection
     * handshake is completed from the console.
     * </p>
     *
     * @param createConnectionRequest
     * @return A Java Future containing the result of the CreateConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectionResponse> createConnection(CreateConnectionRequest createConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectionRequest, CreateConnectionResponse>()
                            .withOperationName("CreateConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConnectionRequest));
            CompletableFuture<CreateConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a resource that represents the infrastructure where a third-party provider is installed. The host is used
     * when you create connections to an installed third-party provider type, such as GitHub Enterprise Server. You
     * create one host for all connections to that provider.
     * </p>
     * <note>
     * <p>
     * A host created through the CLI or the SDK is in `PENDING` status by default. You can make its status `AVAILABLE`
     * by setting up the host in the console.
     * </p>
     * </note>
     *
     * @param createHostRequest
     * @return A Java Future containing the result of the CreateHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.CreateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateHost" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHostResponse> createHost(CreateHostRequest createHostRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateHostResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateHostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHostRequest, CreateHostResponse>().withOperationName("CreateHost")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHostRequest));
            CompletableFuture<CreateHostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a link to a specified external Git repository. A repository link allows Git sync to monitor and sync
     * changes to files in a specified Git repository.
     * </p>
     *
     * @param createRepositoryLinkRequest
     * @return A Java Future containing the result of the CreateRepositoryLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>ResourceAlreadyExistsException Unable to create resource. Resource already exists.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.CreateRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRepositoryLinkResponse> createRepositoryLink(
            CreateRepositoryLinkRequest createRepositoryLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepositoryLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRepositoryLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateRepositoryLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRepositoryLinkRequest, CreateRepositoryLinkResponse>()
                            .withOperationName("CreateRepositoryLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRepositoryLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRepositoryLinkRequest));
            CompletableFuture<CreateRepositoryLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a sync configuration which allows Amazon Web Services to sync content from a Git repository to update a
     * specified Amazon Web Services resource. Parameters for the sync configuration are determined by the sync type.
     * </p>
     *
     * @param createSyncConfigurationRequest
     * @return A Java Future containing the result of the CreateSyncConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>ResourceAlreadyExistsException Unable to create resource. Resource already exists.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.CreateSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSyncConfigurationResponse> createSyncConfiguration(
            CreateSyncConfigurationRequest createSyncConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSyncConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSyncConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateSyncConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSyncConfigurationRequest, CreateSyncConfigurationResponse>()
                            .withOperationName("CreateSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSyncConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSyncConfigurationRequest));
            CompletableFuture<CreateSyncConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The connection to be deleted.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectionResponse> deleteConnection(DeleteConnectionRequest deleteConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectionRequest, DeleteConnectionResponse>()
                            .withOperationName("DeleteConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConnectionRequest));
            CompletableFuture<DeleteConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The host to be deleted. Before you delete a host, all connections associated to the host must be deleted.
     * </p>
     * <note>
     * <p>
     * A host cannot be deleted if it is in the VPC_CONFIG_INITIALIZING or VPC_CONFIG_DELETING state.
     * </p>
     * </note>
     *
     * @param deleteHostRequest
     * @return A Java Future containing the result of the DeleteHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.DeleteHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteHost" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHostResponse> deleteHost(DeleteHostRequest deleteHostRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteHostResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteHostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHostRequest, DeleteHostResponse>().withOperationName("DeleteHost")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteHostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteHostRequest));
            CompletableFuture<DeleteHostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the association between your connection and a specified external Git repository.
     * </p>
     *
     * @param deleteRepositoryLinkRequest
     * @return A Java Future containing the result of the DeleteRepositoryLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>SyncConfigurationStillExistsException Unable to continue. The sync blocker still exists.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>UnsupportedProviderTypeException The specified provider type is not supported for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.DeleteRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRepositoryLinkResponse> deleteRepositoryLink(
            DeleteRepositoryLinkRequest deleteRepositoryLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRepositoryLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteRepositoryLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRepositoryLinkRequest, DeleteRepositoryLinkResponse>()
                            .withOperationName("DeleteRepositoryLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRepositoryLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRepositoryLinkRequest));
            CompletableFuture<DeleteRepositoryLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the sync configuration for a specified repository and connection.
     * </p>
     *
     * @param deleteSyncConfigurationRequest
     * @return A Java Future containing the result of the DeleteSyncConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.DeleteSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSyncConfigurationResponse> deleteSyncConfiguration(
            DeleteSyncConfigurationRequest deleteSyncConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSyncConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSyncConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSyncConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSyncConfigurationRequest, DeleteSyncConfigurationResponse>()
                            .withOperationName("DeleteSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSyncConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSyncConfigurationRequest));
            CompletableFuture<DeleteSyncConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the connection ARN and details such as status, owner, and provider type.
     * </p>
     *
     * @param getConnectionRequest
     * @return A Java Future containing the result of the GetConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConnectionResponse> getConnection(GetConnectionRequest getConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetConnectionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConnectionRequest, GetConnectionResponse>()
                            .withOperationName("GetConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConnectionRequest));
            CompletableFuture<GetConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the host ARN and details such as status, provider type, endpoint, and, if applicable, the VPC
     * configuration.
     * </p>
     *
     * @param getHostRequest
     * @return A Java Future containing the result of the GetHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetHost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetHostResponse> getHost(GetHostRequest getHostRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetHostResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetHostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHostRequest, GetHostResponse>().withOperationName("GetHost")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetHostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHostRequest));
            CompletableFuture<GetHostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about a repository link. A repository link allows Git sync to monitor and sync changes from files
     * in a specified Git repository.
     * </p>
     *
     * @param getRepositoryLinkRequest
     * @return A Java Future containing the result of the GetRepositoryLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositoryLinkResponse> getRepositoryLink(GetRepositoryLinkRequest getRepositoryLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositoryLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetRepositoryLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositoryLinkRequest, GetRepositoryLinkResponse>()
                            .withOperationName("GetRepositoryLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRepositoryLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositoryLinkRequest));
            CompletableFuture<GetRepositoryLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about the sync status for a repository. A repository sync uses Git sync to push and pull changes
     * from your remote repository.
     * </p>
     *
     * @param getRepositorySyncStatusRequest
     * @return A Java Future containing the result of the GetRepositorySyncStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetRepositorySyncStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetRepositorySyncStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRepositorySyncStatusResponse> getRepositorySyncStatus(
            GetRepositorySyncStatusRequest getRepositorySyncStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositorySyncStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositorySyncStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositorySyncStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRepositorySyncStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRepositorySyncStatusResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetRepositorySyncStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRepositorySyncStatusRequest, GetRepositorySyncStatusResponse>()
                            .withOperationName("GetRepositorySyncStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRepositorySyncStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRepositorySyncStatusRequest));
            CompletableFuture<GetRepositorySyncStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the status of the sync with the Git repository for a specific Amazon Web Services resource.
     * </p>
     *
     * @param getResourceSyncStatusRequest
     * @return A Java Future containing the result of the GetResourceSyncStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetResourceSyncStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetResourceSyncStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceSyncStatusResponse> getResourceSyncStatus(
            GetResourceSyncStatusRequest getResourceSyncStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSyncStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSyncStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSyncStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceSyncStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceSyncStatusResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetResourceSyncStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceSyncStatusRequest, GetResourceSyncStatusResponse>()
                            .withOperationName("GetResourceSyncStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourceSyncStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourceSyncStatusRequest));
            CompletableFuture<GetResourceSyncStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the most recent sync blockers.
     * </p>
     *
     * @param getSyncBlockerSummaryRequest
     * @return A Java Future containing the result of the GetSyncBlockerSummary operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetSyncBlockerSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetSyncBlockerSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSyncBlockerSummaryResponse> getSyncBlockerSummary(
            GetSyncBlockerSummaryRequest getSyncBlockerSummaryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSyncBlockerSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSyncBlockerSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSyncBlockerSummary");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSyncBlockerSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSyncBlockerSummaryResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetSyncBlockerSummaryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSyncBlockerSummaryRequest, GetSyncBlockerSummaryResponse>()
                            .withOperationName("GetSyncBlockerSummary").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSyncBlockerSummaryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSyncBlockerSummaryRequest));
            CompletableFuture<GetSyncBlockerSummaryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about a sync configuration, including the sync type and resource name. A sync configuration
     * allows the configuration to sync (push and pull) changes from the remote repository for a specified branch in a
     * Git repository.
     * </p>
     *
     * @param getSyncConfigurationRequest
     * @return A Java Future containing the result of the GetSyncConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.GetSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSyncConfigurationResponse> getSyncConfiguration(
            GetSyncConfigurationRequest getSyncConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSyncConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSyncConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetSyncConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSyncConfigurationRequest, GetSyncConfigurationResponse>()
                            .withOperationName("GetSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSyncConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSyncConfigurationRequest));
            CompletableFuture<GetSyncConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return A Java Future containing the result of the ListConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectionsResponse> listConnections(ListConnectionsRequest listConnectionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectionsRequest, ListConnectionsResponse>()
                            .withOperationName("ListConnections").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConnectionsRequest));
            CompletableFuture<ListConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the hosts associated with your account.
     * </p>
     *
     * @param listHostsRequest
     * @return A Java Future containing the result of the ListHosts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.ListHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListHosts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListHostsResponse> listHosts(ListHostsRequest listHostsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHostsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHostsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHosts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHostsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListHostsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListHostsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHostsRequest, ListHostsResponse>().withOperationName("ListHosts")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListHostsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listHostsRequest));
            CompletableFuture<ListHostsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the repository links created for connections in your account.
     * </p>
     *
     * @param listRepositoryLinksRequest
     * @return A Java Future containing the result of the ListRepositoryLinks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.ListRepositoryLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListRepositoryLinks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRepositoryLinksResponse> listRepositoryLinks(
            ListRepositoryLinksRequest listRepositoryLinksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRepositoryLinksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRepositoryLinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositoryLinks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRepositoryLinksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRepositoryLinksResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRepositoryLinksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRepositoryLinksRequest, ListRepositoryLinksResponse>()
                            .withOperationName("ListRepositoryLinks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRepositoryLinksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRepositoryLinksRequest));
            CompletableFuture<ListRepositoryLinksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the repository sync definitions for repository links in your account.
     * </p>
     *
     * @param listRepositorySyncDefinitionsRequest
     * @return A Java Future containing the result of the ListRepositorySyncDefinitions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.ListRepositorySyncDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListRepositorySyncDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRepositorySyncDefinitionsResponse> listRepositorySyncDefinitions(
            ListRepositorySyncDefinitionsRequest listRepositorySyncDefinitionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRepositorySyncDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRepositorySyncDefinitionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositorySyncDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRepositorySyncDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRepositorySyncDefinitionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListRepositorySyncDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRepositorySyncDefinitionsRequest, ListRepositorySyncDefinitionsResponse>()
                            .withOperationName("ListRepositorySyncDefinitions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRepositorySyncDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRepositorySyncDefinitionsRequest));
            CompletableFuture<ListRepositorySyncDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of sync configurations for a specified repository.
     * </p>
     *
     * @param listSyncConfigurationsRequest
     * @return A Java Future containing the result of the ListSyncConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.ListSyncConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListSyncConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSyncConfigurationsResponse> listSyncConfigurations(
            ListSyncConfigurationsRequest listSyncConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSyncConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSyncConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSyncConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSyncConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSyncConfigurationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListSyncConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSyncConfigurationsRequest, ListSyncConfigurationsResponse>()
                            .withOperationName("ListSyncConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSyncConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSyncConfigurationsRequest));
            CompletableFuture<ListSyncConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the set of key-value pairs (metadata) that are used to manage the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a specified host with the provided configurations.
     * </p>
     *
     * @param updateHostRequest
     * @return A Java Future containing the result of the UpdateHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException Two conflicting operations have been made on the same resource.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>UnsupportedOperationException The operation is not supported. Check the connection status and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.UpdateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateHost" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateHostResponse> updateHost(UpdateHostRequest updateHostRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHost");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateHostResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateHostResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateHostRequest, UpdateHostResponse>().withOperationName("UpdateHost")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateHostRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateHostRequest));
            CompletableFuture<UpdateHostResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the association between your connection and a specified external Git repository. A repository link allows
     * Git sync to monitor and sync changes to files in a specified Git repository.
     * </p>
     *
     * @param updateRepositoryLinkRequest
     * @return A Java Future containing the result of the UpdateRepositoryLink operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConditionalCheckFailedException The conditional check failed. Try again later.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>UpdateOutOfSyncException The update is out of sync. Try syncing again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.UpdateRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRepositoryLinkResponse> updateRepositoryLink(
            UpdateRepositoryLinkRequest updateRepositoryLinkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepositoryLink");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRepositoryLinkResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateRepositoryLinkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRepositoryLinkRequest, UpdateRepositoryLinkResponse>()
                            .withOperationName("UpdateRepositoryLink").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateRepositoryLinkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateRepositoryLinkRequest));
            CompletableFuture<UpdateRepositoryLinkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to update the status of a sync blocker, resolving the blocker and allowing syncing to continue.
     * </p>
     *
     * @param updateSyncBlockerRequest
     * @return A Java Future containing the result of the UpdateSyncBlocker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>RetryLatestCommitFailedException Retrying the latest commit failed. Try again later.</li>
     *         <li>SyncBlockerDoesNotExistException Unable to continue. The sync blocker does not exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.UpdateSyncBlocker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateSyncBlocker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSyncBlockerResponse> updateSyncBlocker(UpdateSyncBlockerRequest updateSyncBlockerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSyncBlockerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSyncBlockerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSyncBlocker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSyncBlockerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSyncBlockerResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateSyncBlockerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSyncBlockerRequest, UpdateSyncBlockerResponse>()
                            .withOperationName("UpdateSyncBlocker").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSyncBlockerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSyncBlockerRequest));
            CompletableFuture<UpdateSyncBlockerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the sync configuration for your connection and a specified external Git repository.
     * </p>
     *
     * @param updateSyncConfigurationRequest
     * @return A Java Future containing the result of the UpdateSyncConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConcurrentModificationException Exception thrown as a result of concurrent modification to an
     *         application. For example, two individuals attempting to edit the same application at the same time.</li>
     *         <li>InternalServerException Received an internal server exception. Try again later.</li>
     *         <li>InvalidInputException The input is not valid. Verify that the action is typed correctly.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>UpdateOutOfSyncException The update is out of sync. Try syncing again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CodeConnectionsAsyncClient.UpdateSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSyncConfigurationResponse> updateSyncConfiguration(
            UpdateSyncConfigurationRequest updateSyncConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSyncConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSyncConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConcurrentModificationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
                case "SyncBlockerDoesNotExistException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException")
                            .httpStatusCode(400).exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "UnsupportedOperationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
                case "InvalidInputException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidInputException::builder).build());
                case "UnsupportedProviderTypeException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException")
                            .httpStatusCode(400).exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
                case "ResourceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "SyncConfigurationStillExistsException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                                    .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder)
                                    .build());
                case "UpdateOutOfSyncException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                            .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
                case "RetryLatestCommitFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
                case "ConditionalCheckFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "ResourceAlreadyExistsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                            .httpStatusCode(400).exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateSyncConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSyncConfigurationRequest, UpdateSyncConfigurationResponse>()
                            .withOperationName("UpdateSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSyncConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSyncConfigurationRequest));
            CompletableFuture<UpdateSyncConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CodeConnectionsServiceClientConfiguration serviceClientConfiguration() {
        return new CodeConnectionsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeConnectionsException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        CodeConnectionsServiceClientConfigurationBuilder serviceConfigBuilder = new CodeConnectionsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
