/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeconnections;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.codeconnections.internal.CodeConnectionsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.codeconnections.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.codeconnections.model.AccessDeniedException;
import software.amazon.awssdk.services.codeconnections.model.CodeConnectionsException;
import software.amazon.awssdk.services.codeconnections.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codeconnections.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.codeconnections.model.ConflictException;
import software.amazon.awssdk.services.codeconnections.model.CreateConnectionRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateConnectionResponse;
import software.amazon.awssdk.services.codeconnections.model.CreateHostRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateHostResponse;
import software.amazon.awssdk.services.codeconnections.model.CreateRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.CreateSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.CreateSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteHostRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteHostResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.DeleteSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.DeleteSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.model.GetConnectionRequest;
import software.amazon.awssdk.services.codeconnections.model.GetConnectionResponse;
import software.amazon.awssdk.services.codeconnections.model.GetHostRequest;
import software.amazon.awssdk.services.codeconnections.model.GetHostResponse;
import software.amazon.awssdk.services.codeconnections.model.GetRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.GetRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.GetRepositorySyncStatusRequest;
import software.amazon.awssdk.services.codeconnections.model.GetRepositorySyncStatusResponse;
import software.amazon.awssdk.services.codeconnections.model.GetResourceSyncStatusRequest;
import software.amazon.awssdk.services.codeconnections.model.GetResourceSyncStatusResponse;
import software.amazon.awssdk.services.codeconnections.model.GetSyncBlockerSummaryRequest;
import software.amazon.awssdk.services.codeconnections.model.GetSyncBlockerSummaryResponse;
import software.amazon.awssdk.services.codeconnections.model.GetSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.GetSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.model.InternalServerException;
import software.amazon.awssdk.services.codeconnections.model.InvalidInputException;
import software.amazon.awssdk.services.codeconnections.model.LimitExceededException;
import software.amazon.awssdk.services.codeconnections.model.ListConnectionsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListConnectionsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListHostsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListHostsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListRepositoryLinksRequest;
import software.amazon.awssdk.services.codeconnections.model.ListRepositoryLinksResponse;
import software.amazon.awssdk.services.codeconnections.model.ListRepositorySyncDefinitionsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListRepositorySyncDefinitionsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListSyncConfigurationsRequest;
import software.amazon.awssdk.services.codeconnections.model.ListSyncConfigurationsResponse;
import software.amazon.awssdk.services.codeconnections.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codeconnections.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codeconnections.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codeconnections.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codeconnections.model.ResourceUnavailableException;
import software.amazon.awssdk.services.codeconnections.model.RetryLatestCommitFailedException;
import software.amazon.awssdk.services.codeconnections.model.SyncBlockerDoesNotExistException;
import software.amazon.awssdk.services.codeconnections.model.SyncConfigurationStillExistsException;
import software.amazon.awssdk.services.codeconnections.model.TagResourceRequest;
import software.amazon.awssdk.services.codeconnections.model.TagResourceResponse;
import software.amazon.awssdk.services.codeconnections.model.ThrottlingException;
import software.amazon.awssdk.services.codeconnections.model.UnsupportedOperationException;
import software.amazon.awssdk.services.codeconnections.model.UnsupportedProviderTypeException;
import software.amazon.awssdk.services.codeconnections.model.UntagResourceRequest;
import software.amazon.awssdk.services.codeconnections.model.UntagResourceResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateHostRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateHostResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateOutOfSyncException;
import software.amazon.awssdk.services.codeconnections.model.UpdateRepositoryLinkRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateRepositoryLinkResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncBlockerRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncBlockerResponse;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncConfigurationRequest;
import software.amazon.awssdk.services.codeconnections.model.UpdateSyncConfigurationResponse;
import software.amazon.awssdk.services.codeconnections.transform.CreateConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.CreateHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.CreateRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.CreateSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.DeleteSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetConnectionRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetRepositorySyncStatusRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetResourceSyncStatusRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetSyncBlockerSummaryRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.GetSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListConnectionsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListHostsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListRepositoryLinksRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListRepositorySyncDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListSyncConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateHostRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateRepositoryLinkRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateSyncBlockerRequestMarshaller;
import software.amazon.awssdk.services.codeconnections.transform.UpdateSyncConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CodeConnectionsClient}.
 *
 * @see CodeConnectionsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeConnectionsClient implements CodeConnectionsClient {
    private static final Logger log = Logger.loggerFor(DefaultCodeConnectionsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeConnectionsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "CodeConnections" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a connection that can then be given to other Amazon Web Services services like CodePipeline so that it
     * can access third-party code repositories. The connection is in pending status until the third-party connection
     * handshake is completed from the console.
     * </p>
     *
     * @param createConnectionRequest
     * @return Result of the CreateConnection operation returned by the service.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ResourceUnavailableException
     *         Resource not found. Verify the ARN for the host resource and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateConnectionResponse createConnection(CreateConnectionRequest createConnectionRequest)
            throws LimitExceededException, ResourceNotFoundException, ResourceUnavailableException, AwsServiceException,
            SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateConnectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnection");

            return clientHandler.execute(new ClientExecutionParams<CreateConnectionRequest, CreateConnectionResponse>()
                    .withOperationName("CreateConnection").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a resource that represents the infrastructure where a third-party provider is installed. The host is used
     * when you create connections to an installed third-party provider type, such as GitHub Enterprise Server. You
     * create one host for all connections to that provider.
     * </p>
     * <note>
     * <p>
     * A host created through the CLI or the SDK is in `PENDING` status by default. You can make its status `AVAILABLE`
     * by setting up the host in the console.
     * </p>
     * </note>
     *
     * @param createHostRequest
     * @return Result of the CreateHost operation returned by the service.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.CreateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateHost" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateHostResponse createHost(CreateHostRequest createHostRequest) throws LimitExceededException, AwsServiceException,
            SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHostResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHost");

            return clientHandler.execute(new ClientExecutionParams<CreateHostRequest, CreateHostResponse>()
                    .withOperationName("CreateHost").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createHostRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateHostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a link to a specified external Git repository. A repository link allows Git sync to monitor and sync
     * changes to files in a specified Git repository.
     * </p>
     *
     * @param createRepositoryLinkRequest
     * @return Result of the CreateRepositoryLink operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws ResourceAlreadyExistsException
     *         Unable to create resource. Resource already exists.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.CreateRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRepositoryLinkResponse createRepositoryLink(CreateRepositoryLinkRequest createRepositoryLinkRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            LimitExceededException, ResourceAlreadyExistsException, ThrottlingException, AwsServiceException, SdkClientException,
            CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRepositoryLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRepositoryLink");

            return clientHandler.execute(new ClientExecutionParams<CreateRepositoryLinkRequest, CreateRepositoryLinkResponse>()
                    .withOperationName("CreateRepositoryLink").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createRepositoryLinkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRepositoryLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a sync configuration which allows Amazon Web Services to sync content from a Git repository to update a
     * specified Amazon Web Services resource. Parameters for the sync configuration are determined by the sync type.
     * </p>
     *
     * @param createSyncConfigurationRequest
     * @return Result of the CreateSyncConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws ResourceAlreadyExistsException
     *         Unable to create resource. Resource already exists.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.CreateSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/CreateSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSyncConfigurationResponse createSyncConfiguration(CreateSyncConfigurationRequest createSyncConfigurationRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            LimitExceededException, ResourceAlreadyExistsException, ThrottlingException, AwsServiceException, SdkClientException,
            CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSyncConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSyncConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateSyncConfigurationRequest, CreateSyncConfigurationResponse>()
                            .withOperationName("CreateSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createSyncConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateSyncConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The connection to be deleted.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return Result of the DeleteConnection operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConnectionResponse deleteConnection(DeleteConnectionRequest deleteConnectionRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConnectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnection");

            return clientHandler.execute(new ClientExecutionParams<DeleteConnectionRequest, DeleteConnectionResponse>()
                    .withOperationName("DeleteConnection").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The host to be deleted. Before you delete a host, all connections associated to the host must be deleted.
     * </p>
     * <note>
     * <p>
     * A host cannot be deleted if it is in the VPC_CONFIG_INITIALIZING or VPC_CONFIG_DELETING state.
     * </p>
     * </note>
     *
     * @param deleteHostRequest
     * @return Result of the DeleteHost operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ResourceUnavailableException
     *         Resource not found. Verify the ARN for the host resource and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.DeleteHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteHost" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteHostResponse deleteHost(DeleteHostRequest deleteHostRequest) throws ResourceNotFoundException,
            ResourceUnavailableException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHostResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHost");

            return clientHandler.execute(new ClientExecutionParams<DeleteHostRequest, DeleteHostResponse>()
                    .withOperationName("DeleteHost").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteHostRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteHostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the association between your connection and a specified external Git repository.
     * </p>
     *
     * @param deleteRepositoryLinkRequest
     * @return Result of the DeleteRepositoryLink operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws SyncConfigurationStillExistsException
     *         Unable to continue. The sync blocker still exists.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UnsupportedProviderTypeException
     *         The specified provider type is not supported for connections.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.DeleteRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRepositoryLinkResponse deleteRepositoryLink(DeleteRepositoryLinkRequest deleteRepositoryLinkRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            SyncConfigurationStillExistsException, ResourceNotFoundException, ThrottlingException,
            UnsupportedProviderTypeException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRepositoryLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRepositoryLink");

            return clientHandler.execute(new ClientExecutionParams<DeleteRepositoryLinkRequest, DeleteRepositoryLinkResponse>()
                    .withOperationName("DeleteRepositoryLink").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRepositoryLinkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRepositoryLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the sync configuration for a specified repository and connection.
     * </p>
     *
     * @param deleteSyncConfigurationRequest
     * @return Result of the DeleteSyncConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.DeleteSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/DeleteSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSyncConfigurationResponse deleteSyncConfiguration(DeleteSyncConfigurationRequest deleteSyncConfigurationRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            LimitExceededException, ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSyncConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSyncConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteSyncConfigurationRequest, DeleteSyncConfigurationResponse>()
                            .withOperationName("DeleteSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteSyncConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteSyncConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the connection ARN and details such as status, owner, and provider type.
     * </p>
     *
     * @param getConnectionRequest
     * @return Result of the GetConnection operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ResourceUnavailableException
     *         Resource not found. Verify the ARN for the host resource and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetConnectionResponse getConnection(GetConnectionRequest getConnectionRequest) throws ResourceNotFoundException,
            ResourceUnavailableException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConnectionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConnectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConnectionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConnection");

            return clientHandler.execute(new ClientExecutionParams<GetConnectionRequest, GetConnectionResponse>()
                    .withOperationName("GetConnection").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getConnectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the host ARN and details such as status, provider type, endpoint, and, if applicable, the VPC
     * configuration.
     * </p>
     *
     * @param getHostRequest
     * @return Result of the GetHost operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ResourceUnavailableException
     *         Resource not found. Verify the ARN for the host resource and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetHost" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetHostResponse getHost(GetHostRequest getHostRequest) throws ResourceNotFoundException, ResourceUnavailableException,
            AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetHostResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHost");

            return clientHandler.execute(new ClientExecutionParams<GetHostRequest, GetHostResponse>()
                    .withOperationName("GetHost").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getHostRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetHostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about a repository link. A repository link allows Git sync to monitor and sync changes from files
     * in a specified Git repository.
     * </p>
     *
     * @param getRepositoryLinkRequest
     * @return Result of the GetRepositoryLink operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRepositoryLinkResponse getRepositoryLink(GetRepositoryLinkRequest getRepositoryLinkRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRepositoryLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositoryLink");

            return clientHandler.execute(new ClientExecutionParams<GetRepositoryLinkRequest, GetRepositoryLinkResponse>()
                    .withOperationName("GetRepositoryLink").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRepositoryLinkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRepositoryLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about the sync status for a repository. A repository sync uses Git sync to push and pull changes
     * from your remote repository.
     * </p>
     *
     * @param getRepositorySyncStatusRequest
     * @return Result of the GetRepositorySyncStatus operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetRepositorySyncStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetRepositorySyncStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRepositorySyncStatusResponse getRepositorySyncStatus(GetRepositorySyncStatusRequest getRepositorySyncStatusRequest)
            throws AccessDeniedException, InternalServerException, InvalidInputException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRepositorySyncStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRepositorySyncStatusResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRepositorySyncStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRepositorySyncStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRepositorySyncStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRepositorySyncStatusRequest, GetRepositorySyncStatusResponse>()
                            .withOperationName("GetRepositorySyncStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getRepositorySyncStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRepositorySyncStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the status of the sync with the Git repository for a specific Amazon Web Services resource.
     * </p>
     *
     * @param getResourceSyncStatusRequest
     * @return Result of the GetResourceSyncStatus operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetResourceSyncStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetResourceSyncStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceSyncStatusResponse getResourceSyncStatus(GetResourceSyncStatusRequest getResourceSyncStatusRequest)
            throws AccessDeniedException, InternalServerException, InvalidInputException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceSyncStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceSyncStatusResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceSyncStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSyncStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSyncStatus");

            return clientHandler.execute(new ClientExecutionParams<GetResourceSyncStatusRequest, GetResourceSyncStatusResponse>()
                    .withOperationName("GetResourceSyncStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourceSyncStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceSyncStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the most recent sync blockers.
     * </p>
     *
     * @param getSyncBlockerSummaryRequest
     * @return Result of the GetSyncBlockerSummary operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetSyncBlockerSummary
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetSyncBlockerSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSyncBlockerSummaryResponse getSyncBlockerSummary(GetSyncBlockerSummaryRequest getSyncBlockerSummaryRequest)
            throws AccessDeniedException, InternalServerException, InvalidInputException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSyncBlockerSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSyncBlockerSummaryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSyncBlockerSummaryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSyncBlockerSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSyncBlockerSummary");

            return clientHandler.execute(new ClientExecutionParams<GetSyncBlockerSummaryRequest, GetSyncBlockerSummaryResponse>()
                    .withOperationName("GetSyncBlockerSummary").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getSyncBlockerSummaryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSyncBlockerSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about a sync configuration, including the sync type and resource name. A sync configuration
     * allows the configuration to sync (push and pull) changes from the remote repository for a specified branch in a
     * Git repository.
     * </p>
     *
     * @param getSyncConfigurationRequest
     * @return Result of the GetSyncConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.GetSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/GetSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSyncConfigurationResponse getSyncConfiguration(GetSyncConfigurationRequest getSyncConfigurationRequest)
            throws AccessDeniedException, InternalServerException, InvalidInputException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSyncConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSyncConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetSyncConfigurationRequest, GetSyncConfigurationResponse>()
                    .withOperationName("GetSyncConfiguration").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getSyncConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSyncConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return Result of the ListConnections operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConnectionsResponse listConnections(ListConnectionsRequest listConnectionsRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConnectionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnections");

            return clientHandler.execute(new ClientExecutionParams<ListConnectionsRequest, ListConnectionsResponse>()
                    .withOperationName("ListConnections").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listConnectionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConnectionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the hosts associated with your account.
     * </p>
     *
     * @param listHostsRequest
     * @return Result of the ListHosts operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.ListHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListHosts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListHostsResponse listHosts(ListHostsRequest listHostsRequest) throws AwsServiceException, SdkClientException,
            CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHostsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHostsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHostsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHostsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHosts");

            return clientHandler.execute(new ClientExecutionParams<ListHostsRequest, ListHostsResponse>()
                    .withOperationName("ListHosts").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listHostsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListHostsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the repository links created for connections in your account.
     * </p>
     *
     * @param listRepositoryLinksRequest
     * @return Result of the ListRepositoryLinks operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.ListRepositoryLinks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListRepositoryLinks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRepositoryLinksResponse listRepositoryLinks(ListRepositoryLinksRequest listRepositoryLinksRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRepositoryLinksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRepositoryLinksResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRepositoryLinksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRepositoryLinksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositoryLinks");

            return clientHandler.execute(new ClientExecutionParams<ListRepositoryLinksRequest, ListRepositoryLinksResponse>()
                    .withOperationName("ListRepositoryLinks").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRepositoryLinksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRepositoryLinksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the repository sync definitions for repository links in your account.
     * </p>
     *
     * @param listRepositorySyncDefinitionsRequest
     * @return Result of the ListRepositorySyncDefinitions operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.ListRepositorySyncDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListRepositorySyncDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRepositorySyncDefinitionsResponse listRepositorySyncDefinitions(
            ListRepositorySyncDefinitionsRequest listRepositorySyncDefinitionsRequest) throws AccessDeniedException,
            InternalServerException, InvalidInputException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRepositorySyncDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRepositorySyncDefinitionsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRepositorySyncDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRepositorySyncDefinitionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRepositorySyncDefinitions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRepositorySyncDefinitionsRequest, ListRepositorySyncDefinitionsResponse>()
                            .withOperationName("ListRepositorySyncDefinitions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listRepositorySyncDefinitionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRepositorySyncDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of sync configurations for a specified repository.
     * </p>
     *
     * @param listSyncConfigurationsRequest
     * @return Result of the ListSyncConfigurations operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.ListSyncConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListSyncConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSyncConfigurationsResponse listSyncConfigurations(ListSyncConfigurationsRequest listSyncConfigurationsRequest)
            throws AccessDeniedException, InternalServerException, InvalidInputException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSyncConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSyncConfigurationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSyncConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSyncConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSyncConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSyncConfigurationsRequest, ListSyncConfigurationsResponse>()
                            .withOperationName("ListSyncConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listSyncConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSyncConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the set of key-value pairs (metadata) that are used to manage the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws LimitExceededException
     *         Exceeded the maximum limit for connections.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            LimitExceededException, AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from an Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a specified host with the provided configurations.
     * </p>
     *
     * @param updateHostRequest
     * @return Result of the UpdateHost operation returned by the service.
     * @throws ConflictException
     *         Two conflicting operations have been made on the same resource.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ResourceUnavailableException
     *         Resource not found. Verify the ARN for the host resource and try again.
     * @throws UnsupportedOperationException
     *         The operation is not supported. Check the connection status and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.UpdateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateHost" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateHostResponse updateHost(UpdateHostRequest updateHostRequest) throws ConflictException,
            ResourceNotFoundException, ResourceUnavailableException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateHostResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateHostResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHostRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHostRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHost");

            return clientHandler.execute(new ClientExecutionParams<UpdateHostRequest, UpdateHostResponse>()
                    .withOperationName("UpdateHost").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateHostRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateHostRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the association between your connection and a specified external Git repository. A repository link allows
     * Git sync to monitor and sync changes to files in a specified Git repository.
     * </p>
     *
     * @param updateRepositoryLinkRequest
     * @return Result of the UpdateRepositoryLink operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConditionalCheckFailedException
     *         The conditional check failed. Try again later.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UpdateOutOfSyncException
     *         The update is out of sync. Try syncing again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.UpdateRepositoryLink
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateRepositoryLink"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRepositoryLinkResponse updateRepositoryLink(UpdateRepositoryLinkRequest updateRepositoryLinkRequest)
            throws AccessDeniedException, ConditionalCheckFailedException, InternalServerException, InvalidInputException,
            ResourceNotFoundException, ThrottlingException, UpdateOutOfSyncException, AwsServiceException, SdkClientException,
            CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRepositoryLinkResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRepositoryLinkResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRepositoryLinkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRepositoryLinkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRepositoryLink");

            return clientHandler.execute(new ClientExecutionParams<UpdateRepositoryLinkRequest, UpdateRepositoryLinkResponse>()
                    .withOperationName("UpdateRepositoryLink").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateRepositoryLinkRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRepositoryLinkRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to update the status of a sync blocker, resolving the blocker and allowing syncing to continue.
     * </p>
     *
     * @param updateSyncBlockerRequest
     * @return Result of the UpdateSyncBlocker operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws RetryLatestCommitFailedException
     *         Retrying the latest commit failed. Try again later.
     * @throws SyncBlockerDoesNotExistException
     *         Unable to continue. The sync blocker does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.UpdateSyncBlocker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateSyncBlocker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSyncBlockerResponse updateSyncBlocker(UpdateSyncBlockerRequest updateSyncBlockerRequest)
            throws AccessDeniedException, InternalServerException, InvalidInputException, ResourceNotFoundException,
            RetryLatestCommitFailedException, SyncBlockerDoesNotExistException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSyncBlockerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSyncBlockerResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSyncBlockerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSyncBlockerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSyncBlocker");

            return clientHandler.execute(new ClientExecutionParams<UpdateSyncBlockerRequest, UpdateSyncBlockerResponse>()
                    .withOperationName("UpdateSyncBlocker").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateSyncBlockerRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSyncBlockerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the sync configuration for your connection and a specified external Git repository.
     * </p>
     *
     * @param updateSyncConfigurationRequest
     * @return Result of the UpdateSyncConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConcurrentModificationException
     *         Exception thrown as a result of concurrent modification to an application. For example, two individuals
     *         attempting to edit the same application at the same time.
     * @throws InternalServerException
     *         Received an internal server exception. Try again later.
     * @throws InvalidInputException
     *         The input is not valid. Verify that the action is typed correctly.
     * @throws ResourceNotFoundException
     *         Resource not found. Verify the connection resource ARN and try again.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws UpdateOutOfSyncException
     *         The update is out of sync. Try syncing again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeConnectionsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeConnectionsClient.UpdateSyncConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codeconnections-2023-12-01/UpdateSyncConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSyncConfigurationResponse updateSyncConfiguration(UpdateSyncConfigurationRequest updateSyncConfigurationRequest)
            throws AccessDeniedException, ConcurrentModificationException, InternalServerException, InvalidInputException,
            ResourceNotFoundException, ThrottlingException, UpdateOutOfSyncException, AwsServiceException, SdkClientException,
            CodeConnectionsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSyncConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSyncConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "SyncBlockerDoesNotExistException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncBlockerDoesNotExistException").httpStatusCode(400)
                        .exceptionBuilderSupplier(SyncBlockerDoesNotExistException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "InvalidInputException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidInputException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidInputException::builder).build());
            case "UnsupportedProviderTypeException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedProviderTypeException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedProviderTypeException::builder).build());
            case "ResourceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceUnavailableException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceUnavailableException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "SyncConfigurationStillExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("SyncConfigurationStillExistsException")
                        .httpStatusCode(400).exceptionBuilderSupplier(SyncConfigurationStillExistsException::builder).build());
            case "UpdateOutOfSyncException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UpdateOutOfSyncException").httpStatusCode(400)
                        .exceptionBuilderSupplier(UpdateOutOfSyncException::builder).build());
            case "RetryLatestCommitFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("RetryLatestCommitFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(RetryLatestCommitFailedException::builder).build());
            case "ConditionalCheckFailedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "ResourceAlreadyExistsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSyncConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSyncConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeConnections");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSyncConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSyncConfigurationRequest, UpdateSyncConfigurationResponse>()
                            .withOperationName("UpdateSyncConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateSyncConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSyncConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        CodeConnectionsServiceClientConfigurationBuilder serviceConfigBuilder = new CodeConnectionsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeConnectionsException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0");
    }

    @Override
    public final CodeConnectionsServiceClientConfiguration serviceClientConfiguration() {
        return new CodeConnectionsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
