/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an application revision.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GenericRevisionInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<GenericRevisionInfo.Builder, GenericRevisionInfo> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(GenericRevisionInfo::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> DEPLOYMENT_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("deploymentGroups")
            .getter(getter(GenericRevisionInfo::deploymentGroups))
            .setter(setter(Builder::deploymentGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> FIRST_USED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("firstUsedTime").getter(getter(GenericRevisionInfo::firstUsedTime))
            .setter(setter(Builder::firstUsedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("firstUsedTime").build()).build();

    private static final SdkField<Instant> LAST_USED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUsedTime").getter(getter(GenericRevisionInfo::lastUsedTime)).setter(setter(Builder::lastUsedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUsedTime").build()).build();

    private static final SdkField<Instant> REGISTER_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("registerTime").getter(getter(GenericRevisionInfo::registerTime)).setter(setter(Builder::registerTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("registerTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            DEPLOYMENT_GROUPS_FIELD, FIRST_USED_TIME_FIELD, LAST_USED_TIME_FIELD, REGISTER_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final List<String> deploymentGroups;

    private final Instant firstUsedTime;

    private final Instant lastUsedTime;

    private final Instant registerTime;

    private GenericRevisionInfo(BuilderImpl builder) {
        this.description = builder.description;
        this.deploymentGroups = builder.deploymentGroups;
        this.firstUsedTime = builder.firstUsedTime;
        this.lastUsedTime = builder.lastUsedTime;
        this.registerTime = builder.registerTime;
    }

    /**
     * <p>
     * A comment about the revision.
     * </p>
     * 
     * @return A comment about the revision.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeploymentGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeploymentGroups() {
        return deploymentGroups != null && !(deploymentGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The deployment groups for which this is the current target revision.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeploymentGroups} method.
     * </p>
     * 
     * @return The deployment groups for which this is the current target revision.
     */
    public final List<String> deploymentGroups() {
        return deploymentGroups;
    }

    /**
     * <p>
     * When the revision was first used by CodeDeploy.
     * </p>
     * 
     * @return When the revision was first used by CodeDeploy.
     */
    public final Instant firstUsedTime() {
        return firstUsedTime;
    }

    /**
     * <p>
     * When the revision was last used by CodeDeploy.
     * </p>
     * 
     * @return When the revision was last used by CodeDeploy.
     */
    public final Instant lastUsedTime() {
        return lastUsedTime;
    }

    /**
     * <p>
     * When the revision was registered with CodeDeploy.
     * </p>
     * 
     * @return When the revision was registered with CodeDeploy.
     */
    public final Instant registerTime() {
        return registerTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasDeploymentGroups() ? deploymentGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(firstUsedTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUsedTime());
        hashCode = 31 * hashCode + Objects.hashCode(registerTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GenericRevisionInfo)) {
            return false;
        }
        GenericRevisionInfo other = (GenericRevisionInfo) obj;
        return Objects.equals(description(), other.description()) && hasDeploymentGroups() == other.hasDeploymentGroups()
                && Objects.equals(deploymentGroups(), other.deploymentGroups())
                && Objects.equals(firstUsedTime(), other.firstUsedTime()) && Objects.equals(lastUsedTime(), other.lastUsedTime())
                && Objects.equals(registerTime(), other.registerTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GenericRevisionInfo").add("Description", description())
                .add("DeploymentGroups", hasDeploymentGroups() ? deploymentGroups() : null).add("FirstUsedTime", firstUsedTime())
                .add("LastUsedTime", lastUsedTime()).add("RegisterTime", registerTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "deploymentGroups":
            return Optional.ofNullable(clazz.cast(deploymentGroups()));
        case "firstUsedTime":
            return Optional.ofNullable(clazz.cast(firstUsedTime()));
        case "lastUsedTime":
            return Optional.ofNullable(clazz.cast(lastUsedTime()));
        case "registerTime":
            return Optional.ofNullable(clazz.cast(registerTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GenericRevisionInfo, T> g) {
        return obj -> g.apply((GenericRevisionInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GenericRevisionInfo> {
        /**
         * <p>
         * A comment about the revision.
         * </p>
         * 
         * @param description
         *        A comment about the revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The deployment groups for which this is the current target revision.
         * </p>
         * 
         * @param deploymentGroups
         *        The deployment groups for which this is the current target revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentGroups(Collection<String> deploymentGroups);

        /**
         * <p>
         * The deployment groups for which this is the current target revision.
         * </p>
         * 
         * @param deploymentGroups
         *        The deployment groups for which this is the current target revision.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentGroups(String... deploymentGroups);

        /**
         * <p>
         * When the revision was first used by CodeDeploy.
         * </p>
         * 
         * @param firstUsedTime
         *        When the revision was first used by CodeDeploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder firstUsedTime(Instant firstUsedTime);

        /**
         * <p>
         * When the revision was last used by CodeDeploy.
         * </p>
         * 
         * @param lastUsedTime
         *        When the revision was last used by CodeDeploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUsedTime(Instant lastUsedTime);

        /**
         * <p>
         * When the revision was registered with CodeDeploy.
         * </p>
         * 
         * @param registerTime
         *        When the revision was registered with CodeDeploy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder registerTime(Instant registerTime);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private List<String> deploymentGroups = DefaultSdkAutoConstructList.getInstance();

        private Instant firstUsedTime;

        private Instant lastUsedTime;

        private Instant registerTime;

        private BuilderImpl() {
        }

        private BuilderImpl(GenericRevisionInfo model) {
            description(model.description);
            deploymentGroups(model.deploymentGroups);
            firstUsedTime(model.firstUsedTime);
            lastUsedTime(model.lastUsedTime);
            registerTime(model.registerTime);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getDeploymentGroups() {
            if (deploymentGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return deploymentGroups;
        }

        public final void setDeploymentGroups(Collection<String> deploymentGroups) {
            this.deploymentGroups = DeploymentGroupsListCopier.copy(deploymentGroups);
        }

        @Override
        public final Builder deploymentGroups(Collection<String> deploymentGroups) {
            this.deploymentGroups = DeploymentGroupsListCopier.copy(deploymentGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deploymentGroups(String... deploymentGroups) {
            deploymentGroups(Arrays.asList(deploymentGroups));
            return this;
        }

        public final Instant getFirstUsedTime() {
            return firstUsedTime;
        }

        public final void setFirstUsedTime(Instant firstUsedTime) {
            this.firstUsedTime = firstUsedTime;
        }

        @Override
        public final Builder firstUsedTime(Instant firstUsedTime) {
            this.firstUsedTime = firstUsedTime;
            return this;
        }

        public final Instant getLastUsedTime() {
            return lastUsedTime;
        }

        public final void setLastUsedTime(Instant lastUsedTime) {
            this.lastUsedTime = lastUsedTime;
        }

        @Override
        public final Builder lastUsedTime(Instant lastUsedTime) {
            this.lastUsedTime = lastUsedTime;
            return this;
        }

        public final Instant getRegisterTime() {
            return registerTime;
        }

        public final void setRegisterTime(Instant registerTime) {
            this.registerTime = registerTime;
        }

        @Override
        public final Builder registerTime(Instant registerTime) {
            this.registerTime = registerTime;
            return this;
        }

        @Override
        public GenericRevisionInfo build() {
            return new GenericRevisionInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
