/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codedeploy.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that specifies how traffic is shifted from one version of a Lambda function to another version
 * during an Lambda deployment, or from one Amazon ECS task set to another during an Amazon ECS deployment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TrafficRoutingConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<TrafficRoutingConfig.Builder, TrafficRoutingConfig> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(TrafficRoutingConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<TimeBasedCanary> TIME_BASED_CANARY_FIELD = SdkField
            .<TimeBasedCanary> builder(MarshallingType.SDK_POJO).memberName("timeBasedCanary")
            .getter(getter(TrafficRoutingConfig::timeBasedCanary)).setter(setter(Builder::timeBasedCanary))
            .constructor(TimeBasedCanary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeBasedCanary").build()).build();

    private static final SdkField<TimeBasedLinear> TIME_BASED_LINEAR_FIELD = SdkField
            .<TimeBasedLinear> builder(MarshallingType.SDK_POJO).memberName("timeBasedLinear")
            .getter(getter(TrafficRoutingConfig::timeBasedLinear)).setter(setter(Builder::timeBasedLinear))
            .constructor(TimeBasedLinear::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeBasedLinear").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            TIME_BASED_CANARY_FIELD, TIME_BASED_LINEAR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final TimeBasedCanary timeBasedCanary;

    private final TimeBasedLinear timeBasedLinear;

    private TrafficRoutingConfig(BuilderImpl builder) {
        this.type = builder.type;
        this.timeBasedCanary = builder.timeBasedCanary;
        this.timeBasedLinear = builder.timeBasedLinear;
    }

    /**
     * <p>
     * The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a deployment
     * configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TrafficRoutingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a
     *         deployment configuration.
     * @see TrafficRoutingType
     */
    public final TrafficRoutingType type() {
        return TrafficRoutingType.fromValue(type);
    }

    /**
     * <p>
     * The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a deployment
     * configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TrafficRoutingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a
     *         deployment configuration.
     * @see TrafficRoutingType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * A configuration that shifts traffic from one version of a Lambda function or ECS task set to another in two
     * increments. The original and target Lambda function versions or ECS task sets are specified in the deployment's
     * AppSpec file.
     * </p>
     * 
     * @return A configuration that shifts traffic from one version of a Lambda function or ECS task set to another in
     *         two increments. The original and target Lambda function versions or ECS task sets are specified in the
     *         deployment's AppSpec file.
     */
    public final TimeBasedCanary timeBasedCanary() {
        return timeBasedCanary;
    }

    /**
     * <p>
     * A configuration that shifts traffic from one version of a Lambda function or Amazon ECS task set to another in
     * equal increments, with an equal number of minutes between each increment. The original and target Lambda function
     * versions or Amazon ECS task sets are specified in the deployment's AppSpec file.
     * </p>
     * 
     * @return A configuration that shifts traffic from one version of a Lambda function or Amazon ECS task set to
     *         another in equal increments, with an equal number of minutes between each increment. The original and
     *         target Lambda function versions or Amazon ECS task sets are specified in the deployment's AppSpec file.
     */
    public final TimeBasedLinear timeBasedLinear() {
        return timeBasedLinear;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timeBasedCanary());
        hashCode = 31 * hashCode + Objects.hashCode(timeBasedLinear());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrafficRoutingConfig)) {
            return false;
        }
        TrafficRoutingConfig other = (TrafficRoutingConfig) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(timeBasedCanary(), other.timeBasedCanary())
                && Objects.equals(timeBasedLinear(), other.timeBasedLinear());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TrafficRoutingConfig").add("Type", typeAsString()).add("TimeBasedCanary", timeBasedCanary())
                .add("TimeBasedLinear", timeBasedLinear()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "timeBasedCanary":
            return Optional.ofNullable(clazz.cast(timeBasedCanary()));
        case "timeBasedLinear":
            return Optional.ofNullable(clazz.cast(timeBasedLinear()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("timeBasedCanary", TIME_BASED_CANARY_FIELD);
        map.put("timeBasedLinear", TIME_BASED_LINEAR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TrafficRoutingConfig, T> g) {
        return obj -> g.apply((TrafficRoutingConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TrafficRoutingConfig> {
        /**
         * <p>
         * The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a
         * deployment configuration.
         * </p>
         * 
         * @param type
         *        The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a
         *        deployment configuration.
         * @see TrafficRoutingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrafficRoutingType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a
         * deployment configuration.
         * </p>
         * 
         * @param type
         *        The type of traffic shifting (<code>TimeBasedCanary</code> or <code>TimeBasedLinear</code>) used by a
         *        deployment configuration.
         * @see TrafficRoutingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrafficRoutingType
         */
        Builder type(TrafficRoutingType type);

        /**
         * <p>
         * A configuration that shifts traffic from one version of a Lambda function or ECS task set to another in two
         * increments. The original and target Lambda function versions or ECS task sets are specified in the
         * deployment's AppSpec file.
         * </p>
         * 
         * @param timeBasedCanary
         *        A configuration that shifts traffic from one version of a Lambda function or ECS task set to another
         *        in two increments. The original and target Lambda function versions or ECS task sets are specified in
         *        the deployment's AppSpec file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeBasedCanary(TimeBasedCanary timeBasedCanary);

        /**
         * <p>
         * A configuration that shifts traffic from one version of a Lambda function or ECS task set to another in two
         * increments. The original and target Lambda function versions or ECS task sets are specified in the
         * deployment's AppSpec file.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeBasedCanary.Builder} avoiding the
         * need to create one manually via {@link TimeBasedCanary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeBasedCanary.Builder#build()} is called immediately and its
         * result is passed to {@link #timeBasedCanary(TimeBasedCanary)}.
         * 
         * @param timeBasedCanary
         *        a consumer that will call methods on {@link TimeBasedCanary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeBasedCanary(TimeBasedCanary)
         */
        default Builder timeBasedCanary(Consumer<TimeBasedCanary.Builder> timeBasedCanary) {
            return timeBasedCanary(TimeBasedCanary.builder().applyMutation(timeBasedCanary).build());
        }

        /**
         * <p>
         * A configuration that shifts traffic from one version of a Lambda function or Amazon ECS task set to another
         * in equal increments, with an equal number of minutes between each increment. The original and target Lambda
         * function versions or Amazon ECS task sets are specified in the deployment's AppSpec file.
         * </p>
         * 
         * @param timeBasedLinear
         *        A configuration that shifts traffic from one version of a Lambda function or Amazon ECS task set to
         *        another in equal increments, with an equal number of minutes between each increment. The original and
         *        target Lambda function versions or Amazon ECS task sets are specified in the deployment's AppSpec
         *        file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeBasedLinear(TimeBasedLinear timeBasedLinear);

        /**
         * <p>
         * A configuration that shifts traffic from one version of a Lambda function or Amazon ECS task set to another
         * in equal increments, with an equal number of minutes between each increment. The original and target Lambda
         * function versions or Amazon ECS task sets are specified in the deployment's AppSpec file.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimeBasedLinear.Builder} avoiding the
         * need to create one manually via {@link TimeBasedLinear#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimeBasedLinear.Builder#build()} is called immediately and its
         * result is passed to {@link #timeBasedLinear(TimeBasedLinear)}.
         * 
         * @param timeBasedLinear
         *        a consumer that will call methods on {@link TimeBasedLinear.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeBasedLinear(TimeBasedLinear)
         */
        default Builder timeBasedLinear(Consumer<TimeBasedLinear.Builder> timeBasedLinear) {
            return timeBasedLinear(TimeBasedLinear.builder().applyMutation(timeBasedLinear).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private TimeBasedCanary timeBasedCanary;

        private TimeBasedLinear timeBasedLinear;

        private BuilderImpl() {
        }

        private BuilderImpl(TrafficRoutingConfig model) {
            type(model.type);
            timeBasedCanary(model.timeBasedCanary);
            timeBasedLinear(model.timeBasedLinear);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(TrafficRoutingType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final TimeBasedCanary.Builder getTimeBasedCanary() {
            return timeBasedCanary != null ? timeBasedCanary.toBuilder() : null;
        }

        public final void setTimeBasedCanary(TimeBasedCanary.BuilderImpl timeBasedCanary) {
            this.timeBasedCanary = timeBasedCanary != null ? timeBasedCanary.build() : null;
        }

        @Override
        public final Builder timeBasedCanary(TimeBasedCanary timeBasedCanary) {
            this.timeBasedCanary = timeBasedCanary;
            return this;
        }

        public final TimeBasedLinear.Builder getTimeBasedLinear() {
            return timeBasedLinear != null ? timeBasedLinear.toBuilder() : null;
        }

        public final void setTimeBasedLinear(TimeBasedLinear.BuilderImpl timeBasedLinear) {
            this.timeBasedLinear = timeBasedLinear != null ? timeBasedLinear.build() : null;
        }

        @Override
        public final Builder timeBasedLinear(TimeBasedLinear timeBasedLinear) {
            this.timeBasedLinear = timeBasedLinear;
            return this;
        }

        @Override
        public TrafficRoutingConfig build() {
            return new TrafficRoutingConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
