/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codegurureviewer.model.AccessDeniedException;
import software.amazon.awssdk.services.codegurureviewer.model.AssociateRepositoryRequest;
import software.amazon.awssdk.services.codegurureviewer.model.AssociateRepositoryResponse;
import software.amazon.awssdk.services.codegurureviewer.model.CodeGuruReviewerException;
import software.amazon.awssdk.services.codegurureviewer.model.CodeGuruReviewerRequest;
import software.amazon.awssdk.services.codegurureviewer.model.ConflictException;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewResponse;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRecommendationFeedbackRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRecommendationFeedbackResponse;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationResponse;
import software.amazon.awssdk.services.codegurureviewer.model.DisassociateRepositoryRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DisassociateRepositoryResponse;
import software.amazon.awssdk.services.codegurureviewer.model.InternalServerException;
import software.amazon.awssdk.services.codegurureviewer.model.ListCodeReviewsRequest;
import software.amazon.awssdk.services.codegurureviewer.model.ListCodeReviewsResponse;
import software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationFeedbackRequest;
import software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationFeedbackResponse;
import software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.codegurureviewer.model.ListRepositoryAssociationsRequest;
import software.amazon.awssdk.services.codegurureviewer.model.ListRepositoryAssociationsResponse;
import software.amazon.awssdk.services.codegurureviewer.model.NotFoundException;
import software.amazon.awssdk.services.codegurureviewer.model.PutRecommendationFeedbackRequest;
import software.amazon.awssdk.services.codegurureviewer.model.PutRecommendationFeedbackResponse;
import software.amazon.awssdk.services.codegurureviewer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codegurureviewer.model.ThrottlingException;
import software.amazon.awssdk.services.codegurureviewer.model.ValidationException;
import software.amazon.awssdk.services.codegurureviewer.paginators.ListCodeReviewsIterable;
import software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationFeedbackIterable;
import software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationsIterable;
import software.amazon.awssdk.services.codegurureviewer.paginators.ListRepositoryAssociationsIterable;
import software.amazon.awssdk.services.codegurureviewer.transform.AssociateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.DescribeCodeReviewRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.DescribeRecommendationFeedbackRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.DescribeRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.DisassociateRepositoryRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.ListCodeReviewsRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.ListRecommendationFeedbackRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.ListRepositoryAssociationsRequestMarshaller;
import software.amazon.awssdk.services.codegurureviewer.transform.PutRecommendationFeedbackRequestMarshaller;

/**
 * Internal implementation of {@link CodeGuruReviewerClient}.
 *
 * @see CodeGuruReviewerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeGuruReviewerClient implements CodeGuruReviewerClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeGuruReviewerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates an AWS CodeCommit repository with Amazon CodeGuru Reviewer. When you associate an AWS CodeCommit
     * repository with Amazon CodeGuru Reviewer, Amazon CodeGuru Reviewer will provide recommendations for each pull
     * request raised within the repository. You can view recommendations in the AWS CodeCommit repository.
     * </p>
     * <p>
     * You can associate a GitHub repository using the Amazon CodeGuru Reviewer console.
     * </p>
     *
     * @param associateRepositoryRequest
     * @return Result of the AssociateRepository operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.AssociateRepository
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/AssociateRepository"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateRepositoryResponse associateRepository(AssociateRepositoryRequest associateRepositoryRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateRepositoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AssociateRepositoryRequest, AssociateRepositoryResponse>()
                .withOperationName("AssociateRepository").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(associateRepositoryRequest)
                .withMarshaller(new AssociateRepositoryRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the metadaata associated with the code review along with its status.
     * </p>
     *
     * @param describeCodeReviewRequest
     * @return Result of the DescribeCodeReview operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.DescribeCodeReview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/DescribeCodeReview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCodeReviewResponse describeCodeReview(DescribeCodeReviewRequest describeCodeReviewRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCodeReviewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCodeReviewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeCodeReviewRequest, DescribeCodeReviewResponse>()
                .withOperationName("DescribeCodeReview").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeCodeReviewRequest)
                .withMarshaller(new DescribeCodeReviewRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes the customer feedback for a CodeGuru Reviewer recommendation.
     * </p>
     *
     * @param describeRecommendationFeedbackRequest
     * @return Result of the DescribeRecommendationFeedback operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.DescribeRecommendationFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/DescribeRecommendationFeedback"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRecommendationFeedbackResponse describeRecommendationFeedback(
            DescribeRecommendationFeedbackRequest describeRecommendationFeedbackRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRecommendationFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRecommendationFeedbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeRecommendationFeedbackRequest, DescribeRecommendationFeedbackResponse>()
                        .withOperationName("DescribeRecommendationFeedback").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeRecommendationFeedbackRequest)
                        .withMarshaller(new DescribeRecommendationFeedbackRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Describes a repository association.
     * </p>
     *
     * @param describeRepositoryAssociationRequest
     * @return Result of the DescribeRepositoryAssociation operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.DescribeRepositoryAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/DescribeRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRepositoryAssociationResponse describeRepositoryAssociation(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest) throws NotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRepositoryAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeRepositoryAssociationRequest, DescribeRepositoryAssociationResponse>()
                        .withOperationName("DescribeRepositoryAssociation").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeRepositoryAssociationRequest)
                        .withMarshaller(new DescribeRepositoryAssociationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the association between Amazon CodeGuru Reviewer and a repository.
     * </p>
     *
     * @param disassociateRepositoryRequest
     * @return Result of the DisassociateRepository operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.DisassociateRepository
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/DisassociateRepository"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateRepositoryResponse disassociateRepository(DisassociateRepositoryRequest disassociateRepositoryRequest)
            throws NotFoundException, InternalServerException, ValidationException, AccessDeniedException, ConflictException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateRepositoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateRepositoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisassociateRepositoryRequest, DisassociateRepositoryResponse>()
                .withOperationName("DisassociateRepository").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disassociateRepositoryRequest)
                .withMarshaller(new DisassociateRepositoryRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all the code reviews that the customer has created in the past 90 days.
     * </p>
     *
     * @param listCodeReviewsRequest
     * @return Result of the ListCodeReviews operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListCodeReviews
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListCodeReviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCodeReviewsResponse listCodeReviews(ListCodeReviewsRequest listCodeReviewsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCodeReviewsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCodeReviewsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListCodeReviewsRequest, ListCodeReviewsResponse>()
                .withOperationName("ListCodeReviews").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listCodeReviewsRequest)
                .withMarshaller(new ListCodeReviewsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists all the code reviews that the customer has created in the past 90 days.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCodeReviews(software.amazon.awssdk.services.codegurureviewer.model.ListCodeReviewsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListCodeReviewsIterable responses = client.listCodeReviewsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codegurureviewer.paginators.ListCodeReviewsIterable responses = client
     *             .listCodeReviewsPaginator(request);
     *     for (software.amazon.awssdk.services.codegurureviewer.model.ListCodeReviewsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListCodeReviewsIterable responses = client.listCodeReviewsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCodeReviews(software.amazon.awssdk.services.codegurureviewer.model.ListCodeReviewsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCodeReviewsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListCodeReviews
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListCodeReviews"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCodeReviewsIterable listCodeReviewsPaginator(ListCodeReviewsRequest listCodeReviewsRequest)
            throws InternalServerException, ValidationException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, CodeGuruReviewerException {
        return new ListCodeReviewsIterable(this, applyPaginatorUserAgent(listCodeReviewsRequest));
    }

    /**
     * <p>
     * Lists the customer feedback for a CodeGuru Reviewer recommendation for all users. This API will be used from the
     * console to extract the previously given feedback by the user to pre-populate the feedback emojis for all
     * recommendations.
     * </p>
     *
     * @param listRecommendationFeedbackRequest
     * @return Result of the ListRecommendationFeedback operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListRecommendationFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListRecommendationFeedback"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationFeedbackResponse listRecommendationFeedback(
            ListRecommendationFeedbackRequest listRecommendationFeedbackRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecommendationFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecommendationFeedbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListRecommendationFeedbackRequest, ListRecommendationFeedbackResponse>()
                        .withOperationName("ListRecommendationFeedback").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listRecommendationFeedbackRequest)
                        .withMarshaller(new ListRecommendationFeedbackRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the customer feedback for a CodeGuru Reviewer recommendation for all users. This API will be used from the
     * console to extract the previously given feedback by the user to pre-populate the feedback emojis for all
     * recommendations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRecommendationFeedback(software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationFeedbackRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationFeedbackIterable responses = client.listRecommendationFeedbackPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationFeedbackIterable responses = client
     *             .listRecommendationFeedbackPaginator(request);
     *     for (software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationFeedbackResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationFeedbackIterable responses = client.listRecommendationFeedbackPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecommendationFeedback(software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationFeedbackRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecommendationFeedbackRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListRecommendationFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListRecommendationFeedback"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationFeedbackIterable listRecommendationFeedbackPaginator(
            ListRecommendationFeedbackRequest listRecommendationFeedbackRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeGuruReviewerException {
        return new ListRecommendationFeedbackIterable(this, applyPaginatorUserAgent(listRecommendationFeedbackRequest));
    }

    /**
     * <p>
     * Returns the list of all recommendations for a completed code review.
     * </p>
     *
     * @param listRecommendationsRequest
     * @return Result of the ListRecommendations operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationsResponse listRecommendations(ListRecommendationsRequest listRecommendationsRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                .withOperationName("ListRecommendations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listRecommendationsRequest)
                .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the list of all recommendations for a completed code review.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRecommendations(software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationsIterable responses = client.listRecommendationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationsIterable responses = client
     *             .listRecommendationsPaginator(request);
     *     for (software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListRecommendationsIterable responses = client.listRecommendationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecommendations(software.amazon.awssdk.services.codegurureviewer.model.ListRecommendationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecommendationsIterable listRecommendationsPaginator(ListRecommendationsRequest listRecommendationsRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, CodeGuruReviewerException {
        return new ListRecommendationsIterable(this, applyPaginatorUserAgent(listRecommendationsRequest));
    }

    /**
     * <p>
     * Lists repository associations. You can optionally filter on one or more of the following recommendation
     * properties: provider types, states, names, and owners.
     * </p>
     *
     * @param listRepositoryAssociationsRequest
     * @return Result of the ListRepositoryAssociations operation returned by the service.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListRepositoryAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListRepositoryAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRepositoryAssociationsResponse listRepositoryAssociations(
            ListRepositoryAssociationsRequest listRepositoryAssociationsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRepositoryAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRepositoryAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListRepositoryAssociationsRequest, ListRepositoryAssociationsResponse>()
                        .withOperationName("ListRepositoryAssociations").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listRepositoryAssociationsRequest)
                        .withMarshaller(new ListRepositoryAssociationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists repository associations. You can optionally filter on one or more of the following recommendation
     * properties: provider types, states, names, and owners.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRepositoryAssociations(software.amazon.awssdk.services.codegurureviewer.model.ListRepositoryAssociationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListRepositoryAssociationsIterable responses = client.listRepositoryAssociationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codegurureviewer.paginators.ListRepositoryAssociationsIterable responses = client
     *             .listRepositoryAssociationsPaginator(request);
     *     for (software.amazon.awssdk.services.codegurureviewer.model.ListRepositoryAssociationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codegurureviewer.paginators.ListRepositoryAssociationsIterable responses = client.listRepositoryAssociationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRepositoryAssociations(software.amazon.awssdk.services.codegurureviewer.model.ListRepositoryAssociationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRepositoryAssociationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.ListRepositoryAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/ListRepositoryAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRepositoryAssociationsIterable listRepositoryAssociationsPaginator(
            ListRepositoryAssociationsRequest listRepositoryAssociationsRequest) throws InternalServerException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, CodeGuruReviewerException {
        return new ListRepositoryAssociationsIterable(this, applyPaginatorUserAgent(listRepositoryAssociationsRequest));
    }

    /**
     * <p>
     * Stores customer feedback for a CodeGuru-Reviewer recommendation. When this API is called again with different
     * reactions the previous feedback is overwritten.
     * </p>
     *
     * @param putRecommendationFeedbackRequest
     * @return Result of the PutRecommendationFeedback operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request was not found.
     * @throws InternalServerException
     *         The server encountered an internal error and is unable to complete the request.
     * @throws ValidationException
     *         The input fails to satisfy the specified constraints.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeGuruReviewerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeGuruReviewerClient.PutRecommendationFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codeguru-reviewer-2019-09-19/PutRecommendationFeedback"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutRecommendationFeedbackResponse putRecommendationFeedback(
            PutRecommendationFeedbackRequest putRecommendationFeedbackRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, CodeGuruReviewerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRecommendationFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRecommendationFeedbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<PutRecommendationFeedbackRequest, PutRecommendationFeedbackResponse>()
                        .withOperationName("PutRecommendationFeedback").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(putRecommendationFeedbackRequest)
                        .withMarshaller(new PutRecommendationFeedbackRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeGuruReviewerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CodeGuruReviewerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
