/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the recommendation feedback.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecommendationFeedback implements SdkPojo, Serializable,
        ToCopyableBuilder<RecommendationFeedback.Builder, RecommendationFeedback> {
    private static final SdkField<String> CODE_REVIEW_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CodeReviewArn").getter(getter(RecommendationFeedback::codeReviewArn))
            .setter(setter(Builder::codeReviewArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CodeReviewArn").build()).build();

    private static final SdkField<String> RECOMMENDATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecommendationId").getter(getter(RecommendationFeedback::recommendationId))
            .setter(setter(Builder::recommendationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecommendationId").build()).build();

    private static final SdkField<List<String>> REACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Reactions")
            .getter(getter(RecommendationFeedback::reactionsAsStrings))
            .setter(setter(Builder::reactionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Reactions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> USER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("UserId")
            .getter(getter(RecommendationFeedback::userId)).setter(setter(Builder::userId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserId").build()).build();

    private static final SdkField<Instant> CREATED_TIME_STAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimeStamp").getter(getter(RecommendationFeedback::createdTimeStamp))
            .setter(setter(Builder::createdTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimeStamp").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_STAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTimeStamp").getter(getter(RecommendationFeedback::lastUpdatedTimeStamp))
            .setter(setter(Builder::lastUpdatedTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTimeStamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CODE_REVIEW_ARN_FIELD,
            RECOMMENDATION_ID_FIELD, REACTIONS_FIELD, USER_ID_FIELD, CREATED_TIME_STAMP_FIELD, LAST_UPDATED_TIME_STAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String codeReviewArn;

    private final String recommendationId;

    private final List<String> reactions;

    private final String userId;

    private final Instant createdTimeStamp;

    private final Instant lastUpdatedTimeStamp;

    private RecommendationFeedback(BuilderImpl builder) {
        this.codeReviewArn = builder.codeReviewArn;
        this.recommendationId = builder.recommendationId;
        this.reactions = builder.reactions;
        this.userId = builder.userId;
        this.createdTimeStamp = builder.createdTimeStamp;
        this.lastUpdatedTimeStamp = builder.lastUpdatedTimeStamp;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the <a
     * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CodeReview.html"> <code>CodeReview</code> </a>
     * object.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the <a
     *         href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CodeReview.html">
     *         <code>CodeReview</code> </a> object.
     */
    public String codeReviewArn() {
        return codeReviewArn;
    }

    /**
     * <p>
     * The recommendation ID that can be used to track the provided recommendations. Later on it can be used to collect
     * the feedback.
     * </p>
     * 
     * @return The recommendation ID that can be used to track the provided recommendations. Later on it can be used to
     *         collect the feedback.
     */
    public String recommendationId() {
        return recommendationId;
    }

    /**
     * <p>
     * List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear off all
     * your feedback.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasReactions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear
     *         off all your feedback.
     */
    public List<Reaction> reactions() {
        return ReactionsCopier.copyStringToEnum(reactions);
    }

    /**
     * Returns true if the Reactions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasReactions() {
        return reactions != null && !(reactions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear off all
     * your feedback.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasReactions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear
     *         off all your feedback.
     */
    public List<String> reactionsAsStrings() {
        return reactions;
    }

    /**
     * <p>
     * The ID of the user that made the API call.
     * </p>
     * <p>
     * The <code>UserId</code> is an IAM principal that can be specified as an AWS account ID or an Amazon Resource Name
     * (ARN). For more information, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying"
     * > Specifying a Principal</a> in the <i>AWS Identity and Access Management User Guide</i>.
     * </p>
     * 
     * @return The ID of the user that made the API call. </p>
     *         <p>
     *         The <code>UserId</code> is an IAM principal that can be specified as an AWS account ID or an Amazon
     *         Resource Name (ARN). For more information, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying"
     *         > Specifying a Principal</a> in the <i>AWS Identity and Access Management User Guide</i>.
     */
    public String userId() {
        return userId;
    }

    /**
     * <p>
     * The time at which the feedback was created.
     * </p>
     * 
     * @return The time at which the feedback was created.
     */
    public Instant createdTimeStamp() {
        return createdTimeStamp;
    }

    /**
     * <p>
     * The time at which the feedback was last updated.
     * </p>
     * 
     * @return The time at which the feedback was last updated.
     */
    public Instant lastUpdatedTimeStamp() {
        return lastUpdatedTimeStamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codeReviewArn());
        hashCode = 31 * hashCode + Objects.hashCode(recommendationId());
        hashCode = 31 * hashCode + Objects.hashCode(hasReactions() ? reactionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(userId());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimeStamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTimeStamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecommendationFeedback)) {
            return false;
        }
        RecommendationFeedback other = (RecommendationFeedback) obj;
        return Objects.equals(codeReviewArn(), other.codeReviewArn())
                && Objects.equals(recommendationId(), other.recommendationId()) && hasReactions() == other.hasReactions()
                && Objects.equals(reactionsAsStrings(), other.reactionsAsStrings()) && Objects.equals(userId(), other.userId())
                && Objects.equals(createdTimeStamp(), other.createdTimeStamp())
                && Objects.equals(lastUpdatedTimeStamp(), other.lastUpdatedTimeStamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RecommendationFeedback").add("CodeReviewArn", codeReviewArn())
                .add("RecommendationId", recommendationId()).add("Reactions", hasReactions() ? reactionsAsStrings() : null)
                .add("UserId", userId()).add("CreatedTimeStamp", createdTimeStamp())
                .add("LastUpdatedTimeStamp", lastUpdatedTimeStamp()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CodeReviewArn":
            return Optional.ofNullable(clazz.cast(codeReviewArn()));
        case "RecommendationId":
            return Optional.ofNullable(clazz.cast(recommendationId()));
        case "Reactions":
            return Optional.ofNullable(clazz.cast(reactionsAsStrings()));
        case "UserId":
            return Optional.ofNullable(clazz.cast(userId()));
        case "CreatedTimeStamp":
            return Optional.ofNullable(clazz.cast(createdTimeStamp()));
        case "LastUpdatedTimeStamp":
            return Optional.ofNullable(clazz.cast(lastUpdatedTimeStamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecommendationFeedback, T> g) {
        return obj -> g.apply((RecommendationFeedback) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecommendationFeedback> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the <a
         * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CodeReview.html"> <code>CodeReview</code>
         * </a> object.
         * </p>
         * 
         * @param codeReviewArn
         *        The Amazon Resource Name (ARN) of the <a
         *        href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CodeReview.html">
         *        <code>CodeReview</code> </a> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder codeReviewArn(String codeReviewArn);

        /**
         * <p>
         * The recommendation ID that can be used to track the provided recommendations. Later on it can be used to
         * collect the feedback.
         * </p>
         * 
         * @param recommendationId
         *        The recommendation ID that can be used to track the provided recommendations. Later on it can be used
         *        to collect the feedback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationId(String recommendationId);

        /**
         * <p>
         * List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear off
         * all your feedback.
         * </p>
         * 
         * @param reactions
         *        List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to
         *        clear off all your feedback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reactionsWithStrings(Collection<String> reactions);

        /**
         * <p>
         * List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear off
         * all your feedback.
         * </p>
         * 
         * @param reactions
         *        List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to
         *        clear off all your feedback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reactionsWithStrings(String... reactions);

        /**
         * <p>
         * List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear off
         * all your feedback.
         * </p>
         * 
         * @param reactions
         *        List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to
         *        clear off all your feedback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reactions(Collection<Reaction> reactions);

        /**
         * <p>
         * List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to clear off
         * all your feedback.
         * </p>
         * 
         * @param reactions
         *        List for storing reactions. Reactions are utf-8 text code for emojis. You can send an empty list to
         *        clear off all your feedback.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reactions(Reaction... reactions);

        /**
         * <p>
         * The ID of the user that made the API call.
         * </p>
         * <p>
         * The <code>UserId</code> is an IAM principal that can be specified as an AWS account ID or an Amazon Resource
         * Name (ARN). For more information, see <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying"
         * > Specifying a Principal</a> in the <i>AWS Identity and Access Management User Guide</i>.
         * </p>
         * 
         * @param userId
         *        The ID of the user that made the API call. </p>
         *        <p>
         *        The <code>UserId</code> is an IAM principal that can be specified as an AWS account ID or an Amazon
         *        Resource Name (ARN). For more information, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying"
         *        > Specifying a Principal</a> in the <i>AWS Identity and Access Management User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userId(String userId);

        /**
         * <p>
         * The time at which the feedback was created.
         * </p>
         * 
         * @param createdTimeStamp
         *        The time at which the feedback was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimeStamp(Instant createdTimeStamp);

        /**
         * <p>
         * The time at which the feedback was last updated.
         * </p>
         * 
         * @param lastUpdatedTimeStamp
         *        The time at which the feedback was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTimeStamp(Instant lastUpdatedTimeStamp);
    }

    static final class BuilderImpl implements Builder {
        private String codeReviewArn;

        private String recommendationId;

        private List<String> reactions = DefaultSdkAutoConstructList.getInstance();

        private String userId;

        private Instant createdTimeStamp;

        private Instant lastUpdatedTimeStamp;

        private BuilderImpl() {
        }

        private BuilderImpl(RecommendationFeedback model) {
            codeReviewArn(model.codeReviewArn);
            recommendationId(model.recommendationId);
            reactionsWithStrings(model.reactions);
            userId(model.userId);
            createdTimeStamp(model.createdTimeStamp);
            lastUpdatedTimeStamp(model.lastUpdatedTimeStamp);
        }

        public final String getCodeReviewArn() {
            return codeReviewArn;
        }

        @Override
        public final Builder codeReviewArn(String codeReviewArn) {
            this.codeReviewArn = codeReviewArn;
            return this;
        }

        public final void setCodeReviewArn(String codeReviewArn) {
            this.codeReviewArn = codeReviewArn;
        }

        public final String getRecommendationId() {
            return recommendationId;
        }

        @Override
        public final Builder recommendationId(String recommendationId) {
            this.recommendationId = recommendationId;
            return this;
        }

        public final void setRecommendationId(String recommendationId) {
            this.recommendationId = recommendationId;
        }

        public final Collection<String> getReactions() {
            if (reactions instanceof SdkAutoConstructList) {
                return null;
            }
            return reactions;
        }

        @Override
        public final Builder reactionsWithStrings(Collection<String> reactions) {
            this.reactions = ReactionsCopier.copy(reactions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reactionsWithStrings(String... reactions) {
            reactionsWithStrings(Arrays.asList(reactions));
            return this;
        }

        @Override
        public final Builder reactions(Collection<Reaction> reactions) {
            this.reactions = ReactionsCopier.copyEnumToString(reactions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reactions(Reaction... reactions) {
            reactions(Arrays.asList(reactions));
            return this;
        }

        public final void setReactions(Collection<String> reactions) {
            this.reactions = ReactionsCopier.copy(reactions);
        }

        public final String getUserId() {
            return userId;
        }

        @Override
        public final Builder userId(String userId) {
            this.userId = userId;
            return this;
        }

        public final void setUserId(String userId) {
            this.userId = userId;
        }

        public final Instant getCreatedTimeStamp() {
            return createdTimeStamp;
        }

        @Override
        public final Builder createdTimeStamp(Instant createdTimeStamp) {
            this.createdTimeStamp = createdTimeStamp;
            return this;
        }

        public final void setCreatedTimeStamp(Instant createdTimeStamp) {
            this.createdTimeStamp = createdTimeStamp;
        }

        public final Instant getLastUpdatedTimeStamp() {
            return lastUpdatedTimeStamp;
        }

        @Override
        public final Builder lastUpdatedTimeStamp(Instant lastUpdatedTimeStamp) {
            this.lastUpdatedTimeStamp = lastUpdatedTimeStamp;
            return this;
        }

        public final void setLastUpdatedTimeStamp(Instant lastUpdatedTimeStamp) {
            this.lastUpdatedTimeStamp = lastUpdatedTimeStamp;
        }

        @Override
        public RecommendationFeedback build() {
            return new RecommendationFeedback(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
