/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssociateRepositoryRequest extends CodeGuruReviewerRequest implements
        ToCopyableBuilder<AssociateRepositoryRequest.Builder, AssociateRepositoryRequest> {
    private static final SdkField<Repository> REPOSITORY_FIELD = SdkField.<Repository> builder(MarshallingType.SDK_POJO)
            .memberName("Repository").getter(getter(AssociateRepositoryRequest::repository)).setter(setter(Builder::repository))
            .constructor(Repository::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Repository").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(AssociateRepositoryRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(AssociateRepositoryRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_FIELD,
            CLIENT_REQUEST_TOKEN_FIELD, TAGS_FIELD));

    private final Repository repository;

    private final String clientRequestToken;

    private final Map<String, String> tags;

    private AssociateRepositoryRequest(BuilderImpl builder) {
        super(builder);
        this.repository = builder.repository;
        this.clientRequestToken = builder.clientRequestToken;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The repository to associate.
     * </p>
     * 
     * @return The repository to associate.
     */
    public Repository repository() {
        return repository;
    }

    /**
     * <p>
     * Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository associations
     * if there are failures and retries.
     * </p>
     * 
     * @return Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository
     *         associations if there are failures and retries.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label with two
     * parts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>, or
     * <code>Secret</code>). Tag keys are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>, <code>Production</code>,
     * or a team name). Omitting the tag value is the same as using an empty string. Like tag keys, tag values are case
     * sensitive.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label with
     *         two parts: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>,
     *         or <code>Secret</code>). Tag keys are case sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>,
     *         <code>Production</code>, or a team name). Omitting the tag value is the same as using an empty string.
     *         Like tag keys, tag values are case sensitive.
     *         </p>
     *         </li>
     */
    public Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(repository());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssociateRepositoryRequest)) {
            return false;
        }
        AssociateRepositoryRequest other = (AssociateRepositoryRequest) obj;
        return Objects.equals(repository(), other.repository())
                && Objects.equals(clientRequestToken(), other.clientRequestToken()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AssociateRepositoryRequest").add("Repository", repository())
                .add("ClientRequestToken", clientRequestToken()).add("Tags", hasTags() ? tags() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Repository":
            return Optional.ofNullable(clazz.cast(repository()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssociateRepositoryRequest, T> g) {
        return obj -> g.apply((AssociateRepositoryRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CodeGuruReviewerRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AssociateRepositoryRequest> {
        /**
         * <p>
         * The repository to associate.
         * </p>
         * 
         * @param repository
         *        The repository to associate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repository(Repository repository);

        /**
         * <p>
         * The repository to associate.
         * </p>
         * This is a convenience that creates an instance of the {@link Repository.Builder} avoiding the need to create
         * one manually via {@link Repository#builder()}.
         *
         * When the {@link Consumer} completes, {@link Repository.Builder#build()} is called immediately and its result
         * is passed to {@link #repository(Repository)}.
         * 
         * @param repository
         *        a consumer that will call methods on {@link Repository.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #repository(Repository)
         */
        default Builder repository(Consumer<Repository.Builder> repository) {
            return repository(Repository.builder().applyMutation(repository).build());
        }

        /**
         * <p>
         * Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository
         * associations if there are failures and retries.
         * </p>
         * 
         * @param clientRequestToken
         *        Amazon CodeGuru Reviewer uses this value to prevent the accidental creation of duplicate repository
         *        associations if there are failures and retries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label with two
         * parts:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>, or
         * <code>Secret</code>). Tag keys are case sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>,
         * <code>Production</code>, or a team name). Omitting the tag value is the same as using an empty string. Like
         * tag keys, tag values are case sensitive.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        An array of key-value pairs used to tag an associated repository. A tag is a custom attribute label
         *        with two parts: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A <i>tag key</i> (for example, <code>CostCenter</code>, <code>Environment</code>, <code>Project</code>
         *        , or <code>Secret</code>). Tag keys are case sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An optional field known as a <i>tag value</i> (for example, <code>111122223333</code>,
         *        <code>Production</code>, or a team name). Omitting the tag value is the same as using an empty string.
         *        Like tag keys, tag values are case sensitive.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeGuruReviewerRequest.BuilderImpl implements Builder {
        private Repository repository;

        private String clientRequestToken;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AssociateRepositoryRequest model) {
            super(model);
            repository(model.repository);
            clientRequestToken(model.clientRequestToken);
            tags(model.tags);
        }

        public final Repository.Builder getRepository() {
            return repository != null ? repository.toBuilder() : null;
        }

        @Override
        public final Builder repository(Repository repository) {
            this.repository = repository;
            return this;
        }

        public final void setRepository(Repository.BuilderImpl repository) {
            this.repository = repository != null ? repository.build() : null;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AssociateRepositoryRequest build() {
            return new AssociateRepositoryRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
