/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a repository association. The <a
 * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_DescribeRepositoryAssociation.html">
 * <code>DescribeRepositoryAssociation</code> </a> operation returns a <code>RepositoryAssociation</code> object.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RepositoryAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<RepositoryAssociation.Builder, RepositoryAssociation> {
    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationId").getter(getter(RepositoryAssociation::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationId").build()).build();

    private static final SdkField<String> ASSOCIATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssociationArn").getter(getter(RepositoryAssociation::associationArn))
            .setter(setter(Builder::associationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociationArn").build()).build();

    private static final SdkField<String> CONNECTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionArn").getter(getter(RepositoryAssociation::connectionArn))
            .setter(setter(Builder::connectionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RepositoryAssociation::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Owner")
            .getter(getter(RepositoryAssociation::owner)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<String> PROVIDER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProviderType").getter(getter(RepositoryAssociation::providerTypeAsString))
            .setter(setter(Builder::providerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProviderType").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(RepositoryAssociation::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateReason").getter(getter(RepositoryAssociation::stateReason)).setter(setter(Builder::stateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateReason").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_STAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTimeStamp").getter(getter(RepositoryAssociation::lastUpdatedTimeStamp))
            .setter(setter(Builder::lastUpdatedTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTimeStamp").build())
            .build();

    private static final SdkField<Instant> CREATED_TIME_STAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimeStamp").getter(getter(RepositoryAssociation::createdTimeStamp))
            .setter(setter(Builder::createdTimeStamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimeStamp").build()).build();

    private static final SdkField<KMSKeyDetails> KMS_KEY_DETAILS_FIELD = SdkField
            .<KMSKeyDetails> builder(MarshallingType.SDK_POJO).memberName("KMSKeyDetails")
            .getter(getter(RepositoryAssociation::kmsKeyDetails)).setter(setter(Builder::kmsKeyDetails))
            .constructor(KMSKeyDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KMSKeyDetails").build()).build();

    private static final SdkField<S3RepositoryDetails> S3_REPOSITORY_DETAILS_FIELD = SdkField
            .<S3RepositoryDetails> builder(MarshallingType.SDK_POJO).memberName("S3RepositoryDetails")
            .getter(getter(RepositoryAssociation::s3RepositoryDetails)).setter(setter(Builder::s3RepositoryDetails))
            .constructor(S3RepositoryDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3RepositoryDetails").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSOCIATION_ID_FIELD,
            ASSOCIATION_ARN_FIELD, CONNECTION_ARN_FIELD, NAME_FIELD, OWNER_FIELD, PROVIDER_TYPE_FIELD, STATE_FIELD,
            STATE_REASON_FIELD, LAST_UPDATED_TIME_STAMP_FIELD, CREATED_TIME_STAMP_FIELD, KMS_KEY_DETAILS_FIELD,
            S3_REPOSITORY_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String associationId;

    private final String associationArn;

    private final String connectionArn;

    private final String name;

    private final String owner;

    private final String providerType;

    private final String state;

    private final String stateReason;

    private final Instant lastUpdatedTimeStamp;

    private final Instant createdTimeStamp;

    private final KMSKeyDetails kmsKeyDetails;

    private final S3RepositoryDetails s3RepositoryDetails;

    private RepositoryAssociation(BuilderImpl builder) {
        this.associationId = builder.associationId;
        this.associationArn = builder.associationArn;
        this.connectionArn = builder.connectionArn;
        this.name = builder.name;
        this.owner = builder.owner;
        this.providerType = builder.providerType;
        this.state = builder.state;
        this.stateReason = builder.stateReason;
        this.lastUpdatedTimeStamp = builder.lastUpdatedTimeStamp;
        this.createdTimeStamp = builder.createdTimeStamp;
        this.kmsKeyDetails = builder.kmsKeyDetails;
        this.s3RepositoryDetails = builder.s3RepositoryDetails;
    }

    /**
     * <p>
     * The ID of the repository association.
     * </p>
     * 
     * @return The ID of the repository association.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) identifying the repository association.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) identifying the repository association.
     */
    public final String associationArn() {
        return associationArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Amazon Web Services CodeStar Connections connection. Its format is
     * <code>arn:aws:codestar-connections:region-id:aws-account_id:connection/connection-id</code>. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_Connection.html">
     * <code>Connection</code> </a> in the <i>Amazon Web Services CodeStar Connections API Reference</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an Amazon Web Services CodeStar Connections connection. Its format is
     *         <code>arn:aws:codestar-connections:region-id:aws-account_id:connection/connection-id</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_Connection.html">
     *         <code>Connection</code> </a> in the <i>Amazon Web Services CodeStar Connections API Reference</i>.
     */
    public final String connectionArn() {
        return connectionArn;
    }

    /**
     * <p>
     * The name of the repository.
     * </p>
     * 
     * @return The name of the repository.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The owner of the repository. For an Amazon Web Services CodeCommit repository, this is the Amazon Web Services
     * account ID of the account that owns the repository. For a GitHub, GitHub Enterprise Server, or Bitbucket
     * repository, this is the username for the account that owns the repository. For an S3 repository, it can be the
     * username or Amazon Web Services account ID.
     * </p>
     * 
     * @return The owner of the repository. For an Amazon Web Services CodeCommit repository, this is the Amazon Web
     *         Services account ID of the account that owns the repository. For a GitHub, GitHub Enterprise Server, or
     *         Bitbucket repository, this is the username for the account that owns the repository. For an S3
     *         repository, it can be the username or Amazon Web Services account ID.
     */
    public final String owner() {
        return owner;
    }

    /**
     * <p>
     * The provider type of the repository association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #providerType} will
     * return {@link ProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #providerTypeAsString}.
     * </p>
     * 
     * @return The provider type of the repository association.
     * @see ProviderType
     */
    public final ProviderType providerType() {
        return ProviderType.fromValue(providerType);
    }

    /**
     * <p>
     * The provider type of the repository association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #providerType} will
     * return {@link ProviderType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #providerTypeAsString}.
     * </p>
     * 
     * @return The provider type of the repository association.
     * @see ProviderType
     */
    public final String providerTypeAsString() {
        return providerType;
    }

    /**
     * <p>
     * The state of the repository association.
     * </p>
     * <p>
     * The valid repository association states are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Associated</b>: The repository association is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Associating</b>: CodeGuru Reviewer is:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru Reviewer review.
     * </p>
     * <note>
     * <p>
     * If your repository <code>ProviderType</code> is <code>GitHub</code>, <code>GitHub Enterprise Server</code>, or
     * <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks in your repository to trigger CodeGuru Reviewer
     * reviews. If you delete these webhooks, reviews of code in your repository cannot be triggered.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your repository.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Failed</b>: The repository failed to associate or disassociate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and source code
     * access.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a new
     * association with this repository if you want to review source code in it later. You can control access to code
     * reviews created in an associated repository with tags after it has been disassociated. For more information, see
     * <a href="https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html">Using
     * tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RepositoryAssociationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the repository association.</p>
     *         <p>
     *         The valid repository association states are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Associated</b>: The repository association is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Associating</b>: CodeGuru Reviewer is:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru Reviewer
     *         review.
     *         </p>
     *         <note>
     *         <p>
     *         If your repository <code>ProviderType</code> is <code>GitHub</code>,
     *         <code>GitHub Enterprise Server</code>, or <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks in
     *         your repository to trigger CodeGuru Reviewer reviews. If you delete these webhooks, reviews of code in
     *         your repository cannot be triggered.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your
     *         repository.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Failed</b>: The repository failed to associate or disassociate.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and
     *         source code access.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a new
     *         association with this repository if you want to review source code in it later. You can control access to
     *         code reviews created in an associated repository with tags after it has been disassociated. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html"
     *         >Using tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User
     *         Guide</i>.
     *         </p>
     *         </li>
     * @see RepositoryAssociationState
     */
    public final RepositoryAssociationState state() {
        return RepositoryAssociationState.fromValue(state);
    }

    /**
     * <p>
     * The state of the repository association.
     * </p>
     * <p>
     * The valid repository association states are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Associated</b>: The repository association is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Associating</b>: CodeGuru Reviewer is:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru Reviewer review.
     * </p>
     * <note>
     * <p>
     * If your repository <code>ProviderType</code> is <code>GitHub</code>, <code>GitHub Enterprise Server</code>, or
     * <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks in your repository to trigger CodeGuru Reviewer
     * reviews. If you delete these webhooks, reviews of code in your repository cannot be triggered.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your repository.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Failed</b>: The repository failed to associate or disassociate.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and source code
     * access.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a new
     * association with this repository if you want to review source code in it later. You can control access to code
     * reviews created in an associated repository with tags after it has been disassociated. For more information, see
     * <a href="https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html">Using
     * tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RepositoryAssociationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the repository association.</p>
     *         <p>
     *         The valid repository association states are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Associated</b>: The repository association is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Associating</b>: CodeGuru Reviewer is:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru Reviewer
     *         review.
     *         </p>
     *         <note>
     *         <p>
     *         If your repository <code>ProviderType</code> is <code>GitHub</code>,
     *         <code>GitHub Enterprise Server</code>, or <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks in
     *         your repository to trigger CodeGuru Reviewer reviews. If you delete these webhooks, reviews of code in
     *         your repository cannot be triggered.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your
     *         repository.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Failed</b>: The repository failed to associate or disassociate.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and
     *         source code access.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a new
     *         association with this repository if you want to review source code in it later. You can control access to
     *         code reviews created in an associated repository with tags after it has been disassociated. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html"
     *         >Using tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User
     *         Guide</i>.
     *         </p>
     *         </li>
     * @see RepositoryAssociationState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A description of why the repository association is in the current state.
     * </p>
     * 
     * @return A description of why the repository association is in the current state.
     */
    public final String stateReason() {
        return stateReason;
    }

    /**
     * <p>
     * The time, in milliseconds since the epoch, when the repository association was last updated.
     * </p>
     * 
     * @return The time, in milliseconds since the epoch, when the repository association was last updated.
     */
    public final Instant lastUpdatedTimeStamp() {
        return lastUpdatedTimeStamp;
    }

    /**
     * <p>
     * The time, in milliseconds since the epoch, when the repository association was created.
     * </p>
     * 
     * @return The time, in milliseconds since the epoch, when the repository association was created.
     */
    public final Instant createdTimeStamp() {
        return createdTimeStamp;
    }

    /**
     * <p>
     * A <code>KMSKeyDetails</code> object that contains:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The encryption option for this repository association. It is either owned by Amazon Web Services Key Management
     * Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (<code>CUSTOMER_MANAGED_CMK</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The ID of the Amazon Web Services KMS key that is associated with this respository association.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A <code>KMSKeyDetails</code> object that contains:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The encryption option for this repository association. It is either owned by Amazon Web Services Key
     *         Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (
     *         <code>CUSTOMER_MANAGED_CMK</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The ID of the Amazon Web Services KMS key that is associated with this respository association.
     *         </p>
     *         </li>
     */
    public final KMSKeyDetails kmsKeyDetails() {
        return kmsKeyDetails;
    }

    /**
     * Returns the value of the S3RepositoryDetails property for this object.
     * 
     * @return The value of the S3RepositoryDetails property for this object.
     */
    public final S3RepositoryDetails s3RepositoryDetails() {
        return s3RepositoryDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(associationArn());
        hashCode = 31 * hashCode + Objects.hashCode(connectionArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(providerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateReason());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTimeStamp());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimeStamp());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyDetails());
        hashCode = 31 * hashCode + Objects.hashCode(s3RepositoryDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RepositoryAssociation)) {
            return false;
        }
        RepositoryAssociation other = (RepositoryAssociation) obj;
        return Objects.equals(associationId(), other.associationId()) && Objects.equals(associationArn(), other.associationArn())
                && Objects.equals(connectionArn(), other.connectionArn()) && Objects.equals(name(), other.name())
                && Objects.equals(owner(), other.owner()) && Objects.equals(providerTypeAsString(), other.providerTypeAsString())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(stateReason(), other.stateReason())
                && Objects.equals(lastUpdatedTimeStamp(), other.lastUpdatedTimeStamp())
                && Objects.equals(createdTimeStamp(), other.createdTimeStamp())
                && Objects.equals(kmsKeyDetails(), other.kmsKeyDetails())
                && Objects.equals(s3RepositoryDetails(), other.s3RepositoryDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RepositoryAssociation").add("AssociationId", associationId())
                .add("AssociationArn", associationArn()).add("ConnectionArn", connectionArn()).add("Name", name())
                .add("Owner", owner()).add("ProviderType", providerTypeAsString()).add("State", stateAsString())
                .add("StateReason", stateReason()).add("LastUpdatedTimeStamp", lastUpdatedTimeStamp())
                .add("CreatedTimeStamp", createdTimeStamp()).add("KMSKeyDetails", kmsKeyDetails())
                .add("S3RepositoryDetails", s3RepositoryDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "AssociationArn":
            return Optional.ofNullable(clazz.cast(associationArn()));
        case "ConnectionArn":
            return Optional.ofNullable(clazz.cast(connectionArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "ProviderType":
            return Optional.ofNullable(clazz.cast(providerTypeAsString()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateReason":
            return Optional.ofNullable(clazz.cast(stateReason()));
        case "LastUpdatedTimeStamp":
            return Optional.ofNullable(clazz.cast(lastUpdatedTimeStamp()));
        case "CreatedTimeStamp":
            return Optional.ofNullable(clazz.cast(createdTimeStamp()));
        case "KMSKeyDetails":
            return Optional.ofNullable(clazz.cast(kmsKeyDetails()));
        case "S3RepositoryDetails":
            return Optional.ofNullable(clazz.cast(s3RepositoryDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RepositoryAssociation, T> g) {
        return obj -> g.apply((RepositoryAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RepositoryAssociation> {
        /**
         * <p>
         * The ID of the repository association.
         * </p>
         * 
         * @param associationId
         *        The ID of the repository association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) identifying the repository association.
         * </p>
         * 
         * @param associationArn
         *        The Amazon Resource Name (ARN) identifying the repository association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationArn(String associationArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Amazon Web Services CodeStar Connections connection. Its format is
         * <code>arn:aws:codestar-connections:region-id:aws-account_id:connection/connection-id</code>. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_Connection.html">
         * <code>Connection</code> </a> in the <i>Amazon Web Services CodeStar Connections API Reference</i>.
         * </p>
         * 
         * @param connectionArn
         *        The Amazon Resource Name (ARN) of an Amazon Web Services CodeStar Connections connection. Its format
         *        is <code>arn:aws:codestar-connections:region-id:aws-account_id:connection/connection-id</code>. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_Connection.html">
         *        <code>Connection</code> </a> in the <i>Amazon Web Services CodeStar Connections API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionArn(String connectionArn);

        /**
         * <p>
         * The name of the repository.
         * </p>
         * 
         * @param name
         *        The name of the repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The owner of the repository. For an Amazon Web Services CodeCommit repository, this is the Amazon Web
         * Services account ID of the account that owns the repository. For a GitHub, GitHub Enterprise Server, or
         * Bitbucket repository, this is the username for the account that owns the repository. For an S3 repository, it
         * can be the username or Amazon Web Services account ID.
         * </p>
         * 
         * @param owner
         *        The owner of the repository. For an Amazon Web Services CodeCommit repository, this is the Amazon Web
         *        Services account ID of the account that owns the repository. For a GitHub, GitHub Enterprise Server,
         *        or Bitbucket repository, this is the username for the account that owns the repository. For an S3
         *        repository, it can be the username or Amazon Web Services account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(String owner);

        /**
         * <p>
         * The provider type of the repository association.
         * </p>
         * 
         * @param providerType
         *        The provider type of the repository association.
         * @see ProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProviderType
         */
        Builder providerType(String providerType);

        /**
         * <p>
         * The provider type of the repository association.
         * </p>
         * 
         * @param providerType
         *        The provider type of the repository association.
         * @see ProviderType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProviderType
         */
        Builder providerType(ProviderType providerType);

        /**
         * <p>
         * The state of the repository association.
         * </p>
         * <p>
         * The valid repository association states are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Associated</b>: The repository association is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Associating</b>: CodeGuru Reviewer is:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru Reviewer
         * review.
         * </p>
         * <note>
         * <p>
         * If your repository <code>ProviderType</code> is <code>GitHub</code>, <code>GitHub Enterprise Server</code>,
         * or <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks in your repository to trigger CodeGuru Reviewer
         * reviews. If you delete these webhooks, reviews of code in your repository cannot be triggered.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your
         * repository.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Failed</b>: The repository failed to associate or disassociate.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and source
         * code access.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a new
         * association with this repository if you want to review source code in it later. You can control access to
         * code reviews created in an associated repository with tags after it has been disassociated. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html">Using
         * tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the repository association.</p>
         *        <p>
         *        The valid repository association states are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Associated</b>: The repository association is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Associating</b>: CodeGuru Reviewer is:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru
         *        Reviewer review.
         *        </p>
         *        <note>
         *        <p>
         *        If your repository <code>ProviderType</code> is <code>GitHub</code>,
         *        <code>GitHub Enterprise Server</code>, or <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks
         *        in your repository to trigger CodeGuru Reviewer reviews. If you delete these webhooks, reviews of code
         *        in your repository cannot be triggered.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your
         *        repository.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Failed</b>: The repository failed to associate or disassociate.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and
         *        source code access.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a
         *        new association with this repository if you want to review source code in it later. You can control
         *        access to code reviews created in an associated repository with tags after it has been disassociated.
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html"
         *        >Using tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User
         *        Guide</i>.
         *        </p>
         *        </li>
         * @see RepositoryAssociationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RepositoryAssociationState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the repository association.
         * </p>
         * <p>
         * The valid repository association states are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Associated</b>: The repository association is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Associating</b>: CodeGuru Reviewer is:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru Reviewer
         * review.
         * </p>
         * <note>
         * <p>
         * If your repository <code>ProviderType</code> is <code>GitHub</code>, <code>GitHub Enterprise Server</code>,
         * or <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks in your repository to trigger CodeGuru Reviewer
         * reviews. If you delete these webhooks, reviews of code in your repository cannot be triggered.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your
         * repository.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Failed</b>: The repository failed to associate or disassociate.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and source
         * code access.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a new
         * association with this repository if you want to review source code in it later. You can control access to
         * code reviews created in an associated repository with tags after it has been disassociated. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html">Using
         * tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User Guide</i>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the repository association.</p>
         *        <p>
         *        The valid repository association states are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Associated</b>: The repository association is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Associating</b>: CodeGuru Reviewer is:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Setting up pull request notifications. This is required for pull requests to trigger a CodeGuru
         *        Reviewer review.
         *        </p>
         *        <note>
         *        <p>
         *        If your repository <code>ProviderType</code> is <code>GitHub</code>,
         *        <code>GitHub Enterprise Server</code>, or <code>Bitbucket</code>, CodeGuru Reviewer creates webhooks
         *        in your repository to trigger CodeGuru Reviewer reviews. If you delete these webhooks, reviews of code
         *        in your repository cannot be triggered.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        Setting up source code access. This is required for CodeGuru Reviewer to securely clone code in your
         *        repository.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Failed</b>: The repository failed to associate or disassociate.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Disassociating</b>: CodeGuru Reviewer is removing the repository's pull request notifications and
         *        source code access.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Disassociated</b>: CodeGuru Reviewer successfully disassociated the repository. You can create a
         *        new association with this repository if you want to review source code in it later. You can control
         *        access to code reviews created in an associated repository with tags after it has been disassociated.
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/codeguru/latest/reviewer-ug/auth-and-access-control-using-tags.html"
         *        >Using tags to control access to associated repositories</a> in the <i>Amazon CodeGuru Reviewer User
         *        Guide</i>.
         *        </p>
         *        </li>
         * @see RepositoryAssociationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RepositoryAssociationState
         */
        Builder state(RepositoryAssociationState state);

        /**
         * <p>
         * A description of why the repository association is in the current state.
         * </p>
         * 
         * @param stateReason
         *        A description of why the repository association is in the current state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReason(String stateReason);

        /**
         * <p>
         * The time, in milliseconds since the epoch, when the repository association was last updated.
         * </p>
         * 
         * @param lastUpdatedTimeStamp
         *        The time, in milliseconds since the epoch, when the repository association was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTimeStamp(Instant lastUpdatedTimeStamp);

        /**
         * <p>
         * The time, in milliseconds since the epoch, when the repository association was created.
         * </p>
         * 
         * @param createdTimeStamp
         *        The time, in milliseconds since the epoch, when the repository association was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimeStamp(Instant createdTimeStamp);

        /**
         * <p>
         * A <code>KMSKeyDetails</code> object that contains:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The encryption option for this repository association. It is either owned by Amazon Web Services Key
         * Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (<code>CUSTOMER_MANAGED_CMK</code>
         * ).
         * </p>
         * </li>
         * <li>
         * <p>
         * The ID of the Amazon Web Services KMS key that is associated with this respository association.
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyDetails
         *        A <code>KMSKeyDetails</code> object that contains:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The encryption option for this repository association. It is either owned by Amazon Web Services Key
         *        Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (
         *        <code>CUSTOMER_MANAGED_CMK</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The ID of the Amazon Web Services KMS key that is associated with this respository association.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyDetails(KMSKeyDetails kmsKeyDetails);

        /**
         * <p>
         * A <code>KMSKeyDetails</code> object that contains:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The encryption option for this repository association. It is either owned by Amazon Web Services Key
         * Management Service (KMS) (<code>AWS_OWNED_CMK</code>) or customer managed (<code>CUSTOMER_MANAGED_CMK</code>
         * ).
         * </p>
         * </li>
         * <li>
         * <p>
         * The ID of the Amazon Web Services KMS key that is associated with this respository association.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link KMSKeyDetails.Builder} avoiding the need
         * to create one manually via {@link KMSKeyDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link KMSKeyDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #kmsKeyDetails(KMSKeyDetails)}.
         * 
         * @param kmsKeyDetails
         *        a consumer that will call methods on {@link KMSKeyDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kmsKeyDetails(KMSKeyDetails)
         */
        default Builder kmsKeyDetails(Consumer<KMSKeyDetails.Builder> kmsKeyDetails) {
            return kmsKeyDetails(KMSKeyDetails.builder().applyMutation(kmsKeyDetails).build());
        }

        /**
         * Sets the value of the S3RepositoryDetails property for this object.
         *
         * @param s3RepositoryDetails
         *        The new value for the S3RepositoryDetails property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3RepositoryDetails(S3RepositoryDetails s3RepositoryDetails);

        /**
         * Sets the value of the S3RepositoryDetails property for this object.
         *
         * This is a convenience method that creates an instance of the {@link S3RepositoryDetails.Builder} avoiding the
         * need to create one manually via {@link S3RepositoryDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3RepositoryDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #s3RepositoryDetails(S3RepositoryDetails)}.
         * 
         * @param s3RepositoryDetails
         *        a consumer that will call methods on {@link S3RepositoryDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3RepositoryDetails(S3RepositoryDetails)
         */
        default Builder s3RepositoryDetails(Consumer<S3RepositoryDetails.Builder> s3RepositoryDetails) {
            return s3RepositoryDetails(S3RepositoryDetails.builder().applyMutation(s3RepositoryDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String associationId;

        private String associationArn;

        private String connectionArn;

        private String name;

        private String owner;

        private String providerType;

        private String state;

        private String stateReason;

        private Instant lastUpdatedTimeStamp;

        private Instant createdTimeStamp;

        private KMSKeyDetails kmsKeyDetails;

        private S3RepositoryDetails s3RepositoryDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(RepositoryAssociation model) {
            associationId(model.associationId);
            associationArn(model.associationArn);
            connectionArn(model.connectionArn);
            name(model.name);
            owner(model.owner);
            providerType(model.providerType);
            state(model.state);
            stateReason(model.stateReason);
            lastUpdatedTimeStamp(model.lastUpdatedTimeStamp);
            createdTimeStamp(model.createdTimeStamp);
            kmsKeyDetails(model.kmsKeyDetails);
            s3RepositoryDetails(model.s3RepositoryDetails);
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        @Transient
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final String getAssociationArn() {
            return associationArn;
        }

        public final void setAssociationArn(String associationArn) {
            this.associationArn = associationArn;
        }

        @Override
        @Transient
        public final Builder associationArn(String associationArn) {
            this.associationArn = associationArn;
            return this;
        }

        public final String getConnectionArn() {
            return connectionArn;
        }

        public final void setConnectionArn(String connectionArn) {
            this.connectionArn = connectionArn;
        }

        @Override
        @Transient
        public final Builder connectionArn(String connectionArn) {
            this.connectionArn = connectionArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        @Transient
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        public final String getProviderType() {
            return providerType;
        }

        public final void setProviderType(String providerType) {
            this.providerType = providerType;
        }

        @Override
        @Transient
        public final Builder providerType(String providerType) {
            this.providerType = providerType;
            return this;
        }

        @Override
        @Transient
        public final Builder providerType(ProviderType providerType) {
            this.providerType(providerType == null ? null : providerType.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(RepositoryAssociationState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateReason() {
            return stateReason;
        }

        public final void setStateReason(String stateReason) {
            this.stateReason = stateReason;
        }

        @Override
        @Transient
        public final Builder stateReason(String stateReason) {
            this.stateReason = stateReason;
            return this;
        }

        public final Instant getLastUpdatedTimeStamp() {
            return lastUpdatedTimeStamp;
        }

        public final void setLastUpdatedTimeStamp(Instant lastUpdatedTimeStamp) {
            this.lastUpdatedTimeStamp = lastUpdatedTimeStamp;
        }

        @Override
        @Transient
        public final Builder lastUpdatedTimeStamp(Instant lastUpdatedTimeStamp) {
            this.lastUpdatedTimeStamp = lastUpdatedTimeStamp;
            return this;
        }

        public final Instant getCreatedTimeStamp() {
            return createdTimeStamp;
        }

        public final void setCreatedTimeStamp(Instant createdTimeStamp) {
            this.createdTimeStamp = createdTimeStamp;
        }

        @Override
        @Transient
        public final Builder createdTimeStamp(Instant createdTimeStamp) {
            this.createdTimeStamp = createdTimeStamp;
            return this;
        }

        public final KMSKeyDetails.Builder getKmsKeyDetails() {
            return kmsKeyDetails != null ? kmsKeyDetails.toBuilder() : null;
        }

        public final void setKmsKeyDetails(KMSKeyDetails.BuilderImpl kmsKeyDetails) {
            this.kmsKeyDetails = kmsKeyDetails != null ? kmsKeyDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder kmsKeyDetails(KMSKeyDetails kmsKeyDetails) {
            this.kmsKeyDetails = kmsKeyDetails;
            return this;
        }

        public final S3RepositoryDetails.Builder getS3RepositoryDetails() {
            return s3RepositoryDetails != null ? s3RepositoryDetails.toBuilder() : null;
        }

        public final void setS3RepositoryDetails(S3RepositoryDetails.BuilderImpl s3RepositoryDetails) {
            this.s3RepositoryDetails = s3RepositoryDetails != null ? s3RepositoryDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder s3RepositoryDetails(S3RepositoryDetails s3RepositoryDetails) {
            this.s3RepositoryDetails = s3RepositoryDetails;
            return this;
        }

        @Override
        public RepositoryAssociation build() {
            return new RepositoryAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
