/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The type of a code review. There are two code review types:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>PullRequest</code> - A code review that is automatically triggered by a pull request on an associated
 * repository.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>RepositoryAnalysis</code> - A code review that analyzes all code under a specified branch in an associated
 * repository. The associated repository is specified using its ARN in <a
 * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CreateCodeReview"> <code>CreateCodeReview</code>
 * </a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CodeReviewType implements SdkPojo, Serializable, ToCopyableBuilder<CodeReviewType.Builder, CodeReviewType> {
    private static final SdkField<RepositoryAnalysis> REPOSITORY_ANALYSIS_FIELD = SdkField
            .<RepositoryAnalysis> builder(MarshallingType.SDK_POJO).memberName("RepositoryAnalysis")
            .getter(getter(CodeReviewType::repositoryAnalysis)).setter(setter(Builder::repositoryAnalysis))
            .constructor(RepositoryAnalysis::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RepositoryAnalysis").build())
            .build();

    private static final SdkField<List<String>> ANALYSIS_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AnalysisTypes")
            .getter(getter(CodeReviewType::analysisTypesAsStrings))
            .setter(setter(Builder::analysisTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalysisTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_ANALYSIS_FIELD,
            ANALYSIS_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final RepositoryAnalysis repositoryAnalysis;

    private final List<String> analysisTypes;

    private CodeReviewType(BuilderImpl builder) {
        this.repositoryAnalysis = builder.repositoryAnalysis;
        this.analysisTypes = builder.analysisTypes;
    }

    /**
     * <p>
     * A code review that analyzes all code under a specified branch in an associated repository. The associated
     * repository is specified using its ARN in <a
     * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CreateCodeReview">
     * <code>CreateCodeReview</code> </a>.
     * </p>
     * 
     * @return A code review that analyzes all code under a specified branch in an associated repository. The associated
     *         repository is specified using its ARN in <a
     *         href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CreateCodeReview">
     *         <code>CreateCodeReview</code> </a>.
     */
    public final RepositoryAnalysis repositoryAnalysis() {
        return repositoryAnalysis;
    }

    /**
     * <p>
     * They types of analysis performed during a repository analysis or a pull request review. You can specify either
     * <code>Security</code>, <code>CodeQuality</code>, or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAnalysisTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return They types of analysis performed during a repository analysis or a pull request review. You can specify
     *         either <code>Security</code>, <code>CodeQuality</code>, or both.
     */
    public final List<AnalysisType> analysisTypes() {
        return AnalysisTypesCopier.copyStringToEnum(analysisTypes);
    }

    /**
     * Returns true if the AnalysisTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAnalysisTypes() {
        return analysisTypes != null && !(analysisTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * They types of analysis performed during a repository analysis or a pull request review. You can specify either
     * <code>Security</code>, <code>CodeQuality</code>, or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAnalysisTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return They types of analysis performed during a repository analysis or a pull request review. You can specify
     *         either <code>Security</code>, <code>CodeQuality</code>, or both.
     */
    public final List<String> analysisTypesAsStrings() {
        return analysisTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(repositoryAnalysis());
        hashCode = 31 * hashCode + Objects.hashCode(hasAnalysisTypes() ? analysisTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CodeReviewType)) {
            return false;
        }
        CodeReviewType other = (CodeReviewType) obj;
        return Objects.equals(repositoryAnalysis(), other.repositoryAnalysis()) && hasAnalysisTypes() == other.hasAnalysisTypes()
                && Objects.equals(analysisTypesAsStrings(), other.analysisTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CodeReviewType").add("RepositoryAnalysis", repositoryAnalysis())
                .add("AnalysisTypes", hasAnalysisTypes() ? analysisTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RepositoryAnalysis":
            return Optional.ofNullable(clazz.cast(repositoryAnalysis()));
        case "AnalysisTypes":
            return Optional.ofNullable(clazz.cast(analysisTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CodeReviewType, T> g) {
        return obj -> g.apply((CodeReviewType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CodeReviewType> {
        /**
         * <p>
         * A code review that analyzes all code under a specified branch in an associated repository. The associated
         * repository is specified using its ARN in <a
         * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CreateCodeReview">
         * <code>CreateCodeReview</code> </a>.
         * </p>
         * 
         * @param repositoryAnalysis
         *        A code review that analyzes all code under a specified branch in an associated repository. The
         *        associated repository is specified using its ARN in <a
         *        href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CreateCodeReview">
         *        <code>CreateCodeReview</code> </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryAnalysis(RepositoryAnalysis repositoryAnalysis);

        /**
         * <p>
         * A code review that analyzes all code under a specified branch in an associated repository. The associated
         * repository is specified using its ARN in <a
         * href="https://docs.aws.amazon.com/codeguru/latest/reviewer-api/API_CreateCodeReview">
         * <code>CreateCodeReview</code> </a>.
         * </p>
         * This is a convenience that creates an instance of the {@link RepositoryAnalysis.Builder} avoiding the need to
         * create one manually via {@link RepositoryAnalysis#builder()}.
         *
         * When the {@link Consumer} completes, {@link RepositoryAnalysis.Builder#build()} is called immediately and its
         * result is passed to {@link #repositoryAnalysis(RepositoryAnalysis)}.
         * 
         * @param repositoryAnalysis
         *        a consumer that will call methods on {@link RepositoryAnalysis.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #repositoryAnalysis(RepositoryAnalysis)
         */
        default Builder repositoryAnalysis(Consumer<RepositoryAnalysis.Builder> repositoryAnalysis) {
            return repositoryAnalysis(RepositoryAnalysis.builder().applyMutation(repositoryAnalysis).build());
        }

        /**
         * <p>
         * They types of analysis performed during a repository analysis or a pull request review. You can specify
         * either <code>Security</code>, <code>CodeQuality</code>, or both.
         * </p>
         * 
         * @param analysisTypes
         *        They types of analysis performed during a repository analysis or a pull request review. You can
         *        specify either <code>Security</code>, <code>CodeQuality</code>, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisTypesWithStrings(Collection<String> analysisTypes);

        /**
         * <p>
         * They types of analysis performed during a repository analysis or a pull request review. You can specify
         * either <code>Security</code>, <code>CodeQuality</code>, or both.
         * </p>
         * 
         * @param analysisTypes
         *        They types of analysis performed during a repository analysis or a pull request review. You can
         *        specify either <code>Security</code>, <code>CodeQuality</code>, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisTypesWithStrings(String... analysisTypes);

        /**
         * <p>
         * They types of analysis performed during a repository analysis or a pull request review. You can specify
         * either <code>Security</code>, <code>CodeQuality</code>, or both.
         * </p>
         * 
         * @param analysisTypes
         *        They types of analysis performed during a repository analysis or a pull request review. You can
         *        specify either <code>Security</code>, <code>CodeQuality</code>, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisTypes(Collection<AnalysisType> analysisTypes);

        /**
         * <p>
         * They types of analysis performed during a repository analysis or a pull request review. You can specify
         * either <code>Security</code>, <code>CodeQuality</code>, or both.
         * </p>
         * 
         * @param analysisTypes
         *        They types of analysis performed during a repository analysis or a pull request review. You can
         *        specify either <code>Security</code>, <code>CodeQuality</code>, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisTypes(AnalysisType... analysisTypes);
    }

    static final class BuilderImpl implements Builder {
        private RepositoryAnalysis repositoryAnalysis;

        private List<String> analysisTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CodeReviewType model) {
            repositoryAnalysis(model.repositoryAnalysis);
            analysisTypesWithStrings(model.analysisTypes);
        }

        public final RepositoryAnalysis.Builder getRepositoryAnalysis() {
            return repositoryAnalysis != null ? repositoryAnalysis.toBuilder() : null;
        }

        @Override
        public final Builder repositoryAnalysis(RepositoryAnalysis repositoryAnalysis) {
            this.repositoryAnalysis = repositoryAnalysis;
            return this;
        }

        public final void setRepositoryAnalysis(RepositoryAnalysis.BuilderImpl repositoryAnalysis) {
            this.repositoryAnalysis = repositoryAnalysis != null ? repositoryAnalysis.build() : null;
        }

        public final Collection<String> getAnalysisTypes() {
            if (analysisTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return analysisTypes;
        }

        @Override
        public final Builder analysisTypesWithStrings(Collection<String> analysisTypes) {
            this.analysisTypes = AnalysisTypesCopier.copy(analysisTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder analysisTypesWithStrings(String... analysisTypes) {
            analysisTypesWithStrings(Arrays.asList(analysisTypes));
            return this;
        }

        @Override
        public final Builder analysisTypes(Collection<AnalysisType> analysisTypes) {
            this.analysisTypes = AnalysisTypesCopier.copyEnumToString(analysisTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder analysisTypes(AnalysisType... analysisTypes) {
            analysisTypes(Arrays.asList(analysisTypes));
            return this;
        }

        public final void setAnalysisTypes(Collection<String> analysisTypes) {
            this.analysisTypes = AnalysisTypesCopier.copy(analysisTypes);
        }

        @Override
        public CodeReviewType build() {
            return new CodeReviewType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
