/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Metadata about a rule. Rule metadata includes an ID, a name, a list of tags, and a short and long description.
 * CodeGuru Reviewer uses rules to analyze code. A rule's recommendation is included in analysis results if code is
 * detected that violates the rule.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RuleMetadata implements SdkPojo, Serializable, ToCopyableBuilder<RuleMetadata.Builder, RuleMetadata> {
    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("RuleId")
            .getter(getter(RuleMetadata::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleId").build()).build();

    private static final SdkField<String> RULE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleName").getter(getter(RuleMetadata::ruleName)).setter(setter(Builder::ruleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleName").build()).build();

    private static final SdkField<String> SHORT_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShortDescription").getter(getter(RuleMetadata::shortDescription))
            .setter(setter(Builder::shortDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShortDescription").build()).build();

    private static final SdkField<String> LONG_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LongDescription").getter(getter(RuleMetadata::longDescription)).setter(setter(Builder::longDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LongDescription").build()).build();

    private static final SdkField<List<String>> RULE_TAGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RuleTags")
            .getter(getter(RuleMetadata::ruleTags))
            .setter(setter(Builder::ruleTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_ID_FIELD,
            RULE_NAME_FIELD, SHORT_DESCRIPTION_FIELD, LONG_DESCRIPTION_FIELD, RULE_TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ruleId;

    private final String ruleName;

    private final String shortDescription;

    private final String longDescription;

    private final List<String> ruleTags;

    private RuleMetadata(BuilderImpl builder) {
        this.ruleId = builder.ruleId;
        this.ruleName = builder.ruleName;
        this.shortDescription = builder.shortDescription;
        this.longDescription = builder.longDescription;
        this.ruleTags = builder.ruleTags;
    }

    /**
     * <p>
     * The ID of the rule.
     * </p>
     * 
     * @return The ID of the rule.
     */
    public final String ruleId() {
        return ruleId;
    }

    /**
     * <p>
     * The name of the rule.
     * </p>
     * 
     * @return The name of the rule.
     */
    public final String ruleName() {
        return ruleName;
    }

    /**
     * <p>
     * A short description of the rule.
     * </p>
     * 
     * @return A short description of the rule.
     */
    public final String shortDescription() {
        return shortDescription;
    }

    /**
     * <p>
     * A long description of the rule.
     * </p>
     * 
     * @return A long description of the rule.
     */
    public final String longDescription() {
        return longDescription;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRuleTags() {
        return ruleTags != null && !(ruleTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags that are associated with the rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleTags} method.
     * </p>
     * 
     * @return Tags that are associated with the rule.
     */
    public final List<String> ruleTags() {
        return ruleTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        hashCode = 31 * hashCode + Objects.hashCode(ruleName());
        hashCode = 31 * hashCode + Objects.hashCode(shortDescription());
        hashCode = 31 * hashCode + Objects.hashCode(longDescription());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleTags() ? ruleTags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RuleMetadata)) {
            return false;
        }
        RuleMetadata other = (RuleMetadata) obj;
        return Objects.equals(ruleId(), other.ruleId()) && Objects.equals(ruleName(), other.ruleName())
                && Objects.equals(shortDescription(), other.shortDescription())
                && Objects.equals(longDescription(), other.longDescription()) && hasRuleTags() == other.hasRuleTags()
                && Objects.equals(ruleTags(), other.ruleTags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RuleMetadata").add("RuleId", ruleId()).add("RuleName", ruleName())
                .add("ShortDescription", shortDescription()).add("LongDescription", longDescription())
                .add("RuleTags", hasRuleTags() ? ruleTags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        case "RuleName":
            return Optional.ofNullable(clazz.cast(ruleName()));
        case "ShortDescription":
            return Optional.ofNullable(clazz.cast(shortDescription()));
        case "LongDescription":
            return Optional.ofNullable(clazz.cast(longDescription()));
        case "RuleTags":
            return Optional.ofNullable(clazz.cast(ruleTags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RuleMetadata, T> g) {
        return obj -> g.apply((RuleMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RuleMetadata> {
        /**
         * <p>
         * The ID of the rule.
         * </p>
         * 
         * @param ruleId
         *        The ID of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);

        /**
         * <p>
         * The name of the rule.
         * </p>
         * 
         * @param ruleName
         *        The name of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleName(String ruleName);

        /**
         * <p>
         * A short description of the rule.
         * </p>
         * 
         * @param shortDescription
         *        A short description of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shortDescription(String shortDescription);

        /**
         * <p>
         * A long description of the rule.
         * </p>
         * 
         * @param longDescription
         *        A long description of the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder longDescription(String longDescription);

        /**
         * <p>
         * Tags that are associated with the rule.
         * </p>
         * 
         * @param ruleTags
         *        Tags that are associated with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleTags(Collection<String> ruleTags);

        /**
         * <p>
         * Tags that are associated with the rule.
         * </p>
         * 
         * @param ruleTags
         *        Tags that are associated with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleTags(String... ruleTags);
    }

    static final class BuilderImpl implements Builder {
        private String ruleId;

        private String ruleName;

        private String shortDescription;

        private String longDescription;

        private List<String> ruleTags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RuleMetadata model) {
            ruleId(model.ruleId);
            ruleName(model.ruleName);
            shortDescription(model.shortDescription);
            longDescription(model.longDescription);
            ruleTags(model.ruleTags);
        }

        public final String getRuleId() {
            return ruleId;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final String getRuleName() {
            return ruleName;
        }

        public final void setRuleName(String ruleName) {
            this.ruleName = ruleName;
        }

        @Override
        public final Builder ruleName(String ruleName) {
            this.ruleName = ruleName;
            return this;
        }

        public final String getShortDescription() {
            return shortDescription;
        }

        public final void setShortDescription(String shortDescription) {
            this.shortDescription = shortDescription;
        }

        @Override
        public final Builder shortDescription(String shortDescription) {
            this.shortDescription = shortDescription;
            return this;
        }

        public final String getLongDescription() {
            return longDescription;
        }

        public final void setLongDescription(String longDescription) {
            this.longDescription = longDescription;
        }

        @Override
        public final Builder longDescription(String longDescription) {
            this.longDescription = longDescription;
            return this;
        }

        public final Collection<String> getRuleTags() {
            if (ruleTags instanceof SdkAutoConstructList) {
                return null;
            }
            return ruleTags;
        }

        public final void setRuleTags(Collection<String> ruleTags) {
            this.ruleTags = RuleTagsCopier.copy(ruleTags);
        }

        @Override
        public final Builder ruleTags(Collection<String> ruleTags) {
            this.ruleTags = RuleTagsCopier.copy(ruleTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleTags(String... ruleTags) {
            ruleTags(Arrays.asList(ruleTags));
            return this;
        }

        @Override
        public RuleMetadata build() {
            return new RuleMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
