/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.codegurureviewer.CodeGuruReviewerClient;
import software.amazon.awssdk.services.codegurureviewer.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.codegurureviewer.model.CodeGuruReviewerRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewResponse;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationResponse;
import software.amazon.awssdk.services.codegurureviewer.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultCodeGuruReviewerWaiter implements CodeGuruReviewerWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final CodeGuruReviewerClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeRepositoryAssociationResponse> repositoryAssociationSucceededWaiter;

    private final Waiter<DescribeCodeReviewResponse> codeReviewCompletedWaiter;

    private DefaultCodeGuruReviewerWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = CodeGuruReviewerClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.repositoryAssociationSucceededWaiter = Waiter.builder(DescribeRepositoryAssociationResponse.class)
                .acceptors(repositoryAssociationSucceededWaiterAcceptors())
                .overrideConfiguration(repositoryAssociationSucceededWaiterConfig(builder.overrideConfiguration)).build();
        this.codeReviewCompletedWaiter = Waiter.builder(DescribeCodeReviewResponse.class)
                .acceptors(codeReviewCompletedWaiterAcceptors())
                .overrideConfiguration(codeReviewCompletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeCodeReviewResponse> waitUntilCodeReviewCompleted(
            DescribeCodeReviewRequest describeCodeReviewRequest) {
        return codeReviewCompletedWaiter.run(() -> client.describeCodeReview(applyWaitersUserAgent(describeCodeReviewRequest)));
    }

    @Override
    public WaiterResponse<DescribeCodeReviewResponse> waitUntilCodeReviewCompleted(
            DescribeCodeReviewRequest describeCodeReviewRequest, WaiterOverrideConfiguration overrideConfig) {
        return codeReviewCompletedWaiter.run(() -> client.describeCodeReview(applyWaitersUserAgent(describeCodeReviewRequest)),
                codeReviewCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeRepositoryAssociationResponse> waitUntilRepositoryAssociationSucceeded(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest) {
        return repositoryAssociationSucceededWaiter.run(() -> client
                .describeRepositoryAssociation(applyWaitersUserAgent(describeRepositoryAssociationRequest)));
    }

    @Override
    public WaiterResponse<DescribeRepositoryAssociationResponse> waitUntilRepositoryAssociationSucceeded(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest, WaiterOverrideConfiguration overrideConfig) {
        return repositoryAssociationSucceededWaiter.run(
                () -> client.describeRepositoryAssociation(applyWaitersUserAgent(describeRepositoryAssociationRequest)),
                repositoryAssociationSucceededWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeRepositoryAssociationResponse>> repositoryAssociationSucceededWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeRepositoryAssociationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RepositoryAssociation").field("State").value(), "Associated");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RepositoryAssociation").field("State").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RepositoryAssociation.State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RepositoryAssociation").field("State").value(), "Associating");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCodeReviewResponse>> codeReviewCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCodeReviewResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("CodeReview").field("State").value(), "Completed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("CodeReview").field("State").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (CodeReview.State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("CodeReview").field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration repositoryAssociationSucceededWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration codeReviewCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static CodeGuruReviewerWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends CodeGuruReviewerRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements CodeGuruReviewerWaiter.Builder {
        private CodeGuruReviewerClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public CodeGuruReviewerWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public CodeGuruReviewerWaiter.Builder client(CodeGuruReviewerClient client) {
            this.client = client;
            return this;
        }

        public CodeGuruReviewerWaiter build() {
            return new DefaultCodeGuruReviewerWaiter(this);
        }
    }
}
