/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurureviewer.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.codegurureviewer.CodeGuruReviewerAsyncClient;
import software.amazon.awssdk.services.codegurureviewer.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.codegurureviewer.model.CodeGuruReviewerRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeCodeReviewResponse;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationRequest;
import software.amazon.awssdk.services.codegurureviewer.model.DescribeRepositoryAssociationResponse;
import software.amazon.awssdk.services.codegurureviewer.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultCodeGuruReviewerAsyncWaiter implements CodeGuruReviewerAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final CodeGuruReviewerAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeRepositoryAssociationResponse> repositoryAssociationSucceededWaiter;

    private final AsyncWaiter<DescribeCodeReviewResponse> codeReviewCompletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultCodeGuruReviewerAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = CodeGuruReviewerAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.repositoryAssociationSucceededWaiter = AsyncWaiter.builder(DescribeRepositoryAssociationResponse.class)
                .acceptors(repositoryAssociationSucceededWaiterAcceptors())
                .overrideConfiguration(repositoryAssociationSucceededWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.codeReviewCompletedWaiter = AsyncWaiter.builder(DescribeCodeReviewResponse.class)
                .acceptors(codeReviewCompletedWaiterAcceptors())
                .overrideConfiguration(codeReviewCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCodeReviewResponse>> waitUntilCodeReviewCompleted(
            DescribeCodeReviewRequest describeCodeReviewRequest) {
        return codeReviewCompletedWaiter.runAsync(() -> client
                .describeCodeReview(applyWaitersUserAgent(describeCodeReviewRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCodeReviewResponse>> waitUntilCodeReviewCompleted(
            DescribeCodeReviewRequest describeCodeReviewRequest, WaiterOverrideConfiguration overrideConfig) {
        return codeReviewCompletedWaiter.runAsync(
                () -> client.describeCodeReview(applyWaitersUserAgent(describeCodeReviewRequest)),
                codeReviewCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeRepositoryAssociationResponse>> waitUntilRepositoryAssociationSucceeded(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest) {
        return repositoryAssociationSucceededWaiter.runAsync(() -> client
                .describeRepositoryAssociation(applyWaitersUserAgent(describeRepositoryAssociationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeRepositoryAssociationResponse>> waitUntilRepositoryAssociationSucceeded(
            DescribeRepositoryAssociationRequest describeRepositoryAssociationRequest, WaiterOverrideConfiguration overrideConfig) {
        return repositoryAssociationSucceededWaiter.runAsync(
                () -> client.describeRepositoryAssociation(applyWaitersUserAgent(describeRepositoryAssociationRequest)),
                repositoryAssociationSucceededWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeRepositoryAssociationResponse>> repositoryAssociationSucceededWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeRepositoryAssociationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RepositoryAssociation").field("State").value(), "Associated");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("RepositoryAssociation").field("State").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (RepositoryAssociation.State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("RepositoryAssociation").field("State").value(), "Associating");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCodeReviewResponse>> codeReviewCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCodeReviewResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("CodeReview").field("State").value(), "Completed");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("CodeReview").field("State").value(), "Failed");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (CodeReview.State=Failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("CodeReview").field("State").value(), "Pending");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration repositoryAssociationSucceededWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration codeReviewCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static CodeGuruReviewerAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends CodeGuruReviewerRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements CodeGuruReviewerAsyncWaiter.Builder {
        private CodeGuruReviewerAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public CodeGuruReviewerAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public CodeGuruReviewerAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public CodeGuruReviewerAsyncWaiter.Builder client(CodeGuruReviewerAsyncClient client) {
            this.client = client;
            return this;
        }

        public CodeGuruReviewerAsyncWaiter build() {
            return new DefaultCodeGuruReviewerAsyncWaiter(this);
        }
    }
}
