/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codegurusecurity.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a finding that was detected in your code.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Finding implements SdkPojo, Serializable, ToCopyableBuilder<Finding.Builder, Finding> {
    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Finding::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Finding::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> DETECTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("detectorId").getter(getter(Finding::detectorId)).setter(setter(Builder::detectorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("detectorId").build()).build();

    private static final SdkField<String> DETECTOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("detectorName").getter(getter(Finding::detectorName)).setter(setter(Builder::detectorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("detectorName").build()).build();

    private static final SdkField<List<String>> DETECTOR_TAGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("detectorTags")
            .getter(getter(Finding::detectorTags))
            .setter(setter(Builder::detectorTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("detectorTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GENERATOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("generatorId").getter(getter(Finding::generatorId)).setter(setter(Builder::generatorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("generatorId").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Finding::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<Remediation> REMEDIATION_FIELD = SdkField.<Remediation> builder(MarshallingType.SDK_POJO)
            .memberName("remediation").getter(getter(Finding::remediation)).setter(setter(Builder::remediation))
            .constructor(Remediation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("remediation").build()).build();

    private static final SdkField<Resource> RESOURCE_FIELD = SdkField.<Resource> builder(MarshallingType.SDK_POJO)
            .memberName("resource").getter(getter(Finding::resource)).setter(setter(Builder::resource))
            .constructor(Resource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resource").build()).build();

    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ruleId")
            .getter(getter(Finding::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ruleId").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("severity").getter(getter(Finding::severityAsString)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("severity").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Finding::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(Finding::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(Finding::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(Finding::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final SdkField<Vulnerability> VULNERABILITY_FIELD = SdkField.<Vulnerability> builder(MarshallingType.SDK_POJO)
            .memberName("vulnerability").getter(getter(Finding::vulnerability)).setter(setter(Builder::vulnerability))
            .constructor(Vulnerability::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vulnerability").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATED_AT_FIELD,
            DESCRIPTION_FIELD, DETECTOR_ID_FIELD, DETECTOR_NAME_FIELD, DETECTOR_TAGS_FIELD, GENERATOR_ID_FIELD, ID_FIELD,
            REMEDIATION_FIELD, RESOURCE_FIELD, RULE_ID_FIELD, SEVERITY_FIELD, STATUS_FIELD, TITLE_FIELD, TYPE_FIELD,
            UPDATED_AT_FIELD, VULNERABILITY_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant createdAt;

    private final String description;

    private final String detectorId;

    private final String detectorName;

    private final List<String> detectorTags;

    private final String generatorId;

    private final String id;

    private final Remediation remediation;

    private final Resource resource;

    private final String ruleId;

    private final String severity;

    private final String status;

    private final String title;

    private final String type;

    private final Instant updatedAt;

    private final Vulnerability vulnerability;

    private Finding(BuilderImpl builder) {
        this.createdAt = builder.createdAt;
        this.description = builder.description;
        this.detectorId = builder.detectorId;
        this.detectorName = builder.detectorName;
        this.detectorTags = builder.detectorTags;
        this.generatorId = builder.generatorId;
        this.id = builder.id;
        this.remediation = builder.remediation;
        this.resource = builder.resource;
        this.ruleId = builder.ruleId;
        this.severity = builder.severity;
        this.status = builder.status;
        this.title = builder.title;
        this.type = builder.type;
        this.updatedAt = builder.updatedAt;
        this.vulnerability = builder.vulnerability;
    }

    /**
     * <p>
     * The time when the finding was created.
     * </p>
     * 
     * @return The time when the finding was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * A description of the finding.
     * </p>
     * 
     * @return A description of the finding.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The identifier for the detector that detected the finding in your code. A detector is a defined rule based on
     * industry standards and AWS best practices.
     * </p>
     * 
     * @return The identifier for the detector that detected the finding in your code. A detector is a defined rule
     *         based on industry standards and AWS best practices.
     */
    public final String detectorId() {
        return detectorId;
    }

    /**
     * <p>
     * The name of the detector that identified the security vulnerability in your code.
     * </p>
     * 
     * @return The name of the detector that identified the security vulnerability in your code.
     */
    public final String detectorName() {
        return detectorName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DetectorTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDetectorTags() {
        return detectorTags != null && !(detectorTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more tags or categorizations that are associated with a detector. These tags are defined by type,
     * programming language, or other classification such as maintainability or consistency.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDetectorTags} method.
     * </p>
     * 
     * @return One or more tags or categorizations that are associated with a detector. These tags are defined by type,
     *         programming language, or other classification such as maintainability or consistency.
     */
    public final List<String> detectorTags() {
        return detectorTags;
    }

    /**
     * <p>
     * The identifier for the component that generated a finding such as AmazonCodeGuruSecurity.
     * </p>
     * 
     * @return The identifier for the component that generated a finding such as AmazonCodeGuruSecurity.
     */
    public final String generatorId() {
        return generatorId;
    }

    /**
     * <p>
     * The identifier for a finding.
     * </p>
     * 
     * @return The identifier for a finding.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * An object that contains the details about how to remediate a finding.
     * </p>
     * 
     * @return An object that contains the details about how to remediate a finding.
     */
    public final Remediation remediation() {
        return remediation;
    }

    /**
     * <p>
     * The resource where Amazon CodeGuru Security detected a finding.
     * </p>
     * 
     * @return The resource where Amazon CodeGuru Security detected a finding.
     */
    public final Resource resource() {
        return resource;
    }

    /**
     * <p>
     * The identifier for the rule that generated the finding.
     * </p>
     * 
     * @return The identifier for the rule that generated the finding.
     */
    public final String ruleId() {
        return ruleId;
    }

    /**
     * <p>
     * The severity of the finding. Severity can be critical, high, medium, low, or informational. For information on
     * severity levels, see <a
     * href="https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
     * >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link Severity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the finding. Severity can be critical, high, medium, low, or informational. For
     *         information on severity levels, see <a href=
     *         "https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
     *         >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
     * @see Severity
     */
    public final Severity severity() {
        return Severity.fromValue(severity);
    }

    /**
     * <p>
     * The severity of the finding. Severity can be critical, high, medium, low, or informational. For information on
     * severity levels, see <a
     * href="https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
     * >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severity} will
     * return {@link Severity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityAsString}.
     * </p>
     * 
     * @return The severity of the finding. Severity can be critical, high, medium, low, or informational. For
     *         information on severity levels, see <a href=
     *         "https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
     *         >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
     * @see Severity
     */
    public final String severityAsString() {
        return severity;
    }

    /**
     * <p>
     * The status of the finding. A finding status can be open or closed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the finding. A finding status can be open or closed.
     * @see Status
     */
    public final Status status() {
        return Status.fromValue(status);
    }

    /**
     * <p>
     * The status of the finding. A finding status can be open or closed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the finding. A finding status can be open or closed.
     * @see Status
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The title of the finding.
     * </p>
     * 
     * @return The title of the finding.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The type of finding.
     * </p>
     * 
     * @return The type of finding.
     */
    public final String type() {
        return type;
    }

    /**
     * <p>
     * The time when the finding was last updated. Findings are updated when you remediate them or when the finding code
     * location changes.
     * </p>
     * 
     * @return The time when the finding was last updated. Findings are updated when you remediate them or when the
     *         finding code location changes.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * An object that describes the detected security vulnerability.
     * </p>
     * 
     * @return An object that describes the detected security vulnerability.
     */
    public final Vulnerability vulnerability() {
        return vulnerability;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(detectorId());
        hashCode = 31 * hashCode + Objects.hashCode(detectorName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDetectorTags() ? detectorTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(generatorId());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(remediation());
        hashCode = 31 * hashCode + Objects.hashCode(resource());
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        hashCode = 31 * hashCode + Objects.hashCode(severityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(vulnerability());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Finding)) {
            return false;
        }
        Finding other = (Finding) obj;
        return Objects.equals(createdAt(), other.createdAt()) && Objects.equals(description(), other.description())
                && Objects.equals(detectorId(), other.detectorId()) && Objects.equals(detectorName(), other.detectorName())
                && hasDetectorTags() == other.hasDetectorTags() && Objects.equals(detectorTags(), other.detectorTags())
                && Objects.equals(generatorId(), other.generatorId()) && Objects.equals(id(), other.id())
                && Objects.equals(remediation(), other.remediation()) && Objects.equals(resource(), other.resource())
                && Objects.equals(ruleId(), other.ruleId()) && Objects.equals(severityAsString(), other.severityAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(title(), other.title())
                && Objects.equals(type(), other.type()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(vulnerability(), other.vulnerability());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Finding").add("CreatedAt", createdAt()).add("Description", description())
                .add("DetectorId", detectorId()).add("DetectorName", detectorName())
                .add("DetectorTags", hasDetectorTags() ? detectorTags() : null).add("GeneratorId", generatorId()).add("Id", id())
                .add("Remediation", remediation()).add("Resource", resource()).add("RuleId", ruleId())
                .add("Severity", severityAsString()).add("Status", statusAsString()).add("Title", title()).add("Type", type())
                .add("UpdatedAt", updatedAt()).add("Vulnerability", vulnerability()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "detectorId":
            return Optional.ofNullable(clazz.cast(detectorId()));
        case "detectorName":
            return Optional.ofNullable(clazz.cast(detectorName()));
        case "detectorTags":
            return Optional.ofNullable(clazz.cast(detectorTags()));
        case "generatorId":
            return Optional.ofNullable(clazz.cast(generatorId()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "remediation":
            return Optional.ofNullable(clazz.cast(remediation()));
        case "resource":
            return Optional.ofNullable(clazz.cast(resource()));
        case "ruleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        case "severity":
            return Optional.ofNullable(clazz.cast(severityAsString()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        case "type":
            return Optional.ofNullable(clazz.cast(type()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "vulnerability":
            return Optional.ofNullable(clazz.cast(vulnerability()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Finding, T> g) {
        return obj -> g.apply((Finding) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Finding> {
        /**
         * <p>
         * The time when the finding was created.
         * </p>
         * 
         * @param createdAt
         *        The time when the finding was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * A description of the finding.
         * </p>
         * 
         * @param description
         *        A description of the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The identifier for the detector that detected the finding in your code. A detector is a defined rule based on
         * industry standards and AWS best practices.
         * </p>
         * 
         * @param detectorId
         *        The identifier for the detector that detected the finding in your code. A detector is a defined rule
         *        based on industry standards and AWS best practices.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectorId(String detectorId);

        /**
         * <p>
         * The name of the detector that identified the security vulnerability in your code.
         * </p>
         * 
         * @param detectorName
         *        The name of the detector that identified the security vulnerability in your code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectorName(String detectorName);

        /**
         * <p>
         * One or more tags or categorizations that are associated with a detector. These tags are defined by type,
         * programming language, or other classification such as maintainability or consistency.
         * </p>
         * 
         * @param detectorTags
         *        One or more tags or categorizations that are associated with a detector. These tags are defined by
         *        type, programming language, or other classification such as maintainability or consistency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectorTags(Collection<String> detectorTags);

        /**
         * <p>
         * One or more tags or categorizations that are associated with a detector. These tags are defined by type,
         * programming language, or other classification such as maintainability or consistency.
         * </p>
         * 
         * @param detectorTags
         *        One or more tags or categorizations that are associated with a detector. These tags are defined by
         *        type, programming language, or other classification such as maintainability or consistency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectorTags(String... detectorTags);

        /**
         * <p>
         * The identifier for the component that generated a finding such as AmazonCodeGuruSecurity.
         * </p>
         * 
         * @param generatorId
         *        The identifier for the component that generated a finding such as AmazonCodeGuruSecurity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder generatorId(String generatorId);

        /**
         * <p>
         * The identifier for a finding.
         * </p>
         * 
         * @param id
         *        The identifier for a finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * An object that contains the details about how to remediate a finding.
         * </p>
         * 
         * @param remediation
         *        An object that contains the details about how to remediate a finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remediation(Remediation remediation);

        /**
         * <p>
         * An object that contains the details about how to remediate a finding.
         * </p>
         * This is a convenience method that creates an instance of the {@link Remediation.Builder} avoiding the need to
         * create one manually via {@link Remediation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Remediation.Builder#build()} is called immediately and its result
         * is passed to {@link #remediation(Remediation)}.
         * 
         * @param remediation
         *        a consumer that will call methods on {@link Remediation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #remediation(Remediation)
         */
        default Builder remediation(Consumer<Remediation.Builder> remediation) {
            return remediation(Remediation.builder().applyMutation(remediation).build());
        }

        /**
         * <p>
         * The resource where Amazon CodeGuru Security detected a finding.
         * </p>
         * 
         * @param resource
         *        The resource where Amazon CodeGuru Security detected a finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resource(Resource resource);

        /**
         * <p>
         * The resource where Amazon CodeGuru Security detected a finding.
         * </p>
         * This is a convenience method that creates an instance of the {@link Resource.Builder} avoiding the need to
         * create one manually via {@link Resource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Resource.Builder#build()} is called immediately and its result is
         * passed to {@link #resource(Resource)}.
         * 
         * @param resource
         *        a consumer that will call methods on {@link Resource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resource(Resource)
         */
        default Builder resource(Consumer<Resource.Builder> resource) {
            return resource(Resource.builder().applyMutation(resource).build());
        }

        /**
         * <p>
         * The identifier for the rule that generated the finding.
         * </p>
         * 
         * @param ruleId
         *        The identifier for the rule that generated the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);

        /**
         * <p>
         * The severity of the finding. Severity can be critical, high, medium, low, or informational. For information
         * on severity levels, see <a
         * href="https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
         * >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity of the finding. Severity can be critical, high, medium, low, or informational. For
         *        information on severity levels, see <a href=
         *        "https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
         *        >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
         * @see Severity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Severity
         */
        Builder severity(String severity);

        /**
         * <p>
         * The severity of the finding. Severity can be critical, high, medium, low, or informational. For information
         * on severity levels, see <a
         * href="https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
         * >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
         * </p>
         * 
         * @param severity
         *        The severity of the finding. Severity can be critical, high, medium, low, or informational. For
         *        information on severity levels, see <a href=
         *        "https://docs.aws.amazon.com/codeguru/latest/security-ug/findings-overview.html#severity-distribution"
         *        >Finding severity</a> in the <i>Amazon CodeGuru Security User Guide</i>.
         * @see Severity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Severity
         */
        Builder severity(Severity severity);

        /**
         * <p>
         * The status of the finding. A finding status can be open or closed.
         * </p>
         * 
         * @param status
         *        The status of the finding. A finding status can be open or closed.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the finding. A finding status can be open or closed.
         * </p>
         * 
         * @param status
         *        The status of the finding. A finding status can be open or closed.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(Status status);

        /**
         * <p>
         * The title of the finding.
         * </p>
         * 
         * @param title
         *        The title of the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The type of finding.
         * </p>
         * 
         * @param type
         *        The type of finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * The time when the finding was last updated. Findings are updated when you remediate them or when the finding
         * code location changes.
         * </p>
         * 
         * @param updatedAt
         *        The time when the finding was last updated. Findings are updated when you remediate them or when the
         *        finding code location changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * An object that describes the detected security vulnerability.
         * </p>
         * 
         * @param vulnerability
         *        An object that describes the detected security vulnerability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vulnerability(Vulnerability vulnerability);

        /**
         * <p>
         * An object that describes the detected security vulnerability.
         * </p>
         * This is a convenience method that creates an instance of the {@link Vulnerability.Builder} avoiding the need
         * to create one manually via {@link Vulnerability#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Vulnerability.Builder#build()} is called immediately and its
         * result is passed to {@link #vulnerability(Vulnerability)}.
         * 
         * @param vulnerability
         *        a consumer that will call methods on {@link Vulnerability.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vulnerability(Vulnerability)
         */
        default Builder vulnerability(Consumer<Vulnerability.Builder> vulnerability) {
            return vulnerability(Vulnerability.builder().applyMutation(vulnerability).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant createdAt;

        private String description;

        private String detectorId;

        private String detectorName;

        private List<String> detectorTags = DefaultSdkAutoConstructList.getInstance();

        private String generatorId;

        private String id;

        private Remediation remediation;

        private Resource resource;

        private String ruleId;

        private String severity;

        private String status;

        private String title;

        private String type;

        private Instant updatedAt;

        private Vulnerability vulnerability;

        private BuilderImpl() {
        }

        private BuilderImpl(Finding model) {
            createdAt(model.createdAt);
            description(model.description);
            detectorId(model.detectorId);
            detectorName(model.detectorName);
            detectorTags(model.detectorTags);
            generatorId(model.generatorId);
            id(model.id);
            remediation(model.remediation);
            resource(model.resource);
            ruleId(model.ruleId);
            severity(model.severity);
            status(model.status);
            title(model.title);
            type(model.type);
            updatedAt(model.updatedAt);
            vulnerability(model.vulnerability);
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDetectorId() {
            return detectorId;
        }

        public final void setDetectorId(String detectorId) {
            this.detectorId = detectorId;
        }

        @Override
        public final Builder detectorId(String detectorId) {
            this.detectorId = detectorId;
            return this;
        }

        public final String getDetectorName() {
            return detectorName;
        }

        public final void setDetectorName(String detectorName) {
            this.detectorName = detectorName;
        }

        @Override
        public final Builder detectorName(String detectorName) {
            this.detectorName = detectorName;
            return this;
        }

        public final Collection<String> getDetectorTags() {
            if (detectorTags instanceof SdkAutoConstructList) {
                return null;
            }
            return detectorTags;
        }

        public final void setDetectorTags(Collection<String> detectorTags) {
            this.detectorTags = DetectorTagsCopier.copy(detectorTags);
        }

        @Override
        public final Builder detectorTags(Collection<String> detectorTags) {
            this.detectorTags = DetectorTagsCopier.copy(detectorTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder detectorTags(String... detectorTags) {
            detectorTags(Arrays.asList(detectorTags));
            return this;
        }

        public final String getGeneratorId() {
            return generatorId;
        }

        public final void setGeneratorId(String generatorId) {
            this.generatorId = generatorId;
        }

        @Override
        public final Builder generatorId(String generatorId) {
            this.generatorId = generatorId;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Remediation.Builder getRemediation() {
            return remediation != null ? remediation.toBuilder() : null;
        }

        public final void setRemediation(Remediation.BuilderImpl remediation) {
            this.remediation = remediation != null ? remediation.build() : null;
        }

        @Override
        public final Builder remediation(Remediation remediation) {
            this.remediation = remediation;
            return this;
        }

        public final Resource.Builder getResource() {
            return resource != null ? resource.toBuilder() : null;
        }

        public final void setResource(Resource.BuilderImpl resource) {
            this.resource = resource != null ? resource.build() : null;
        }

        @Override
        public final Builder resource(Resource resource) {
            this.resource = resource;
            return this;
        }

        public final String getRuleId() {
            return ruleId;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        @Override
        public final Builder severity(Severity severity) {
            this.severity(severity == null ? null : severity.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(Status status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final Vulnerability.Builder getVulnerability() {
            return vulnerability != null ? vulnerability.toBuilder() : null;
        }

        public final void setVulnerability(Vulnerability.BuilderImpl vulnerability) {
            this.vulnerability = vulnerability != null ? vulnerability.build() : null;
        }

        @Override
        public final Builder vulnerability(Vulnerability vulnerability) {
            this.vulnerability = vulnerability;
            return this;
        }

        @Override
        public Finding build() {
            return new Finding(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
