/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents information about a webhook and its definition.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebhookDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<WebhookDefinition.Builder, WebhookDefinition> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(WebhookDefinition::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TARGET_PIPELINE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetPipeline").getter(getter(WebhookDefinition::targetPipeline))
            .setter(setter(Builder::targetPipeline))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetPipeline").build()).build();

    private static final SdkField<String> TARGET_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetAction").getter(getter(WebhookDefinition::targetAction)).setter(setter(Builder::targetAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetAction").build()).build();

    private static final SdkField<List<WebhookFilterRule>> FILTERS_FIELD = SdkField
            .<List<WebhookFilterRule>> builder(MarshallingType.LIST)
            .memberName("filters")
            .getter(getter(WebhookDefinition::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<WebhookFilterRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(WebhookFilterRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> AUTHENTICATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authentication").getter(getter(WebhookDefinition::authenticationAsString))
            .setter(setter(Builder::authentication))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authentication").build()).build();

    private static final SdkField<WebhookAuthConfiguration> AUTHENTICATION_CONFIGURATION_FIELD = SdkField
            .<WebhookAuthConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("authenticationConfiguration")
            .getter(getter(WebhookDefinition::authenticationConfiguration))
            .setter(setter(Builder::authenticationConfiguration))
            .constructor(WebhookAuthConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authenticationConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            TARGET_PIPELINE_FIELD, TARGET_ACTION_FIELD, FILTERS_FIELD, AUTHENTICATION_FIELD, AUTHENTICATION_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String targetPipeline;

    private final String targetAction;

    private final List<WebhookFilterRule> filters;

    private final String authentication;

    private final WebhookAuthConfiguration authenticationConfiguration;

    private WebhookDefinition(BuilderImpl builder) {
        this.name = builder.name;
        this.targetPipeline = builder.targetPipeline;
        this.targetAction = builder.targetAction;
        this.filters = builder.filters;
        this.authentication = builder.authentication;
        this.authenticationConfiguration = builder.authenticationConfiguration;
    }

    /**
     * <p>
     * The name of the webhook.
     * </p>
     * 
     * @return The name of the webhook.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name of the pipeline you want to connect to the webhook.
     * </p>
     * 
     * @return The name of the pipeline you want to connect to the webhook.
     */
    public final String targetPipeline() {
        return targetPipeline;
    }

    /**
     * <p>
     * The name of the action in a pipeline you want to connect to the webhook. The action must be from the source
     * (first) stage of the pipeline.
     * </p>
     * 
     * @return The name of the action in a pipeline you want to connect to the webhook. The action must be from the
     *         source (first) stage of the pipeline.
     */
    public final String targetAction() {
        return targetAction;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined rules must
     * pass for the request to be accepted and the pipeline started.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilters} method.
     * </p>
     * 
     * @return A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined rules
     *         must pass for the request to be accepted and the pipeline started.
     */
    public final List<WebhookFilterRule> filters() {
        return filters;
    }

    /**
     * <p>
     * Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For information about the authentication scheme implemented by GITHUB_HMAC, see <a
     * href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub Developer
     * website.
     * </p>
     * </li>
     * <li>
     * <p>
     * IP rejects webhooks trigger requests unless they originate from an IP address in the IP range whitelisted in the
     * authentication configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authentication}
     * will return {@link WebhookAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #authenticationAsString}.
     * </p>
     * 
     * @return Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For information about the authentication scheme implemented by GITHUB_HMAC, see <a
     *         href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub Developer
     *         website.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IP rejects webhooks trigger requests unless they originate from an IP address in the IP range whitelisted
     *         in the authentication configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
     *         </p>
     *         </li>
     * @see WebhookAuthenticationType
     */
    public final WebhookAuthenticationType authentication() {
        return WebhookAuthenticationType.fromValue(authentication);
    }

    /**
     * <p>
     * Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For information about the authentication scheme implemented by GITHUB_HMAC, see <a
     * href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub Developer
     * website.
     * </p>
     * </li>
     * <li>
     * <p>
     * IP rejects webhooks trigger requests unless they originate from an IP address in the IP range whitelisted in the
     * authentication configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authentication}
     * will return {@link WebhookAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #authenticationAsString}.
     * </p>
     * 
     * @return Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For information about the authentication scheme implemented by GITHUB_HMAC, see <a
     *         href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub Developer
     *         website.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IP rejects webhooks trigger requests unless they originate from an IP address in the IP range whitelisted
     *         in the authentication configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
     *         </p>
     *         </li>
     * @see WebhookAuthenticationType
     */
    public final String authenticationAsString() {
        return authentication;
    }

    /**
     * <p>
     * Properties that configure the authentication applied to incoming webhook trigger requests. The required
     * properties depend on the authentication type. For GITHUB_HMAC, only the <code>SecretToken </code>property must be
     * set. For IP, only the <code>AllowedIPRange </code>property must be set to a valid CIDR range. For
     * UNAUTHENTICATED, no properties can be set.
     * </p>
     * 
     * @return Properties that configure the authentication applied to incoming webhook trigger requests. The required
     *         properties depend on the authentication type. For GITHUB_HMAC, only the <code>SecretToken </code>property
     *         must be set. For IP, only the <code>AllowedIPRange </code>property must be set to a valid CIDR range. For
     *         UNAUTHENTICATED, no properties can be set.
     */
    public final WebhookAuthConfiguration authenticationConfiguration() {
        return authenticationConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(targetPipeline());
        hashCode = 31 * hashCode + Objects.hashCode(targetAction());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilters() ? filters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(authenticationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebhookDefinition)) {
            return false;
        }
        WebhookDefinition other = (WebhookDefinition) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(targetPipeline(), other.targetPipeline())
                && Objects.equals(targetAction(), other.targetAction()) && hasFilters() == other.hasFilters()
                && Objects.equals(filters(), other.filters())
                && Objects.equals(authenticationAsString(), other.authenticationAsString())
                && Objects.equals(authenticationConfiguration(), other.authenticationConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WebhookDefinition").add("Name", name()).add("TargetPipeline", targetPipeline())
                .add("TargetAction", targetAction()).add("Filters", hasFilters() ? filters() : null)
                .add("Authentication", authenticationAsString())
                .add("AuthenticationConfiguration", authenticationConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "targetPipeline":
            return Optional.ofNullable(clazz.cast(targetPipeline()));
        case "targetAction":
            return Optional.ofNullable(clazz.cast(targetAction()));
        case "filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "authentication":
            return Optional.ofNullable(clazz.cast(authenticationAsString()));
        case "authenticationConfiguration":
            return Optional.ofNullable(clazz.cast(authenticationConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WebhookDefinition, T> g) {
        return obj -> g.apply((WebhookDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebhookDefinition> {
        /**
         * <p>
         * The name of the webhook.
         * </p>
         * 
         * @param name
         *        The name of the webhook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the pipeline you want to connect to the webhook.
         * </p>
         * 
         * @param targetPipeline
         *        The name of the pipeline you want to connect to the webhook.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetPipeline(String targetPipeline);

        /**
         * <p>
         * The name of the action in a pipeline you want to connect to the webhook. The action must be from the source
         * (first) stage of the pipeline.
         * </p>
         * 
         * @param targetAction
         *        The name of the action in a pipeline you want to connect to the webhook. The action must be from the
         *        source (first) stage of the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetAction(String targetAction);

        /**
         * <p>
         * A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined rules must
         * pass for the request to be accepted and the pipeline started.
         * </p>
         * 
         * @param filters
         *        A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined
         *        rules must pass for the request to be accepted and the pipeline started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<WebhookFilterRule> filters);

        /**
         * <p>
         * A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined rules must
         * pass for the request to be accepted and the pipeline started.
         * </p>
         * 
         * @param filters
         *        A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined
         *        rules must pass for the request to be accepted and the pipeline started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(WebhookFilterRule... filters);

        /**
         * <p>
         * A list of rules applied to the body/payload sent in the POST request to a webhook URL. All defined rules must
         * pass for the request to be accepted and the pipeline started.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<WebhookFilterRule>.Builder} avoiding
         * the need to create one manually via {@link List<WebhookFilterRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<WebhookFilterRule>.Builder#build()} is called immediately
         * and its result is passed to {@link #filters(List<WebhookFilterRule>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link List<WebhookFilterRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(List<WebhookFilterRule>)
         */
        Builder filters(Consumer<WebhookFilterRule.Builder>... filters);

        /**
         * <p>
         * Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For information about the authentication scheme implemented by GITHUB_HMAC, see <a
         * href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub Developer
         * website.
         * </p>
         * </li>
         * <li>
         * <p>
         * IP rejects webhooks trigger requests unless they originate from an IP address in the IP range whitelisted in
         * the authentication configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authentication
         *        Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For information about the authentication scheme implemented by GITHUB_HMAC, see <a
         *        href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub
         *        Developer website.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IP rejects webhooks trigger requests unless they originate from an IP address in the IP range
         *        whitelisted in the authentication configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
         *        </p>
         *        </li>
         * @see WebhookAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebhookAuthenticationType
         */
        Builder authentication(String authentication);

        /**
         * <p>
         * Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For information about the authentication scheme implemented by GITHUB_HMAC, see <a
         * href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub Developer
         * website.
         * </p>
         * </li>
         * <li>
         * <p>
         * IP rejects webhooks trigger requests unless they originate from an IP address in the IP range whitelisted in
         * the authentication configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authentication
         *        Supported options are GITHUB_HMAC, IP, and UNAUTHENTICATED.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For information about the authentication scheme implemented by GITHUB_HMAC, see <a
         *        href="https://developer.github.com/webhooks/securing/">Securing your webhooks</a> on the GitHub
         *        Developer website.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IP rejects webhooks trigger requests unless they originate from an IP address in the IP range
         *        whitelisted in the authentication configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNAUTHENTICATED accepts all webhook trigger requests regardless of origin.
         *        </p>
         *        </li>
         * @see WebhookAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WebhookAuthenticationType
         */
        Builder authentication(WebhookAuthenticationType authentication);

        /**
         * <p>
         * Properties that configure the authentication applied to incoming webhook trigger requests. The required
         * properties depend on the authentication type. For GITHUB_HMAC, only the <code>SecretToken </code>property
         * must be set. For IP, only the <code>AllowedIPRange </code>property must be set to a valid CIDR range. For
         * UNAUTHENTICATED, no properties can be set.
         * </p>
         * 
         * @param authenticationConfiguration
         *        Properties that configure the authentication applied to incoming webhook trigger requests. The
         *        required properties depend on the authentication type. For GITHUB_HMAC, only the
         *        <code>SecretToken </code>property must be set. For IP, only the <code>AllowedIPRange </code>property
         *        must be set to a valid CIDR range. For UNAUTHENTICATED, no properties can be set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationConfiguration(WebhookAuthConfiguration authenticationConfiguration);

        /**
         * <p>
         * Properties that configure the authentication applied to incoming webhook trigger requests. The required
         * properties depend on the authentication type. For GITHUB_HMAC, only the <code>SecretToken </code>property
         * must be set. For IP, only the <code>AllowedIPRange </code>property must be set to a valid CIDR range. For
         * UNAUTHENTICATED, no properties can be set.
         * </p>
         * This is a convenience method that creates an instance of the {@link WebhookAuthConfiguration.Builder}
         * avoiding the need to create one manually via {@link WebhookAuthConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link WebhookAuthConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #authenticationConfiguration(WebhookAuthConfiguration)}.
         * 
         * @param authenticationConfiguration
         *        a consumer that will call methods on {@link WebhookAuthConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationConfiguration(WebhookAuthConfiguration)
         */
        default Builder authenticationConfiguration(Consumer<WebhookAuthConfiguration.Builder> authenticationConfiguration) {
            return authenticationConfiguration(WebhookAuthConfiguration.builder().applyMutation(authenticationConfiguration)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String targetPipeline;

        private String targetAction;

        private List<WebhookFilterRule> filters = DefaultSdkAutoConstructList.getInstance();

        private String authentication;

        private WebhookAuthConfiguration authenticationConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(WebhookDefinition model) {
            name(model.name);
            targetPipeline(model.targetPipeline);
            targetAction(model.targetAction);
            filters(model.filters);
            authentication(model.authentication);
            authenticationConfiguration(model.authenticationConfiguration);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getTargetPipeline() {
            return targetPipeline;
        }

        public final void setTargetPipeline(String targetPipeline) {
            this.targetPipeline = targetPipeline;
        }

        @Override
        public final Builder targetPipeline(String targetPipeline) {
            this.targetPipeline = targetPipeline;
            return this;
        }

        public final String getTargetAction() {
            return targetAction;
        }

        public final void setTargetAction(String targetAction) {
            this.targetAction = targetAction;
        }

        @Override
        public final Builder targetAction(String targetAction) {
            this.targetAction = targetAction;
            return this;
        }

        public final List<WebhookFilterRule.Builder> getFilters() {
            List<WebhookFilterRule.Builder> result = WebhookFiltersCopier.copyToBuilder(this.filters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilters(Collection<WebhookFilterRule.BuilderImpl> filters) {
            this.filters = WebhookFiltersCopier.copyFromBuilder(filters);
        }

        @Override
        public final Builder filters(Collection<WebhookFilterRule> filters) {
            this.filters = WebhookFiltersCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(WebhookFilterRule... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<WebhookFilterRule.Builder>... filters) {
            filters(Stream.of(filters).map(c -> WebhookFilterRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getAuthentication() {
            return authentication;
        }

        public final void setAuthentication(String authentication) {
            this.authentication = authentication;
        }

        @Override
        public final Builder authentication(String authentication) {
            this.authentication = authentication;
            return this;
        }

        @Override
        public final Builder authentication(WebhookAuthenticationType authentication) {
            this.authentication(authentication == null ? null : authentication.toString());
            return this;
        }

        public final WebhookAuthConfiguration.Builder getAuthenticationConfiguration() {
            return authenticationConfiguration != null ? authenticationConfiguration.toBuilder() : null;
        }

        public final void setAuthenticationConfiguration(WebhookAuthConfiguration.BuilderImpl authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration != null ? authenticationConfiguration.build() : null;
        }

        @Override
        public final Builder authenticationConfiguration(WebhookAuthConfiguration authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration;
            return this;
        }

        @Override
        public WebhookDefinition build() {
            return new WebhookDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
