/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents information about an action declaration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActionDeclaration implements SdkPojo, Serializable,
        ToCopyableBuilder<ActionDeclaration.Builder, ActionDeclaration> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ActionDeclaration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<ActionTypeId> ACTION_TYPE_ID_FIELD = SdkField.<ActionTypeId> builder(MarshallingType.SDK_POJO)
            .memberName("actionTypeId").getter(getter(ActionDeclaration::actionTypeId)).setter(setter(Builder::actionTypeId))
            .constructor(ActionTypeId::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionTypeId").build()).build();

    private static final SdkField<Integer> RUN_ORDER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("runOrder").getter(getter(ActionDeclaration::runOrder)).setter(setter(Builder::runOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("runOrder").build()).build();

    private static final SdkField<Map<String, String>> CONFIGURATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("configuration")
            .getter(getter(ActionDeclaration::configuration))
            .setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> COMMANDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("commands")
            .getter(getter(ActionDeclaration::commands))
            .setter(setter(Builder::commands))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("commands").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<OutputArtifact>> OUTPUT_ARTIFACTS_FIELD = SdkField
            .<List<OutputArtifact>> builder(MarshallingType.LIST)
            .memberName("outputArtifacts")
            .getter(getter(ActionDeclaration::outputArtifacts))
            .setter(setter(Builder::outputArtifacts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputArtifacts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OutputArtifact> builder(MarshallingType.SDK_POJO)
                                            .constructor(OutputArtifact::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<InputArtifact>> INPUT_ARTIFACTS_FIELD = SdkField
            .<List<InputArtifact>> builder(MarshallingType.LIST)
            .memberName("inputArtifacts")
            .getter(getter(ActionDeclaration::inputArtifacts))
            .setter(setter(Builder::inputArtifacts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputArtifacts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InputArtifact> builder(MarshallingType.SDK_POJO)
                                            .constructor(InputArtifact::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> OUTPUT_VARIABLES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("outputVariables")
            .getter(getter(ActionDeclaration::outputVariables))
            .setter(setter(Builder::outputVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputVariables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(ActionDeclaration::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(ActionDeclaration::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(ActionDeclaration::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<Integer> TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeoutInMinutes").getter(getter(ActionDeclaration::timeoutInMinutes))
            .setter(setter(Builder::timeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeoutInMinutes").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            ACTION_TYPE_ID_FIELD, RUN_ORDER_FIELD, CONFIGURATION_FIELD, COMMANDS_FIELD, OUTPUT_ARTIFACTS_FIELD,
            INPUT_ARTIFACTS_FIELD, OUTPUT_VARIABLES_FIELD, ROLE_ARN_FIELD, REGION_FIELD, NAMESPACE_FIELD,
            TIMEOUT_IN_MINUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final ActionTypeId actionTypeId;

    private final Integer runOrder;

    private final Map<String, String> configuration;

    private final List<String> commands;

    private final List<OutputArtifact> outputArtifacts;

    private final List<InputArtifact> inputArtifacts;

    private final List<String> outputVariables;

    private final String roleArn;

    private final String region;

    private final String namespace;

    private final Integer timeoutInMinutes;

    private ActionDeclaration(BuilderImpl builder) {
        this.name = builder.name;
        this.actionTypeId = builder.actionTypeId;
        this.runOrder = builder.runOrder;
        this.configuration = builder.configuration;
        this.commands = builder.commands;
        this.outputArtifacts = builder.outputArtifacts;
        this.inputArtifacts = builder.inputArtifacts;
        this.outputVariables = builder.outputVariables;
        this.roleArn = builder.roleArn;
        this.region = builder.region;
        this.namespace = builder.namespace;
        this.timeoutInMinutes = builder.timeoutInMinutes;
    }

    /**
     * <p>
     * The action declaration's name.
     * </p>
     * 
     * @return The action declaration's name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Specifies the action type and the provider of the action.
     * </p>
     * 
     * @return Specifies the action type and the provider of the action.
     */
    public final ActionTypeId actionTypeId() {
        return actionTypeId;
    }

    /**
     * <p>
     * The order in which actions are run.
     * </p>
     * 
     * @return The order in which actions are run.
     */
    public final Integer runOrder() {
        return runOrder;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configuration property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfiguration() {
        return configuration != null && !(configuration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The action's configuration. These are key-value pairs that specify input values for an action. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#action-requirements"
     * >Action Structure Requirements in CodePipeline</a>. For the list of configuration properties for the
     * CloudFormation action type in CodePipeline, see <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-action-reference.html"
     * >Configuration Properties Reference</a> in the <i>CloudFormation User Guide</i>. For template snippets with
     * examples, see <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-parameter-override-functions.html"
     * >Using Parameter Override Functions with CodePipeline Pipelines</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * <p>
     * The values can be represented in either JSON or YAML format. For example, the JSON configuration item format is
     * as follows:
     * </p>
     * <p>
     * <i>JSON:</i>
     * </p>
     * <p>
     * <code>"Configuration" : { Key : Value },</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfiguration} method.
     * </p>
     * 
     * @return The action's configuration. These are key-value pairs that specify input values for an action. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#action-requirements"
     *         >Action Structure Requirements in CodePipeline</a>. For the list of configuration properties for the
     *         CloudFormation action type in CodePipeline, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-action-reference.html"
     *         >Configuration Properties Reference</a> in the <i>CloudFormation User Guide</i>. For template snippets
     *         with examples, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-parameter-override-functions.html"
     *         >Using Parameter Override Functions with CodePipeline Pipelines</a> in the <i>CloudFormation User
     *         Guide</i>.</p>
     *         <p>
     *         The values can be represented in either JSON or YAML format. For example, the JSON configuration item
     *         format is as follows:
     *         </p>
     *         <p>
     *         <i>JSON:</i>
     *         </p>
     *         <p>
     *         <code>"Configuration" : { Key : Value },</code>
     */
    public final Map<String, String> configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Commands property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommands() {
        return commands != null && !(commands instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The shell commands to run with your compute action in CodePipeline. All commands are supported except multi-line
     * formats. While CodeBuild logs and permissions are used, you do not need to create any resources in CodeBuild.
     * </p>
     * <note>
     * <p>
     * Using compute time for this action will incur separate charges in CodeBuild.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommands} method.
     * </p>
     * 
     * @return The shell commands to run with your compute action in CodePipeline. All commands are supported except
     *         multi-line formats. While CodeBuild logs and permissions are used, you do not need to create any
     *         resources in CodeBuild.</p> <note>
     *         <p>
     *         Using compute time for this action will incur separate charges in CodeBuild.
     *         </p>
     */
    public final List<String> commands() {
        return commands;
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputArtifacts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputArtifacts() {
        return outputArtifacts != null && !(outputArtifacts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name or ID of the result of the action declaration, such as a test or build artifact.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputArtifacts} method.
     * </p>
     * 
     * @return The name or ID of the result of the action declaration, such as a test or build artifact.
     */
    public final List<OutputArtifact> outputArtifacts() {
        return outputArtifacts;
    }

    /**
     * For responses, this returns true if the service returned a value for the InputArtifacts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInputArtifacts() {
        return inputArtifacts != null && !(inputArtifacts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name or ID of the artifact consumed by the action, such as a test or build artifact.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputArtifacts} method.
     * </p>
     * 
     * @return The name or ID of the artifact consumed by the action, such as a test or build artifact.
     */
    public final List<InputArtifact> inputArtifacts() {
        return inputArtifacts;
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputVariables property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputVariables() {
        return outputVariables != null && !(outputVariables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of variables that are to be exported from the compute action. This is specifically CodeBuild environment
     * variables as used for that action.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputVariables} method.
     * </p>
     * 
     * @return The list of variables that are to be exported from the compute action. This is specifically CodeBuild
     *         environment variables as used for that action.
     */
    public final List<String> outputVariables() {
        return outputVariables;
    }

    /**
     * <p>
     * The ARN of the IAM service role that performs the declared action. This is assumed through the roleArn for the
     * pipeline.
     * </p>
     * 
     * @return The ARN of the IAM service role that performs the declared action. This is assumed through the roleArn
     *         for the pipeline.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The action declaration's Amazon Web Services Region, such as us-east-1.
     * </p>
     * 
     * @return The action declaration's Amazon Web Services Region, such as us-east-1.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The variable namespace associated with the action. All variables produced as output by this action fall under
     * this namespace.
     * </p>
     * 
     * @return The variable namespace associated with the action. All variables produced as output by this action fall
     *         under this namespace.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * A timeout duration in minutes that can be applied against the ActionType’s default timeout value specified in <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html">Quotas for CodePipeline </a>. This
     * attribute is available only to the manual approval ActionType.
     * </p>
     * 
     * @return A timeout duration in minutes that can be applied against the ActionType’s default timeout value
     *         specified in <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html">Quotas for
     *         CodePipeline </a>. This attribute is available only to the manual approval ActionType.
     */
    public final Integer timeoutInMinutes() {
        return timeoutInMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(actionTypeId());
        hashCode = 31 * hashCode + Objects.hashCode(runOrder());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfiguration() ? configuration() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCommands() ? commands() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputArtifacts() ? outputArtifacts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInputArtifacts() ? inputArtifacts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputVariables() ? outputVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMinutes());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActionDeclaration)) {
            return false;
        }
        ActionDeclaration other = (ActionDeclaration) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(actionTypeId(), other.actionTypeId())
                && Objects.equals(runOrder(), other.runOrder()) && hasConfiguration() == other.hasConfiguration()
                && Objects.equals(configuration(), other.configuration()) && hasCommands() == other.hasCommands()
                && Objects.equals(commands(), other.commands()) && hasOutputArtifacts() == other.hasOutputArtifacts()
                && Objects.equals(outputArtifacts(), other.outputArtifacts()) && hasInputArtifacts() == other.hasInputArtifacts()
                && Objects.equals(inputArtifacts(), other.inputArtifacts()) && hasOutputVariables() == other.hasOutputVariables()
                && Objects.equals(outputVariables(), other.outputVariables()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(region(), other.region()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(timeoutInMinutes(), other.timeoutInMinutes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ActionDeclaration").add("Name", name()).add("ActionTypeId", actionTypeId())
                .add("RunOrder", runOrder()).add("Configuration", hasConfiguration() ? configuration() : null)
                .add("Commands", hasCommands() ? commands() : null)
                .add("OutputArtifacts", hasOutputArtifacts() ? outputArtifacts() : null)
                .add("InputArtifacts", hasInputArtifacts() ? inputArtifacts() : null)
                .add("OutputVariables", hasOutputVariables() ? outputVariables() : null).add("RoleArn", roleArn())
                .add("Region", region()).add("Namespace", namespace()).add("TimeoutInMinutes", timeoutInMinutes()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "actionTypeId":
            return Optional.ofNullable(clazz.cast(actionTypeId()));
        case "runOrder":
            return Optional.ofNullable(clazz.cast(runOrder()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "commands":
            return Optional.ofNullable(clazz.cast(commands()));
        case "outputArtifacts":
            return Optional.ofNullable(clazz.cast(outputArtifacts()));
        case "inputArtifacts":
            return Optional.ofNullable(clazz.cast(inputArtifacts()));
        case "outputVariables":
            return Optional.ofNullable(clazz.cast(outputVariables()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "timeoutInMinutes":
            return Optional.ofNullable(clazz.cast(timeoutInMinutes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ActionDeclaration, T> g) {
        return obj -> g.apply((ActionDeclaration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActionDeclaration> {
        /**
         * <p>
         * The action declaration's name.
         * </p>
         * 
         * @param name
         *        The action declaration's name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Specifies the action type and the provider of the action.
         * </p>
         * 
         * @param actionTypeId
         *        Specifies the action type and the provider of the action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionTypeId(ActionTypeId actionTypeId);

        /**
         * <p>
         * Specifies the action type and the provider of the action.
         * </p>
         * This is a convenience method that creates an instance of the {@link ActionTypeId.Builder} avoiding the need
         * to create one manually via {@link ActionTypeId#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ActionTypeId.Builder#build()} is called immediately and its
         * result is passed to {@link #actionTypeId(ActionTypeId)}.
         * 
         * @param actionTypeId
         *        a consumer that will call methods on {@link ActionTypeId.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actionTypeId(ActionTypeId)
         */
        default Builder actionTypeId(Consumer<ActionTypeId.Builder> actionTypeId) {
            return actionTypeId(ActionTypeId.builder().applyMutation(actionTypeId).build());
        }

        /**
         * <p>
         * The order in which actions are run.
         * </p>
         * 
         * @param runOrder
         *        The order in which actions are run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runOrder(Integer runOrder);

        /**
         * <p>
         * The action's configuration. These are key-value pairs that specify input values for an action. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#action-requirements"
         * >Action Structure Requirements in CodePipeline</a>. For the list of configuration properties for the
         * CloudFormation action type in CodePipeline, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-action-reference.html"
         * >Configuration Properties Reference</a> in the <i>CloudFormation User Guide</i>. For template snippets with
         * examples, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-parameter-override-functions.html"
         * >Using Parameter Override Functions with CodePipeline Pipelines</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * <p>
         * The values can be represented in either JSON or YAML format. For example, the JSON configuration item format
         * is as follows:
         * </p>
         * <p>
         * <i>JSON:</i>
         * </p>
         * <p>
         * <code>"Configuration" : { Key : Value },</code>
         * </p>
         * 
         * @param configuration
         *        The action's configuration. These are key-value pairs that specify input values for an action. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#action-requirements"
         *        >Action Structure Requirements in CodePipeline</a>. For the list of configuration properties for the
         *        CloudFormation action type in CodePipeline, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-action-reference.html"
         *        >Configuration Properties Reference</a> in the <i>CloudFormation User Guide</i>. For template snippets
         *        with examples, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline-parameter-override-functions.html"
         *        >Using Parameter Override Functions with CodePipeline Pipelines</a> in the <i>CloudFormation User
         *        Guide</i>.</p>
         *        <p>
         *        The values can be represented in either JSON or YAML format. For example, the JSON configuration item
         *        format is as follows:
         *        </p>
         *        <p>
         *        <i>JSON:</i>
         *        </p>
         *        <p>
         *        <code>"Configuration" : { Key : Value },</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(Map<String, String> configuration);

        /**
         * <p>
         * The shell commands to run with your compute action in CodePipeline. All commands are supported except
         * multi-line formats. While CodeBuild logs and permissions are used, you do not need to create any resources in
         * CodeBuild.
         * </p>
         * <note>
         * <p>
         * Using compute time for this action will incur separate charges in CodeBuild.
         * </p>
         * </note>
         * 
         * @param commands
         *        The shell commands to run with your compute action in CodePipeline. All commands are supported except
         *        multi-line formats. While CodeBuild logs and permissions are used, you do not need to create any
         *        resources in CodeBuild.</p> <note>
         *        <p>
         *        Using compute time for this action will incur separate charges in CodeBuild.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commands(Collection<String> commands);

        /**
         * <p>
         * The shell commands to run with your compute action in CodePipeline. All commands are supported except
         * multi-line formats. While CodeBuild logs and permissions are used, you do not need to create any resources in
         * CodeBuild.
         * </p>
         * <note>
         * <p>
         * Using compute time for this action will incur separate charges in CodeBuild.
         * </p>
         * </note>
         * 
         * @param commands
         *        The shell commands to run with your compute action in CodePipeline. All commands are supported except
         *        multi-line formats. While CodeBuild logs and permissions are used, you do not need to create any
         *        resources in CodeBuild.</p> <note>
         *        <p>
         *        Using compute time for this action will incur separate charges in CodeBuild.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commands(String... commands);

        /**
         * <p>
         * The name or ID of the result of the action declaration, such as a test or build artifact.
         * </p>
         * 
         * @param outputArtifacts
         *        The name or ID of the result of the action declaration, such as a test or build artifact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputArtifacts(Collection<OutputArtifact> outputArtifacts);

        /**
         * <p>
         * The name or ID of the result of the action declaration, such as a test or build artifact.
         * </p>
         * 
         * @param outputArtifacts
         *        The name or ID of the result of the action declaration, such as a test or build artifact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputArtifacts(OutputArtifact... outputArtifacts);

        /**
         * <p>
         * The name or ID of the result of the action declaration, such as a test or build artifact.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.OutputArtifact.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codepipeline.model.OutputArtifact#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.OutputArtifact.Builder#build()} is called
         * immediately and its result is passed to {@link #outputArtifacts(List<OutputArtifact>)}.
         * 
         * @param outputArtifacts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.OutputArtifact.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputArtifacts(java.util.Collection<OutputArtifact>)
         */
        Builder outputArtifacts(Consumer<OutputArtifact.Builder>... outputArtifacts);

        /**
         * <p>
         * The name or ID of the artifact consumed by the action, such as a test or build artifact.
         * </p>
         * 
         * @param inputArtifacts
         *        The name or ID of the artifact consumed by the action, such as a test or build artifact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputArtifacts(Collection<InputArtifact> inputArtifacts);

        /**
         * <p>
         * The name or ID of the artifact consumed by the action, such as a test or build artifact.
         * </p>
         * 
         * @param inputArtifacts
         *        The name or ID of the artifact consumed by the action, such as a test or build artifact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputArtifacts(InputArtifact... inputArtifacts);

        /**
         * <p>
         * The name or ID of the artifact consumed by the action, such as a test or build artifact.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact.Builder#build()} is called
         * immediately and its result is passed to {@link #inputArtifacts(List<InputArtifact>)}.
         * 
         * @param inputArtifacts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputArtifacts(java.util.Collection<InputArtifact>)
         */
        Builder inputArtifacts(Consumer<InputArtifact.Builder>... inputArtifacts);

        /**
         * <p>
         * The list of variables that are to be exported from the compute action. This is specifically CodeBuild
         * environment variables as used for that action.
         * </p>
         * 
         * @param outputVariables
         *        The list of variables that are to be exported from the compute action. This is specifically CodeBuild
         *        environment variables as used for that action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputVariables(Collection<String> outputVariables);

        /**
         * <p>
         * The list of variables that are to be exported from the compute action. This is specifically CodeBuild
         * environment variables as used for that action.
         * </p>
         * 
         * @param outputVariables
         *        The list of variables that are to be exported from the compute action. This is specifically CodeBuild
         *        environment variables as used for that action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputVariables(String... outputVariables);

        /**
         * <p>
         * The ARN of the IAM service role that performs the declared action. This is assumed through the roleArn for
         * the pipeline.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the IAM service role that performs the declared action. This is assumed through the roleArn
         *        for the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The action declaration's Amazon Web Services Region, such as us-east-1.
         * </p>
         * 
         * @param region
         *        The action declaration's Amazon Web Services Region, such as us-east-1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The variable namespace associated with the action. All variables produced as output by this action fall under
         * this namespace.
         * </p>
         * 
         * @param namespace
         *        The variable namespace associated with the action. All variables produced as output by this action
         *        fall under this namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * A timeout duration in minutes that can be applied against the ActionType’s default timeout value specified in
         * <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html">Quotas for CodePipeline </a>.
         * This attribute is available only to the manual approval ActionType.
         * </p>
         * 
         * @param timeoutInMinutes
         *        A timeout duration in minutes that can be applied against the ActionType’s default timeout value
         *        specified in <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html">Quotas
         *        for CodePipeline </a>. This attribute is available only to the manual approval ActionType.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMinutes(Integer timeoutInMinutes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private ActionTypeId actionTypeId;

        private Integer runOrder;

        private Map<String, String> configuration = DefaultSdkAutoConstructMap.getInstance();

        private List<String> commands = DefaultSdkAutoConstructList.getInstance();

        private List<OutputArtifact> outputArtifacts = DefaultSdkAutoConstructList.getInstance();

        private List<InputArtifact> inputArtifacts = DefaultSdkAutoConstructList.getInstance();

        private List<String> outputVariables = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private String region;

        private String namespace;

        private Integer timeoutInMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(ActionDeclaration model) {
            name(model.name);
            actionTypeId(model.actionTypeId);
            runOrder(model.runOrder);
            configuration(model.configuration);
            commands(model.commands);
            outputArtifacts(model.outputArtifacts);
            inputArtifacts(model.inputArtifacts);
            outputVariables(model.outputVariables);
            roleArn(model.roleArn);
            region(model.region);
            namespace(model.namespace);
            timeoutInMinutes(model.timeoutInMinutes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final ActionTypeId.Builder getActionTypeId() {
            return actionTypeId != null ? actionTypeId.toBuilder() : null;
        }

        public final void setActionTypeId(ActionTypeId.BuilderImpl actionTypeId) {
            this.actionTypeId = actionTypeId != null ? actionTypeId.build() : null;
        }

        @Override
        public final Builder actionTypeId(ActionTypeId actionTypeId) {
            this.actionTypeId = actionTypeId;
            return this;
        }

        public final Integer getRunOrder() {
            return runOrder;
        }

        public final void setRunOrder(Integer runOrder) {
            this.runOrder = runOrder;
        }

        @Override
        public final Builder runOrder(Integer runOrder) {
            this.runOrder = runOrder;
            return this;
        }

        public final Map<String, String> getConfiguration() {
            if (configuration instanceof SdkAutoConstructMap) {
                return null;
            }
            return configuration;
        }

        public final void setConfiguration(Map<String, String> configuration) {
            this.configuration = ActionConfigurationMapCopier.copy(configuration);
        }

        @Override
        public final Builder configuration(Map<String, String> configuration) {
            this.configuration = ActionConfigurationMapCopier.copy(configuration);
            return this;
        }

        public final Collection<String> getCommands() {
            if (commands instanceof SdkAutoConstructList) {
                return null;
            }
            return commands;
        }

        public final void setCommands(Collection<String> commands) {
            this.commands = CommandListCopier.copy(commands);
        }

        @Override
        public final Builder commands(Collection<String> commands) {
            this.commands = CommandListCopier.copy(commands);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder commands(String... commands) {
            commands(Arrays.asList(commands));
            return this;
        }

        public final List<OutputArtifact.Builder> getOutputArtifacts() {
            List<OutputArtifact.Builder> result = OutputArtifactListCopier.copyToBuilder(this.outputArtifacts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputArtifacts(Collection<OutputArtifact.BuilderImpl> outputArtifacts) {
            this.outputArtifacts = OutputArtifactListCopier.copyFromBuilder(outputArtifacts);
        }

        @Override
        public final Builder outputArtifacts(Collection<OutputArtifact> outputArtifacts) {
            this.outputArtifacts = OutputArtifactListCopier.copy(outputArtifacts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputArtifacts(OutputArtifact... outputArtifacts) {
            outputArtifacts(Arrays.asList(outputArtifacts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputArtifacts(Consumer<OutputArtifact.Builder>... outputArtifacts) {
            outputArtifacts(Stream.of(outputArtifacts).map(c -> OutputArtifact.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<InputArtifact.Builder> getInputArtifacts() {
            List<InputArtifact.Builder> result = InputArtifactListCopier.copyToBuilder(this.inputArtifacts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInputArtifacts(Collection<InputArtifact.BuilderImpl> inputArtifacts) {
            this.inputArtifacts = InputArtifactListCopier.copyFromBuilder(inputArtifacts);
        }

        @Override
        public final Builder inputArtifacts(Collection<InputArtifact> inputArtifacts) {
            this.inputArtifacts = InputArtifactListCopier.copy(inputArtifacts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputArtifacts(InputArtifact... inputArtifacts) {
            inputArtifacts(Arrays.asList(inputArtifacts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputArtifacts(Consumer<InputArtifact.Builder>... inputArtifacts) {
            inputArtifacts(Stream.of(inputArtifacts).map(c -> InputArtifact.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getOutputVariables() {
            if (outputVariables instanceof SdkAutoConstructList) {
                return null;
            }
            return outputVariables;
        }

        public final void setOutputVariables(Collection<String> outputVariables) {
            this.outputVariables = OutputVariableListCopier.copy(outputVariables);
        }

        @Override
        public final Builder outputVariables(Collection<String> outputVariables) {
            this.outputVariables = OutputVariableListCopier.copy(outputVariables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputVariables(String... outputVariables) {
            outputVariables(Arrays.asList(outputVariables));
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final Integer getTimeoutInMinutes() {
            return timeoutInMinutes;
        }

        public final void setTimeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
        }

        @Override
        public final Builder timeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
            return this;
        }

        @Override
        public ActionDeclaration build() {
            return new ActionDeclaration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
